;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Package: CLIM-USER; Base: 10; Lowercase: Yes -*-

(in-package :clim-user)

;; An OUTPUT-DESTINATION is a type whose printed representation consists
;; of the type of output destination (Window, File, Printer) followed by
;; another argument (a pathname or printer name).  The two fields are
;; separated by a space.
(define-presentation-type output-destination ())
     
;; In a real application, this would probably be something else...
(defvar *printer-name-alist*
	`(("Boston Globe" boston-globe)
	  ("New York Times" new-york-times)
	  ("Pravda" pravda)
	  ("Boston Comic News" comic-news)))

;; ...and backquoting into here is not so hot either
(defparameter *destination-type-alist*
  `((:window :value (:window nil nil nil))
    (:pathname :value (:pathname pathname "pathname" "Enter a pathname"))
    (:printer :value (:printer (member-alist ,*printer-name-alist*)
		      "printer" "Enter the name of a printer"))))

(clim:define-presentation-method clim:accept 
    ((type output-destination) stream (view clim:textual-view) &key)
  ;; Since #\Space separates the fields, make it a delimiter gesture
  ;; so that the calls to ACCEPT will terminate when the user types
  ;; a space character.  This allows allows this type to interact
  ;; correctly with command-line input.
  (clim:with-delimiter-gestures (#\Space)
    (let (dtype place delimiter)
      ;; Read the destination type using ACCEPT.  Establish a "help"
      ;; context to prompt the user to enter a destination type, otherwise
      ;; ACCEPT will just tell the user to enter a MEMBER-ALIST.
      (clim:with-accept-help 
	  ((:subhelp #'(lambda (stream action string)
			 (declare (ignore action string))
			 (write-string "Enter the destination type." stream))))
        (setq dtype (clim:accept `(clim:member-alist ,*destination-type-alist*)
				 :stream stream :view view :prompt "type")))
      ;; DTYPE has information regarding the next type to read in
      (destructuring-bind (dtype type prompt help) dtype
	(when (eql type nil)
	  (return-from clim:accept (list dtype nil)))
	;; Read the delimiter -- it should be a space, but if it is not,
	;; signal a parse error.
	(setq delimiter (stream-peek-char stream))
	(cond ((char-equal delimiter #\Space)
	       ;; The delimiter was a space, so remove it from the input
	       ;; buffer and read the next integer.
	       (clim:stream-read-char stream)
	       (clim:with-accept-help 
		   ((:subhelp #'(lambda (stream action string)
				  (declare (ignore action string))
				  (write-string help stream))))
		 (setq place (clim:accept type
					  :stream stream :view view :prompt prompt))))
	      (t (clim:simple-parse-error "Invalid delimiter: ~S" delimiter)))
	;; Return the result, leaving the final delimiter in place.
	(list dtype place)))))

(clim:define-presentation-method clim:present
    (object (type output-destination) stream (view clim:textual-view) &key)
  ;; Just print the two parts of the object separated by a space.
  (destructuring-bind (dtype place) object
    (if place
	(format stream "~:(~A~) ~A" dtype place)
	(format stream "~:(~A~)" dtype))))

;; Only lists of two elements whose first element is one of :WINDOW,
;; :PATHNAME, or :PRINTER are of this type
(clim:define-presentation-method clim:presentation-typep
    (object (type output-destination))
  (and (listp object)
       (= (length object) 2)
       (member (first object) '(:window :pathname :printer))
       (not (null (assoc (first object) *destination-type-alist*)))))
