;;; -*- Mode: Lisp; Syntax: ANSI-Common-Lisp; Package: CLIM-USER; Base: 10; Lowercase: Yes -*-

(in-package :clim-user)

(defun squares-by-rows (&optional (stream *standard-output*))
  ;; The output within this is in tabular form
  (clim:formatting-table (stream :x-spacing '(2 :character))
    ;; The output within this constitutes a row
    (clim:formatting-row (stream)
      (clim:with-text-face (stream :italic)
        ;; These are the cells in the row
	(clim:formatting-cell (stream :align-x :center) "N")
        (clim:formatting-cell (stream :align-x :center) "N**2")
        (clim:formatting-cell (stream :align-x :center) "(sqrt N)")))
    (do ((i 1 (1+ i)))
        ((> i 10))
      ;; The output within this constitutes a row
      (clim:formatting-row (stream)
        ;; The next three are the cells in the row
        (clim:formatting-cell (stream :align-x :right)
          (format stream "~D" i))
        (clim:formatting-cell (stream :align-x :right)
          (format stream "~D" (* i i)))
        (clim:formatting-cell (stream :align-x :right)
          (format stream "~4$" (sqrt i)))))))

(defun squares-by-columns (&optional (stream *standard-output*))
  (clim:formatting-table (stream :x-spacing '(2 :character))
    ;; The output within this constitutes a column
    (clim:formatting-column (stream)
      (clim:with-text-face (stream :italic)
	;; These are the cells in the column
	(clim:formatting-cell (stream :align-x :center) "N")
        (clim:formatting-cell (stream :align-x :center) "N**2")
        (clim:formatting-cell (stream :align-x :center) "(sqrt N)")))
    (do ((i 1 (1+ i)))
        ((> i 10))
      ;; The output within this constitutes a column
      (clim:formatting-column (stream)
	;; The next three are the cells in the column
        (clim:formatting-cell (stream :align-x :right)
          (format stream "~D" i))
        (clim:formatting-cell (stream :align-x :right)
          (format stream "~D" (* i i)))
        (clim:formatting-cell (stream :align-x :right)
          (format stream "~4$" (sqrt i)))))))

;; Nested table formatting, with borders
(defun multiplication-tables (&optional (stream *standard-output*))
  (flet ((table (stream factor)
           (clim:surrounding-output-with-border (stream)
             ;; Indicate that we want the table broken into three columns
	     (clim:formatting-table (stream :multiple-columns 3)
               (do ((i 1 (1+ i)))
                   ((> i 9))
                 (clim:formatting-row (stream)
                   (clim:formatting-cell (stream :align-x :right)
                     (format stream "~D" (* i factor)))))))))
    ;; Three columnsm with nice spacing for the borders
    (clim:formatting-table (stream :multiple-columns 3
                                   :y-spacing 10 :x-spacing 10)
      (do ((i 1 (1+ i)))
          ((> i 9))
        (clim:formatting-row (stream)
          (clim:formatting-cell (stream :align-x :center)
            (table stream i)))))))


;;; A poor implementation of a tabular input using tables and dialogs
;;; Not recommended for spreadsheets, but it uses interesting techniques

(defparameter *default-cell-width* '(6 :character))

;; A spreadsheet consists of a bunch of rows, a row consisting of column
;; sums, and a total
(defclass spreadsheet ()
    ((rows :reader spreadsheet-rows)
     (column-sums :reader spreadsheet-column-sums)
     (total :accessor spreadsheet-total :initform 0.0)
     (cell-width :reader spreadsheet-cell-width
		 :initarg :cell-width :initform *default-cell-width*)))

(defmethod initialize-instance :after ((spreadsheet spreadsheet)
				       &key nrows ncells &allow-other-keys)
  (with-slots (rows column-sums) spreadsheet
    (setf rows (make-array nrows))
    ;; Initialize all the requested rows
    (dotimes (i nrows)
      (setf (aref rows i) (make-instance 'spreadsheet-row :ncells ncells)))
    ;; Initialize the totals row
    (setf column-sums (make-instance 'spreadsheet-row :ncells ncells))))

;; A row consists of a bunch of cells and a row sum.
(defclass spreadsheet-row ()
    ((cells :reader spreadsheet-row-cells)
     (row-sum :accessor spreadsheet-row-sum :initform 0.0)))
  
(defmethod initialize-instance :after ((row spreadsheet-row)
				       &key ncells &allow-other-keys)
  (with-slots (cells) row
    (setq cells (make-array ncells :element-type 'float :initial-element 0.0))))

;; Formatting a spreadsheet row consists of getting as input a value
;; for each cell in the row, using as a default the value that is
;; there right now.
(defun format-spreadsheet-row (row stream cell-width)
  (let ((cells (spreadsheet-row-cells row)))
    (clim:formatting-row (stream)
      (dotimes (i (length cells))
	;; Each cell has its own call to ACCEPT
	(clim:formatting-cell (stream :align-x :right :min-width cell-width)
	  (setf (aref cells i)
		(clim:accept 'float
			     :default (aref cells i)
			     :prompt nil :prompt-mode :raw
			     :query-identifier (list cells i)
			     :stream stream))))
      (present-sum (spreadsheet-row-sum row) stream cell-width))
    (setf (spreadsheet-row-sum row) (reduce #'+ cells))))

(defun present-sum (sum stream cell-width)
  (clim:formatting-cell (stream :align-x :right :min-width cell-width)
    (clim:with-text-face (stream :bold)
      (clim:present sum 'float :stream stream))))

(defun spreadsheet (nrows ncells
		    &key (cell-width *default-cell-width*) (stream *query-io*))
  (let* ((spreadsheet (make-instance 'spreadsheet
				     :nrows nrows :ncells ncells
				     :cell-width cell-width))
	 (rows (spreadsheet-rows spreadsheet))
	 (column-totals (spreadsheet-row-cells (spreadsheet-column-sums spreadsheet))))
    ;; Need :RESYNCHRONIZE-EVERY-PASS T to get the totals recomputed
    (clim:accepting-values (stream :own-window nil
				   :resynchronize-every-pass t)
      (clim:formatting-table (stream :x-spacing '(2 :character)
				     :equalize-column-widths t)
	(map nil #'(lambda (row)
		     (format-spreadsheet-row row stream cell-width)) rows)
	(clim:formatting-row (stream)
	  (map nil #'(lambda (cell)
		       (present-sum cell stream cell-width)) column-totals)
	  (present-sum (spreadsheet-total spreadsheet) stream cell-width)))
      (setf (spreadsheet-total spreadsheet) 0.0)
      (dotimes (i ncells)
	(setf (aref column-totals i) 0.0)
	(dotimes (j nrows)
	  (incf (aref column-totals i) (aref (spreadsheet-row-cells (aref rows j)) i)))
	(incf (spreadsheet-total spreadsheet) (aref column-totals i))))
    spreadsheet))


;; Draws a class graph with multiple roots
(defun graph-classes (classes &optional (orientation :horizontal)
                                        (stream *standard-output*))
  (clim:format-graph-from-roots 
    (mapcar #'clos:find-class classes)
    #'(lambda (class stream)
        (clim:surrounding-output-with-border (stream)
          (format stream "~S" (clos:class-name class))))
    #'clos:class-direct-subclasses
    :merge-duplicates t :orientation orientation
    :stream stream))

;; Using redisplay with graphs
(defun redisplay-graph (&optional (stream *standard-output*))
  (macrolet ((make-node (&key name children)
	       `(list* ,name ,children))
	     (node-name (node)
	       `(car ,node))
	     (node-children (node)
	       `(cdr ,node)))
    (let* ((3a (make-node :name "3A"))
	   (3b (make-node :name "3B"))
	   (2a (make-node :name "2A"))
	   (2b (make-node :name "2B"))
	   (2c (make-node :name "2C"))
	   (1a (make-node :name "1A" :children (list 2a 2b)))
	   (1b (make-node :name "1B" :children (list 2b 2c)))
	   (root (make-node :name "0" :children (list 1a 1b)))
	   (graph
	     (clim:updating-output (stream :unique-id root)
	       (clim:format-graph-from-root
		 root
		 #'(lambda (node s)
		     (clim:updating-output (s :cache-value node)
		       (write-string (node-name node) s)))
		 #'cdr				;i.e., #'node-children
		 :stream stream))))
      (loop
	(sleep 2)
	(setf (node-children 2b) (list 3a 3b))
	(clim:redisplay graph stream)
	(sleep 2)
	(setf (node-children 2b) nil)
	(clim:redisplay graph stream)))))


(defparameter *gettysburg-address*
  (concatenate 'string
    "Fourscore and seven years ago our forefathers brought forth "
    "on this continent a new nation, conceived in Liberty, and "
    "dedicated to the proposition that all men are created equal. "
    "Now we are engaged in a great civil war, testing whether that "
    "nation, or any nation so conceived and so dedicated, can long "
    "endure.  We are met on a great battlefield of that war.  We have "
    "come to dedicate a portion of that field, as a final resting "
    "place for those who here gave their lives that that nation might "
    "live.  It is altogether fitting and proper that we do so.  But, "
    "in a larger sense, we cannot dedicate -- we cannot consecrate -- "
    "we cannot hallow this ground.  The brave men, living and dead, "
    "who struggled here, have consecrated it far above our poor power "
    "to add or detract.  The world will little note, nor long "
    "remember, what we say here, but it can never forget what they "
    "did here.  It is for us, the living, rather, to be dedicated "
    "here to the unfinished work which they who fought here have thus "
    "far so nobly advanced.  It is rather for us to be here dedicated "
    "to the great task remaining before us -- that from these honored "
    "dead we take increased devotion to that cause for which they "
    "gave the last full measure of devotion -- that we here highly "
    "resolve that these dead shall not have died in vain -- that this "
    "nation, under God, shall have a new birth of freedom -- and that "
    "government of the people, by the people, and for the people, "
    "shall not perish from the earth."))

(defun fill-gettysburg (&optional (stream *standard-output*))
  (clim:filling-output (stream :fill-width '(60 :character))
    (write-string *gettysburg-address* stream))
  (values))

(defparameter *some-names*
  '("Sleepy" "Sneezy" "Sleazy" "Happy" "Grumpy" "Dopey" "Doc"
    "Dasher" "Dancer" "Donner" "Blitzen" "Bashful"))

(defun fill-some-names (&optional (stream *standard-output*))
  (clim:filling-output (stream :fill-width '(40 :character)
			       :after-line-break "  ")
    (clim:format-textual-list *some-names* #'princ
			      :stream stream
			      :separator ", " :conjunction "and")
    (write-char #\. stream)))
