;;; -*- Mode: LISP; Syntax: Common-Lisp; Package: CLIM; Base: 10; Lowercase: Yes -*-

|#
Date: Tue, 17 Aug 1993 10:24 -0400
From: Scott McKay <SWM@stony-brook.scrc.symbolics.com>
Subject: Drag and drop translators for CLIM 1.1
To: clim@bbn.com
Message-Id: <19930817142415.7.SWM@SUMMER.SCRC.Symbolics.COM>

At Stefan Bernemann's request, I put my old CLIM 1.1 implementation of
drag and drop translators into the CLIM library in the following file:

  BRAZIL.CAMBRIDGE.APPLE.COM:/pub/clim/clim-1/drag-and-drop.lisp

I haven't tested this in a while, and I have made a bunch of improvements
to the version that went into CLIM 2.0, but somebody may find it useful
anyway.  Enjoy.
|#

(in-package "CLIM")

(export 'clim::(define-drag-and-drop-translator
		frame-drag-and-drop-feedback
		frame-drag-and-drop-highlighting)
	'clim)

;;; Direct manipulation (click and drag) translators

;;--- Kludge until we can get ":LEFT :HOLD" working...
(define-gesture-name :select-hold :button :left :shifts (:shift))

(defclass drag-and-drop-translator (presentation-translator)
    ((destination-type :initarg :destination-type
		       :reader presentation-translator-destination-type)
     (real-body :initarg :real-body
		:reader presentation-translator-real-body)
     (real-tester :initform nil :initarg :real-tester
		  :reader presentation-translator-real-tester)
     (real-documentation :initform nil :initarg :real-documentation
			 :reader presentation-translator-real-documentation)
     (feedback :initform nil :initarg :feedback
	       :reader presentation-translator-feedback)
     (highlighting :initform nil :initarg :highlighting
		   :reader presentation-translator-highlighting)))

(defmacro define-drag-and-drop-translator
	  (name
	   (from-type to-type destination-type command-table
	    &key (gesture ':select-hold) tester
		 documentation (menu t) priority
		 feedback highlighting)
	   arglist
	   &body body)
  #+Genera (declare (zwei:indentation 1 3 3 1))
  (with-warnings-for-definition name define-presentation-translator
    (multiple-value-bind (body-function body-name)
	(write-translator-function (cons arglist body) name 'real-body
				   '(destination-object destination-presentation))
      (multiple-value-bind (tester-function tester-name)
	  (when tester
	    (write-translator-function tester name 'real-tester
				       '(destination-object destination-presentation)))
	(multiple-value-bind (doc-function doc-name)
	    (if (stringp documentation)
		(values nil documentation)
	        (when documentation
		  (write-translator-function 
		    documentation name 'real-doc
		    '(destination-object destination-presentation stream))))
	  (multiple-value-bind (feedback-function feedback-name)
	      (if (symbolp feedback)
		  (values nil feedback)
		  (write-drag-and-drop-function 
		    feedback name 'feedback
		    '(frame presentation stream initial-x initial-y new-x new-y state)))
	    (multiple-value-bind (highlighting-function highlighting-name)
		(if (symbolp highlighting)
		    (values nil highlighting)
		    (write-drag-and-drop-function
		      highlighting name 'highlighting
		      '(frame presentation stream state)))
	      (when (null feedback)
		(setq feedback 'frame-drag-and-drop-feedback))
	      (when (null highlighting)
		(setq highlighting 'frame-drag-and-drop-highlighting))
	      `(progn
		 (define-presentation-translator-1 ,name
		     (,from-type ,to-type ,command-table
		      :gesture ,gesture
		      :tester nil
		      :tester-definitive t
		      :documentation "Drag presentation"
		      :pointer-documentation "Drag presentation"
		      :menu ,menu
		      :priority ,priority
		      :translator-class drag-and-drop-translator
		      :destination-type ',destination-type
		      :real-body ',body-name
		      :feedback ,feedback
		      :highlighting ,highlighting
		      ,@(when tester `(:real-tester ',tester-name))
		      ,@(when documentation `(:real-documentation ',doc-name)))
		     ,*translator-function-arglist*
		   (drag-and-drop-translator-internal ,@*translator-function-arglist*))
		 ,body-function
		 ,tester-function
		 ,doc-function
		 ,feedback-function
		 ,highlighting-function))))))))

(defun write-drag-and-drop-function (function translator-name clause-name args)
  (let ((function-name (gensymbol translator-name clause-name))
	(arglist (first function))
	(body (rest function)))
    (multiple-value-bind (arglist ignores)
	(canonicalize-and-match-lambda-lists args arglist)
      (values `(defun ,function-name ,arglist
		 ,@(and ignores `((declare (ignore ,@ignores))))
		 ,@body)
	      function-name))))

;; All the drag-and-drop translators that satisfy the current from-type,
;; to-type, and gesture will get here.  However, the translator now being
;; run is not necessarily the right one.  What we do here is to track the 
;; pointer until the destination presentation has been reached, choose the
;; correct translator and run it.
(defun drag-and-drop-translator-internal (object presentation context-type
					  frame event window x y)
  (let* ((translators
	   (let ((all-translators
		   (find-presentation-translators
		     (presentation-type presentation) context-type frame)))
	     ;; This find all translators that could conceivably apply
	     (mapcan #'(lambda (translator)
			 (when (and (typep translator 'drag-and-drop-translator)
				    (test-presentation-translator
				      translator presentation context-type
				      frame window x y :event event))
			   (list translator)))
		     all-translators)))
	 (destination-types
	   ;; So we can let TRACKING-POINTER do the work of finding applicable
	   ;; destination presentations...
	   (mapcar #'presentation-translator-destination-type translators))
	 ;; Get the feedback and highlighting functions
	 ;;--- This should really check that they are all the same
	 (feedback
	   (and translators (presentation-translator-feedback (first translators))))
	 (highlighting
	   (and translators (presentation-translator-highlighting (first translators))))
	 (initial-x x)
	 (initial-y y)
	 last-x last-y
	 (from-presentation presentation)
	 ;; The presentation we finally landed on, and the event used
	 (destination nil)
	 (destination-event nil)
	 ;; Defeat the normal gesture processing stuff
	 (*input-wait-handler* nil)
	 (*input-wait-test* nil)
	 (*pointer-button-press-handler* nil))
    (read-gesture :stream window :timeout 0)	;eat the initial mouse click
    (unhighlight-highlighted-presentation window)
    (when (null translators)
      (return-from drag-and-drop-translator-internal nil))
    (flet ((find-translator (destination)
	     (let* ((translator
		      (find (presentation-type destination) translators
			    :key #'presentation-translator-destination-type
			    :test #'presentation-subtypep))
		    (tester
		      (and translator (presentation-translator-real-tester translator))))
	       (when (or (null tester)
			 (funcall tester
				  object presentation context-type
				  frame event window x y
				  (presentation-object destination) destination))
		 translator))))
      (declare (dynamic-extent #'find-translator))
      (macrolet ((feedback (x y state)
		   `(funcall feedback frame from-presentation window 
				      initial-x initial-y ,x ,y ,state))
		 (highlight (presentation state)
		   `(when ,presentation
		      (funcall highlighting frame ,presentation window 
					    ,state))))
	(block drag-presentation
	  (tracking-pointer (window :context-type `((or ,@destination-types))
				    :multiple-window t)
	    (:presentation-button-press (presentation event)
	     (highlight destination :unhighlight)
	     (setq destination presentation
		   destination-event event)
	     (return-from drag-presentation))
	    (:pointer-button-press (event)
	     (highlight destination :unhighlight)
	     (setq destination nil
		   destination-event event)
	     (return-from drag-presentation))
	    (:presentation (presentation window x y)
	     (when last-x
	       (feedback last-x last-y :unhighlight))
	     (setq last-x x
		   last-y y)
	     (feedback x y :highlight)
	     (when (not (eq presentation destination))
	       (highlight destination :unhighlight)
	       (setq destination presentation)
	       (highlight presentation :highlight)
	       #+++ignore				;--- documentation
	       (let ((translator (find-translator destination)))
		 (when translator
		   (let ((documentation
			   (presentation-translator-real-documentation translator)))
		     (if (stringp documentation)
			 (write-string documentation doc-stream)
		         (funcall documentation 
				  (presentation-object presentation) presentation context-type
				  frame event window x y
				  (presentation-object destination) destination
				  doc-stream)))))))
	    (:pointer-motion (window x y)
	     (highlight destination :unhighlight)
	     (setq destination nil)
	     (when last-x
	       (feedback last-x last-y :unhighlight))
	     (setq last-x x
		   last-y y)
	     (feedback x y :highlight))))
	(when last-x
	  (feedback last-x last-y :unhighlight)))
      ;; The user has put down the presentation, figure out what to do
      ;;--- What if there is more than one translator?
      (let ((translator (find-translator destination)))
	(when translator
	  (multiple-value-bind (result-object result-type)
	      (funcall (presentation-translator-real-body translator)
		       object presentation context-type
		       frame event window x y
		       (presentation-object destination) destination)
	    (setq result-type (or result-type (evacuate-list context-type)))
	    ;; Find the tag to throw to, and do so
	    (dolist (this-context *input-context*)
	      (let* ((context-type (input-context-type this-context))
		     (tag (input-context-tag this-context)))
		(when (presentation-subtypep result-type context-type)
		  (throw tag
		    (values result-object result-type destination-event)))))))))))

;; NEW-X and NEW-Y are in stream coordinates.
(defmethod frame-drag-and-drop-feedback ((frame application-frame) presentation stream
					 initial-x initial-y new-x new-y state)
  (declare (ignore initial-x initial-y state))	;we'll just use XOR
  (multiple-value-bind (width height) (bounding-rectangle-size presentation)
    (with-output-recording-options (stream :record-p nil)
      (draw-rectangle* stream new-x new-y (+ new-x width) (+ new-y height)
		       :filled nil :ink +flipping-ink+ :line-dashes #(4 4)))))

(defmethod frame-drag-and-drop-highlighting ((frame application-frame) presentation stream
					     state)
  (ecase state
    (:highlight
      (set-highlighted-presentation stream presentation))
    (:unhighlight
      (unhighlight-highlighted-presentation stream))))

;; Used to extract all presentations of a certain type in a "surrounded region"
(defun presentations-of-type-in-region (type stream region &optional (sort t))
  (let ((collection nil))
    (labels ((collect (record x-offset y-offset)
	       (if (presentationp record)
		   (when (presentation-typep (presentation-object record) type)
		     (pushnew record collection))
		 (multiple-value-bind (xoff yoff)
		     (output-record-start-position* record)
		   (map-over-output-record-elements-overlapping-region 
		     record region #'collect (+ x-offset xoff) (+ y-offset yoff))))))
      (declare (dynamic-extent #'collect))
      (collect (output-recording-stream-output-record stream) 0 0)
      (when (and collection sort)
	(setq collection (sort-presentations-by-position collection)))
      collection)))

(defun sort-presentations-by-position (presentations)
  (flet ((position-lessp (p1 p2)
	   (unless (eq (output-record-parent p1) (output-record-parent p2))
	     ;;--- Convert to absolute coordinates if they don't share parents
	     )
	   (multiple-value-bind (x1 y1) (bounding-rectangle-position* p1)
	     (multiple-value-bind (x2 y2) (bounding-rectangle-position* p2)
	       (if (= y1 y2)
		   (< x1 x2)
		   (< y1 y2))))))
    (declare (dynamic-extent #'position-lessp))
    (sort presentations #'position-lessp)))


#||
()

(define-command (com-halt-machine :command-table clim-demo::lisp-listener :name t)
    ()
  (format t "~&Would halt machine"))

(define-presentation-type world ())
(define-presentation-type compiler ())
(define-presentation-type trashcan ())

(define-drag-and-drop-translator dm-load-file
    (pathname command world clim-demo::lisp-listener
     :documentation "Load this file")
    (object)
  `(clim-demo::com-load-file ,object))

(define-drag-and-drop-translator dm-compile-file
    (pathname command compiler clim-demo::lisp-listener
     :documentation "Compile this file")
    (object)
  `(clim-demo::com-compile-file ,object))

(define-drag-and-drop-translator dm-hardcopy-file
    (pathname command clim-demo::printer clim-demo::lisp-listener
     :documentation "Hardcopy this file")
    (object destination-object)
  `(clim-demo::com-hardcopy-file ,object ,destination-object))

(define-drag-and-drop-translator dm-delete-file
    (pathname command trashcan clim-demo::lisp-listener
     :documentation "Delete this file")
    (object)
  `(clim-demo::com-delete-file ,object))

(define-drag-and-drop-translator dm-halt-machine
    (world command trashcan clim-demo::lisp-listener
     :documentation "Halt the machine")
    ()
  `(com-halt-machine))

(defun draw-icons (&optional (stream *standard-output*))
  (formatting-table (stream :inter-row-spacing 4)
    (loop for (type label object) in '((world "The World")
				       (compiler "The Compiler")
				       (trashcan "The Trashcan")
				       (clim-demo::printer "LautScribner" vermicelli))
	  doing
      (formatting-row (stream)
	(formatting-cell (stream)
	  (with-output-as-presentation (:stream stream
					:object (or object type)
					:type type)
	    (write-string label stream))))))
  (fresh-line stream)
  (present #p"sap:>swm>lispm-init.lisp" 'pathname :stream stream))

||#
