;;; -*- Mode: LISP; Syntax: Common-lisp; Package: User; Base: 10 -*-

;;; Rule Base: Franz to Common Lisp

;;;----------------------------------------------------------------------
;;; Package
;;;----------------------------------------------------------------------

; (in-package 'franz-to-common) ; in a version to come

;;;------------------------------------------------------------------------------------------
;;; Laden der Read-Macros und Rule-Macros
;;;------------------------------------------------------------------------------------------

(eval-when (compile eval)
  (unless (get 'tl::init-rule-handling 'tl::version)
    (load "../kernel/init-rule-handling")))

(init-rule-file)

(defruleset standard (expr))

(defrule  append Standard 1
	(fl::|append| ?list nil)
	==>
	(copy-list ?list)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:30:01 1987")))

(defrule  append1 Standard 1
	(fl::|append1| ?list ?elem)
	==>
	(append ?list (list ?elem))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:31:41 1987")))

(defrule fasl Standard 1
   (fl::|fasl| ?file)
   ==>
   (load ?file)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "16.10.1987, 20:48")))

(defrule  nconc1 Standard 1
	(fl::|nconc1| ?list ?elem)
	==>
	(nconc ?list (list ?elem))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:31:41 1987")))

(defrule  assoc Standard 1
	(fl::|assoc| ?item ?list)
	==>
	(assoc ?item ?list :test #'equal)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:32:30 1987")))

(defrule  assq Standard 1
	(fl::|assq| ?item ?list)
	==>
	(assoc ?item ?list :test #'eq)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:33:19 1987")))

(defrule  rassq Standard 1
	(fl::|rassq| ?item ?list)
	==>
	(rassoc ?item ?list :test #'eq)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:57:23 1987")))

(defrule  get_pname.make-symbol Standard 1
	(fl::|get_pname| (??:{intern make-symbol} ?string))
	==>				       ; intern u. make-symbol sind
	?string				       ; Common-Lisp-Funktionen
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:25:28 1987")))

(defrule  concat Standard 1
	(fl::|concat| ?*args)
	==>
	(intern 
	  (concatenate 'string 
		       ?*,(mapcar #'(lambda (arg)
				      (cond ((stringp arg) arg)
					    ((and (listp arg)
						  (eq (car arg) 'quote)
						  (symbolp (cadr arg)))
					     (string-upcase (cadr arg)))
					    (t `(princ-to-string ,arg))))
				  ?*args)))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:34:48 1987")))

(defrule  uconcat Standard 1
	(fl::|uconcat| ?*args)
	==>
	(make-symbol 
	  (concatenate 'string 
		       ?*,(mapcar #'(lambda (arg)
				      (cond ((stringp arg) arg)
					    ((and (listp arg)
						  (eq (car arg) 'quote)
						  (symbolp (cadr arg)))
					     (string-upcase (cadr arg)))
					    (t `(princ-to-string ,arg))))
				  ?*args)))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:16:06 1987")))

(defrule backquote-default Standard 1
   `??
   ==>
   ?$left$
   (Franz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "09.10.1987, 18:37")))

(defrule comma-default Standard 1
   , ??
   ==>
   ?$left$
   (Franz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "09.10.1987, 18:01")))

(defrule  copy Standard 1
	(fl::|copy| ?expr)
	==>
	(copy-tree ?expr)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:39:34 1987")))

(defrule  def.lambda Standard 1
	(fl::|def| ?name (fl::|lambda| ?arglist ?*body))
	==>
	(defun ?name ?arglist ?*body)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:40:34 1987")))

(defrule  def.macro Standard 1
	(fl::|def| ?name (fl::|macro| (?form) ?*body))
	==>
	(defmacro ?name (&whole ?form &rest ignore)
	  ?*body)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 19:16:42 1987")))

(defrule  def.nlambda/lexpr Standard 1
	(fl::|def| ?name (?type:{fl::|nlambda| fl::|lexpr|} ?arglist ?*body))
	==>
	(def ?name (?,(Symbol ?type) ?arglist ?*body))
	(protocol-format "%Warning: Function type ~s doesn't exist ~
                                  in Common-Lisp~%" ?type)
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "16.12.1987, 19:28")
 (author "Andreas Girgensohn") (created "Fri Jan 23 13:52:29 1987")))

(defrule  defun Standard 1
	(??:{fl::|defun| fl::|de|} ?name ?arglist:listp ?*body)
	==>
	(defun ?name ?arglist ?*body)
	(Franz-to-Common Machine)
	(NoProtocol)
((modified-by "Schwab, Thomas")(modified "16.10.1987, 15:00")
 (author "Andreas Girgensohn") (created "Thu Jan 22 12:37:23 1987")))

(defrule  defun.macro Standard 1
	(fl::|defun| ?name fl::|macro| (?form) ?*body)
	==>
	(defmacro ?name (&whole ?form &rest ignore)
	  ?*body)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 19:25:15 1987")))

(defrule  defun.expr Standard 1
	(fl::|defun| ?name fl::|expr| ?arglist:listp ?*body)
	==>
	(defun ?name ?arglist ?*body)
	(Franz-to-Common Machine)
	(Protocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 13:49:16 1987")))

(defrule  defun.fexpr Standard 1
	(fl::|defun| ?name fl::|fexpr| ?arglist ?*body)
	==>
	(defun ?name fexpr ?arglist ?*body)
	(protocol-format "%Warning: Function type fexpr doesn't exist in Common-Lisp~%~
                            Define a corresponding macro for ~s" ?name)
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "16.12.1987, 19:30")
 (author "Andreas Girgensohn") (created "Fri Jan 23 13:55:28 1987")))

(defrule  defun.args Standard 0
	(fl::|defun| ?name ?args:symbolp ?*body)
	==>
	(defun ?name ?args ?*body)
	(protocol-format "%Warning: Function type lexpr doesn't exist in Common-Lisp~%~
                            Define a corresponding macro for ~s" ?name)
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "16.12.1987, 19:32")
 (author "Andreas Girgensohn") (created "Fri Jan 23 13:57:56 1987")))

(defrule  defprop Standard 1
	(fl::|defprop| ?sym ?val ?ind)
	==>
	(setf (get '?sym '?ind) '?val)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:41:30 1987")))

(defrule  putprop Standard 1
	(fl::|putprop| ?sym ?val ?ind)
	==>
	(setf ?,(Standard ?`(get ?sym ?ind)) ?val)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:42:33 1987")))

(defrule  get.1 Standard 1
	(fl::|get| '?sym:symbolp ?key)
	==>
	(get '?sym ?key)
	(Franz-to-Common)
	(NoProtocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 15:29:58 1987")))

(defrule  get.2 Standard 1
	(fl::|get| '?list:consp ?key)
	==>
	(getf '?,(cdr ?list) ?key)
	(Franz-to-Common)
	(Protocol)
((modified-by "Andreas Girgensohn") (modified "Fri Jan 23 14:01:29 1987")
 (author "Andreas Girgensohn") (created "Thu Jan 22 15:32:06 1987")))

(defrule  get.3 Standard 0
	(fl::|get| ?sym ?key)
	==>
	(get ?sym ?key)
	(protocol-format "%Warning: ~s~%~
                          Use getf if ~s is a disembodied property-list~%" ?$left$ ?sym)
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel") (modified "16.12.1987, 18:59")
 (author "Andreas Girgensohn") (created "Thu Jan 22 15:32:54 1987")))

;;; CCC wird vieleicht gebraucht
; (defvar *protocolprot*)

(defrule  break Standard 1
	(fl::|break| ?*args)
	==>
	?,(cond ((null ?*args) '(break ""))
		       ((cdr ?*args) `(and ,(cadr ?*args)
					   (break "~a" ',(car ?*args))))
		       (t `(break "~a" ',(car ?*args))))
	(protocol-format "%Warning: ~s~%break returns only nil~%" ?$left$)
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "16.12.1987, 19:36")
 (author "Andreas Girgensohn") (created "Mon Jan 19 21:43:00 1987")))

(defrule  catch.1 Standard 1
	(fl::|catch| ?form)
	==>
	(catch nil ?form)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:45:34 1987")))

(defrule  catch.2 Standard 1
	(fl::|catch| ?form ?tag)
	==>
	(catch '?tag ?form)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:46:03 1987")))

(defrule  delete.1 Standard 1
	(fl::|delete| ?item ?list)
	==>
	(delete ?item ?list :test #'equal)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:46:43 1987")))

(defrule  delete.2 Standard 1
	(fl::|delete| ?item ?list ?count)
	==>
	(delete ?item ?list :test #'equal :count ?count)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:47:28 1987")))

(defrule  delq.1 Standard 1
	(??:{fl::|delq| fl::|dremove|} ?item ?list)
	==>
	(delete ?item ?list :test #'eq)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:49:00 1987")
 (modified "Matthias Ressel")))

(defrule  delq.2 Standard 1
	(??:{fl::|delq| fl::|dremove|} ?item ?list ?count)
	==>
	(delete ?item ?list :test #'eq :count ?count)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:50:01 1987")))

(defrule  dreverse Standard 1
	(fl::|dreverse| ?list)
	==>
	(nreverse ?list)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:50:30 1987")))

(defrule  dsubst Standard 1
	(fl::|dsubst| ?new ?old ?tree)
	==>
	(nsubst ?new ?old ?tree :test #'equal)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:51:28 1987")))

(defrule  subst Standard 1
	(fl::|subst| ?new ?old ?tree)
	==>
	(subst ?new ?old ?tree :test #'equal)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:52:17 1987")))

(defrule  remove Standard 1
	(fl::|remove| ?item ?list)
	==>
	(remove ?item ?list :test #'equal)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 14:04:44 1987")))

(defrule  Cnth Standard 1
	(fl::|Cnth| ?list ?index)
	==>
	(nthcdr ?,(if (numberp ?index)
		      (1- ?index)
		      ?`(1- ?index))
		?list)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:40:49 1987")))

(defrule  nthelem Standard 1
	(fl::|nthelem| ?ind ?list)
	==>
	(nth ?,(if (numberp ?ind) (1- ?ind) ?`(1- ?ind)) ?list)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:51:28 1987")))

(defrule  cad*r Standard 1
	(?cad*r:cad*rp ?form)
	==>
	?,(expand-cad*r ?`(?cad*r ?form))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 14:06:14 1987")))

;;; CCC cad*rp sollte noch abpruefen, ob sym in Package fl liegt.
;;; CCC Dies ist geschehen.

(defun cad*rp (sym)
  (and (symbolp sym)
       (eq (symbol-package sym) *franz-package*)
       (let ((pname (symbol-name sym)))
	 (and (not (equal pname ""))
	      (eql (schar pname 0) #\c)
	      (let ((sym-length (length pname)))
		(and (> sym-length 6)
		     (eql (schar pname (1- sym-length)) #\r)
		     (every #'(lambda (ch)
				(member ch '(#\a #\d)))
			    (subseq pname 1 (1- sym-length)))))))))
 
(defun expand-cad*r (form)
  (do ((reversed-a-d-s (cdr (nreverse (cdr (coerce (symbol-name (car form)) 'list))))
		       (cddddr reversed-a-d-s))
       (result (cadr form)))
      ((null reversed-a-d-s) result)
    (setq result (list 
		   (intern
		     (string-upcase
		       (coerce 
		       `(#\C ,.(nreverse (ldiff reversed-a-d-s (cddddr reversed-a-d-s))) #\R)
		       'string)))
		   result))))

(defrule  dtpr Standard 1
	(fl::|dtpr| ?x)
	==>
	(consp ?x)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:52:43 1987")))

(defrule  eqstr Standard 1
	(fl::|eqstr| ?x ?y)
	==>
	(equal ?x ?y)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:53:25 1987")))

(defrule error.0 Standard 1
   (fl::|error|)
   ==>
   (error "")
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "22.10.1987, 13:12")))

(defrule  error.1 Standard 1
	(fl::|error| ?message)
	==>
	(error "~a" ?message)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:54:42 1987")))

(defrule  error.2 Standard 1
	(fl::|error| ?mess1 ?mess2)
	==>
	(error "~a ~a" ?mess1 ?mess2)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:55:20 1987")))

(defrule  ferror Standard 1
	(fl::|ferror| ?*args)
	==>
	(error ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:55:58 1987")))

(defrule  err Standard 1
	(fl::|err| ?*rest)
	==>
	(error "")
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Mon Jan 19 21:56:24 1987")))

(defrule  gensym Standard 1
	(fl::|gensym| ?leader)
	==>
	(gensym ?,(if (and (consp ?leader)
			   (eq (car ?leader) 'quote)
			   (symbolp (cadr ?leader)))
		      (string-upcase (cadr ?leader))
		      ?`(string ?leader)))
	(Franz-to-Common)
	(Protocol)
((modified-by "Andreas Girgensohn") (modified "Thu Jan 22 19:47:49 1987")
 (author "Andreas Girgensohn") (created "Mon Jan 19 21:58:45 1987")))

(defrule  get_pname Standard 1
	(fl::|get_pname| ?sym)
	==>
	(symbol-name ?sym)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:43:26 1987")))

(defrule  getchar Standard 1
	(??:{fl::|getchar| fl::|getcharn| fl::|nthchar|} ?sym ?ind)
	==>
	(char (string ?sym) ?,(if (numberp ?ind) (1- ?ind) ?`(1- ?ind)))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:44:10 1987")))

(defrule  ascii Standard 1
	(fl::|ascii| ?int)
	==>
	(intern (string (int-char ?int)))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 13:10:39 1987")))

(defrule  getd Standard 1
	(fl::|getd| ?sym)
	==>
	(and (fboundp ?sym) (symbol-function ?sym))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:46:28 1987")))

(defrule  putd Standard 1
	(fl::|putd| ?sym ?func)
	==>
	(setf (symbol-function ?sym) ?func)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:47:19 1987")))

(defrule  map Standard 1
	(fl::|map| ?*args)
	==>
	(mapl ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:47:51 1987")))

(defrule  member Standard 1
	(fl::|member| ?item ?list)
	==>
	(member ?item ?list :test #'equal)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:48:28 1987")))

(defrule  memq Standard 1
	(fl::|memq| ?item ?list)
	==>
	(member ?item ?list :test #'eq)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:48:57 1987")))

(defrule  merge Standard 1
	(fl::|merge| ?list1 ?list2 ?pred)
	==>
	(merge 'list ?list1 ?list2 ?pred)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:49:46 1987")))

(defrule  ncons Standard 1
	(fl::|ncons| ?expr)
	==>
	(list ?expr)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:50:12 1987")))

(defrule  neq Standard 1
	(fl::|neq| ?x ?y)
	==>
	(not (eq ?x ?y))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:50:39 1987")))

(defrule  nequal Standard 1
	(fl::|nequal| ?x ?y)
	==>
	(not (equal ?x ?y))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:50:39 1987")))

(defrule  numbp Standard 1
	(fl::|numbp| ?x)
	==>
	(numberp ?x)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:52:44 1987")))

(defrule  onep Standard 1
	(fl::|onep| ?x)
	==>
	(eql ?x 1)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:53:08 1987")))

(defrule  patom Standard 1
	(fl::|patom| ?*args)
	==>
	(princ ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:53:37 1987")))

(defrule  print Standard 1
	(fl::|print| ?*args)
	==>
	(prin1 ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:53:37 1987")))

(defrule probef Standard 1
   (fl::|probef| ?file)
   ==>
   (probe-file ?file)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "20.10.1987, 15:54")))

(defrule  plist Standard 1
	(fl::|plist| ?sym)
	==>
	(symbol-plist ?sym)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:54:07 1987")))

(defrule  setplist Standard 1
	(fl::|setplist| ?sym ?list)
	==>
	(setf (symbol-plist ?sym) ?list)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:54:44 1987")))

(defrule  pp-form Standard 1
	(fl::|pp-form| ?*args)
	==>
	(pprint ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:55:33 1987")))

(defrule minus-with-1-arg Standard 1
   (fl::- ?arg)
   ==>
   (- ?arg 0)
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "16.12.1987, 23:30")))

(defrule minus-with->2-args Standard 1
   (fl::- ?arg1 ?arg2 ?*args)
   ==>
   (- ?arg1 ?arg2 ?*args)
   (Franz-to-common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "17.12.1987, 1:26")))

(defrule  quotient.2 Standard 1
	(fl::|quotient| 1 ?x)
	==>
	(/ ?x)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:56:36 1987")))

;; CCC Fuer die read-Funktionen ist noch die Eingabe von
;;     1. whitespace und
;;     2. characters zu klaeren
(defrule  read.1 Standard 1			
	(fl::|read|)
	==>
	(read nil nil)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:58:01 1987")))

(defrule  read.2 Standard 1
	(fl::|read| ?port ?*eof)
	==>
	(read ?port nil ?*eof)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:57:31 1987")))

(defrule  readc.1 Standard 1
	(fl::|readc| ?port)
	==>
	(read-char ?port nil)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:59:18 1987")))

(defrule  readc.2 Standard 1
	(fl::|readc| ?port ?*eof)
	==>
	(read-char ?port nil ?*eof)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:00:13 1987")))

(defrule selectq-to-case Standard 1
	(??:{fl::|selectq| fl::|caseq|} ?*args)
	==>
	(case ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Matthias Ressel")))

(defrule drain.1 Standard 1
   (fl::|drain|)
   ==>
   (force-output)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "18.10.1987, 20:59")))

#||
(defrule drain.2 Standard 1
   (fl::|drain| ?port)
   ==>
   (force-output ?port)
   (debug-format "~s ==> ~s is also possible" ?$left$ ?`(clear-input ?port))
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "18.10.1987, 20:59")))
||#

(defrule drain.2 Standard 1
   (fl::|drain| *standard-input*)
   ==>
   (clear-input *standard-input*)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "18.10.1987, 20:59")))

(defrule drain.3 Standard 1
   (fl::|drain| *standard-output*)
   ==>
   (force-output *standard-output*)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "18.10.1987, 20:59")))

(defrule  tyi.1 Standard 1
	(fl::|tyi|)
	==>
	(read-char nil nil -1)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:00:56 1987")))

(defrule  tyi.2 Standard 1
	(fl::|tyi| ?port)
	==>
	(read-char ?port nil -1)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:01:41 1987")))

(defrule  tyipeek.1 Standard 1
	(fl::|tyipeek|)
	==>
	(peek-char nil nil nil -1)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:02:38 1987")))

(defrule  tyipeek.2 Standard 1
	(fl::|tyipeek| ?port)
	==>
	(peek-char nil ?port nil -1)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:03:23 1987")))

(defrule  tyo.1 Standard 2
	(fl::|tyo| ?char:numberp ?*port)
	==>
	(write-char ?,(int-char ?char) ?*port)
	(Franz-to-Common)
	(Protocol)
((modified-by "Andreas Girgensohn")(modified "17.10.1987, 17:44")
 (author "Andreas Girgensohn") (created "Tue Jan 20 18:01:41 1987")))

(defrule  tyo.2 Standard 1
	(fl::|tyo| ?char ?*port)
	==>
	(write-char ?char ?*port)
	(Franz-to-Common)
	(Protocol)
((modified-by "Andreas Girgensohn")(modified "17.10.1987, 17:44")
 (author "Andreas Girgensohn") (created "Tue Jan 20 18:01:41 1987")))

(defrule  untyi.1 Standard 2
	(fl::|untyi| ?char:numberp ?*port)
	==>
	(unread-char ?,(int-char ?char) ?*port)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:04:00 1987")))

(defrule  untyi.2 Standard 1
	(fl::|untyi| ?char ?*port)
	==>
	(unread-char ?char ?*port)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:04:00 1987")))

(defrule  terpr Standard 1
	(fl::|terpr| ?*port)
	==>
	(terpri ?*port)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:12:14 1987")))

(defrule  infile Standard 1
	(fl::|infile| ?name)
	==>
	(open ?name :direction :input)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:43:13 1987")))

(defrule  outfile.1 Standard 1
	(fl::|outfile| ?name)
	==>
	(open ?name :direction :output)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:43:43 1987")))

(defun append-p (st)
  (cond ((stringp st) (member (schar st 0) '(#\A #\a)))
	((or (not (consp st))
	     (not (eq (car st) 'quote)))
	 nil)
	((symbolp (cadr st)) (member (schar (symbol-name (cadr st)) 0) '(#\A #\a)))
	((stringp (cadr st)) (member (schar (cadr st) 0) '(#\A #\a)))))

(defrule  outfile.2 Standard 1
	(fl::|outfile| ?name ?append:append-p)
	==>
	(open ?name :direction :output :if-exists :append)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:43:43 1987")))

(defrule  portp Standard 1
	(fl::|portp| ?port)
	==>
	(streamp ?port)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:45:03 1987")))

(defrule  pntlen Standard 1
	(??:{fl::|pntlen| fl::|flatc|} ?form)
	==>
	(length (princ-to-string ?form))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 13:05:02 1987")))

(defrule  flatsize Standard 1
	(fl::|flatsize| ?form)
	==>
	(length (prin1-to-string ?form))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 13:05:38 1987")))

(defrule  readlist Standard 1
	(fl::|readlist| ?list)
	==>
	(read-from-string (coerce ?list 'string))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:05:01 1987")))

(defrule  makereadtable.1 Standard 1
	(fl::|makereadtable|)
	==>
	(copy-readtable)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 13:30:23 1987")))

(defrule  makereadtable.2 Standard 1
	(fl::|makereadtable| nil)
	==>
	(copy-readtable)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 13:31:14 1987")))

(defrule  makereadtable.3 Standard 0
	(fl::|makereadtable| ?flag)
	==>
	(copy-readtable nil)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 13:32:28 1987")))

(defrule  sortcar Standard 1
	(fl::|sortcar| ?list ?pred)
	==>
	(sort ?list ?pred :key #'car)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:05:51 1987")))

(defrule  substring.1 Standard 1
	(fl::|substring| ?string ?index)
	==>
	(subseq ?,(if (stringp ?string) ?string ?`(string ?string))
		?,(if (numberp ?index) (1- ?index) ?`(1- ?index)))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:08:16 1987")))

(defrule  substring.2 Standard 1
	(fl::|substring| ?string ?index ?length)
	==>
	?,(transform-substring ?string ?index ?length)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:09:25 1987")))

(defun transform-substring (string index length)
  (let* ((st (cond ((stringp string) string)
		   ((atom string) `(string ,string))
		   (t 'tmp-string)))
	 (args (list st
		     (if (numberp index) (1- index) `(1- ,index))
		     `(min (length ,st)
			   ,(if (and (numberp index) (numberp length))
				(+ index length -1)
			        `(+ ,index ,length -1))))))
    (if (atom string)
	`(subseq . ,args)
        `(let ((tmp-string (string ,string)))
	   (subseq . ,args)))))
    
  
(defrule  sum Standard 1
	(fl::|sum| ?*args)
	==>
	(+ ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:11:13 1987")))

(defrule  symeval Standard 1
	(fl::|symeval| ?sym)
	==>
	(symbol-value ?sym)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:11:45 1987")))

(defrule  throw.1 Standard 1
	(fl::|throw| ?val)
	==>
	(throw nil ?val)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:12:49 1987")))

(defrule  throw.2 Standard 1
	(fl::|throw| ?val ?tag)
	==>
	(throw '?tag ?val)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:13:22 1987")))

(defrule  type Standard 1
	(fl::|type| ?expr)
	==>
	(type-of ?expr)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 18:15:20 1987")))

(defrule  boole.0 Standard 1
	(fl::|boole| 0 ?*rest)
	==>
	0
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:12:30 1987")))

(defrule  boole.1 Standard 1
	(fl::|boole| 1 ?x ?y ?*rest)
	==>
	(logand ?x ?y ?*rest)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:14:19 1987")))

(defrule  boole.2 Standard 1
	(fl::|boole| 2 ?x ?y)
	==>
	(logandc1 ?x ?y)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.3 Standard 1
	(fl::|boole| 3 ?x ?y)
	==>
	?y
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.4 Standard 1
	(fl::|boole| 4 ?x ?y)
	==>
	(logandc2 ?x ?y)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.5 Standard 1
	(fl::|boole| 5 ?x ?y)
	==>
	?x
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.6 Standard 1
	(fl::|boole| 6 ?x ?y ?*rest)
	==>
	(logxor ?x ?y ?*rest)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:14:19 1987")))

(defrule  boole.7 Standard 1
	(fl::|boole| 7 ?x ?y ?*rest)
	==>
	(logior ?x ?y ?*rest)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:14:19 1987")))

(defrule  boole.8 Standard 1
	(fl::|boole| 8 ?x ?y)
	==>
	(lognor ?x ?y)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.9 Standard 1
	(fl::|boole| 9 ?x ?y ?*rest)
	==>
	(logeqv ?x ?y ?*rest)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:14:19 1987")))

(defrule  boole.10 Standard 1
	(fl::|boole| 10 ?x ?y)
	==>
	(lognot ?x)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.11 Standard 1
	(fl::|boole| 11 ?x ?y)
	==>
	(logorc1 ?x ?y)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.12 Standard 1
	(fl::|boole| 12 ?x ?y)
	==>
	(lognot ?y)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.13 Standard 1
	(fl::|boole| 13 ?x ?y)
	==>
	(logorc2 ?x ?y)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.14 Standard 1
	(fl::|boole| 14 ?x ?y)
	==>
	(lognand ?x ?y)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:16:53 1987")))

(defrule  boole.15 Standard 1
	(fl::|boole| 15 ?*rest)
	==>
	-1
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:12:30 1987")))

(defrule  bcdp Standard 1
	(fl::|bcdp| ?func)
	==>
	(compiled-function-p ?func)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:33:55 1987")))

(defrule  copysymbol Standard 1
	(fl::|copysymbol| ?*args)
	==>
	(copy-symbol ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:34:51 1987")))

(defrule  incr Standard 1
	(fl::|incr| ?x)
	==>
	(incf ?x)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:36:04 1987")))

(defrule  decr Standard 1
	(fl::|decr| ?x)
	==>
	(decf ?x)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:36:39 1987")))

(defrule  If Standard 1
	(fl::|If| ?*args)
	==>
	(if ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:37:09 1987")))

(defrule  lessp.1 Standard 1
	(??:{fl::|<&| fl::|lessp|} ?*args)
	==>
	(< ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:38:22 1987")))

(defrule  less-equal.1 Standard 1
	(fl::|<=&| ?*args)
	==>
	(<= ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Wed Jan 21 15:40:34 1987")))

(defrule  equal Standard 1
	(fl::|=&| ?*args)
	==>
	(= ?*args)
	(Franz-to-Common Machine)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:38:17 1987")))

(defrule  greaterp.1 Standard 1
	(??:{fl::|>&| fl::|greaterp|} ?*args)
	==>
	(> ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:39:25 1987")))

(defrule  greater-equal.1 Standard 1
	(fl::|>=&| ?*args)
	==>
	(>= ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:40:12 1987")))

(defrule  alphalessp.1 Standard 1
	(fl::|alphalessp| ?x ?y)
	==>
	(string< ?,(if (stringp ?x) ?x ?`(string ?x))
		 ?,(if (stringp ?y) ?y ?`(string ?y)))
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:50:04 1987")))

(defrule  let.1 Standard 1
	(?foo:{fl::|let| fl::|let*|} ?vars:not-destructuring-p ?*body)
	==>
	(?,(transform-franz-symbol ?foo) ?vars ?*body)
	(Franz-to-Common)
	(NoProtocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:40:12 1987")))

(defun not-destructuring-p (vars)
  (notany #'(lambda (var)
	      (and (consp var)
		   (consp (car var))))
	  vars))

(defun self-evaluating-p (expr)
  (or (member expr '(t nil) :test #'eq)
      (numberp expr)
      (stringp expr)
      (keywordp expr)))

(defrule  quote.keyword Standard 1
	(??:{quote fl::|quote|} ?keyword:self-evaluating-p)
	==>
	?keyword
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Fri Jan 23 13:46:03 1987")))

(defrule  absval Standard 1
	(fl::|absval| ?x)
	==>
	(abs ?x)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:41:11 1987")))

(defrule  fixp Standard 1
	(fl::|fixp| ?x)
	==>
	(integerp ?x)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:41:35 1987")))

(defrule  fix Standard 1
	(fl::|fix| ?x)
	==>
	(floor ?x)				; surprisingly (fix -3.2) 
	(Franz-to-Common)			;              => -4 !
	(Protocol)
((modified-by "Matthias Ressel")(modified "05.01.1988, 17:39")
 (author "Andreas Girgensohn") (created "Thu Jan 22 12:41:57 1987")))

(defrule  lexpr-funcall Standard 1
	(fl::|lexpr-funcall| ?*args)
	==>
	(apply ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:42:27 1987")))

(defrule  remob Standard 1
	(fl::|remob| ?sym)
	==>
	(unintern ?sym)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:45:26 1987")))

(defrule  lsh Standard 1
	(fl::|lsh| ?x ?count)
	==>
	(ash ?x ?count)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:45:54 1987")))

(defrule  expt Standard 1
	(fl::|^| ?base ?exp)
	==>
	(expt ?base ?exp)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:46:27 1987")))

(defrule  vref Standard 1
	(fl::|vref| ?vector ?index)
	==>
	(svref ?vector ?index)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:47:10 1987")))

(defrule  vset Standard 1
	(fl::|vset| ?vector ?index ?val)
	==>
	(setf (svref ?vector ?index) ?val)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:47:40 1987")))

(defrule  vectorp Standard 1
	(fl::|vectorp| ?vector)
	==>
	(simple-vector-p ?vector)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:48:12 1987")))

(defrule  new-vector.1 Standard 1
	(fl::|new-vector| ?length)
	==>
	(make-array ?length :initial-contents nil)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 13:02:08 1987")))

(defrule  new-vector.2 Standard 1
	(fl::|new-vector| ?length ?init)
	==>
	(make-array ?length :initial-contents ?init)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 13:02:43 1987")))

(defrule  hunk Standard 1
	(fl::|hunk| ?*args)
	==>
	(vector ?*args)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:57:57 1987")))

(defrule  hunkp Standard 1
	(fl::|hunkp| ?hunk)
	==>
	(simple-vector-p ?hunk)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:58:26 1987")))

(defrule  cxr Standard 1
	(fl::|cxr| ?index ?hunk)
	==>
	(svref ?hunk ?index)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:58:52 1987")))

(defrule  rplacx Standard 1
	(fl::|rplacx| ?index ?hunk ?val)
	==>
	(setf (svref ?hunk ?index) ?val)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 12:59:27 1987")))

(defrule  makhunk.1 Standard 1
	(fl::|makhunk| ?length:numberp)
	==>
	(make-array ?length :initial-contents nil)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 13:00:19 1987")))

(defrule  makhunk.2 Standard 1
	(fl::|makhunk| ?list)
	==>
	(apply #'vector ?list)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 13:00:56 1987")))

(defrule  store Standard 1
	(fl::|store| ?aref ?val)
	==>
	(setf ?aref ?val)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Thu Jan 22 13:06:59 1987")))

;;; for macro

(defvar *expand-for* t)

(defrule  for Standard 1
	(fl::|for| ?*forms)
	*expand-for*
	==>
	?,(expand-for ?*forms)
	(Franz-to-Common)
	(Protocol)
((author "Andreas Girgensohn") (created "Tue Jan 20 17:47:51 1987")))

(defun expand-for (forms)
  (let ((vars (vars-for forms))
	(args (args-for forms))
	(test (test-for forms))
	(type (type-for forms))
	(body (body-for forms)))
    (if (and (eq type 'fl::|do|)
	     (eql (length vars) 1))
	(let ((body (add-test test (make-body test type body))))
	  `(dolist (,(car vars) ,(car args))
		   . ,(if (and (consp body) (eq (car body) 'progn))
			  (cdr body)
			(list body))))
      `(,(make-mapfn test type)
	#',(make-lambda vars (add-test test (make-body test type body)))
	. ,args))))
  
(defun type-for (forms)
  (let ((item (item-for '(fl::|do| fl::|save| fl::|splice| fl::|filter|) forms)))
    (cond (item (car item))
	  ((error "No body in for loop")))))
  
(defun vars-for (forms)
  (mapcan #'(lambda (x)
              (when (is-var-form x) (list (var-var-form x))))
          forms))

(defun args-for (forms)
  (mapcan #'(lambda (x)
              (when (is-var-form x) (list (args-var-form x))))
          forms))

(defun is-var-form (x)
  (and (eql (length x) 3) (eq (cadr x) 'fl::|in|)))
  
(defun var-var-form (x)
  (car x))

(defun args-var-form (x)
  (caddr x))
  
(defun test-for (forms)
  (let ((item (item-for '(fl::|when|) forms)))
    (when item (cadr item))))
  
(defun body-for (forms)
  (let ((item (item-for '(fl::|do| fl::|save| fl::|splice| fl::|filter|) forms)))
    (cond ((not item) (error "No body in for loop"))
	  ((null (cddr item)) (cadr item))
	  ((cons 'progn (cdr item))))))

(defun item-for (keywords forms)
  (dolist (key keywords)
    (let ((item (assoc key forms :test #'eq)))
      (when item
	(return item)))))

(defun make-mapfn (test type)
  (cond ((eq type 'fl::|do|) 'mapc)
	((not (eq type 'fl::|save|)) 'mapcan)
	((null test) 'mapcar)
	('mapcan)))
  
(defun make-body (test type body)
  (cond ((eq type 'fl::|filter|)
	 `(let ((x ,body)) (when x (list x))))
	((or (not (eq type 'fl::|save|)) (null test)) body)
	((list 'list body))))
  
(defun add-test (test body)
  (cond ((null test) body)
	((null body) test)
	(t (cons 'when
		 (if (and (consp body)
			  (eq (car body) 'progn))
		     (cons test (cdr body))
		     (list test body))))))
  
(defun make-lambda (vars body)
  (cond ((equal vars (cdr body)) (car body))
	((eq (car body) 'progn) (cons 'lambda (cons vars (cdr body))))
	((list 'lambda vars body))))
  
;;; not necessary but pretty  --  Andreas

(defrule if/progn-to-when Standard 2
   (fl::|if| ?cond (progn ?*body))
   ==>
   (when ?cond ?*body)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "22.10.1987, 13:53")))

(defrule if-to-when Standard 1
   (fl::|if| ?cond ?then)
   ==>
   (when ?cond ?then)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "22.10.1987, 13:53")))

(defrule cond-to-when Standard 1
   (fl::|cond| (?cond ?*body))
   ==>
   (when ?cond ?*body)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "22.10.1987, 13:54")))

(defrule cond-to-if Standard 1
   (fl::|cond| (?cond ?then) (t ?else))
   ==>
   (if ?cond ?then ?else)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "22.10.1987, 13:55")))

(defrule and/progn-to-when Standard 1
   (fl::|and| ?cond (progn ?*body))
   ==>
   (when ?cond ?*body)
   (Franz-to-Common)
   (Protocol)
 ((author "Andreas Girgensohn") (created "22.10.1987, 13:56")))

(defrule  global-declare Standard 0
	(fl::|declare| ?*declarations)
	==>
	?$left$
	(protocol-format "%Warning: ~s~%global declarations are not allowed~%~
                          use defvar, proclaim ... or local declarations (preferred)~%"
			 ?$left$)
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "16.12.1987, 19:35")
 (author "Andreas Girgensohn") (created "Thu Jan 22 19:06:03 1987")))

(defrule comment-on-symbolics Standard 1
   (fl::|comment| ?*args)
   ==>
   ?$left$
   ;; (protocol-format "%Note: You might use zl:comment") 
   (Franz-to-Common)
   (NoProtocol)
 ((author "Matthias Ressel") (created "05.01.1988, 17:55")))

(defrule  not-implemented Standard 0	       ; zuletzt probieren
	(?foo:{fl::|*array| fl::|*break| fl::|*catch| fl::|*dif| fl::|*invmod| fl::|*makhunk|
               fl::|*mod| fl::|*rplacx| fl::|*rset| fl::|*throw| fl::|Divide|
               fl::|Emuldiv| fl::|I-throw-err| fl::|add-syntax-class| fl::|aexplode|
               fl::|aexplodec| fl::|aexploden| fl::|allocate| fl::|allsym| fl::|any-zerop|
               fl::|apply*| fl::|arg| fl::|argv| fl::|array| fl::|array-type| fl::|arraycall|
               fl::|arraydims| fl::|arrayp| fl::|arrayref| fl::|baktrace| fl::|baktraceprint|
               fl::|bcdcad| fl::|bcdcall| fl::|bignum-leftshift| fl::|bignum-to-list|
               fl::|bigp| fl::|bindstack| fl::|break-err-handler| fl::|case-scan|
               fl::|charcnt| fl::|charsinbuf| fl::|cib| fl::|comment| fl::|compile|
               fl::|concatl| fl::|condclosefile| fl::|copyint*| fl::|cprintf| fl::|cputim|
               fl::|cpy1| fl::|cr| fl::|de-compose| fl::|debug| fl::|debug-err-handler|
               fl::|debugging| fl::|def| fl::|defcmacro| fl::|defsetf| fl::|defun| fl::|deref|
               fl::|desetq| fl::|df| fl::|dir-home| fl::|dir-lisplib| fl::|dm|
               fl::|double-to-float| fl::|drm| fl::|dsm| fl::|dumplisp| fl::|environment|
               fl::|environment-files| fl::|environment-syntax| fl::|eq-tyimode|
               fl::|err-with-message| fl::|errset| fl::|ev-arraycall| fl::|eval1| fl::|every|
               fl::|ex| fl::|exec| fl::|exece| fl::|exit| fl::|exl| fl::|explode|
               fl::|explodec| fl::|exploden| fl::|fasl-a-file| fl::|fclosure|
               fl::|fclosure-alist| fl::|fclosure-function| fl::|fclosure-list|
               fl::|fclosurep| fl::|feature-present| fl::|ffasl| fl::|fileopen| fl::|filepos|
               fl::|filestat| fl::|fillarray| fl::|fillarrayarray| fl::|float-to-double|
               fl::|fork| fl::|framedump| fl::|franz-reset| fl::|franz-top-level|
               fl::|freturn| fl::|frexp| fl::|gc| fl::|gcafter| fl::|getaccess|
               fl::|getaddress| fl::|getaux| fl::|getdata| fl::|getdelta| fl::|getdisc|
               fl::|getentry| fl::|getenv| fl::|getl| fl::|getlength| fl::|getsyntax|
               fl::|haipart| fl::|hashtabstat| fl::|haulong| fl::|help| fl::|hunk-to-list|
               fl::|implode| fl::|in-tyimode| fl::|include| fl::|include-file|
               fl::|include-if| fl::|includef| fl::|includef-if| fl::|initsym| fl::|initsym1|
               fl::|insert| fl::|killcopy| fl::|kwote| fl::|lambdacvt| fl::|lconc|
               fl::|linelength| fl::|lineread| fl::|list-to-bignum| fl::|listarray|
               fl::|listify| fl::|liszt-declare| fl::|liszt-define| fl::|litatom|
               fl::|load-autorunobject| fl::|load-file| fl::|load-if-needed| fl::|loop|
               fl::|lsubst| fl::|make-equivalent| fl::|make-fclosure-with-alist|
               fl::|make-list-array| fl::|maknam| fl::|maknum| fl::|marray| fl::|merge1|
               fl::|mergelists| fl::|mfunction| fl::|monitor| fl::|msg| fl::|msg-print|
               fl::|msg-tyo-char| fl::|namestack| fl::|new-vectori-byte|
               fl::|new-vectori-long| fl::|new-vectori-word| fl::|newsym| fl::|nwritn|
               fl::|oblist| fl::|opval| fl::|pp| fl::|process| fl::|prtpageused| fl::|ptime|
               fl::|ptr| fl::|purcopy| fl::|purep| fl::|putaccess| fl::|putaux| fl::|putdata|
               fl::|putdelta| fl::|putdisc| fl::|putlength| fl::|pv%| fl::|quote!|
               fl::|quote!-expr-mac| fl::|quotify| fl::|quotify1| fl::|ratom| fl::|rematom|
               fl::|removeaddress| fl::|remq| fl::|remsym| fl::|remsym1| fl::|replace|
               fl::|reset| fl::|resetio| fl::|roman-char| fl::|roman-step| fl::|rot|
               fl::|sassoc| fl::|sassq| fl::|scons| fl::|segment| fl::|set-in-closure|
               fl::|setarg| fl::|setf-check| fl::|cad+r| fl::|setsyntax| fl::|shell|
               fl::|showstack| fl::|showstack-baktrace| fl::|signal| fl::|signp| fl::|sizeof|
               fl::|sl-print| fl::|sload| fl::|small-segment| fl::|some| fl::|sortcarhelp|
               fl::|sortmerge| fl::|splitlst| fl::|sprintf| fl::|sstatus| fl::|status|
               fl::|sticky-bignum-leftshift| fl::|storeintern| fl::|strip-extension|
               fl::|sublishelp| fl::|subpair| fl::|subpr| fl::|subst-eq| fl::|subst-eqp|
               fl::|substeq| fl::|substequal| fl::|substringn| fl::|symeval-in-fclosure|
               fl::|symstat| fl::|syscall| fl::|tab| fl::|tconc| fl::|termcapexe|
               fl::|termcapinit| fl::|tilde-expand| fl::|time-string| fl::|top-eval|
               fl::|top-init| fl::|top-level| fl::|top-print| fl::|top-prompt| fl::|top-read|
               fl::|top-reader-off| fl::|top-reader-on| fl::|trace-funp| fl::|truename|
               fl::|tyimode| fl::|uncompile| fl::|undef-fun-handler| fl::|valuep|
               fl::|vector-dump| fl::|vectori-byte| fl::|vectori-long| fl::|vectori-word|
               fl::|vectorip| fl::|vget| fl::|vi| fl::|vil| fl::|vprop| fl::|vputprop|
               fl::|vsetprop| fl::|vsize-byte| fl::|vsize-word| fl::|wait| fl::|without-path|
               fl::|xcons| fl::|zapline|}
	 ?*args)
	==>
	(?foo ?*args)
	(protocol-format "%Warning: ~s~%~s is not implemented in Common-Lisp~%"
			 ?$left$ ?foo)
	(Franz-to-Common)
	(NoProtocol)
((modified-by "Matthias Ressel")(modified "16.12.1987, 19:50")
 (author "Andreas Girgensohn") (created "Tue Jan 20 17:22:52 1987")))

(defrule  no-difference Standard 0
  (?foo:{fl::* fl::+ fl::1+ fl::1- fl::< fl::<= fl::= fl::> fl::>= fl::|abs|
         fl::|and| fl::|append| fl::|apply|  fl::|atan| fl::|atom|
	 fl::|boundp| fl::|caaaar| fl::|caaadr| fl::|caaar|
         fl::|caadar| fl::|caaddr| fl::|caadr| fl::|caar| fl::|cadaar| fl::|cadadr|
         fl::|cadar| fl::|caddar| fl::|cadddr| fl::|caddr| fl::|cadr| fl::|car| fl::|cdaaar|
         fl::|cdaar| fl::|cdadar| fl::|cdaddr| fl::|cdadr| fl::|cdar| fl::|cddaar|
         fl::|cddadr| fl::|cddar| fl::|cdddar| fl::|cddddr| fl::|cdddr| fl::|cddr| fl::|cdr|
         fl::|close| fl::|cond| fl::|cons| fl::|defvar| fl::|defun| fl::|defmacro| fl::|do| fl::|eq|
	 fl::|equal| fl::|eval| fl::|eval-when| fl::|evenp|
	 fl::|exp| fl::|expt| fl::|float| fl::|floatp| fl::|format| fl::|funcall|
         fl::|function| fl::|get| fl::|go| fl::|if| fl::|intern| fl::|last| fl::|length|
         fl::|list| fl::|list*| fl::|listp| fl::|load| fl::|macroexpand|
         fl::|makunbound| fl::|mapc| fl::|mapcan| fl::|mapcar| fl::|mapcon| fl::|maplist|
         fl::|max| fl::|min| fl::|minusp| fl::|nconc| fl::|not| fl::|nreconc| fl::|nreverse|
         fl::|nthcdr|
         fl::|null| fl::|numberp| fl::|oddp| fl::|or| fl::|plusp| fl::|pop| fl::|princ|
	 fl::|prog|
         fl::|prog1| fl::|prog2| fl::|progn| fl::|push| fl::|quote|
	 fl::|remprop| fl::|return|
	 fl::|reverse| fl::|rplaca|
         fl::|rplacd| fl::|set| fl::|setf| fl::|setq| fl::|sin|
	 fl::|sort| fl::|sqrt| fl::|stringp|
	 fl::|symbolp| fl::|terpri| fl::|unwind-protect| fl::|zerop|}
	 ?*args)
	  ==>
	  (?,(transform-franz-symbol ?foo) ?*args)
	  (Franz-to-Common)
	  (NoProtocol)
  ((modified-by "Bauer, Joachim")(modified "29.01.1988, 15:50")
   (author "Andreas Girgensohn") (created "Tue Jan 20 17:40:23 1987")))

;;;------------------------------------------------------------------------------------------
;;; Ruecksetzen der Readersyntax
;;;------------------------------------------------------------------------------------------

(clean-rule-file)				; resets Reader-Syntax
						; if file compiles up to here
