; -*- mode:     CL -*- ----------------------------------------------------- ;
; File:         zebu-generator.l
; Description:  Generate Domain
; Author:       Joachim H. Laubsch
; Created:      25-Feb-92
; Modified:     Tue Mar  9 17:46:02 1993 (Joachim H. Laubsch)
; Language:     CL
; Package:      ZEBU
; Status:       Experimental (Do Not Distribute) 
; RCS $Header: $
;
; (c) Copyright 1992, Hewlett-Packard Company
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
; Revisions:
; RCS $Log: $
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
(IN-PACKAGE  "ZEBU")

(proclaim '(special *identifier-continue-chars* *identifier-start-chars*))

;----------------------------------------------------------------------------;
; generate-domain-file
;---------------------
; Generate the DEFSTRUCT calls to define the domain & dump to FILE
; When using the meta-grammar, printers will be compiled too.

; file is open when generate-domain-file is called.
; return true if anything was written.

; If DEFSTRUCT is used in the grammar file  -- *domain-structs*  is not
; () -- the domain does not need to be generated.

(defun generate-domain-file (file port &aux domain printers written?)
  (unless *domain-structs*
    (when (setq domain (prepare-domain)) ; sets *domain-type-hierarchy*
      (when (string= (grammar-name *compiler-grammar*)
		     "zebu-mg")
	(format t "~%Compiling Printers ..")
	(setq printers (gen-printers)))))
  (format t "~%Writing domain to ~a~%" file)
  ;; Dump out hierarchy
  (dolist (f (or (nreverse *domain-structs*)
		 (generate-domain domain printers) written?))
    (pprint f port)
    (terpri port)
    (setq written? t)))

;----------------------------------------------------------------------------;
; generate-domain
;----------------
; Given domain D and an alist PRINTERS with pairs (<type> . <print-function>)
; return a list of DEFSTRUCT calls

(defun generate-domain (d printers &aux code)
  (flet ((parse-slots (l)
	   (mapcar #'(lambda (s)
		       (if (atom s)
			   s
			 `(,(car s) nil :type (or null ,(cadr s)))))
		   l)))
    (flet ((slots (x)
	     (do ((xrest x (cddr xrest)))
		 ((null xrest) nil)
	       (if (eq (car xrest) ':slots)
		   (return (parse-slots (cadr xrest))))))
	   (make-struct (name include slots constructor?)
	     `(defstruct (,name
			  (:include ,include)
			  ,@(let ((fn (assoc name printers)))
			      (when fn
				`((:print-function ,(cdr fn)))))
			  ,@(unless constructor?
			      (list '(:constructor nil)))
			  )
	       ,@slots)))
      (labels ((generate-domain-aux (sub super args constructor?)
		 (unless (eq sub super)
		   (push (make-struct sub super (slots args) constructor?)
			 code))
		 (do ((xrest args (cddr xrest))) ((null xrest))
		   (when (eq (car xrest) ':subtype)
		     (let ((newsub (cadr xrest)))
		       (if (atom newsub)
			   (push (make-struct newsub sub nil t) code)
			 (generate-domain-aux
			  (car newsub) sub (cdr newsub) t)))))))
	(generate-domain-aux (car d) 'kb-domain (rest d) nil)
	(nreverse code)))))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;          Internal representation of the domain hierarchy as a tree
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defstruct type-tree-node
  -label
  -subtypes
  -supertype				; back link
  -slots
  )

(defvar *domain-type-hierarchy*)	; a backlinked tree
(defvar *domain-HT* (make-hash-table))	; a dictionary label --> node

;----------------------------------------------------------------------------;
; prepare-domain
;---------------
; convert a domain D (as read from a grammar file) into the tree representation
; 
(defun prepare-domain ()
  (clrhash *domain-HT*)
  (flet ((new-node (label supertype slots)
	   (let ((new (make-type-tree-node
		       :-label label :-supertype supertype :-slots slots)))
	     (setf (gethash label *domain-HT*) new))))
    (let* ((top (new-node ':TOP nil nil)))
      (setf *domain-type-hierarchy* top)
      (setf (type-tree-node--subtypes top)
	    (cons (new-node 'kb-sequence top '(first rest separator))
		  (nconc (mapcar #'(lambda (s)
				     (new-node (intern s) top nil))
				 *open-categories*)
			 (mapcar #'(lambda (c) (new-node (car c) top nil))
				 *lex-cats*))))
      (labels ((add-node (node point)
		 (if (consp node)
		     (let* ((label (car node))
                            (slots (cadr (member ':slots node)))
			    (new-point (new-node label point slots)))
		       (push new-point (type-tree-node--subtypes point))
		       (do ((args (cdr node) (cddr args)))
			   ((null args))
			 (when (eq (car args) ':subtype)
			   (add-node (cadr args) new-point))))
		   (let ((new-point (new-node node point nil)))
		     (push new-point (type-tree-node--subtypes point))))))
	(let ((domain (or (get-grammar-options-key ':DOMAIN)
			  (process-domain-definition))))
	  (when domain
	    (add-node domain top)
	    domain))))))

#||
(prepare-domain '(cl-user::arith-exp
		  :subtype (cl-user::factor :slots (-value))
		  :subtype (cl-user::*-op   :slots (-arg1 -arg2))
		  :subtype (cl-user::+-op   :slots (-arg1 -arg2))
		  :subtype (cl-user::expression :slots (-value))))
||#

(defun process-domain-definition (&aux (R (list 'KB-domain)))
  (flet ((super (type)
	   (declare (type domain-type type))
	   (let ((s (find type *domain-types* :key #'domain-type--type)))
	     (when s (domain-type--supertype s)))))
    (labels ((type> (a b)
	       (declare (type domain-type a b))
	       (when b
		 (let ((super-b (domain-type--supertype b)))
		   (or (eq super-b (domain-type--type a))
		       (type> a (super b))))))
	     (find-super (node supertype)
	       ;; node is the list form of the domain def
	       (if (null node)
		   'Nil
		 (if (eq (car node) supertype)
		     node
		   (do ((n (cdr node) (cddr n)))
		       ((null n) nil)
		     (when (eq (car n) ':subtype)
		       (let ((r (find-super (cadr n) supertype)))
			 (when r (return r)))))))))
      ;; if there is a supertype in *domain-types* that is 
      ;; undefined, define it as a subtype of KB-domain
      (dolist (node *domain-types*)
	(let ((supertype (domain-type--supertype node)))
	  (unless (or (eq supertype 'KB-domain)
		      (find supertype *domain-types*
			    :key #'domain-type--type))
	    (push (make-domain-type
		   :-supertype 'KB-domain
		   :-type supertype)
		  *domain-types*))))
      ;; transform the sorted list to the external :DOMAIN notation
      (dolist (node (sort *domain-types* #'type>))
	(let ((supertype (domain-type--supertype node))
	      (type (domain-type--type node))
	      (slots (domain-type--slots node)))
	  (let ((super (or (find-super R supertype) R)))
	    (nconc super `(:subtype
			   (,type ,@(if slots `(:slots ,slots))))))))
      ;; (pprint R)
      R)))

;----------------------------------------------------------------------------;
; do-forall-supertypes
;---------------------
; Iterate fn over all supertypes of type. Type is the label of a
; type-tree-node in *domain-HT*
; Note that every type is its own supertype.

(defun do-forall-supertypes (type fn)
  (let ((node (gethash type *domain-HT*)))
    (labels ((doit (node)
	       (when (type-tree-node-p node)
		 (funcall fn node)
		 (doit (type-tree-node--supertype node)))))
      (when node
	(doit node)))))

;----------------------------------------------------------------------------;
; legal-slot-of
;--------------
; Is slot-label a legal name of a slot of a type named TYPE?
; 
(defun legal-slot-of (slot-label type)
  (do-forall-supertypes
      type
      #'(lambda (node)
	  (dolist (slot (type-tree-node--slots node))
	    (when (eq (if (consp slot)
			  (car slot)
			slot)
		      slot-label)
	      (return-from legal-slot-of t))))))

;----------------------------------------------------------------------------;
; slot-type-of-type
;------------------
; slot-label is a legal-slot-of type
; if slot-label has a type restriction (<slot-label> <type-restriction>)
;    this restriction will be returned
; else :TOP wil be returned
(defun slot-type-of-type (slot-label type)
  (do-forall-supertypes
      type
    #'(lambda (node)
	(dolist (slot (type-tree-node--slots node))
	  (if (consp slot)
	      (when (eq (car slot) slot-label)
		(return-from slot-type-of-type (cadr slot)))
	    (when (eq slot slot-label)
	      (return-from slot-type-of-type :TOP))))))
  )

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                        Generate the print-functions
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;; collect for each type the lhs and production-rhs in the alist
;; type-prod-AL ( (<type> . %1=( %2=(<lhs> . <production-rhs>) .. )) ..)
;; type-print-fn-AL ( (<type> . (lambda ..)) ..)
;; Return type-print-fn-AL

(defun gen-printers (&aux type-prod-AL type-print-fn-AL user-def-types
			  (print-fn-argl (mapcar #'intern 
						 '("ITEM" "STREAM" "LEVEL"))))
  (maphash #'(lambda (key val) (declare (ignore val))
		     (unless (or (member key '(:TOP KB-DOMAIN KB-SEQUENCE))
				 (member (symbol-name key)
					 *open-categories*
					 :test #'string=)
				 (assoc key *lex-cats*))
		       (push key user-def-types)))
	   *domain-ht*)
  (dolist (zb-rule *zb-rules*)
    (let ((lhs (car zb-rule)))
      (dolist (prod (zb-rule--productions (cdr zb-rule)))
	(let ((semantics (production-rhs--semantics prod)))
	  (when semantics
	    (let* ((type (feat-term--type semantics))
		   (type-node (gethash type *domain-HT*))
		   (bdg  (assoc type type-prod-AL))
		   (slots (feat-term--slots semantics)))
	      ;; warn about inconsistent use of the types	      
	      (if (null type-node)
		  (warn "Type: ~S is not defined in this domain" type)
		(dolist (slot slots)
		  (let ((slot-label (label-value-pair--label slot))
			(slot-value (label-value-pair--value slot)))
		    (if (legal-slot-of slot-label type)
			(let ((slot-type (slot-type-of-type
					  slot-label type)))
			  (unless (every
				   #'(lambda (sub)
				       (is-subtype-of sub slot-type))
				   (infer-type-disj-of-expr slot-value))
			    (warn "~S type restriction of ~S violated by ~S"
				  slot-type slot-label slot-value)))
		      (warn "Slot: ~S is not defined for ~S"
			    slot-label type)))))
	      (if bdg
		  (push (cons lhs prod)
			(cdr bdg))
		(push (cons type (list (cons lhs prod)))
		      type-prod-AL))
	      (do-forall-supertypes type
		#'(lambda (node)
		    (setq user-def-types
			  (delete (type-tree-node--label node)
				  user-def-types)))) ))))))
  (when user-def-types
    (warn "Types ~S were defined but not used." user-def-types))
  ;; now generate the print-function for each type
  ;; unless one has been predefined (via the << foo >> Syntax)
  (dolist (e type-prod-AL)
    (let* ((type (car e))
	   (domain-type (find type *domain-types*
			      :key #'domain-type--type))
	   (fun (when domain-type
		  (domain-type-print-function domain-type))))
      (unless fun
	(let ((%1   (cdr e))
	      clauses good-bdgs unused-bdgs)
	  (dolist (%2 %1)
	    (push (gen-print-case %2) clauses))
	  ;; <clause> = short-lambda-list syntax binding-list
	  (multiple-value-bind (cond-clauses bindings)
	      (gen-clauses clauses)
	    ;; split bindings in good ones and unused ones
	    (dolist (b bindings)
	      (if (null (cdr b))
		  (pushnew b unused-bdgs)
		(pushnew b good-bdgs)))
	    ;; the last cond-clause can have antecedent T
	    (setf (caar (last cond-clauses)) t)
	    (setf fun `(lambda (,@print-fn-argl 
				,@(when good-bdgs `(&aux .,good-bdgs)))
			(declare (ignore ,(third print-fn-argl)
				  ,@unused-bdgs
				  .,(if (not good-bdgs)
					`(,(car print-fn-argl))
				      '())))
			,(if (cdr cond-clauses)
			     `(cond ,@cond-clauses)
			   ;; the condition must be true
			   (cadar cond-clauses)))))))
      (push (cons type fun) type-print-fn-AL)))
  type-print-fn-AL)

(defun gen-print-case (lhs-rhs-pair)
  (let* ((prod (cdr lhs-rhs-pair))
	 (syntax (production-rhs--syntax prod))
	 (semantics (production-rhs--semantics prod))
	 (ll (mapcan #'(lambda (constituent)
			 (when (symbolp constituent) (list constituent)))
		     syntax))
	 (binding-list (mapcar #'(lambda (var)
				   (let ((p (find-path var semantics)))
				     (if (null p)
					 (progn
					   (warn "Variable ~S not used in semantics of ~S."
						 var (car lhs-rhs-pair))
					   (list var)
					   )
				       (cons var p))))
			       ll)))
    `(,ll ,syntax ,binding-list)))

;----------------------------------------------------------------------------;
; gen-clauses
;------------
; Given clauses of the form: <clause> = short-lambda-list syntax binding-list
; where binding-list = ((<non-terminal-symbol> . <path to access from ITEM>) ..)
; return: (1) ((<test for print-case> <format stmt derived from syntax>) ..)
;         (2) a lambda-list binding the %u .. variables used to accessors
;             derived from the paths.

(defun gen-clauses (clauses
		    &aux (vars-to-use
			  (mapcar #'intern '("%U" "%V" "%W" "%X" "%Y" "%Z")))
		    alist cond-clauses)
  (flet ((memo-path (path)
	   (let ((bdg (assoc path alist :test #'equal)))
	     (if bdg
		 (cdr bdg)
	       (let ((R (pop vars-to-use)))
		 (push (cons path R) alist)
		 R)))))
    (let ((partitioning			; a set of sets with the same print syntax
	   (partition-set #'same-print-syntax clauses)))
      (flet ((make-format (syntax bdgs)
	       `(format ,(intern "STREAM")
		 ,(apply #'concatenate 'string
		   (make-format-string-list syntax))
		 ,@(mapcan #'(lambda (const) 
			       (if (stringp const) 
				   nil
				 (list (memo-path (cdr (assoc const bdgs))))))
		    syntax))))
	(dolist (eq-print-set partitioning)
	  (let (ante)
	    (dolist (eq-print eq-print-set)
	      (let* ((ll     (first eq-print))
		     (bdgs   (third eq-print))
		     type-list		; type-predicates that have to hold
		     (ll-map		; ((<lvar> . <%var>) ..)
		      (mapcar #'(lambda (var)
				  (push (infer-type-predicate var) type-list)
				  (cons var
					(memo-path (cdr (assoc var bdgs)))))
			      ll)))
		(let ((conjuncts
		       (mapcar #'(lambda (lvar type) (list type (cdr lvar)))
			       ll-map (nreverse type-list))))
		  (pushnew (if (cdr conjuncts)
			       `(AND . ,conjuncts)
			     (car conjuncts))
			ante :test #'equal))))
	    (setq ante (if (cdr ante) (cons 'OR ante) (car ante)))
	    (pushnew `(,ante
		       ,(let* ((proto  (first eq-print-set))
			       (syntax (second proto))
			       (bdgs   (third proto)))
			  (make-format syntax bdgs)))
		     cond-clauses :test #'equal)))
	(values cond-clauses
		(mapcar #'(lambda (pair) ; (<path> .  <%var>)
			    (labels ((nest (path)
				       (if (null path)
					   (intern "ITEM")
					 (list (car path) (nest (cdr path))))))
			      (list (cdr pair) (nest (car pair)))))
			alist))))))

;----------------------------------------------------------------------------;
; same-print-syntax
;------------------
; 
; 
(defun same-print-syntax (a b)
  (let ((a-syntax (second a)) (b-syntax (second b)))
    (and (equal (length a-syntax) (length b-syntax))
	 (every #'(lambda (constituent1 constituent2)
		    (or (and (symbolp constituent1) (symbolp constituent2))
			(and (stringp constituent1)
			     (stringp constituent2)
			     (string= constituent1 constituent2))))
		a-syntax b-syntax)			
	 ;; syntax is the same
	 (let ((a-bdgs (third a)) (b-bdgs (third b)))
	   ;; do all variables have the same path
	   (every #'(lambda (u v)
		      (equal (cdr (assoc u a-bdgs))
			     (cdr (assoc v b-bdgs))))
		  (first a)
		  (first b))))))

;----------------------------------------------------------------------------;
; make-format-string-list
;------------------------
; This converts a rhs of a grammar rule (SYNTAX) to a format string. 
; It tries to infer when spaces should be inserted based on the
; parameter *identifier-continue-chars*

(defun make-format-string-list (syntax &aux (token "~a"))
  (do* ((s-tail syntax (cdr s-tail))
	;; t means a separator is present to the left/right
	(left-sep t (let ((prev (car Acc)))
		      (if (eq token prev)
			  nil
			(not (find
			      (aref prev (1- (length prev)))
			      *identifier-continue-chars*)))))
	Acc)
       ((null s-tail) (nreverse Acc))
    (let ((const (car s-tail))
	  (right-sep (if (cdr s-tail)
			 (let ((next (cadr s-tail)))
			   (if (symbolp next)
			       nil
			     (not (find
				   (aref next 0)
				   *identifier-continue-chars*))))
		       t)))
      (if (stringp const)
	  (push (escape-tilde const) Acc)
	(progn				; insert #\Space if no separator
	  ;; (break "~S ~S ~S" left-sep const right-sep)
	  (unless left-sep (push " " Acc))
	  (push token Acc)
	  (unless right-sep (push " " Acc)))))))

(defun escape-tilde (string)
  ;; precede each ~ by ~
  (declare (string string))
  (let* ((R "")
	 (tilde #\~)
	 (p0 0)
	 (p1 (position tilde string :test #'eql)))
    (declare (fixnum p0 p1))
    (if p1
	(loop (setq R (concatenate
		       'string R (subseq string p0 p1) "~~"))
	      (setq p0 (1+ p1))
	      (unless (setq p1 (position tilde string
					 :start p0 :test #'eql))
		(return-from escape-tilde
		  (concatenate 'string R (subseq string p0)))))
      string)))
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                      type inference for non-terminals
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun type->predicate (type)
  (intern (concatenate 'string (symbol-name type) "-P")
	  (symbol-package type)))

(defun infer-type-predicate (var &aux (v (constituent-name var)))
  (case v
    (NUMBER     'numberp)
    (IDENTIFIER 'identifierp)
    (STRING     'stringp)
    (t (let ((type (infer-type v)))
	 (if (eq type ':TOP)
	     (make-top-predicate *domain-type-hierarchy*) 
	   (type->predicate type))))))

(defun make-top-predicate (tree)
  (let ((domain-top (car (type-tree-node--subtypes tree)))
	(x (gentemp)))
    `(lambda (,x)
      (or (,(type->predicate (type-tree-node--label domain-top))
	   ,x)
       (typep ,x '(OR NUMBER SYMBOL STRING))))))

(defun infer-type (v)
  (let ((domain *domain-type-hierarchy*))
    (labels ((least-upper-bound (a b)
	       (cond ((eq a b) a)
		     ((eq a ':TOP) a)
		     ((eq b ' :TOP) b)
		     ((is-subtype-of a b) b)
		     ((is-subtype-of b a) a)
		     (t (let ((a1 (least-upper-bound (supertype a) b))
			      (b1 (least-upper-bound (supertype b) a)))
			  (if (is-subtype-of a1 b1)
			      a1
			    b1)))))
	     (supertype (a)
	       (let ((node (gethash a *domain-HT*)))
		 (check-domain-type node a)
		 (if (eq node domain)
		     ':TOP
		   (type-tree-node--label
		    (type-tree-node--supertype node))))))
      (if (member v '(NUMBER IDENTIFIER STRING))
	  v
	(let ((disj (infer-type-disj v)))
	  (if (null disj)
	      (warn "Could not infer type for ~S" v)
	    (reduce #'least-upper-bound disj)))))))

(defun is-subtype-of (a b)
  (let ((a-node (gethash a *domain-HT*))
	(b-node (gethash b *domain-HT*)))
    ;; (check-domain-type a-node a)
    ;; (check-domain-type b-node b)
    (or (eq a-node b-node)
	(let ((sup (type-tree-node--supertype a-node)))
	  (and
	   sup
	   (is-subtype-of (type-tree-node--label sup) b))))))

(defun check-domain-type (type node)
  (unless type
    (error "~S is not a defined domain type." node)))

(defun infer-type-disj (v &aux (nts (list v)))
  ;; return a list of the possible types for a non-terminal V
  (labels ((infer-type-aux (v disjuncts)
	     (if (or (member v '(NUMBER IDENTIFIER STRING))
		     (assoc v *lex-cats*))
		 (adjoin v disjuncts) 
	       (let ((zb-rule (assoc v *zb-rules*))
		     (types disjuncts))
		 (unless zb-rule
		   (error "No Rule/Non-terminal named ~s found" v))
		 (dolist (prod (zb-rule--productions (cdr zb-rule)) types)
		   (let ((s (production-rhs--semantics prod)))
		     (if s
			 (pushnew (feat-term--type s) types)
		       (let ((nt (find-if #'symbolp
					  (production-rhs--syntax prod))))
			 (unless (or (null nt) (member nt nts))
			   (push nt nts)
			   (setq types
				 (infer-type-aux nt types)))))))))))
    (infer-type-aux v nil)))

(defun infer-type-disj-of-expr (x)
  (typecase x
    (number '(number))
    (string '(string))
    (symbol (infer-type-disj (constituent-name x)))))

;----------------------------------------------------------------------------;
; find-path
;----------
; Given a typed feature-structure feat-term, and a variable V occuring
; somewhere as a value of a slot, return a path to it
; return: (1) if you are there ()
;         (2) if there is no path to v: :FAIL
;         (3) if there is some path: the first one found

(defun find-path (v feat-term)
  (labels ((find-path-aux (avl)
	     (if (atom avl)
		 (if (feat-term-p avl)
		     (find-path-list (feat-term--slots avl)
				     (feat-term--type avl))
		   (if (eq v avl)
		       t
		     :FAIL))
	       :FAIL))
	   (find-path-list (avl type)
	     (dolist (lv-pair avl)
	       (let ((p (find-path-aux (label-value-pair--value lv-pair))))
		 (unless (eq p :FAIL)
		   (return
		     (cons (intern
			    (concatenate
			     'string
			     (symbol-name type) "-"
			     (symbol-name (label-value-pair--label lv-pair))))
			   (if (eq p 't) nil p))))))))
    (find-path-aux feat-term)))

;----------------------------------------------------------------------------;
; partition-set
;--------------
; partition SET according to EQUIV-FN
; for equiv-fn holds (equiv-fn x y) = (equiv-fn y x)

(defun partition-set (equiv-fn set &aux alist)
  (do ((x-set set (cdr x-set))) ((null x-set))
    (let ((x (car x-set)))
      (push (list x) alist)
      (do ((y-set (cdr x-set) (cdr y-set))) ((null y-set))
	(let ((y (car y-set)))
	  (if (funcall equiv-fn x y)
	      (let ((found-association (assoc x alist)))
		(push y (cdr found-association))))))))
  (labels ((partition-set-aux (alist)
	     (if (null alist)
		 '()
	       (let* ((pair1 (car alist))
		      (set1 (reduce #'union
				    (mapcar
				     #'(lambda (p)
					 (let ((found (find-if
						       #'(lambda (p1)
							   (member p1 p))
						       pair1)))
					   (when found
					     (setf alist (delete p alist))
					     p)))
				     (cdr alist)) 
				    :initial-value pair1)))
		 (cons set1
		       (partition-set-aux (cdr alist)))))))
    (partition-set-aux alist)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                                    tests
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

#||

(GEN-PRINTERS (find-grammar "ex1a"))
(infer-type 'cl-user::ee)
(infer-type 'cl-user::f)
(infer-type 'cl-user::tt)
(infer-type-predicate 'IDENTIFIER)
(infer-type-predicate 'cl-user::ee)

(PARTITION-SET #'(lambda (x y)
		   (eql (aref (string x) 0)
			(aref (string y) 0)))
	       '(a aa aaa b bbb bb c cccc))

(PARTITION-SET #'(lambda (x y)
			    (eql (aref (string x) 0)
				 (aref (string y) 0)))
	       '(a b c))

;----------------------------------------------------------------------------;
; partition-set-by-selection-fn
;------------------------------
;;; partition set according to selection-fn

(defun partition-set-by-selection-fn (selection-fn set &aux alist)
  (dolist (item set)
    (let* ((key (funcall selection-fn item))
	   (found-association (assoc key alist :test #'eql)))
      (if found-association 
	  (nconc (cdr found-association) (list item))
	(push (cons key (list item)) alist))))
  (dolist (pair alist)
    (setf (car pair) (cadr pair)
	  (cdr pair) (cddr pair)))
  alist)


(partition-set-by-selection-fn #'evenp '(1 2 3 4 5 6 7 8))

 ==> ((2 4 6 8) (1 3 5 7))
||#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;                           End of zebu-generator.l
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
