;;; -*- Mode: Lisp; Package: gnode -*-

;;; $Header: /afs/athena.mit.edu/course/other/allp/nlp/tools/RCS/gcf.lisp,v 3.3 92/07/08 16:28:58 sfelshin Exp $

;;; Copyright (c) 1986-92 Massachusetts Institute of Technology.
;;; All rights reserved.
;;; Brought to you by the Athena Language Learning Project.
;;; "Bigger!  Better!  Slower!  28% More Fun!"

(in-package :gnode)

;;; Graphically display semantic structure, i.e., case frames of nodes.
(provide-module :gcf)

#||(global '(gcf *gcf-verbose-snags* *gcf-all-tmappings*))||#

(require-module :iter :macros t)
#||(use-package :iter)||#

(require-module :links :macros t)
#||(use-package :links)||#

(require-module :feat-support :macros t)
(require-module :features :compile nil)
#||(use-package :features)||#

(require-module :meaning-support)
#||(use-package :interlingua)||#

(require-module :pstates)
;;; We need pstate-support for at least PS-TOP, which is an inline.  If
;;; we compile module gnode in a non-parsing lisp, PS-TOP won't be compiled
;;; inline.  There must be a better way.
(require-module :pstate-support :inlines t :when (load-subsystem-p :parsing))
#||(use-package :parser)||#

(require-module :snags)
#||(use-package :snags)||#

(require-module :cuff-links)		;for IMEANINGS-OF-DMEANING and SUBROLE
(require-module :cf-structs :inlines t :accessors t)
(require-module :cf-snags)
#||(use-package :cf)||#

(require-module :nords :accessors t)
#||(use-package :grammar)||#

(require-module :transfer-functions)
#||(use-package :transfer-functions)||#

(require-module :gnode-tools :macros t)
(require-module :gnode-support)


;;; ================================================================

(defvar *gcf-verbose-snags* t)
(defvar *gcf-all-tmappings* nil)

(defun assign-cf-subscripts (node box)
  (declare (type nord node) (type (or null snord-box) box))
  (let ((nodes-seen ()))
    (labels ((assign (node)
	       (when (and *gnode-snords* box)
		 (setq node (snord node box)))
	       (if (node-p node)
		   (let ((cf (node-cf node)))
		     (when (and (null cf)
				*gnode-new-bindings*
				box)
		       ;; If theta-binder is PRO, still won't have CF.
		       (setq node (theta-binder node box)
			     cf (node-cf node)))
		     (push node nodes-seen)
		     (when cf
		       (let ((tmapping (car (case-frame-tmappings cf))))
			 (when tmapping
			   (dolist (am (tmapping-alist tmapping))
			     (assign (arg-match-node am)))
			   (dolist (misfit (tmapping-misfits tmapping))
			     (assign (misfit-node misfit))))
			 (dolist (subnode (case-frame-mods cf))
			   (assign subnode)))))
		 (push node nodes-seen))))
      (assign node))
    (dolist (bound nodes-seen) 
      (unless (gethash bound *subscript-table*)
	;; Don't look for old binding if new binding is present.
	;; Handle bound words under new binding system.
	(let ((binder (if (and *gnode-new-bindings*
			       box
			       (bind-partition bound box))
			  (dolist (binding (bind-partition bound box))
			    (when *gnode-snords*
			      (setq binding (snord binding box)))
			    (when (and (not (eq binding bound))
				       (member binding nodes-seen))
			      (return binding)))
			(and *gnode-old-bindings*
			     (node-p bound)
			     (node-binding bound))))) 
	  (when binder 
	    (setf (gethash bound *subscript-table*)
		  (or (gethash binder *subscript-table*)
		      (setf (gethash binder *subscript-table*)
			    (incf *current-subscript*))))))))))


;;; Assume that only maximal projections get snorded.  Thus we don't
;;; have to chase down snords of preds, specs, or rels.
(defun gcf (nodes &key
		  ((:verbose-snags *gcf-verbose-snags*) *gcf-verbose-snags*)
		  ((:snords *gnode-snords*) *gnode-snords*)
		  ((:all-tmappings *gcf-all-tmappings*) *gcf-all-tmappings*)
		  ((:old-bindings *gnode-old-bindings*) *gnode-old-bindings*)
		  ((:new-bindings *gnode-new-bindings*) *gnode-new-bindings*)
		  (box nil))
  (unless (listp nodes)
    (setq nodes (list nodes)))
  (dolist (node nodes)
    (let ((pbox box))
      (when (typep node 'pstate)
	(setq node (ps-top (setq pbox node))))
      (let ((*current-node-plist* '()))
        (declare (special *current-node-plist*))
        (labels ((gcf1 (node &optional line-shape)
                   (reset-subscripts)
	           (assign-cf-subscripts node pbox)
	           (let ((*reported-snags* ()))
	             (declare (special *reported-snags*))
	             (gcf2 node pbox line-shape))))
          (case (nord-type node)
            (:paragraph
             (with-gnode-child "PARAGRAPH"
               (dolist (child (node-children node))
                 ;; Spanish production looks like
                 ;;   PARAGRAPH => SENTENCE* FINAL-SENTENCE
                 ;; so first child may be :empty -- specially don't print
                 ;; any case frame for this one, but for convenience, just
                 ;; don't print cf of any empty child.
                 (unless (eq (nord-type child) :empty)
                   ;; Call GCF1 so that subscripts get reset.
                   (gcf1 child "b2")))))
	    (t
	     (gcf1 node))))
        (window-title (collect-leaves node box) "CF: ")
        #+:ccl (make-instance 'tree-window
                 :window-title (getf *current-node-plist*
                                     :window-title)
                 :node (first (getf *current-node-plist*
                                    :children))))))
  (reset-subscripts)
  nodes)

(defun gcf-word-dmeaning (word &optional tmapping)
  (format nil "~(~S~)"
	  (if tmapping
	      (tmapping-meaning tmapping)
	    (word-dmeaning word))))

(defun gcf-theta-role-stuff (&optional arg-spec theta-spec)
  (gnode-command "b1")
  (when arg-spec
    (when (arg-spec-case arg-spec)
      (gnode-format "e~(~S~)" (arg-spec-case arg-spec)))
    (when (arg-spec-type arg-spec)
      (gnode-format "f~(~S~)" (arg-spec-type arg-spec)))
    (if (free-arg-spec-p arg-spec)
	(when (free-arg-spec-features arg-spec)
	  (gnode-command
	   "t" "!" (feature-name (arg-spec-features arg-spec))))
      (gnode-format "t[fixed ~:[...~;~:*~S~]]"
		    (let ((nord (fixed-arg-spec-nord arg-spec)))
		      (and (word-p nord)
			   (word-dmeaning nord))))))
  (let ((ako (or (and arg-spec
		      (not (eq (arg-spec-ako arg-spec)
			       :basic-concept))
		      (arg-spec-ako arg-spec))
		 (and theta-spec
		      (not (eq (theta-spec-ako theta-spec)
			       :basic-concept))
		      (theta-spec-ako theta-spec))))
	(not-ako (or (and arg-spec
			  (not (eq (arg-spec-not-ako arg-spec)
				   :basic-concept))
			  (arg-spec-not-ako arg-spec))
		     (and theta-spec
			  (not (eq (theta-spec-not-ako theta-spec)
				   :basic-concept))
			  (theta-spec-not-ako theta-spec)))))
    (cond ((and ako not-ako)
	   (gnode-format "s~(~S, not ~S~)" ako not-ako))
	  (ako
	   (gnode-format "s~(~S~)" ako))
	  (not-ako
	   (gnode-format "snot ~(~S~)" not-ako)))))

(defun gcf-snags (tmapping snord-box)
  (let ((snags (tmapping-snags tmapping))
	theta-spec arg-spec node)
    (dolist (snag snags)
      (setq snag (cdr snag))
      (unless (or (typep snag 'misfit)	;handled elsewhere
	          (typep snag 'ignore-snag)
	          (typep snag 'ako-drift)
	          (and (typep snag 'sem-unlikely)
		       (not *gnode-attend*)))
        (setq theta-spec (snag-theta-spec snag)
	      arg-spec (snag-arg-spec snag))
        (cond
	 ((typep snag 'wrong-voice)
	  (with-gnode-child "VOICE"
;;;	   (gnode-command "b1")
	    (with-gnode-child "has"
	      (with-gnode-child (theta-grid-voice (tmapping-grid tmapping))))
	    (if (voice-should-be snag)
              (with-gnode-child "should be"
                (with-gnode-child (voice-should-be snag)))
	      (with-gnode-child "shouldn't be"
	        (with-gnode-child (voice-shouldnt-be snag))))))
	 ((typep snag 'times-missing)
	  (with-gnode-child "MISSING"
;;;	   (gnode-command "b1")
	    (with-gnode-child (theta-spec-theta-role theta-spec)
	      (gcf-theta-role-stuff
	       (if (rest (theta-spec-arg-specs theta-spec))
                 nil
		 (first (theta-spec-arg-specs theta-spec)))
	       theta-spec))))
	 (t
	  (unless theta-spec
	    (when arg-spec
	      (setq theta-spec
		    (arg-match-theta-spec
		     (find arg-spec (tmapping-alist tmapping)
			   :key #'arg-match-arg-spec)))))
          (setq node (snag-node snag))
	  (with-gnode-child (snag-short-name snag)
;;;	   (gnode-command "b1")
	    (with-gnode-child
              (cond (arg-spec (arg-spec-theta-role arg-spec))
                    (theta-spec (theta-spec-theta-role theta-spec))
                    ((typep snag 'v-arg-beyond-s-arg)
                     (arg-match-theta-role (or (v-arg-match snag)
                                               (s-arg-match snag)))))
	      (cond
	       ((typep snag 'sem-unlikely)
		(multiple-value-bind (string face)
		                     (word-string-and-face (node-pred node) snord-box)
		  (with-gnode-child string
		    (gnode-command face))))
	       ((typep snag 'missing-feat)
		(gnode-command "t" "has: " (nord-features node))
		(gnode-format
		 "sneeds: !~A"
		 (feature-name (arg-spec-features arg-spec))))
	       ((typep snag 'wrong-ako-type)
		(with-gnode-child 'has
		  (let ((temp ()))
		    (dolist (low-tm (case-frame-tmappings
				     (node-cf (snag-node snag))))
		      (iterate (im (imeanings-of-dmeaning
				    (last-meaning (tmapping-meaning low-tm))))
			(if (get im 'leaf-ako)
                          (pushnew (get im 'leaf-ako) temp)
                          (dolist (ako (get im 'ako))
                            (pushnew ako temp :test #'eq)))))
		    (labels ((foo (list)
			       (when list
				 (with-gnode-child
                                   (format nil "~(~S~)" (car list))
				   (foo (cdr list))))))
		      (foo temp)))) 
		(with-gnode-child (if (eq (snag-sign snag) :not)
                                    '|MUST NOT HAVE|
                                    'needs)
		  (with-gnode-child
                    (format nil "~(~S~)"
                            (if (eq (snag-sign snag) :not)
                              (arg-spec-not-ako arg-spec)
                              (arg-spec-ako arg-spec))))))
	       (t nil))))))))))

(defun gcf-tmapping (tmapping cf box)
  (with-gnode-child (gcf-word-dmeaning
		     (case-frame-pred cf) tmapping)
;    (gnode-command "b2")
    ;; Print first tmapping's information.
    (when tmapping
      (gnode-command "e" (theta-grid-voice (tmapping-grid tmapping)))
      (dolist (am (tmapping-alist tmapping))
	(let ((node (arg-match-node am))
	      (arg-spec (arg-match-arg-spec am))
	      (role (arg-match-theta-role am)))
	  (with-gnode-child role
	    (gcf-theta-role-stuff arg-spec)
	    (if (linked-p role :pleonastic 'subrole)
		(gcf-pleo node box)
	      (gcf2 node box)))))
      (dolist (misfit (tmapping-misfits tmapping))
	(let ((node (misfit-node misfit)))
	  (with-gnode-child "MISFIT"
	    (gnode-command "b1")
	    (gnode-format "e~(~S~)" (misfit-case misfit))
	    (gcf2 node box))))

      (when (listp (tmapping-postponed tmapping))
	(dolist (post (tmapping-postponed tmapping))
	  (with-gnode-child "???"
	    (gnode-command "b1")
	    (gnode-format "e~(~S~)" (cf::postponement-case post))
	    (gcf2 (cf::postponement-node post) box))))

      (local-penalty "f" (tmapping-snags tmapping))
      (when *gcf-verbose-snags*
	(gcf-snags tmapping box)))))

(defun gcf2 (nord box &optional line-shape)
  (declare (type (or null snord-box) box) (type nord nord))
  (when (and *gnode-snords* box)
    (setq nord (snord nord box)))
  (etypecase nord
    (word
      (with-gnode-child (format nil "~(~S~)" (word-dmeaning nord))))
    (node
      (let ((cf (node-cf nord))) 
	(when (and (null cf)
		   *gnode-new-bindings*
		   box)
	  ;; If theta-binder is PRO, still won't have CF, but that's okay.
	  (setq nord (theta-binder nord box)
		cf (node-cf nord))) 
	(cond
	  (cf
	   (let ((pred (case-frame-pred cf)))
	     ;; Print dmeanings of predicate and ancillary case
	     ;; frame stuff.
	     (with-gnode-child (format nil "~(~S~)" (known-meaning pred))
	       (gnode-command "b5")
	       (when line-shape (gnode-command line-shape))
	       (when (node-subscript nord)
		 (gnode-command "s" (subscript-char nord)))
	       (let ((spec (case-frame-spec cf))
		     (rel (case-frame-rel cf))
		     divider)
		 (when (node-p spec)
		   (gnode-format "t~A" (node-features spec))
		   (setq spec nil))
		 (setq divider (if (and spec rel) " " ""))
		 (setq rel (etypecase rel
			     (word (gcf-word-dmeaning rel))
			     ((or null node) ""))) ; there are empty nodes
		 (setq spec (etypecase spec
			      (word (gcf-word-dmeaning spec))
			      (null "")))
		 (gnode-command "e" rel divider spec))

	       (when (case-frame-mods cf)
		 (with-gnode-child "MODS"
;;;		   (gnode-command "b2")
		   (gnode-command "l1")
		   (dolist (node (reverse (case-frame-mods cf)))
		     (with-gnode-child "MOD"
		       (gnode-command "b1")
		       (gcf2 node box)))))

	       (if *gcf-all-tmappings*
		   (dolist (tmapping (case-frame-tmappings cf))
		     (gcf-tmapping tmapping cf box))
		 (gcf-tmapping (car (case-frame-tmappings cf))
			       cf box)))))
	  ((labeled-p nord !conjoined)
	   #||
	   (dolist (child (node-children nord))
	     (gcf2 child box "b2"))
	   ||#
	   ;; The node will be labeled conjoined even if it is one of
	   ;; its children which is actually conjoined.  In this case,
	   ;; the appropriate child will have the same type.  This
	   ;; occurs when more stuff, like punctuation, needs to be
	   ;; consed onto a conjoined node.
	   (let (conj)
	     (loop
	      (setq conj (get-child nord :conj))
	      (cond (conj
		     (return))
		    ((setq nord (get-child nord (nord-type nord)))
		     t)			;do nothing, loop around
		    (t (return))))
	     (when conj
	       (with-gnode-child (gcf-word-dmeaning conj)
		 (when line-shape (gnode-command line-shape))
		 (dolist (child (node-children nord))
		   (unless (eq child conj)
		     (gcf2 child box "b2"))))))) 
	  ;; PRO isn't supposed to have a case frame so pass it through to
	  ;; the next clause.
	  ((and (labeled-p nord !pro)
		*gnode-old-bindings*
		(node-binding nord))
	   (gcf2 (node-binding nord) box line-shape))
	  ;; Node with no case frame.
	  (t
	   ;; Use prin1-to-string to keep the colon.
	   (with-gnode-child (prin1-to-string (node-type nord))
	     (when line-shape (gnode-command line-shape))
	     (when (node-subscript nord)
	       (gnode-command "s" (subscript-char nord)))
	     )))))))

(defun gcf-pleo (nord box)
  (declare (type (or null snord-box) box) (type nord nord))
  (when (and *gnode-snords* box)
    (setq nord (snord nord box)))
  (etypecase nord
    (word
     (multiple-value-bind (string face)
	 (word-string-and-face nord box)
      (with-gnode-child string
	(gnode-command "b5")
	(gnode-command face))))
    (node
     ;; If it was null on the surface, print it like gcf2
     ;; would have printed it.  Otherwise, print the surface
     ;; string under a triangle.
     (let ((leaves (collect-leaves nord box)))
       (cond
	 (leaves
	  ;; Ignore faces returned by WORD-STRING-AND-FACE and just use a2,
	  ;; the word face.  Got any better ideas?
	  (with-gnode-child
	      (format nil "~(~{~S~^ ~}~)"
		      (mapcar #'(lambda (word)
				  (word-string-and-face word box))
			      leaves))
	    (gnode-command "a2")
	    (when (node-subscript nord)
	      (gnode-command "s" (subscript-char nord)))
	    (gnode-command "b4")))
	 (t
	  (gcf2 nord box)))))))
