;;;; -*- Mode:Common-Lisp; Package:PHOENIX; Fonts:(MEDFNT); Base:10 -*-
;;;; *-* File: Titanic: /usr/users/eksl/systems/clip/demos/rtk-experiment-definition.lisp *-*
;;;; *-* Last-edit: Friday, September 17, 1993  15:49:58; Edited-By: WESTY *-* 
;;;; *-* Machine: Count (Explorer II, Microcode 489) *-*
;;;; *-* Software: TI Common Lisp System 6.49 *-*
;;;; *-* Lisp: TI Common Lisp System 6.49  *-*

;;;; **************************************************************************
;;;; **************************************************************************
;;;; *                                                                        *
;;;; *                        Real Time Knob Experiment                       *
;;;; *                                                                        *
;;;; **************************************************************************
;;;; **************************************************************************
;;;
;;; Written by: David L. Westbrook
;;;             Experimental Knowledge Systems Laboratory
;;;             Paul R. Cohen, Principal Investigator
;;;             David L. Westbrook, Systems Manager
;;;             David M. Hart, Laboratory Manager
;;;             Department of Computer Science
;;;             University of Massachusetts
;;;             Amherst, Massachusetts 01003.
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;;
;;;  03-28-91 Created.  (Westy)
;;;  11-13-91 Fixed `agents-lost' so that it really works.  (Westy)
;;;
;;; * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
;;; --*--

(in-package 'PHOENIX)

(proclaim '(optimize (speed 3) (safety 1)))

;;; --*--
;;; ***************************************************************************

(define-debugging-tag real-time-knob-experiment)

;; !!!NOTE THIS!!!
(setf *write-using-new-experiment-data-format?* t)

;;;----------------------------------------------------------------------------
;;; Parameters

(defparameter *real-time-knob-values* '(1 3 5 7 9 11))

(defparameter *fire-sector-extension* 5000)    

(defparameter *data-directory-path* "ph:data.rtk2;")

(defparameter *wind-speed* 3)

;;;----------------------------------------------------------------------------

;; Add this to system proper someday.
(defun current-scenario ()
  (get (fire-system) :scenario))

(defun set-current-scenario (new-scenario)
  ;; Make sure this works.
  (setf (get (fire-system) :scenario) new-scenario))

(defsetf current-scenario set-current-scenario)

;;;----------------------------------------------------------------------------

(defun run-real-time-knob-experiment (trials hours &key
                                          (wind-speed *wind-speed*)
                                          (real-time-knob-values *real-time-knob-values*)
					  (use-exp-style t)
                                          (output-file (conjure-up-filename))
                                          (comments ""))
  (setf *wind-speed* wind-speed)
  (setf *real-time-knob-values* real-time-knob-values)

  (assert (zerop (mod trials (length *real-time-knob-values*))) (trials)
          "try using a number of trials that is a multiple of ~d; then you will get a complete set of trials"
          (length *real-time-knob-values*))
  
  (unless (current-scenario)
     ;; Create Scenario now loads the file if necessary (assuming that
     ;; the file is named coherently with regard to the scenario name).
    (create-scenario 'basic :no-menu))
  ;; We have our own script, so get rid of any and
  (when (scenario.script (current-scenario))
    (setf (scenario.script (current-scenario)) nil)
    ;; <hack> reset things in case we loaded a script.
    (send (fire-system) :reset-scenario))
  (run-experiment 'real-time-knob-experiment trials hours :output-file output-file 
                  :extra-header comments :use-exp-style use-exp-style))

;;;----------------------------------------------------------------------------

(defun real-time-knob-experiment-init-before-experiment (use-exp-style)
  (when use-exp-style
    (setf (interaction-style t) 'experiment))
  (setf *reset-random-state-during-scenario-initialization?* nil)
  (setf tv:*screen-saver-time-delay* nil)
  ;; This is prevents "white boxes".  Errors will be signalled with notifications.  Setting *task-error-hook* to
  ;; non-nil (done below in `ra-exp-init-before-trial') will catch things before "white boxes" would be printed,
  ;; but between trials we don't want things to stop.  See comment in `real-time-knob-experiment-after-trial'.
  (setf *query-task-errors* nil)
  ;; Don't allow fires to skip over fire lines.
  (send (fire-simulation) :set-spotting-scale-factor 0)
  ;; Get rid of flank attack, etc.
  (modify-knowledge-base))

(defun real-time-knob-experiment-init-before-trial ()
  (gc-immediately :silent t)
  ;; Moved this from `ra-exp-init-before-experiment' (see `ra-exp-after-trial')
  (setf *phoenix-error-behavior* :RESTART)
  (setf *phoenix-error-restart-hook* 'shutdown-and-go-to-next-trial)
  (setf *task-error-hook* 'shutdown-and-go-to-next-trial)
  ;; Step thru independent variables.
  (setf (send (find-agent-by-name 'fireboss) :cpu-usec/internal-time)
        (minutes/cpu-sec->cpu-usec/internal-time
          (nth-elt-of-cross-product-as-multiple-values (1- (trial-number)) *real-time-knob-values*)))
  (when-debugging-format real-time-knob-experiment
    "Setting fireboss minutes/cpu-sec to ~d"
    (cpu-usec/internal-time->minutes/cpu-sec
      (send (find-agent-by-name 'fireboss) :cpu-usec/internal-time))))

(defun real-time-knob-experiment-after-trial ()
  ;; Reset these so that errors which occur during reset don't cascade.
  (setf *phoenix-error-behavior* :STOP)
  (setf *phoenix-error-restart-hook* nil)
  ;; This is now done in the `shutdown-trial' method.
  (write-current-experiment-data))

(defun real-time-knob-experiment-reset-after-experiment ()
  (setf (interaction-style t) 'normal)
  (setf tv:*screen-saver-time-delay* 20)
  (setf *query-task-errors* t))

;;;----------------------------------------------------------------------------

(define-experiment real-time-knob-experiment (use-exp-style)
  "Simple experiments for exercising and testing the real time knob."
  :before-experiment (real-time-knob-experiment-init-before-experiment use-exp-style)
  :before-trial (real-time-knob-experiment-init-before-trial)
  :after-trial (real-time-knob-experiment-after-trial)
  :after-experiment (real-time-knob-experiment-reset-after-experiment)
  :instrumentation (trial-number
                    real-time-knob-setting
                    number-of-bulldozers
                    wind-speed
                    plan-to-contain-fire
                    all-plans-to-contain-fire
		    fires-started
                    shutdown-time
                    number-of-fires-contained
                    total-fire-line-built
                    r-factor
                    area-burned
                    agents-lost
                    all-agent-instrumentation
                    fireboss-instrumentation
                    bulldozer-instrumentation
		    )

  :script
  ((setup-starting-conditions "12:29"
     (progn
       (send (fire-system) :alter-environment-parameter 'wind-direction 315)                              
       (send (fire-system) :alter-environment-parameter 'wind-speed *wind-speed*)))
   (start-fire "12:30"
     (send (fire-system) :start-fire 700 (point 50000 40000)))))

;; The only major hack in here. This shuts the whole thing down when
;; we finish executing the plan that is working on the fire.
(defmethod (phoenix-agent :after :execute-action) (tl-entry time-to-execute)
  (declare (ignore time-to-execute))
  (when (eq self (find-agent-by-name 'fireboss))
    (let ((plan-to-contain-fire-final-action 
	    (f:get-value (tl-entry-of-plan-to-contain-fire) 'has-end-action)))
      (when (and
	      plan-to-contain-fire-final-action 
	      (eq (name-of tl-entry)
		  (name-of plan-to-contain-fire-final-action))
	      (eq (tl-status plan-to-contain-fire-final-action)
		  :completed))
	(schedule-arbitrary-function
	  'shutdown-trial-of-current-experiment
	  (1+ (current-time))
	  'end-of-trial)))))

;;;----------------------------------------------------------------------------
;;; Instrumentation definitions...

(defclip real-time-knob-setting ()
  "Returns the minutes per cpu-second setting currently being used by the fireboss."
  (:report-key "Real time knob setting")

  (cpu-usec/internal-time->minutes/cpu-sec
    (send (find-agent-by-name 'fireboss) :cpu-usec/internal-time)))

;;;----------------------------------------------------------------------------

(defclip wind-speed ()
  (:report-key "Wind speed")
  
  (send (fire-system) :get-environment-parameter 'wind-speed))

;;;----------------------------------------------------------------------------

(defclip number-of-bulldozers ()
  (:report-key "Number of bulldozers")

  (length (find-agent-by-name 'bulldozer :multiple-allowed t)))

;;;----------------------------------------------------------------------------

;;; This could probably be done using format, as well.
(defmacro round-to-n-significant-decimal-digits (x n &optional (divisor 1))
  "Returns a single-float with n significant digits. This number-conses a bunch so use it sparingly.
Remember - for single-float's the maximum total significant digits is 7 or 8!"
  (utils:make-variables (factor)
    `(let ((,factor (expt 10 ,n)))
       (coerce (/ (round (* (coerce ,x 'double-float) ,factor) ,divisor) ,factor) 'float))))

(defmacro pct (part wh)
  `(if (zerop ,wh) 0 (* 100.0 (/ ,part ,wh))))

(defmacro /-safe (dividend divisor)
  `(if (zerop ,divisor) 0 (/ ,dividend ,divisor)))

(defclip area-burned ()
  (:report-key "Area burned (sq. km)")

  (round-to-n-significant-decimal-digits
    (send (real-world-firemap) :fire-state)
    2
    1e6))

;;;----------------------------------------------------------------------------

(defvar *fires-started* nil) ;; Do we need this anymore? No.

(defclip fires-started ()
  (:report-key "Fires started")

  ;; to prevent having to build extraneous instrumented fires we cache them in here.

  (setf *fires-started* nil)
  (map-over-fires (fire) (:delete-fire-frames nil)
    (push fire *fires-started*))
  (length *fires-started*))

(defclip shutdown-time ()
  (:report-key "Shutdown time (hours)")

  (round-to-n-significant-decimal-digits
    (current-time)
    2
    3600))

;;;----------------------------------------------------------------------------

; Unneeded if we shutdown right after fighting fire.
; ;; Added this to record when fires are contained.
;(defmethod (fireboss :after :em-mark-fire-as-under-control) ()
;  (free-operations
;    (let ((fire (variable-value 'fire)))
;      (when (eq (f:get-value* fire 'status) 'under-control)
;        (f:put-value fire 'perimeter-when-controlled
;                     (fire-perimeter-polyline (fire-origin fire)
;                                              *fire-perimeter-resolution*))
;        (push fire *fires-contained*)))))

(defclip fires-contained ()
  (:report-key "Fires contained")
  (mapcan #'(lambda (fire)
              (when (eq (f:get-value* fire 'status) 'under-control)
                (list fire)))
          (f:get-values* 'actual-fire 'instance+inv)))

(defclip number-of-fires-contained ()
  (:report-key "Fires extinguished")

  (length (fires-contained)))

;;;----------------------------------------------------------------------------

(defun expand-extent-in-all-directions-by (extent distance-in-meters)
  (let ((temp-point (point distance-in-meters distance-in-meters)))
    (extent (point-clip (point-difference (extent-upper-left extent) temp-point))
	    (point-clip (point-sum (extent-lower-right extent) temp-point)))))
	    
(defun length-of-built-line-in-extent (extent)
  (let ((line-length 0))
    (dofiremap (point :upper-left (extent-upper-left extent)
                      :lower-right (extent-lower-right extent))
      (do-feature-edges (edge point (real-world-firemap) :edge-type :dynamic)
	(incf line-length (feature-edge-length edge))))
    (values line-length)))

(defclip r-factor ()
  (:report-key "R factor")

  (round-to-n-significant-decimal-digits
    (/-safe (total-fire-line-built) (total-perimeter))
    3))

(defclip total-perimeter ()
  ()
  (reduce #'+ (fires-contained)
          :key #'(lambda (fire)
                   (fast-polyline-length
                     (fire-perimeter-polyline (fire-origin fire)
                                              *fire-perimeter-resolution*)
                     t))))

(defclip total-fire-line-built ()
  (:report-key "Fireline Built (meters)")
  (reduce #'+ (fires-contained)
          :key #'(lambda (fire)
                   (length-of-built-line-in-extent
                     (expand-extent-in-all-directions-by
                       (accurate-fire-extent (fire-center-of-mass fire)
                                             (point-on-polyline-furthest-from
                                               (fire-center-of-mass fire)
                                               (fire-boundary fire)
                                               nil))
                       *fire-sector-extension*)))))

;;;----------------------------------------------------------------------------

(defclip agents-lost ()
  ()
  (mapcan #'(lambda (agent)
              (when (eq (f:get-value* (send agent :self-frame)  'status) :dead)
                (list (name-of agent))))
          (all-agents)))

;;;----------------------------------------------------------------------------

;; Added this so that "~a overall utilization" below will print out the name
;; only. [I'm suspicious about whether this works all the time. I know
;; it does if I compile this by hand.]
(defmethod print-object ((agent phoenix-agent) stream)
  (if *print-escape*
    (call-next-method agent stream)
    (format stream "~a" (name agent))))

(defclip all-agent-instrumentation ()
  "Records the utilization of all the agents."
  (:components (agent-overall-utilization
                 agent-cognitive-utilization
                 agent-message-handling-time-pct
                 agent-action-selection-time-pct
                 agent-error-recovery-cost
                 agent-error-recovery-percentage-of-cognitive-time
                 number-of-frames-on-timeline
                 )
   :map-function (cons 
               (find-agent-by-name 'fireboss)
               (find-agent-by-name 'bulldozer :multiple-allowed t))))

(defclip agent-overall-utilization (agent)
  (:report-key "~a overall utilization")
  (round-to-n-significant-decimal-digits
    (pct (task-cumulative-cpu-time agent) (current-time))
    1))

(defclip agent-cognitive-utilization (agent)
  (:report-key "~a cognitive utilization")
  (round-to-n-significant-decimal-digits
     (pct (phoenix-agent-cumulative-action-execution-time agent)
          (current-time))
     1))

(defclip agent-message-handling-time-pct (agent)
  (:report-key "~a message handling pct")
  (round-to-n-significant-decimal-digits
     (pct (phoenix-agent-cumulative-message-handling-time agent)
          (current-time))
     1))

(defclip agent-action-selection-time-pct (agent)
  (:report-key "~a action selection pct")
  (round-to-n-significant-decimal-digits
     (pct (phoenix-agent-cumulative-next-action-selection-time agent)
          (current-time))
     1))

(defclip agent-error-recovery-cost (agent)
  (:report-key "~a error recovery cost")
  
  (f:using-frame-system ((name-of agent))
    (reduce #'+ (gather-recovery-method-instances (name-of agent))
	    :key #'determine-recovery-cost)))

(defclip agent-error-recovery-percentage-of-cognitive-time (agent)
  (:report-key "~a ER % of cognitive time")

  (round-to-n-significant-decimal-digits
    (pct (agent-error-recovery-cost (name-of agent))
         (phoenix-agent-cumulative-action-execution-time agent))
    1))
  
(defclip number-of-frames-on-timeline (agent)
  (:report-key "~a number of frames on timeline")
  (f:using-frame-system ((name-of agent))
    (unwind-protect
        (let ((cnt 0))
          (labels ((count-frame (frame)
                     (unless (f:get-value frame 'counted)
                       (incf cnt)
                       (f:put-value frame 'counted t)
                       (count-frames-after frame)
                       (count-frames-below frame)))
                   (count-frames-below (start-frame)
                     (dolist (frame (tl-has-components start-frame))
                       (count-frame frame)))
                   (count-frames-after (start-frame)
                     (dolist (frame (tl-next-actions start-frame))
                       (count-frame frame))))
            
        (dolist (frame (tl-has-start-actions (f:get-value 'initial-timeline 'instance+inv)))
          (count-frame frame)))

          (values cnt))
      
      (f:map-frames #'(lambda (frame) (f:delete-all-values frame 'counted))))))

;;;----------------------------------------------------------------------------

(defclip fireboss-instrumentation ()
  "Instrumentation for the fireboss."
  (:components (agent-total-envelope-time)
   :map-function (list (find-agent-by-name 'fireboss))))

(defclip agent-total-envelope-time (agent)
  (:report-key "~a total envelope time")
  (f:using-frame-system ((name-of agent))
    (reduce #'+
	    #+Phoenix (f:pattern-match #p(instance {f:value-in-hierarchy-of
						   '(ako instance) 'plan-envelope}))
	    #-Phoenix nil
            :key #'plan-computation)))

;;;----------------------------------------------------------------------------

(defclip bulldozer-instrumentation ()
  "Instrumentation for the bulldozers."
  (:components (reflexes-executed)
   :map-function (find-agent-by-name 'bulldozer :multiple-allowed t)))

(defclip reflexes-executed (agent)
  (:report-key "~a reflexes executed")
  
  (reduce #'+ (standard-agent-model-reflexes agent)
          :key #'reflex-execution-count))

;;;----------------------------------------------------------------------------

(defclip count-of-deadly-object-in-path-messages ()
  (:enable-form (trace-message-patterns
                  '(:message-type :msg-reflex :type :error :reason :deadly-object-in-path))
   :disable-form (untrace-message-patterns
                   '(:message-type :msg-reflex :type :error :reason :deadly-object-in-path)))

  (message-pattern-count '(:message-type :msg-reflex :type :error :reason :deadly-object-in-path)))

;;;----------------------------------------------------------------------------

(defun the-first-fire-started ()
  (first (last (f:get-values* 'actual-fire 'instance+inv))))

(defclip plan-to-contain-fire ()
  (:report-key "Plan to Contain Fire")
  
  (get-primitive-action
    (tl-entry-of-plan-to-contain-fire)))

(defun tl-entry-of-plan-to-contain-fire ()
  (f:using-frame-system ((name-of (find-agent-by-name 'fireboss)))
    (let (top-level-actions)
      (dolist (possible (f:get-values* 'act-deal-with-new-fire 'instance+inv))
	(when (equal (f:framer (the-first-fire-started)) (variable-value 'fire :action possible))
	  (push possible top-level-actions)))
      (f:get-value* (first (sort top-level-actions #'> :key #'(lambda (x) (f:get-value* x 'creation-time))))
		    'has-end-action))))

(defclip all-plans-to-contain-fire ()
  (:report-key "All plans to Contain Fire")
  (f:using-frame-system ((name-of (find-agent-by-name 'fireboss)))
    (mapcar #'(lambda (act-deal-with-our-fire)
                (get-primitive-action (f:get-value* act-deal-with-our-fire 'has-end-action)))
            (sort 
              (mapcan #'(lambda (act-deal-with-new-fire)
                          (when (equal (f:framer (the-first-fire-started))
                                       (variable-value 'fire :action act-deal-with-new-fire))
                            (list act-deal-with-new-fire)))
                      (f:get-values* 'act-deal-with-new-fire 'instance+inv))
              #'< :key #'(lambda (x) (f:get-value* x 'creation-time))))))
  
;;;----------------------------------------------------------------------------
;;; Support code

;; Add this to experiment interface someday.
(defun conjure-up-filename ()
  (multiple-value-bind (nil minute hour date month year nil) (get-decoded-time)
    (format nil "~a~a-~d-~d-~d-~d-~d.lisp"
            *data-directory-path*
            net:local-pretty-host-name
            month
            date
            (- year 1900)
            hour
            minute)))

(defun modify-knowledge-base ()
  "Some of the plans are simply too fragile or don't include envelopes"
  (f:put-value* 'plan-flank-attack 'key 
		'(#p(instance environment-info wind-speed {f:evaluate '(< f:*value* 0)})))
  (f:put-value* 'plan-two-bulldozer-rendezvous-and-surround-fire 'key 
		'(#p(instance environment-info wind-speed {f:evaluate '(< f:*value* 0)}))))

(defun gather-recovery-method-instances (&optional 
					     (agent-name (f:frame-system-name f:*current-frame-system*)))
  "Collect the set of instances of recovery methods being used"
  (f:using-frame-system (agent-name)
  (let (error-actions 
	methods)
    ;;find all the errors and envelope violations that occured
    (dolist (possible (f:get-values* 'act-deal-with-error 'instance+inv))
      (when (f:get-value* possible 'creation-time)
	(push possible error-actions)))
    (dolist (possible-instances (f:get-values* 'act-deal-with-error 'ako+inv))
      (dolist (possible (f:get-values* possible-instances 'instance+inv))
	(when (f:get-value* possible 'creation-time)
	  (push possible error-actions))))

    (dolist (next-error error-actions)
      (setf methods (append (f:get-values* next-error 'has-component)
			    ;;; old-components is a list of lists of components
			    (mapcar #'first (f:get-values* next-error 'old-components))
			    methods)))
    methods)))

;;; ***************************************************************************
;;; EOF














