;; -*- Mode:LISP; Package:(BOXER GLOBAL 1000); Base:8.; Fonts:CPTFONT -*-

;;; (C) Copyright 1985 Massachusetts Institute of Technology
;;;
;;; Permission to use, copy, modify, distribute, and sell this software
;;; and its documentation for any purpose is hereby granted without fee,
;;; provided that the above copyright notice appear in all copies and that
;;; both that copyright notice and this permission notice appear in
;;; supporting documentation, and that the name of M.I.T. not be used in
;;; advertising or publicity pertaining to distribution of the software
;;; without specific, written prior permission.  M.I.T. makes no
;;; representations about the suitability of this software for any
;;; purpose.  It is provided "as is" without express or implied warranty.
;;;

;;;;KEY-NAMES

;;; This file defines :BOXER-FUNCTION names for the various keystrokes and
;;; mouse clicks the user can type. This file just defines names for those
;;; keys, other files (COMx) should do DEFBOXER-FUNCTIONs to define what
;;; those keys should do. In terms of ZWEI, this file is below the level
;;; of COMTABs (more at the level of kbd-convert-to-software-char) and the
;;; files which do the DEFBOXER-FUNCTIONs are at the level of COMTABs.

;;; In order to provide fast conversion of LISPM character codes to
;;; BOXER key names, we use an array to look them up in. This is kind
;;; of like ZWEI.

(DEFVAR KEY-NAMES (MAKE-ARRAY '(#-TI 170. #+TI 190. 16.))
  "KEY-NAMES is an art-q array of dimensions 170. by 16.. It is used
   to assign symbol names to keys on the keyboard. An array is used so
   that when a key is pressed the symbol name for the key can be found
   more quickly.")

(DEFVAR MOUSE-CLICK-NAMES (MAKE-ARRAY '(3. 2. 16.))
  "MOUSE-CLICK-NAMES is the symbolic dispatch table for mouse clicks.  The first dimension
   specifies the position (L = 0, M = 1, R = 2), the second position specifies the
   number of times the mouse was clicked (minus 1, i.e. #/mouse-r-2 would be a 1) and
   the last dimension specifies any shifts (i.e. ctrl, meta, etc.)")

(DEFVAR MOUSE-STATE-NAMES (MAKE-ARRAY '(3. 2. 16.))
  "MOUSE-CLICK-NAMES is the symbolic dispatch table for mouse states.  The first dimension
   specifies the position (L = 0, M = 1, R = 2), the second position specifies the
   state of the mouse (0 = down, 1 = up) and the last dimension specifies any 
   shifts (i.e. ctrl, meta, etc.)")

(DEFVAR *BOXER-KEYSTROKE-HISTORY* NIL
  "A list of all the keys pressed. ")

(DEFVAR *BOXER-COMMAND-KEY-ALIST* NIL
  "An association list of key names and command names. ")

(DEFUN DEFINE-KEY-NAME (KEY-NAME KEY-CODE)
  (COND ((NUMBERP KEY-CODE)
	 (ASET KEY-NAME
	       KEY-NAMES
	       (LDB %%KBD-CHAR KEY-CODE) (LDB %%KBD-CONTROL-META KEY-CODE)))
	((SYMBOLP KEY-CODE)
	 (FERROR "~%Can't store symbols in key-names.~
                  ~%In order to teach Boxer how to handle a new kind of symbol~
                  ~%in its input buffer you should define a function to handle~
                  ~%the symbol on the symbol's :BOXER-INPUT property. When Boxer~
                  ~%sees that symbol in its input buffer it will call that function~
                  ~%with the symbol as its only argument."))
	((LISTP KEY-CODE)
	 (FERROR "~%Can't store blips in key-names.~
                  ~%In order to teach the editor how to handle a new kind of blip in~
                  ~%in its input buffer you should define a function to handle the~
                  ~%blip on the :BOXER-INPUT property of the symbol which is the car~
                  ~%of the blip. When Boxer sees a blip with that symbol as its car~
                  ~%in its input buffer it will call that function with the blip as~
                  ~%its only argument."))
	(T
	 (FERROR "~S is a completely unknown type of Boxer Input." KEY-CODE))))

(DEFUN LOOKUP-KEY-NAME (KEY-CODE)
  (AND (FIXNUMP KEY-CODE)
       (>= (LDB %%KBD-CHAR KEY-CODE) 0)
       (<= (LDB %%KBD-CHAR KEY-CODE) #-TI 169. #+TI 189.)
       (>= (LDB %%KBD-CONTROL-META KEY-CODE) 0)
       (<= (LDB %%KBD-CONTROL-META KEY-CODE) 15.)
       (AREF KEY-NAMES (LDB %%KBD-CHAR KEY-CODE)
		       (LDB %%KBD-CONTROL-META KEY-CODE))))



(DEFVAR BU:*KEY-CODE-BEING-HANDLED* NIL)

(DEFVAR BU:*KEY-NAME-BEING-HANDLED* NIL)

(DEFUN HANDLE-BOXER-INPUT (INPUT)
  (increment-key-tick)				;for use with multiple-kill hack
  (PUSH INPUT *BOXER-KEYSTROKE-HISTORY*)
  (COND ((FIXNUMP INPUT)
	 ;; Some sort of Lispm key code. Try to lookup a name for it. If it
	 ;; has a name BOXER-FUNCALL that name with the special variables:
	 ;;   BU:*KEY-CODE-BEING-HANDLED* bound to the key code
	 ;;   BU:*KEY-NAME-BEING-HANDLED* bound to the key name
	 (LET ((KEY-NAME (LOOKUP-KEY-NAME INPUT)))
	   (COND ((AND (NOT-NULL KEY-NAME) (BOXER-FDEFINED? KEY-NAME))
		  (LET ((BU:*KEY-CODE-BEING-HANDLED* INPUT)
			(BU:*KEY-NAME-BEING-HANDLED* KEY-NAME))
		    (BOXER-FUNCALL KEY-NAME)))
		 (T
		  (UNHANDLED-BOXER-INPUT INPUT)))))
	((SYMBOLP INPUT)
	 ;; Some sort of symbol in the input stream.
	 (LET ((HANDLER (GET INPUT ':BOXER-INPUT)))
	   (COND ((NOT-NULL HANDLER)
		  (LET ((BU:*KEY-CODE-BEING-HANDLED* NIL)
			(BU:*KEY-NAME-BEING-HANDLED* NIL))
		  (FUNCALL HANDLER INPUT)))
		 (T
		  (UNHANDLED-BOXER-INPUT INPUT)))))
	((LISTP INPUT)
	 ;; Some sort of a blip in the input stream. Usually this is a mouse
	 ;; click, although it can be anything.
	 (LET ((HANDLER (GET (CAR INPUT) ':BOXER-INPUT)))
	   (COND ((NOT-NULL HANDLER)
		  (LET ((BU:*KEY-CODE-BEING-HANDLED* NIL)
			(BU:*KEY-NAME-BEING-HANDLED* NIL))
		  (FUNCALL HANDLER INPUT)))
		 (T
		  (UNHANDLED-BOXER-INPUT INPUT)))))))

(DEFUN UNHANDLED-BOXER-INPUT (IGNORE)
  ;; For now just be obnoxious
  (BEEP))




(DEFUN DEFINE-KEY-AND-ALL-ITS-SHIFTED-KEY-NAMES (KEY-NAME KEY-CODE)
  (LET* ((C-KEY-NAME   (INTERN-IN-BU-PACKAGE (FORMAT NIL "CTRL-~A" KEY-NAME)))
	 (M-KEY-NAME   (INTERN-IN-BU-PACKAGE (FORMAT NIL "META-~A" KEY-NAME)))
	 (C-M-KEY-NAME (INTERN-IN-BU-PACKAGE (FORMAT NIL "CTRL-META-~A" KEY-NAME)))
	 
	 (C-KEY-CODE   (DPB 1 %%KBD-CONTROL-META KEY-CODE))
	 (M-KEY-CODE   (DPB 2 %%KBD-CONTROL-META KEY-CODE))
	 (C-M-KEY-CODE (DPB 3 %%KBD-CONTROL-META KEY-CODE)))
    (DEFINE-KEY-NAME KEY-NAME     KEY-CODE)
    (DEFINE-KEY-NAME C-KEY-NAME   C-KEY-CODE)
    (DEFINE-KEY-NAME M-KEY-NAME   M-KEY-CODE)
    (DEFINE-KEY-NAME C-M-KEY-NAME C-M-KEY-CODE)))

(EVAL-WHEN (LOAD) 
  
  ;; Give names to all the standard character keys. (A - Z) The upper and lower
  ;; case versions of these keys both share the same name, so a function bound
  ;; to that key will need to look at BU:*KEY-CODE-BEING-HANDLED* if it wants
  ;; to know whether the uppercase or lowercase key was typed.
  (LOOP FOR KEY-CODE FROM 101 TO 132
	FOR KEY-NAME = (INTERN-IN-BU-PACKAGE (FORMAT NIL "~C-KEY" KEY-CODE))
	DO
	 (DEFINE-KEY-AND-ALL-ITS-SHIFTED-KEY-NAMES KEY-NAME KEY-CODE)
	 (DEFINE-KEY-AND-ALL-ITS-SHIFTED-KEY-NAMES KEY-NAME (+ KEY-CODE 40)))

  ;; Now give names to all the rest of the keys that we can use the format ~C
  ;; directive to get a name for. Basically these are all the random single
  ;; symbol things on the keyboard like ! @ # ~ : etc.
  (LOOP FOR KEY-CODE FROM 0 TO #O177
	UNLESS (OR (AND (>= KEY-CODE 101) (<= KEY-CODE 132))
		   (AND (>= KEY-CODE 141) (<= KEY-CODE 172)))
	DO
	 (DEFINE-KEY-AND-ALL-ITS-SHIFTED-KEY-NAMES
	   (INTERN-IN-BU-PACKAGE (FORMAT NIL "~C-KEY" KEY-CODE))
	   KEY-CODE))
  
  
  ;; Give names to all the keys that we can't use the format ~C directive
  ;; to get a name for. Basically these are keys like SPACE, RUBOUT etc.
  ;; Now I know that there is a place in zwei, where it knows how to do
  ;; this, and that I could use that if I wanted to, but I would like this
  ;; to work in the next system release.
  (LOOP FOR KEY-THAT-FORMAT-~C-LOSES-ON IN '((BU:SPACE-KEY        #\SPACE)
					     (BU:RETURN-KEY       #\RETURN)
					     (BU:RUBOUT-KEY       #\RUBOUT)
					     (BU:BREAK-KEY        #\BREAK)
					     (BU:HELP-KEY         #\HELP)
					     (BU:LINE-KEY         #\LINE)
					     (BU:END-KEY          #\END)
					     (BU:CLEAR-INPUT-KEY  #\CLEAR-INPUT)
				       #-3600(BU:STATUS-KEY       #\STATUS)

				       #+CADR(BU:ALTMODE-KEY      #\ALTMODE)
				       #+3600(BU:COMPLETE-KEY     #\COMPLETE)
				       #+3600(BU:ESCAPE-KEY     #\ESCAPE)
				       #+TI  (BU:ESCAPE-KEY       #\ESCAPE)
					     
				       #-3600(BU:CLEAR-SCREEN-KEY #\CLEAR-SCREEN)
				       #+3600(BU:PAGE-KEY         #\PAGE)

				             (BU:QUOTE-KEY        #\QUOTE)
				       
				       #+CADR(BU:ROMAN-I-KEY      #\ROMAN-I)
				       #+CADR(BU:ROMAN-II-KEY     #\ROMAN-II)
				       #+CADR(BU:ROMAN-III-KEY    #\ROMAN-III)
				       #+CADR(BU:ROMAN-IV-KEY     #\ROMAN-IV)

				       #+TI  (BU:UNDO-KEY         #\UNDO)
				       #+TI  (BU:F1-KEY           #\F1)
				       #+TI  (BU:F2-KEY           #\F2)
				       #+TI  (BU:F3-KEY           #\F3)
				       #+TI  (BU:F4-KEY           #\F4)
				       
				       #+CADR(BU:HAND-DOWN-KEY    #\HAND-DOWN)
				       #+CADR(BU:HAND-UP-KEY      #\HAND-UP)
				       #+CADR(BU:HAND-LEFT-KEY    #\HAND-LEFT)
				       #+CADR(BU:HAND-RIGHT-KEY   #\HAND-RIGHT)
                                       #+3600 (BU:SQUARE-KEY       #\SQUARE)
                                       #+3600 (BU:SCROLL-KEY       #\SCROLL)
				       #+3600 (BU:CIRCLE-KEY       #\CIRCLE)
			               #+3600 (BU:TRIANGLE-KEY     #\TRIANGLE)
				             )
	DO (DEFINE-KEY-AND-ALL-ITS-SHIFTED-KEY-NAMES
	     (CAR  KEY-THAT-FORMAT-~C-LOSES-ON)
	     (CADR KEY-THAT-FORMAT-~C-LOSES-ON)))
  )



;;; Give Boxer-Function Names to all the standard mouse-clicks

(DEFUN LOOKUP-CLICK-NAME (CLICK &OPTIONAL (COMTAB MOUSE-CLICK-NAMES))
  (AND (FIXNUMP CLICK)
       (>= (LDB %%KBD-MOUSE-BUTTON CLICK) 0)
       (<= (LDB %%KBD-MOUSE-BUTTON CLICK) 2.)
       (>= (LDB %%KBD-CONTROL-META CLICK) 0)
       (<= (LDB %%KBD-CONTROL-META CLICK) 15.)
       (AREF COMTAB (LDB %%KBD-MOUSE-BUTTON CLICK)
	            (LDB %%KBD-MOUSE-N-CLICKS CLICK)
	            (LDB %%KBD-CONTROL-META CLICK))))

(DEFUN LOOKUP-STATE-NAME (STATE &OPTIONAL (COMTAB MOUSE-STATE-NAMES))
  (AND (FIXNUMP STATE)
       (>= (LDB %%KBD-MOUSE-BUTTON STATE) 0)
       (<= (LDB %%KBD-MOUSE-BUTTON STATE) 2.)
       (>= (LDB %%KBD-CONTROL-META STATE) 0)
       (<= (LDB %%KBD-CONTROL-META STATE) 15.)
       (AREF COMTAB (LDB %%KBD-MOUSE-BUTTON STATE)
	            (LDB %%KBD-MOUSE-UP-STATE STATE)
	            (LDB %%KBD-CONTROL-META STATE))))

(DEFUN DEFINE-CLICK-NAME (CLICK-NAME CLICK COMTAB STATE-SPECIFIER)
  (COND ((NUMBERP CLICK)
	 (ASET CLICK-NAME
	       COMTAB
	       (LDB %%KBD-MOUSE-BUTTON CLICK)
	       (LDB STATE-SPECIFIER CLICK)
	       (LDB %%KBD-CONTROL-META CLICK)))
	(T
	 (FERROR "~S is a completely unknown type of Boxer Input." CLICK))))

(DEFUN DEFINE-CLICK-AND-ALL-ITS-SHIFTED-CLICK-NAMES (CLICK-NAME CLICK-CODE COMTAB
						     STATE-SPECIFIER)
  (LET* ((C-CLICK-NAME   (INTERN-IN-BU-PACKAGE (FORMAT NIL "CTRL-~A" CLICK-NAME)))
	 (M-CLICK-NAME   (INTERN-IN-BU-PACKAGE (FORMAT NIL "META-~A" CLICK-NAME)))
	 (C-M-CLICK-NAME (INTERN-IN-BU-PACKAGE (FORMAT NIL "CTRL-META-~A" CLICK-NAME)))

	 (C-CLICK-CODE   (DPB 1 %%KBD-CONTROL-META CLICK-CODE))
	 (M-CLICK-CODE   (DPB 2 %%KBD-CONTROL-META CLICK-CODE))
	 (C-M-CLICK-CODE (DPB 3 %%KBD-CONTROL-META CLICK-CODE)))
    
    (DEFINE-CLICK-NAME (INTERN-IN-BU-PACKAGE CLICK-NAME) CLICK-CODE COMTAB STATE-SPECIFIER)
    (DEFINE-CLICK-NAME C-CLICK-NAME   C-CLICK-CODE   COMTAB STATE-SPECIFIER)
    (DEFINE-CLICK-NAME M-CLICK-NAME   M-CLICK-CODE   COMTAB STATE-SPECIFIER)
    (DEFINE-CLICK-NAME C-M-CLICK-NAME C-M-CLICK-CODE COMTAB STATE-SPECIFIER)))

(DEFUN DEFINE-CLICK-AND-ALL-ITS-MULTIPLE-CLICK-NAMES (CLICK-NAME CLICK-CODE COMTAB)
  (LET ((1-CLICK-NAME (FORMAT NIL "~A-ONCE" CLICK-NAME))
	(2-CLICK-NAME (FORMAT NIL "~A-TWICE" CLICK-NAME))

	(1-CLICK-CODE (DPB 0 %%KBD-MOUSE-N-CLICKS CLICK-CODE))
	(2-CLICK-CODE (DPB 1 %%KBD-MOUSE-N-CLICKS CLICK-CODE)))
    (DEFINE-CLICK-AND-ALL-ITS-SHIFTED-CLICK-NAMES
      1-CLICK-NAME 1-CLICK-CODE COMTAB %%KBD-MOUSE-N-CLICKS)
    (DEFINE-CLICK-AND-ALL-ITS-SHIFTED-CLICK-NAMES
      2-CLICK-NAME 2-CLICK-CODE COMTAB %%KBD-MOUSE-N-CLICKS)))

(DEFUN DEFINE-INPUT-STATE-AND-ALL-ITS-MULTIPLE-STATE-NAMES (STATE-NAME STATE-CODE COMTAB)
  (LET ((1-STATE-NAME (FORMAT NIL "~A-DOWN" STATE-NAME))
	(2-STATE-NAME (FORMAT NIL "~A-UP" STATE-NAME))

	(1-STATE-CODE (DPB 0 %%KBD-MOUSE-UP-STATE STATE-CODE))
	(2-STATE-CODE (DPB 1 %%KBD-MOUSE-UP-STATE STATE-CODE)))
    (DEFINE-CLICK-AND-ALL-ITS-SHIFTED-CLICK-NAMES
      1-STATE-NAME 1-STATE-CODE COMTAB %%KBD-MOUSE-UP-STATE)
    (DEFINE-CLICK-AND-ALL-ITS-SHIFTED-CLICK-NAMES
      2-STATE-NAME 2-STATE-CODE COMTAB %%KBD-MOUSE-UP-STATE)))

(DEFUN DEFINE-NAMES-FOR-EACH-MOUSE-BUTTON (&OPTIONAL
					   (COMTAB MOUSE-CLICK-NAMES)
					   (DEF-FCN
					     'DEFINE-CLICK-AND-ALL-ITS-MULTIPLE-CLICK-NAMES)
					   (DEVICE "MOUSE")
					   (CLICK-CODE (DPB 1 %%KBD-MOUSE 0)))
  "This is the top level function to call in order to define symbolic names for clicks on
   a pointing device.  It will make symbolic names for left,middle,right and single,double
   or shifted clicks on some input device. "
  (LET ((L-CLICK-NAME (FORMAT NIL "~A-LEFT" DEVICE))
	(M-CLICK-NAME (FORMAT NIL "~A-MIDDLE" DEVICE))
	(R-CLICK-NAME (FORMAT NIL "~A-RIGHT" DEVICE))
	
        (L-CLICK-CODE (DPB 0 %%KBD-MOUSE-BUTTON CLICK-CODE))
	(M-CLICK-CODE (DPB 1 %%KBD-MOUSE-BUTTON CLICK-CODE))
	(R-CLICK-CODE (DPB 2 %%KBD-MOUSE-BUTTON CLICK-CODE)))
    (FUNCALL DEF-FCN L-CLICK-NAME L-CLICK-CODE COMTAB)
    (FUNCALL DEF-FCN M-CLICK-NAME M-CLICK-CODE COMTAB)
    (FUNCALL DEF-FCN R-CLICK-NAME R-CLICK-CODE COMTAB)))

(EVAL-WHEN (LOAD)
  (DEFINE-NAMES-FOR-EACH-MOUSE-BUTTON)
  (DEFINE-NAMES-FOR-EACH-MOUSE-BUTTON
    MOUSE-STATE-NAMES 'DEFINE-INPUT-STATE-AND-ALL-ITS-MULTIPLE-STATE-NAMES)
  )

(DEFUN (:PROPERTY :MOUSE-CLICK :BOXER-INPUT) (BLIP)
  (LET* ((WINDOW (SECOND BLIP))
	 (CLICK  (THIRD  BLIP))
	 (X-POS  (FOURTH BLIP))
	 (Y-POS  (FIFTH  BLIP))
	 (CLICK-NAME (LOOKUP-CLICK-NAME CLICK)))
    (IF (BOXER-FDEFINED? CLICK-NAME)
	(BOXER-FUNCALL CLICK-NAME WINDOW X-POS Y-POS)
	(UNHANDLED-BOXER-INPUT CLICK))))

(DEFUN (:PROPERTY :MOUSE-HOLD :BOXER-INPUT) (BLIP)
  (LET* ((WINDOW (SECOND BLIP))
	 (STATE  (THIRD  BLIP))
	 (X-POS  (FOURTH BLIP))
	 (Y-POS  (FIFTH  BLIP))
	 (STATE-NAME (LOOKUP-STATE-NAME STATE)))
    (IF (BOXER-FDEFINED? STATE-NAME)
	(BOXER-FUNCALL STATE-NAME WINDOW X-POS Y-POS)
	(UNHANDLED-BOXER-INPUT STATE))))

;;; Documentation Support
(DEFMACRO RECORD-COMMAND-KEY (KEY-NAME COMMAND-NAME)
  `(EVAL-WHEN (COMPILE LOAD EVAL)
     (WHEN (NOT (NULL (ASSQ ,KEY-NAME *BOXER-COMMAND-KEY-ALIST*)))
       (SETQ *BOXER-COMMAND-KEY-ALIST*
	     (DELQ (ASSQ ,KEY-NAME *BOXER-COMMAND-KEY-ALIST*) *BOXER-COMMAND-KEY-ALIST*)))
     (PUSH (CONS ,KEY-NAME ,COMMAND-NAME) *BOXER-COMMAND-KEY-ALIST*)))

;; Note that while there might be several keys for one command, 
;; there can only be one command for each key (at top level)

(DEFUN GET-COMMAND-FOR-KEY (KEY-NAME)
  (CDR (ASSQ KEY-NAME *BOXER-COMMAND-KEY-ALIST*)))

(DEFUN GET-KEYS-FOR-COMMAND (COMMAND)
  (LOOP FOR PAIR IN *BOXER-COMMAND-KEY-ALIST*
	WHEN (EQ COMMAND (CDR PAIR))
	  COLLECT (CAR PAIR)))

;;; Input history

(DEFUN DECODE-INPUT-FOR-PRINTING (INPUT &OPTIONAL (STREAM NIL) &AUX (PREFIX ""))
  (COND ((FIXP INPUT)
	 ;; must be a keystroke
	 (FORMAT STREAM "~A~A~%"
		 (PROG2 (COND-EVERY ((PLUSP (LDB %%KBD-CONTROL INPUT))
				     (SETQ PREFIX (STRING-APPEND "CTRL-" PREFIX)))
				    ((PLUSP (LDB %%KBD-META INPUT))
				     (SETQ PREFIX (STRING-APPEND "META-" PREFIX)))
				    ((PLUSP (LDB %%KBD-SUPER INPUT))
				     (SETQ PREFIX (STRING-APPEND "SUPER-" PREFIX)))
				    ((PLUSP (LDB %%KBD-HYPER INPUT))
				     (SETQ PREFIX (STRING-APPEND "HYPER-" PREFIX))))
			PREFIX)
		 (COND ((= #O40 (LDB %%KBD-CHAR INPUT))
			"SPACE")
		       ((= #O215 (LDB %%KBD-CHAR INPUT))
			"RETURN")
		       (T (FORMAT NIL "~C" (LDB %%KBD-CHAR INPUT))))))
	((LISTP INPUT)
	 ;; some sort of BLIP, probably from the mouse
	 (DECODE-MOUSE-CLICK-FOR-PRINTING (THIRD INPUT) STREAM))	;for now...
	(T INPUT)))

(DEFUN DECODE-MOUSE-CLICK-FOR-PRINTING (CLICK &OPTIONAL (STREAM NIL) &AUX (PREFIX ""))
  (FORMAT STREAM "~AMOUSE-~A~D~%"
	  (PROG2 (COND-EVERY ((PLUSP (LDB %%KBD-CONTROL CLICK))
			      (SETQ PREFIX (STRING-APPEND "CTRL-" PREFIX)))
			     ((PLUSP (LDB %%KBD-META CLICK))
			      (SETQ PREFIX (STRING-APPEND "META-" PREFIX)))
			     ((PLUSP (LDB %%KBD-SUPER CLICK))
			      (SETQ PREFIX (STRING-APPEND "SUPER-" PREFIX)))
			     ((PLUSP (LDB %%KBD-HYPER CLICK))
			      (SETQ PREFIX (STRING-APPEND "HYPER-" PREFIX))))
		 PREFIX)
	  (COND ((= 0 (LDB %%KBD-MOUSE-BUTTON CLICK))
		 "LEFT-")
		((= 1 (LDB %%KBD-MOUSE-BUTTON CLICK))
		 "MIDDLE-")
		((= 2 (LDB %%KBD-MOUSE-BUTTON CLICK))
		 "RIGHT-"))
	  (1+ (LDB %%KBD-MOUSE-N-CLICKS CLICK))))

(DEFUN PRINT-KEYSTROKES (&OPTIONAL (LAST-N (LENGTH *BOXER-KEYSTROKE-HISTORY*)))
  (TERPRI STANDARD-OUTPUT)
  (LOOP FOR INDEX FROM LAST-N DOWNTO 1
	DO (DECODE-INPUT-FOR-PRINTING (NTH INDEX *BOXER-KEYSTROKE-HISTORY*)
				      STANDARD-OUTPUT)))

(DEFUN DUMP-KEYSTROKES (BUFFER-NAME &OPTIONAL(LAST-N (LENGTH *BOXER-KEYSTROKE-HISTORY*)))
  (ZWEI:WITH-EDITOR-STREAM (EDITOR-STREAM ':BUFFER-NAME BUFFER-NAME ':CREATE-P T)
    (LOOP FOR INDEX FROM (- (LENGTH *BOXER-KEYSTROKE-HISTORY*) LAST-N)
	            TO   (1- (LENGTH *BOXER-KEYSTROKE-HISTORY*))
	  DO (DECODE-INPUT-FOR-PRINTING (NTH INDEX *BOXER-KEYSTROKE-HISTORY*)
					EDITOR-STREAM))))
