;-*- mode:lisp; package: Boxer; fonts: cptfont, cptfontb -*-
#|
            Copyright 1985 Massachusetts Institute of Technology

 Permission to use, copy, modify, distribute, and sell this software
 and its documentation for any purpose is hereby granted without fee,
 provided that the above copyright notice appear in all copies and that
 both that copyright notice and this permission notice appear in
 supporting documentation, and that the name of M.I.T. not be used in
 advertising or publicity pertaining to distribution of the software
 without specific, written prior permission.  M.I.T. makes no
 representations about the suitability of this software for any
 purpose.  It is provided "as is" without express or implied warranty.


                                         +-Data--+
                This file is part of the | BOXER | system
                                         +-------+

    This file contains the Definitions for objects used by the BOXER Evaluator
    Variables particular to the Evaluator internals are declared here as well as 
    some useful macros.

|#

;;;; EvBox definitions

;; Stores 2 representations of a row.  The ENTRIES slot is a list of the semantically 
;; interesting values while the ITEMS slot has those same entries interspersed with formatting
;; information. 
;; For Example:
;; A ROW that appears in BOXER as "[] foo  bar ; a comment
;; would have a corresponding EVROW with
;;    ENTRIES = ([] FOO BAR)     and 
;;    ITEMS   = ([] (:SPACES 1) FOO (:SPACES 2) BAR :SEMI-COLON-COMMENT A COMMENT)
;;

(DEFSTRUCT (EVROW (:TYPE :NAMED-ARRAY)
		  :COPIER
		  (:PREDICATE EVROW?)
		  (:CONC-NAME "EVROW-")
		  (:PRINT "#<EVROW ~A>" (PRINT-EVROW-INTERNAL EVROW)))
  (ENTRIES '())
  (ITEMS   '()))

(DEFSTRUCT (EVBOX (:TYPE :NAMED-ARRAY)
		 (:PREDICATE EVBOX?)
		 (:PRINT "#<~A>" (PRINT-EVBOX-INTERNAL EVBOX))
		 (:CONC-NAME "%EVBOX-"))
  (NAME NIL)
  (BINDINGS NIL)
  (ROWS '(())))

(DEFSTRUCT (EVDOIT (:INCLUDE EVBOX)
		  :COPIER
		  (:PREDICATE EVDOIT?)
		  (:PRINT "#<~A>" (PRINT-EVBOX-INTERNAL EVDOIT))
		  :CONSTRUCTOR)
  )

(DEFSTRUCT (EVDATA (:INCLUDE EVBOX)
		  :COPIER
		  (:PREDICATE EVDATA?)
		  (:PRINT "#<~A>" (PRINT-EVBOX-INTERNAL EVDATA))
		  :CONSTRUCTOR)
  )

(DEFSTRUCT (EVPORT (:TYPE :NAMED-ARRAY)
		  (:PREDICATE EVPORT?)
		  (:PRINT "#<EvPORT ~A>" (EVPORT-TARGET EVPORT))
		  :CONSTRUCTOR
		  (:CONC-NAME "%EVPORT-")
		  :COPIER)
  (NAME NIL)
  (TARGET NIL))



;;;; Constructors


;;; shadow out the DEFSTuct created one cause its not smart enough
(DEFUN MAKE-EVDATA-FROM-ROWS (ROW-LIST)
  (MAKE-EVDATA ROWS (MAPCAR #'MAKE-EVROW-FROM-ITEMS ROW-LIST)))

(DEFUN MAKE-EVROW-FROM-ENTRY (ENTRY)
  (MAKE-EVROW ENTRIES (NCONS ENTRY)
	      ITEMS   (NCONS ENTRY)))

(DEFUN MAKE-EVROW-FROM-ENTRIES (ENTRIES)
  (MAKE-EVROW ENTRIES ENTRIES
	      ITEMS ENTRIES))

(DEFUN MAKE-EVROW-FROM-ITEMS (ITEMS)
  (MAKE-EVROW ENTRIES (PARSE-LIST-FOR-EVAL ITEMS)
	      ITEMS ITEMS))

(DEFUN APPEND-EVROWS (&REST EVROWS)
  (MULTIPLE-VALUE-BIND (ENTRIES ITEMS)
      (LOOP FOR EVROW IN EVROWS
	    APPENDING (AND EVROW (EVROW-ENTRIES EVROW)) INTO E
	    APPENDING (AND EVROW (EVROW-ITEMS   EVROW)) INTO I
	    FINALLY
	      (RETURN (VALUES E I)))
  (MAKE-EVROW ENTRIES ENTRIES ITEMS ITEMS)))

(DEFSUBST MAKE-EMPTY-EVROW (&OPTIONAL (SPACES 0))
  (MAKE-EVROW ITEMS (WHEN (> SPACES 0) (NCONS (MAKE-SPACES SPACES)))))



;;;; Printing

(DEFUN PRINT-EVROW-INTERNAL (EVROW)
  (LET ((ROW-ENTRIES (EVROW-ENTRIES EVROW)))
    (FORMAT NIL "~A ~A ~A" (IF (NULL ROW-ENTRIES) "" (CAR ROW-ENTRIES))
	                   (IF (NULL (CADR ROW-ENTRIES)) "" (CADR ROW-ENTRIES))
			   (IF (NULL (CADDR ROW-ENTRIES)) "" "..."))))

(DEFUN PRINT-EVBOX-INTERNAL (EVBOX)
  (FORMAT NIL "~A ~A" (TYPEP EVBOX) (LET ((1ST-ROW (CAR (EVBOX-ROWS EVBOX))))
				      (COND ((NULL 1ST-ROW) "")
					    ((EVROW? 1ST-ROW) (PRINT-EVROW-INTERNAL 1ST-ROW))
					    (T "Bad Row")))))


;;;; Predicates

(DEFSUBST EVAL-BOX? (THING)
  (OR (BOX? THING) (EVBOX? THING)))

(DEFSUBST EVAL-DOIT? (THING)
  (OR (DOIT-BOX? THING) (EVDOIT? THING)))

(DEFSUBST EVAL-DATA? (THING)
  (OR (DATA-BOX? THING) (EVDATA? THING)))

(DEFSUBST EVAL-PORT? (THING)
  (OR (PORT-BOX? THING) (EVPORT? THING)))



;;;; Accessor SUBSTs

(DEFSUBST EVBOX-NAME (EVBOX) (IF (EVPORT? EVBOX)
				 (%EVPORT-NAME EVBOX)
				 (%EVBOX-NAME EVBOX)))

(DEFSUBST EVBOX-BINDINGS (EVBOX) (IF (EVPORT? EVBOX)
				     (GET-LOCAL-ENV (EVPORT-TARGET EVBOX))
				     (%EVBOX-BINDINGS EVBOX)))

(DEFSUBST EVBOX-ROWS (EVBOX) (IF (EVAL-PORT? EVBOX) (FERROR "can't get the rows of a PORT")
				 (%EVBOX-ROWS EVBOX)))

(DEFSUBST EVPORT-TARGET (EVPORT) (%EVPORT-TARGET EVPORT))

;;; somewhat higher level row accessors
(DEFSUBST EVBOX-ROW-ENTRIES (EVBOX)
  (MAPCAR #'EVROW-ENTRIES (EVBOX-ROWS EVBOX)))

(DEFSUBST EVBOX-ROW-ITEMS (EVBOX)
  (MAPCAR #'EVROW-ITEMS (EVBOX-ROWS EVBOX)))

;;; mutator substs

(DEFSUBST SET-EVBOX-NAME (EVBOX NEW-NAME)
  (IF (EVPORT? EVBOX) (SETF (%EVPORT-NAME EVBOX) NEW-NAME)
      (SETF (%EVBOX-NAME EVBOX) NEW-NAME)))

(DEFSUBST SET-EVBOX-BINDINGS (EVBOX NEW-BINDINGS)
  (IF (EVPORT? EVBOX) (SETF (%EVBOX-BINDINGS (EVPORT-TARGET EVBOX)) NEW-BINDINGS)
  (SETF (%EVBOX-BINDINGS EVBOX) NEW-BINDINGS)))

(DEFUN SET-EVBOX-ROWS (EVBOX NEW-ROWS)
  (IF (EVPORT? EVBOX)
      (SET-EVBOX-ROWS (EVPORT-TARGET EVBOX) NEW-ROWS)
      (SETF (%EVBOX-ROWS EVBOX) NEW-ROWS)))

#-LMITI(DEFPROP EVBOX-ROWS    ((EVBOX-ROWS EVBOX) SET-EVBOX-ROWS EVBOX SI:VAL) SETF)

#+LMITI(DEFSETF EVBOX-ROWS (EVBOX) (NEW-ROWS) `(SET-EVBOX-ROWS  ,EVBOX ,NEW-ROWS))

(DEFVAR *SPACING-INFO-SYMBOL* :SPACES)

;;; comments

(DEFVAR *VERTICAL-BAR-COMMENT* :VERTICAL-BAR-COMMENT)
(DEFVAR *SEMI-COLON-COMMENT*   :SEMI-COLON-COMMENT)

(PUTPROP *VERTICAL-BAR-COMMENT* #/| 'CONVERTED-CHARACTER)
(PUTPROP *SEMI-COLON-COMMENT*   #/; 'CONVERTED-CHARACTER)

(DEFVAR *COMMENT-CHA-SYMBOLS* `(,*VERTICAL-BAR-COMMENT* ,*SEMI-COLON-COMMENT*))

(DEFVAR *FUNNY-FUNCTION-ARGLIST-TABLE* (MAKE-HASH-TABLE))

(DEFVAR *SYMBOLS-FOR-INPUT-LINE* '(BU: BU:INPUT BU:INPUTS))

(DEFVAR *EVALUATOR-COPYING-ON?* T
  "A Flag which controls the automatic copying of objects in the evaluator.  ")

(DEFVAR *EVALUATOR-COPYING-FUNCTION* 'SHALLOW-COPY-FOR-EVALUATOR)

(DEFVAR *MULTIPLE-ROW-TOP-LEVEL-UNBOX-ACTION* :FLATTEN
  "What happens when we unbox a box with multiple rows at top level. Valid values are
:ERROR (signal an error), :TRUNCATE (use only the top row) and :FLATTEN (use each row
sequentially). ")

;;; Here are the special markers used to alter the default behavior of objects
;;; in the Evaluator

(DEFVAR EVAL-SPECIAL-MARKERS NIL
  "A list of all the special markers used by the evaluator. ")

(DEFMACRO DEFINE-EVAL-MARKER-PREDICATE (NAME VALUE)
  (LET ((PREDICATE-NAME (INTERN (FORMAT NIL "~A?" (GET-PNAME NAME)))))
    `(DEFSUBST ,PREDICATE-NAME (MARKER)
       (OR (EQ ',VALUE MARKER)
	   (AND (LISTP MARKER) (MEMQ ',VALUE MARKER))))))

(DEFMACRO DEFINE-MARKER-READER-MACRO (NAME VALUE CHA)
  (WHEN (CHA? CHA)
    (LET ((MACRO-NAME (INTERN (FORMAT NIL "BOXER-~A-READER-MACRO" (GET-PNAME NAME)))))
      `(PROGN 'COMPILE
	      (SET-SYNTAX-MACRO-CHAR ,CHA ',MACRO-NAME *BOXER-READTABLE*)
	      (PUTPROP ,VALUE ,CHA 'CONVERTED-CHARACTER)
	      (DEFUN ,MACRO-NAME (LIST-SO-FAR IGNORE)
		  (VALUES (APPEND LIST-SO-FAR (NCONS ,VALUE)) NIL T))))))

(DEFMACRO DEFINE-EVAL-MARKER (NAME VALUE ACTION-TYPE &OPTIONAL (ALIASES NIL) (READER-CHA NIL))
  `(PROGN 'COMPILE
	  (DOLIST (ALIAS ',(APPEND ALIASES (NCONS VALUE)))
	    (PUTPROP ALIAS ',VALUE :BOXER-INPUT-FLAVOR))
	  (DEFCONST ,NAME ',VALUE ',ACTION-TYPE)
	  (DEFINE-MARKER-READER-MACRO ,NAME ',VALUE ,READER-CHA)
	  (DEFPROP ,VALUE ,ACTION-TYPE :ACTION-TYPE)
	  (DEFINE-EVAL-MARKER-PREDICATE ,NAME ,VALUE)
	  (PUSH ',VALUE EVAL-SPECIAL-MARKERS)))

	  

;; for Ports

(DEFUN GET-PORT-TARGET (PORT)
  (IF (PORT-BOX? PORT) (TELL PORT :PORTS)
      (EVPORT-TARGET PORT)))

(DEFSUBST BOX-OR-PORT-TARGET (BOX-OR-PORT)
  "Gets you something that is NOT a port"
  (IF (EVAL-PORT? BOX-OR-PORT) (GET-PORT-TARGET BOX-OR-PORT) BOX-OR-PORT))

;;; Insure that an EvBox is returned when selecting parts of an EvBox which is using 
;;; the shallow copying representation

(DEFMACRO GUARANTEE-COPY (BOX-OR-EVBOX)
  `(IF (BOX? ,BOX-OR-EVBOX) (MAKE-EVBOX-FROM-BOX ,BOX-OR-EVBOX)
       ,BOX-OR-EVBOX))


;;;; spaces and comments

;;;; How to Deal with spaces and other irrelevant stuff
;;;  in this representation, spaces are represented by a CONS whose CAR is the value of
;;;  *SPACING-INFO-SYMBOL* and whose CDR is the number of spaces

(DEFSUBST MAKE-SPACES (N)
  (CONS *SPACING-INFO-SYMBOL* N))

(DEFSUBST GET-SPACES (SPACER-ITEM)
  (CDR SPACER-ITEM))

(DEFSUBST SPACES? (EVROW-ITEM)
  (AND (LISTP EVROW-ITEM)(EQ (CAR EVROW-ITEM) *SPACING-INFO-SYMBOL*)))

(DEFSUBST COMMENT-CHA? (EVROW-ITEM)
  (MEMQ EVROW-ITEM *COMMENT-CHA-SYMBOLS*))
