;; -*- Mode:LISP; Package:(BOXER GLOBAL 1000); Fonts:cptfont; Base:10. -*-
;;
;; Copyright 1982-1985 Massachusetts Institute of Technology
;;
;; Permission to use, copy, modify, distribute, and sell this software
;; and its documentation for any purpose is hereby granted without fee,
;; provided that the above copyright notice appear in all copies and that
;; both that copyright notice and this permission notice appear in
;; supporting documentation, and that the name of M.I.T. not be used in
;; advertising or publicity pertaining to distribution of the software
;; without specific, written prior permission.  M.I.T. makes no
;; representations about the suitability of this software for any
;; purpose.  It is provided "as is" without express or implied warranty.
;;
;;
;; This file is part of the BOXER system.
;;
;; Written by Gregor (GREGOR@MIT-AI) et al
;;
;; This file contains the defs for Boxer.
;;



;;;; GRAY PATTERNS

;; These are useful for drawing gray areas on the screen.

(DEFUN MAKE-PATTERN (LIST-OF-ROWS)
   (LET ((ARRAY #-LMITI (MAKE-ARRAY `(32. ,(LENGTH LIST-OF-ROWS)) ':TYPE 'ART-1B)
		#+LMITI (MAKE-PIXEL-ARRAY 32. (LENGTH LIST-OF-ROWS) ':TYPE 'ART-1B))
	 (CURRENT-ROW 0) (CURRENT-COLUMN 0))
     (DOLIST (ROW LIST-OF-ROWS)
       (DO () (NIL)
	 (DOLIST (ELEMENT ROW)
	   (IF (> CURRENT-COLUMN 31.) (RETURN NIL))
	   (ASET ELEMENT ARRAY #-LMITI CURRENT-COLUMN CURRENT-ROW #+LMITI CURRENT-COLUMN)
	   (SETQ CURRENT-COLUMN (1+ CURRENT-COLUMN)))
	 (IF (> CURRENT-COLUMN 31.) (RETURN NIL)))
       (SETQ CURRENT-ROW (1+ CURRENT-ROW)
	     CURRENT-COLUMN 0))
     ARRAY))

(DEFVAR *GRAY0* (MAKE-PATTERN
		  '((1 0 0 0 0 1 0 0 0 0)
		    (0 0 1 0 0 0 0 1 0 0)
		    (0 0 0 0 1 0 0 0 0 1)
		    (0 1 0 0 0 0 1 0 0 0)
		    (0 0 0 1 0 0 0 0 1 0))))

(DEFVAR *GRAY1* (MAKE-PATTERN
		  '((1 0 0 0 1 0 0 0)
		    (0 1 0 0 0 1 0 0)
		    (0 0 0 1 0 0 0 1)
		    (0 0 1 0 0 0 1 0))))
(DEFVAR *GRAY2* (MAKE-PATTERN
		  '((1 0 0 0)
		    (0 0 1 0)
		    (0 1 0 0))))
(DEFVAR *GRAY3* (MAKE-PATTERN
		  '((1 0 0 0 1 0 1 0)
		    (0 1 0 1 0 0 0 1)
		    (1 0 0 0 1 0 1 0)
		    (0 1 0 1 0 0 0 1))))
(DEFVAR *GRAY4* (MAKE-PATTERN
		  '((1 0 1 0 1 0 1 0)
		    (0 1 0 0 0 1 0 0)
		    (1 0 1 0 1 0 1 0))))
(DEFVAR *GRAY5* (MAKE-PATTERN
		  '((1 0 1 0 1 0 1 0)
		    (0 1 0 1 0 1 0 1)
		    (1 0 1 0 1 0 1 0)
		    (0 1 0 1 0 1 0 1))))



;;; Random useful macros.

(DEFUN WARN-ABOUT-INTERNAL-FUNCTION (FN-NAME)
  (COMPILER:WARN '(:BAD-STYLE) "~S is an internal function -- you may lose." FN-NAME))

(DEFMACRO BARF (CONDITION . ERROR-INIT-OPTIONS)
  `(ERROR ,CONDITION . ,ERROR-INIT-OPTIONS))

(DEFMACRO NOT-NULL (X)
  `(NOT (NULL ,X)))

(DEFMACRO ENSURE-LIST (ITEM . IGNORE)
  `(IF (AND ,ITEM (NOT (LISTP ,ITEM))) (SETF ,ITEM (NCONS ,ITEM))))

(DEFMACRO LIST-OR-LISTIFY (ITEM)
  `(IF (NOT (LISTP ,ITEM)) (NCONS ,ITEM) ,ITEM))

;;; This is an abbreviation for SEND which also has the feature of quoting
;;; the second argument (or message) automatically.
(DEFMACRO TELL (INSTANCE MESSAGE-NAME . ARGS)    
  (ONCE-ONLY (INSTANCE)				;<<<*** Get this not-null check
    `(AND (NOT-NULL ,INSTANCE)			;<<<*** out of here soon!!!!!!!
	  (SEND ,INSTANCE ',MESSAGE-NAME . ,ARGS))))

;;; This version of tell checks to see if its first agument (the instance)
;;; is nil. If it is, it doesn't try to send the message, and just returns
;;; nil.
(DEFMACRO TELL-CHECK-NIL (INSTANCE MESSAGE-NAME . ARGS)
  (ONCE-ONLY (INSTANCE)
    `(AND (NOT-NULL ,INSTANCE)
	  (SEND ,INSTANCE ',MESSAGE-NAME . ,ARGS))))

(DEFMACRO MAP-TELL (LIST-OF-INSTANCES MESSAGE-NAME . ARGS)
  `(MAPCAR '(LAMBDA (INSTANCE) (SEND INSTANCE ',MESSAGE-NAME . ,ARGS)) ,LIST-OF-INSTANCES))


;;; These list hacking macros are so useful that I expect all MIT arpanet
;;; ports to be tied up for months while everybody copies them.

(DEFMACRO SPLICE-LIST-INTO-LIST (INTO-LIST LIST BEFORE-ITEM)
  `(SETF ,INTO-LIST (SPLICE-LIST-INTO-LIST-1 ,INTO-LIST ,LIST ,BEFORE-ITEM)))

(DEFMACRO SPLICE-ITEM-INTO-LIST (INTO-LIST ITEM BEFORE-ITEM)
  `(SETF ,INTO-LIST (SPLICE-LIST-INTO-LIST-1 ,INTO-LIST `(,,ITEM) ,BEFORE-ITEM)))

(DEFUN SPLICE-LIST-INTO-LIST-1 (INTO-LIST LIST BEFORE-ITEM)
  (LET ((BEFORE-ITEM-POSITION (FIND-POSITION-IN-LIST BEFORE-ITEM INTO-LIST)))
    (COND ((OR (NULL BEFORE-ITEM-POSITION)
	       (= BEFORE-ITEM-POSITION 0))
	   (NCONC LIST INTO-LIST)
	   LIST)
	  (T
	   (DO* ((TAIL INTO-LIST (CDR TAIL))
		 (NEXT-ITEM (CADR TAIL) (CADR TAIL)))
		((EQ NEXT-ITEM BEFORE-ITEM)
		 (NCONC LIST (CDR TAIL))
		 (RPLACD TAIL LIST)
		 INTO-LIST))))))

(DEFMACRO SPLICE-LIST-ONTO-LIST (ONTO-LIST LIST)
  `(SETF ,ONTO-LIST (NCONC ,ONTO-LIST ,LIST)))

(DEFMACRO SPLICE-ITEM-ONTO-LIST (ONTO-LIST ITEM)
  `(SPLICE-LIST-ONTO-LIST ,ONTO-LIST `(,,ITEM)))

;(DEFMACRO SPLICE-LIST-OUT-OF-LIST (&YOW LOSING-BADLY)) ;doesn't make sense

(DEFMACRO SPLICE-ITEM-OUT-OF-LIST (OUT-OF-LIST ITEM)
  `(SETF ,OUT-OF-LIST (DELETE ,ITEM ,OUT-OF-LIST)))

(DEFMACRO SPLICE-ITEM-AND-TAIL-OUT-OF-LIST (OUT-OF-LIST ITEM)
  `(SETF ,OUT-OF-LIST (SPLICE-ITEM-AND-TAIL-OUT-OF-LIST-1 ,OUT-OF-LIST ,ITEM)))

(DEFUN SPLICE-ITEM-AND-TAIL-OUT-OF-LIST-1 (OUT-OF-LIST ITEM)
  (LET ((ITEM-POSITION (FIND-POSITION-IN-LIST ITEM OUT-OF-LIST)))
    (COND ((NULL ITEM-POSITION) OUT-OF-LIST)
	  ((= ITEM-POSITION 0) NIL)
	  (T (RPLACD (NTHCDR (- ITEM-POSITION 1) OUT-OF-LIST) NIL)
	     OUT-OF-LIST))))

(DEFMACRO SPLICE-BETWEEN-ITEMS-OUT-OF-LIST (LIST FROM-ITEM TO-ITEM)
  `(DO ((FROM-ITEM-PREVIOUS-CONS NIL FROM-ITEM-PREVIOUS-CONS)
	(TO-ITEM-PREVIOUS-CONS NIL TO-ITEM-PREVIOUS-CONS)
	(SCAN ,LIST (CDR SCAN)))
       ((OR (NULL SCAN) (NOT-NULL TO-ITEM-PREVIOUS-CONS))
	(COND ((NULL FROM-ITEM-PREVIOUS-CONS)
	       (SETF ,LIST (CDR TO-ITEM-PREVIOUS-CONS)))
	      (T
	       (RPLACD FROM-ITEM-PREVIOUS-CONS (CDR TO-ITEM-PREVIOUS-CONS))))
	(RPLACD TO-ITEM-PREVIOUS-CONS NIL))
     (COND ((EQ (CADR SCAN) ,FROM-ITEM)
	    (SETQ FROM-ITEM-PREVIOUS-CONS SCAN))
	   ((EQ (CADR SCAN) ,TO-ITEM)
	    (SETQ TO-ITEM-PREVIOUS-CONS SCAN)))))


;;;new list splicing macros that use index numbers...

(DEFMACRO SPLICE-LIST-INTO-LIST-AT (INTO-LIST LIST POSITION)
  `(COND ((= ,POSITION 0)
	  (SETF ,INTO-LIST (NCONC ,LIST ,INTO-LIST)))
	 (( ,POSITION (LENGTH ,INTO-LIST))
	  (SETF ,INTO-LIST (NCONC ,INTO-LIST ,LIST)))
	 (T (SETF ,INTO-LIST (NCONC (FIRSTN ,POSITION ,INTO-LIST)
				    ,LIST
				    (NTHCDR ,POSITION ,INTO-LIST))))))

(DEFMACRO SPLICE-ITEM-INTO-LIST-AT (INTO-LIST ITEM POSITION)
  `(SPLICE-LIST-INTO-LIST-AT ,INTO-LIST `(,,ITEM) ,POSITION))

(DEFMACRO SPLICE-ITEM-OUT-OF-LIST-AT (LIST POSITION)
  `(COND ((= ,POSITION 0)
	  (SETF ,LIST (CDR ,LIST)))
	 (( ,POSITION (LENGTH ,LIST))
	  (SETF ,LIST (BUTLAST ,LIST)))
	 (T (SETF ,LIST (NCONC (FIRSTN ,POSITION ,LIST)
			       (NTHCDR (+ ,POSITION 1) ,LIST))))))

(DEFMACRO SPLICE-ITEM-AND-TAIL-OUT-OF-LIST-FROM (LIST POSITION)
  `(COND (( ,POSITION (LENGTH ,LIST)))
	 (T (SETF ,LIST (FIRSTN ,POSITION ,LIST)))))

(DEFMACRO SPLICE-ITEMS-FROM-TO-OUT-OF-LIST (LIST START-POSITION STOP-POSITION)
  `(COND ((> ,START-POSITION ,STOP-POSITION)
	  (FERROR "The Starting number: ~S is greater than the ending number ~S"
		  ,START-POSITION ,STOP-POSITION))
	 (( ,START-POSITION (LENGTH ,LIST)))
	 ((= ,START-POSITION ,STOP-POSITION)
	  (SPLICE-ITEM-OUT-OF-LIST-AT ,LIST ,START-POSITION))
	 (( ,STOP-POSITION (LENGTH ,LIST))
	  (SPLICE-ITEM-AND-TAIL-OUT-OF-LIST-FROM ,LIST ,START-POSITION))
	 (T (SETF ,LIST (NCONC (FIRSTN ,START-POSITION ,LIST)
			       (NTHCDR ,STOP-POSITION ,LIST))))))

(DEFMACRO ITEMS-SPLICED-FROM-TO-FROM-LIST (LIST START-POSITION STOP-POSITION)
  `(COND ((> ,START-POSITION ,STOP-POSITION)
	  (FERROR "The Starting number: ~S is greater than the ending number ~S"
		  ,START-POSITION ,STOP-POSITION))
	 (( ,START-POSITION (LENGTH ,LIST))
	  '())
	 ((= ,START-POSITION ,STOP-POSITION)
	  (LIST (NTH ,START-POSITION ,LIST)))
	 (( ,STOP-POSITION (LENGTH ,LIST))
	  (NTHCDR ,START-POSITION ,LIST))
	 (T (FIRSTN (- ,STOP-POSITION ,START-POSITION)
		    (NTHCDR ,START-POSITION ,LIST)))))



;; COLLECT is straight from the book, and is documented there.
(DEFVAR *COLLECT-VARIABLE*)

(DEFMACRO WITH-COLLECTION (&BODY BODY)
  (LET ((VAR (GENSYM)))
    `(LET ((,VAR NIL))
       (COMPILER-LET ((*COLLECT-VARIABLE* ',VAR))
		     . ,BODY)
       (NREVERSE ,VAR))))

(DEFMACRO COLLECT (THING)
  `(PUSH ,THING ,*COLLECT-VARIABLE*))


(DEFMACRO DOPLIST ((PLIST PROPERTY INDICATOR) &BODY BODY)
  (LET ((PLIST-SYMBOL (GENSYM)))
    `(DO ((,PLIST-SYMBOL ,PLIST (CDDR ,PLIST-SYMBOL))
	  (,PROPERTY) (,INDICATOR))
	 ((NULL ,PLIST-SYMBOL))
       (SETQ ,PROPERTY  (CADR ,PLIST-SYMBOL)
	     ,INDICATOR (CAR ,PLIST-SYMBOL))
       ,@BODY)))

;; Working inside is neat, and is best documented by example:
;;
;;(DEFUN TEST-WORKING-INSIDE-LIST ()
;;  (LET ((TEST-LIST (LIST 1 2 3)))
;;    (FORMAT T "~%Before  -> ~s" TEST-LIST)
;;    (WORKING-INSIDE-LIST (A B C) TEST-LIST (SETQ A 4 B 5 C 6))
;;    (FORMAT T "~%After   -> ~s" TEST-LIST)))
;;
;;(TEST-WORKING-INSIDE-LIST)
;;Before -> (1 2 3)
;;After  -> (4 5 6)

(DEFMACRO WORKING-INSIDE (VARS LOCS &BODY BODY)
  `(LOCAL-DECLARE ((SPECIAL . ,VARS))
     ; Use progv because it returns multiple values.
     (PROGV ',VARS NIL
	    (LOOP FOR VAR-LOC IN (MAPCAR #'VARIABLE-LOCATION ',VARS)
		  FOR VAL-LOC IN ,LOCS
		  DO
		   (%P-STORE-TAG-AND-POINTER VAR-LOC DTP-EXTERNAL-VALUE-CELL-POINTER VAL-LOC))
	    . ,BODY)))

(DEFMACRO WORKING-INSIDE-LIST (VARS LIST &BODY BODY)
  `(WORKING-INSIDE ,VARS (LOCIFY-LIST ,LIST)
     . ,BODY))

(DEFUN LOCIFY-LIST (LIST)
  (LOOP FOR L ON LIST COLLECT (LOCF (CAR L))))

(DEFMACRO MAXIMIZE (VAR . VALS)
  `(SETF ,VAR (MAX ,VAR . ,VALS)))

(DEFMACRO MINIMIZE (VAR . VALS)
  `(SETF ,VAR (MIN ,VAR . ,VALS)))

(DEFMACRO WITH-SUMMATION (&BODY BODY)
  (LET ((SUMMATION-VAR (GENSYM)))
    `(LET ((,SUMMATION-VAR 0))
       (COMPILER-LET ((SUMMATION-VAR ',SUMMATION-VAR))
	 (PROGN . ,BODY)
	 ,SUMMATION-VAR))))

(DEFMACRO SUM (X)
  (LOCAL-DECLARE ((SPECIAL SUMMATION-VAR))
    `(INCF ,SUMMATION-VAR ,X)))

;; BETWEEN
(DEFMACRO BETWEEN? (X A B)
  `(OR (AND (> ,X ,A) (< ,X ,B))
       (AND (< ,X ,A) (> ,X ,B))))

(DEFMACRO INCLUSIVE-BETWEEN? (X A B)
  `(OR (AND ( ,X ,A) ( ,X ,B))
       (AND ( ,X ,A) ( ,X ,B))))

(DEFMACRO DEFTYPE-CHECKING-MACROS (TYPE TYPE-STRING)
  (LET ((PREDICATE-NAME (INTERN (FORMAT NIL "~S?" TYPE)))
	(CHECK-ARG-NAME (INTERN (FORMAT NIL "CHECK-~S-ARG" TYPE))))
    `(PROGN 'COMPILE
       (DEFSUBST  ,PREDICATE-NAME (X) (TYPEP X ',TYPE))
       (DEFMACRO  ,CHECK-ARG-NAME (X) `(CHECK-ARG ,X  ,',PREDICATE-NAME ,,TYPE-STRING)))))




;;;; Flavor hacking stuff.

(DEFMACRO DEFGET-METHOD ((FLAVOR-NAME METHOD-NAME) VAR-NAME)
  `(DEFMETHOD (,FLAVOR-NAME ,METHOD-NAME) ()
     ,VAR-NAME))

(DEFMACRO DEFSET-METHOD ((FLAVOR-NAME METHOD-NAME) VAR-NAME)
  `(DEFMETHOD (,FLAVOR-NAME ,METHOD-NAME) (NEW-VALUE)
     (SETQ ,VAR-NAME NEW-VALUE)))

(DEFMACRO DEFMETHOD-ALIAS ((FLAVOR ALIAS-METHOD) TO-METHOD)
  (IF (LISTP TO-METHOD)
      `(DEFF (:METHOD ,FLAVOR ,ALIAS-METHOD) #'(:METHOD . ,TO-METHOD))
      `(DEFF (:METHOD ,FLAVOR ,ALIAS-METHOD) #'(:METHOD ,FLAVOR ,TO-METHOD))))

(DEFMACRO DEFMETHOD-FORWARD ((FLAVOR-NAME METHOD-NAME) FORM-TO-EVAL-AND-FORWARD-TO)
  (ONCE-ONLY (FORM-TO-EVAL-AND-FORWARD-TO)
    `(DEFMETHOD (,FLAVOR-NAME ,METHOD-NAME) (&REST ARGS)
       (UNLESS (NULL ,FORM-TO-EVAL-AND-FORWARD-TO)
	 (LEXPR-SEND ,FORM-TO-EVAL-AND-FORWARD-TO ARGS)))))

(DEFMACRO DEFMETHODS (METHOD-SPECS ARGS . BODY)
  (LET ((MAIN-METHOD-SPEC (CAR METHOD-SPECS))
	(ALIAS-METHOD-SPECS (CDR METHOD-SPECS)))
    `(PROGN 'COMPILE
      (DEFMETHOD ,MAIN-METHOD-SPEC ,ARGS . ,BODY)
      . ,(LOOP FOR ALIAS-METHOD-SPEC IN ALIAS-METHOD-SPECS
	       COLLECT `(DEFMETHOD-ALIAS ,ALIAS-METHOD-SPEC ,MAIN-METHOD-SPEC)))))

(DEFMACRO DEFGET-METHODS (((FLAVOR-NAME METHOD-NAME) . OTHER-METHOD-NAMES) VAR-NAME)
  `(DEFMETHODS ((,FLAVOR-NAME ,METHOD-NAME)
		. ,(LOOP FOR OTHER-METHOD-NAME IN OTHER-METHOD-NAMES
			 COLLECT `(,FLAVOR-NAME ,OTHER-METHOD-NAME))) ()
     ,VAR-NAME))

(DEFMACRO DEFSET-METHODS (((FLAVOR-NAME METHOD-NAME) . OTHER-METHOD-NAMES) VAR-NAME)
  `(DEFMETHODS ((,FLAVOR-NAME ,METHOD-NAME)
		. ,(LOOP FOR OTHER-METHOD-NAME IN OTHER-METHOD-NAMES
			 COLLECT `(,FLAVOR-NAME ,OTHER-METHOD-NAME))) (NEW-VALUE)
     (SETQ ,VAR-NAME NEW-VALUE)))



(defflavor FLAVOR-HACKING-MIXIN
	()
	()
  (:DOCUMENTATION :MIXIN
   "This mixin attempts to make up for the flavor system's total lossage in not
    providing a way for instances to change their flavor.

    We provide a :SET-FLAVOR message which can be sent to an instance to get it
    to change its flavor. If instances of the current and new flavors both have
    the same shape (same instance variables in the same order) the old instance
    is preserved (only its flavor is changed). If instances of the current and
    new flavors do not have the same shape, then an instance the new flavor is
    created, that instance is sent a :INIT-SELF-FROM-OLD-INSTANCE message, and
    the old instance is structure-forwarded to the new instance. We also provide
    a default version of :INIT-SELF-FROM-OLD-INSTANCE which just copies over all
    the instance variables the two flavors have in common and does not touch the
    rest. Many applications will want to define :AFTER daemons on this method.

    NOTE THAT BOTH THE NEW AND OLD FLAVORS NEED TO HAVE FLAVOR-HACKING-MIXIN MIXED IN."))

;; make this use CHANGE-INSTANCE-FLAVOR
(DEFMETHOD (FLAVOR-HACKING-MIXIN :SET-FLAVOR) (NEW-FLAVOR)
  (SEND SELF ':SET-FLAVOR-DESCRIPTOR (GET NEW-FLAVOR 'SI:FLAVOR)))

(DEFMETHOD (FLAVOR-HACKING-MIXIN :SET-FLAVOR-DESCRIPTOR) (NEW-DESCRIPTOR)
  (LET* ((CURRENT-DESCRIPTOR (%MAKE-POINTER #-3600 DTP-ARRAY-POINTER #+3600 SYS:DTP-ARRAY
					    (%P-POINTER SELF)))
	 (CURRENT-INSTANCE-VARIABLES (SI:FLAVOR-ALL-INSTANCE-VARIABLES CURRENT-DESCRIPTOR))
	 (NEW-INSTANCE-VARIABLES (SI:FLAVOR-ALL-INSTANCE-VARIABLES NEW-DESCRIPTOR)))
    (IF (EQUAL CURRENT-INSTANCE-VARIABLES NEW-INSTANCE-VARIABLES)
	(%P-STORE-POINTER SELF NEW-DESCRIPTOR)
	(LET ((NEW-INSTANCE (INSTANTIATE-FLAVOR (SI:FLAVOR-NAME NEW-DESCRIPTOR) ())))
	  (TELL NEW-INSTANCE :INIT-SELF-FROM-OLD-INSTANCE SELF)))))

(DEFMETHOD (FLAVOR-HACKING-MIXIN :INIT-SELF-FROM-OLD-INSTANCE) (OLD-INSTANCE)
  (LET ((OLD-VARIABLES (SI:FLAVOR-ALL-INSTANCE-VARIABLES 
			 (%MAKE-POINTER #-3600 DTP-ARRAY-POINTER #+3600 SYS:DTP-ARRAY
					(%P-POINTER OLD-INSTANCE))))
	(NEW-VARIABLES (SI:FLAVOR-ALL-INSTANCE-VARIABLES 
			 (%MAKE-POINTER #-3600 DTP-ARRAY-POINTER #+3600 SYS:DTP-ARRAY
					(%P-POINTER SELF)))))
    (LOOP FOR VAR IN NEW-VARIABLES
	  WHEN (AND (MEMQ VAR OLD-VARIABLES)
		    #-LMITI(BOUNDP-IN-INSTANCE OLD-INSTANCE VAR) #+LMITI T)
	    DO (SET-IN-INSTANCE SELF VAR (SYMEVAL-IN-INSTANCE
					   OLD-INSTANCE VAR))))
  (STRUCTURE-FORWARD OLD-INSTANCE SELF))
	


(DEFFLAVOR PLIST-MIXIN
	((PLIST NIL))
	()
  (:DOCUMENTATION :MIXIN
   "This gives instances their very own plist. I thought there was a
    system supplied mixin that did this, but I couldn't find it so I
    figured I would just write my own."))

(DEFMETHOD (PLIST-MIXIN :PLIST) ()
  (LOCF PLIST))

(DEFMETHOD (PLIST-MIXIN :GET) (INDICATOR)
  (GET (LOCF PLIST) INDICATOR))

(DEFMETHOD (PLIST-MIXIN :GETL) (LIST-OF-INDICATORS)
  (GETL (LOCF PLIST) LIST-OF-INDICATORS))

(DEFMETHOD (PLIST-MIXIN :PUTPROP) (X INDICATOR)
  (PUTPROP (LOCF PLIST) X INDICATOR))

(DEFMETHOD (PLIST-MIXIN :REMPROP) (INDICATOR)
  (REMPROP (LOCF PLIST) INDICATOR))



(DEFFLAVOR VIRTUAL-COPY-MIXIN
	((VC-ROWS NIL)				;used by virtual copy
	 (INFERIOR-PORTS   NIL)
	 (INFERIOR-TARGETS NIL))
	()
  (:DOCUMENTATION :MIXIN
   "This has Slots That are used by the Virtual Copy Mechanism. "))

;;; All of the methods are defined in the virtcopy file




(DEFFLAVOR UNIQUE-NAME-MIXIN
	((UNIQUE-NAME NIL))
	()
  (:INIT-KEYWORDS :UNIQUE-NAME)
  (:DOCUMENTATION :MIXIN
   "Giving a flavor this mixin will cause objects of that flavor to have
    a unique-name. It will also use that unique-name scheme to only allow
    one object with a certain unique-name to exist at a time. After the
    object is made it it will set the value of its unique-name to itself,
    and when the object is killed it will set the value of its unique-name
    to nil."))

(DEFMETHOD (UNIQUE-NAME-MIXIN :BEFORE :INIT) (INIT-PLIST)
  (LET ((INITIAL-UNIQUE-NAME (GET INIT-PLIST ':UNIQUE-NAME)))
    (WHEN (NOT-NULL INITIAL-UNIQUE-NAME)
      ;; If there is already a window with this unique-name, then
      ;; it must be an earlier copy of us. Kill that window,
      ;; and set our unique-name.
      (AND (BOUNDP INITIAL-UNIQUE-NAME)
	   (NOT (NULL (EVAL INITIAL-UNIQUE-NAME)))
	   (SEND (EVAL INITIAL-UNIQUE-NAME) ':KILL))
      (TELL SELF :SET-UNIQUE-NAME INITIAL-UNIQUE-NAME))))

(DEFMETHOD (UNIQUE-NAME-MIXIN :AFTER :KILL) (&REST IGNORE)
  (AND (BOUNDP UNIQUE-NAME)
       (EQ (EVAL UNIQUE-NAME) SELF)
       (SET UNIQUE-NAME NIL)))

(DEFMETHOD (UNIQUE-NAME-MIXIN :UNIQUE-NAME) ()
  UNIQUE-NAME)

(DEFMETHOD (UNIQUE-NAME-MIXIN :SET-UNIQUE-NAME) (NEW-UNIQUE-NAME)
  ;; If we already have a unique-name, then make it not point
  ;; to us anymore. Then make our new unique-name point to us,
  ;; and remember that that its our unique-name.
  (WHEN (NOT (NULL UNIQUE-NAME))
	(SET UNIQUE-NAME NIL))
  (SET NEW-UNIQUE-NAME SELF)
  (SETQ UNIQUE-NAME NEW-UNIQUE-NAME))




;;;; Stuff that is particular to boxer.

;;;; DEFVARS

(DEFVAR *BOXER-SYNCHRONOUS-INTERCEPTED-CHARACTERS*
	(REM #'(LAMBDA (LIST ITEM) (MEMBER ITEM LIST))
	     '(#\BREAK #\ABORT)
	     TV:KBD-STANDARD-INTERCEPTED-CHARACTERS)
  "These are the characters which Boxer would like the KBD code to
   intercept and deal with synchronously.")

(DEFVAR *RETURNED-VALUES-NOT-TO-PRINT* '(:NOPRINT NOPRINT :? NIL)
  "Items on this list will not be printed out if they are returned by
   from a doit-key.")

(DEFVAR *INSIDE-LISP-BREAKPOINT-P* NIL)

(DEFVAR *POINT* NIL)

(DEFVAR *MARK* NIL)

(DEFVAR *POINT-BLINKER* NIL)
(DEFVAR *CURSOR-BLINKER-WID* 3.)
(DEFVAR *CURSOR-BLINKER-MIN-HEI* 12.)

(DEFVAR *MOUSE-BLINKER* NIL)

(DEFVAR *SPRITE-BLINKER* NIL)

(DEFVAR *MINIMUM-CURSOR-HEIGHT* 12.
  "The minimum height to draw the cursor so that it doesn't dissapear.")

(DEFVAR *MINIMUM-BOX-WID* 25.
  "The minimum width any box will be drawn on the screen.")

(DEFVAR *MINIMUM-BOX-HEI* 25.
  "The minimum height any box will be drawn on the screen.")

(DEFVAR *MULTIPLICATION* 1)

(DEFVAR *KILL-RING* NIL)

(DEFVAR *COM-MAKE-PORT-CURRENT-PORT* NIL
  "This variable is used to store newly created ports until they are inserted into the
   World. ")

(DEFVAR *CURRENT-FONT-NO* 0
  "The no of the font the user is currently using. This number is used to
   to determine the font-no of newly created chas.")

(DEFVAR *BOLDFACE-FONT-NO* 2
  "The font number of boldface characters.  This relies on the details of what the font
map for the *BOXER-PANE* is.  ")

(DEFVAR *ITALICS-FONT-NO* 3
  "The font number of italics characters.  This relies on the details of what the font
map for the *BOXER-PANE* is.  ")

(DEFVAR *BOXER-READTABLE* (COPY-READTABLE SI:INITIAL-READTABLE))

(DEFVAR *INITIAL-BOX* NIL
  "The initial box the editor starts with, this box cannot be deleted
   killed etc.")

(DEFVAR *CURRENT-SCREEN-BOX* NIL
  "The Lowest Level Screen Box Which Contains the *Point*")

(DEFVAR *MARKED-SCREEN-BOX* NIL
  "The Lowest Level Scren Box Which Contains the *mark*")

(DEFVAR *BOXER-FUNCTIONS* NIL
  "This variable contains a list of symbols for all the 
   lisp functions imported to Boxer.")

;;;Region Variables

(DEFVAR *CURRENT-EDITOR-REGION* NIL)

(DEFVAR *REGION-BEING-DEFINED* NIL
  "Bound to a region which is in the process of being delineated.  NIL Otherwise.")

(DEFVAR *KILLED-REGION-BUFFER* NIL
  "this should be integrated into the generic kill buffer eventually")

(DEFVAR *HIGHLIGHT-YANKED-REGION* NIL
  "Controls whether freshly yanked back region should be highlighted. ")

(DEFVAR REGION-LIST NIL)

;;;Box top variables...

(DEFVAR *FONT-NUMBER-FOR-NAMING* 2.
  "The font number that specifies the font for names and variables. ")

;;;variables that PORTS use...

(DEFVAR *PORT-HASH-TABLE* NIL			;this is ONLY used by the old the file system
  "This variable is a table consisting of boxes which are being
   ported to and their TRUE-NAMES. FLUSH AS SOON AS THE FASDUMPER WORKS.")

;;;these are used by the file system

(DEFVAR *BOX-STORAGE-ARRAY* NIL			;this is ONLY used by the old file system
  "This is used for intermediate storage of the box array
   during fast-saves and fast-reads")

(DEFVAR *BOX-STORAGE-LIST* NIL			;this is ONLY used by the old file system
  "This is used for intermediate storage of the box array
   during saves and reads")

(DEFVAR *FILE-PORT-HASH-TABLE* NIL		;this is ONLY used by the old file system
  "This variable is a table consisting of boxes which are being
   ported to along with their TRUE-NAMES. This table is used only
   by file streams. FLUSH AS SOON AS THE FASDUMPER WORKS. ")

(DEFVAR *RENAME-QUEUE* NIL			;this is ONLY used by the old file system
  "A list of boxes which have TRUE-NAME's which need to be changed
   since other boxes may already have those names.")

(DEFVAR *FILE-PORT-QUEUE* NIL			;this is ONLY used by the old file system
  "A list of port boxes waiting for their ported to box to be built 
   by the file system.")


;;;; Variables Having To Do With Redisplay.

(DEFVAR *REDISPLAYABLE-WINDOWS* NIL
  "This is a list of all the windows which should be redisplayed when
   REDISPLAY is called. Windows which have the redisplayable-window-mixin
   take care of adding/removing themselves to/from this list automatically.")

(DEFVAR *REDISPLAY-WINDOW* NIL
  "Inside of REDISPLAYING-WINDOW, this variable is bound to the window
   being redisplayed.")

(DEFVAR *OUTERMOST-BOX* NIL
  "Inside of REDISPLAYING-WINDOW, this variable is bound to the window
   being redisplayed's outermost-box. This is the box which currently
   fills that window.")

(DEFVAR *OUTERMOST-SCREEN-BOX* NIL
  "Inside of REDISPLAYING-WINDOW, this variable is bound to the window
   being redisplayed's outermost-screen-box. This is the screen box which
   represents that window outermost-box.")

(DEFVAR *REDISPLAY-CLUES* NIL
  "A list of redisplay-clues. This are hints left behind by the editor
   to help the redisplay code figure out what is going on.")

(DEFVAR *COMPLETE-REDISPLAY-IN-PROGRESS?* NIL
  "Binding this variable to T around a call to redisplay will 'force'
   the redisplay. That is it will cause a complete redisplay of the
   screen. FORCE-REDISPLAY-WINDOW uses this.")

(DEFVAR *SPACE-AROUND-OUTERMOST-SCREEN-BOX* 9.
  "This is the number of pixels between the outside of the outermost screen
   box and the inside of the window. This space exists to allow the user to
   move the mouse out of the outermost box.")

(DEFVAR *TICK* 0
  "This is the global variable used by the (TICK) function to generate
   a continuously increasing series of integers. This is mostly used by
   the redisplay code although it wouldn't mess things up if (TICK)
   was called by other sections of code.")

(DEFVAR *BOX-ZOOM-WAITING-TIME* 1
  "The amount of time spent waiting between the individual steps when zooming a box. ")

(DEFVAR *CONTROL-CHARACTER-DISPLAY-PREFIX* #/
  "For display of control characters (all of them until we decide on different prefixes")

(DEFUN TICK ()
  (SETQ *TICK* (+ *TICK* 1)))

(DEFVAR *OUTERMOST-SCREEN-BOX-STACK* NIL
  "Keeps track of the previous outermost screen boxes so that they can be returned to. ")

;;;editor variables...

(DEFVAR *COLUMN* 0
  "the cha-no of the point for use with cntrl-p and cntrl-n")

(DEFVAR *WORD-DELIMITERS* '(#/< #/> #/  #/- #/, #/. #/' #/: #/ #/|))

(DEFVAR *FUNCTION-DELIMITERS* '(#/   #/, #/: #/ #/|))

(DEFVAR *KILL-BUFFER-ROW* NIL)

(DEFVAR *BOXER-VERSION-INFO* NIL
  "This variable keeps track of what version of boxer is currently loaded
   and being used.  Versions for general release are numbered while specific
   development versions have associated names.")


;;;;windows that boxer uses, and other related things

(DEFVAR *BOXER-FRAME* NIL
  "This frame contains *turtle-pane* *boxer-pane* etc.")

(DEFVAR *NAME-PANE* NIL)

(DEFVAR *BOXER-PANE* NIL
  "The pane which contains the actual boxer screen editor.")

;For the error handler to peek at until we get a real evaluator.

(DEFVAR *CURRENT-FUNCTION-BEING-FUNCALLED* "Toplevel")

(DEFVAR *BOXER-ERROR-HANDLER-P* T
  "If the value of this variable is non-nil, errors inside of Boxer will
   be passed to the regular LISPM error handler instead of the Boxer
   error handler.")

;;; STEPPING VARS

(defvar *step-flag* nil "Controls whether the (interim) stepper is in operation.")

(defvar *box-copy-for-stepping* nil "Should be an evaluator variable, when we have one.  A
copy of the the currently-executing box, placed in the stepping window.  The :funcall method
needs the actual box so it can flash lights inside it.")

;;; graphics variables

(DEFVAR *DEFAULT-TURTLE-BOX-WID* 326
  "The default width of any newly created turtle box. ")

(DEFVAR *DEFAULT-TURTLE-BOX-HEI* 217
  "The default height of any newly created turtle box. ")

(DEFVAR *DEFAULT-GRAPHICS-BOX-WID* 326
  "The default width of any newly created graphics box. ")

(DEFVAR *DEFAULT-GRAPHICS-BOX-HEI* 217
  "The default height of any newly created graphics box. ")

;;; Binding variables

(DEFVAR *EXPORTING-BOX-MARKER* ':EXPORT
  "This is a marker used by the binding code to mark subboxes which want to export some or
all of their bindings into the superior environment. ")

(DEFVAR *EXPORT-ALL-VARIABLES-MARKER* ':ALL
  "The prescence of this marker in the EXPORTS slot of a box indicates that ALL of the box's
bindings are to be exported to other boxes. ")



(DEFFLAVOR BOXER-FRAME
	()
	(UNIQUE-NAME-MIXIN TV:BORDERED-CONSTRAINT-FRAME)
  (:DEFAULT-INIT-PLIST
   :UNIQUE-NAME '*BOXER-FRAME*))

(DEFFLAVOR NAME-PANE
	()
	(UNIQUE-NAME-MIXIN
	 TV:PANE-MIXIN
	 TV:WINDOW)
  (:DEFAULT-INIT-PLIST
   :UNIQUE-NAME              '*NAME-PANE*
   :SAVE-BITS                T
   :DEEXPOSED-TYPEOUT-ACTION ':PERMIT
   :LABEL                    NIL
   :BLINKER-P                NIL
   :FONT-MAP                 `(,FONTS:MEDFNB)))

(DEFFLAVOR BOXER-PANE
	()
	(UNIQUE-NAME-MIXIN
	 REDISPLAYABLE-WINDOW-MIXIN
	 TV:PROCESS-MIXIN
	 TV:PANE-MIXIN
 #+LMITI TV:ANY-TYI-MIXIN
	 TV:WINDOW)
  (:DEFAULT-INIT-PLIST
   :UNIQUE-NAME              '*BOXER-PANE*
   :SAVE-BITS                T
   :DEEXPOSED-TYPEOUT-ACTION ':PERMIT
   :LABEL                    NIL
   :BLINKER-P                NIL
   :FONT-MAP                 `(,FONTS:MEDFNT ,FONTS:CPTFONT ,FONTS:MEDFNB)

   :ASYNCHRONOUS-CHARACTERS  ()))


(DEFFLAVOR CURSOR-BLINKER
	()
	(TV:RECTANGULAR-BLINKER)
  (:DEFAULT-INIT-PLIST :VISIBILITY NIL
		       :FOLLOW-P T
		       :WIDTH *CURSOR-BLINKER-WID*
		       :HEIGHT *CURSOR-BLINKER-MIN-HEI*))

;;; We need to use our own blinkers because the standard mouse blinkers in SYmbolics REL6 use 
;;; FAST-TRACKING-MIXIN which doesn't allow us to turn the mouse blinker off

(DEFFLAVOR BOXER-MOUSE-BLINKER
	()
	(#+SYMBOLICS TV:MOUSE-BLINKER-MIXIN
	 #-SYMBOLICS TV:MOUSE-BLINKER-FAST-TRACKING-MIXIN
	 TV:CHARACTER-BLINKER)
  (:DEFAULT-INIT-PLIST :VISIBILITY NIL
		       :font 'fonts:mouse
		       :char 6))

;;; Sprite Blinker by Jeremy
;;; This blinker is the rectangle which is used to highlight sprites
;;; The slots remember which screen box and which turtle were highlighted.

(defflavor Sprite-blinker
	((selected-sprite nil)
	 (sprite-screen-box nil))
	(tv:hollow-rectangular-blinker)
  :settable-instance-variables
  :gettable-instance-variables)
  



;;;;EDITOR OBJECT DEFINITIONS

;;;cha is only available as a component of the box flavor
;;;normal chas are now fixnums store in the superior row's chas-array

(DEFFLAVOR CHA
	((SUPERIOR-ROW NIL)
	 (CHA-CODE #\SPACE)		  ;if this is the symbol :BOX then the
					  ;cha is actually a box, if this
					  ;is not the symbol :BOX then it is the
					  ;cha code of this cha
	 (FONT-NO *CURRENT-FONT-NO*))	  ;this only makes sense if cha-code
					  ;is actually a cha code
	(ACTUAL-OBJ-MIXIN  PLIST-MIXIN)
  (:INIT-KEYWORDS :SUPERIOR-ROW :CHA-CODE :FONT)
  (:DEFAULT-INIT-PLIST :CHA-CODE #\SPACE)
  (:DOCUMENTATION :MIXIN
   "Chas are no longer meant to be instantiated.  The flavor exists only as a mixin to the
Box flavor. "))

(DEFFLAVOR POP-UP-BOX-MIXIN
	()
	()
  (:DOCUMENTATION :MIXIN
   "Makes the box go away when it is exited.  Removal is executed by an :AFTER demon."))

(DEFSUBST CHA? (CHA) (FIXNUMP CHA))

(DEFVAR %%BOXER-CHA-CODE-FIELD #O0010)

(DEFVAR %%BOXER-FONT-NO-FIELD #O1404)

(DEFVAR %%BOXER-CHA-CTRL-FIELD #O1004)

(DEFVAR %%BOXER-CHA-CODE-AND-CTRL-FIELD #O0014)

(DEFVAR %%NUMBER-FIELD #O0004
  "Byte specifier for getting the number out of a keycode for a number key (i.e. ctrl-2). ")

(DEFUN MAKE-CHA (CHA-CODE &OPTIONAL(FONT-NO *CURRENT-FONT-NO*))
  (DPB FONT-NO %%BOXER-FONT-NO-FIELD CHA-CODE))

(DEFSUBST CHA-CODE-NO-CTRL (CHA)
  (IF (CHA? CHA)
      (LDB %%BOXER-CHA-CODE-FIELD CHA)
      ':BOX))

(DEFSUBST FONT-NO (CHA)
  (IF (CHA? CHA)
      (LDB %%BOXER-FONT-NO-FIELD CHA)
      NIL))

(DEFSUBST CTRL-CODE (CHA)
  (IF (CHA? CHA)
      (LDB %%BOXER-CHA-CTRL-FIELD CHA)
      NIL))

(DEFSUBST CHA-CODE (CHA)
  (IF (CHA? CHA)
      (LDB %%BOXER-CHA-CODE-AND-CTRL-FIELD CHA)
      NIL))

(DEFSUBST NUMBER-CODE (CHA)
  (IF (CHA? CHA)
      (LDB %%NUMBER-FIELD CHA)
      NIL))

(DEFSUBST SET-FONT-NO (CHA FN)
  (IF (CHA? CHA)
      (DPB FN %%BOXER-FONT-NO-FIELD CHA)
      CHA))

(DEFSUBST SET-CTRL-CODE (CHA CD)
  (IF (CHA? CHA)
      (DPB CD %%BOXER-CHA-CTRL-FIELD CHA)
      CHA))

(DEFFLAVOR ROW
	((SUPERIOR-BOX NIL)
	 (PREVIOUS-ROW NIL)
	 (NEXT-ROW NIL)
	 (CHAS-ARRAY NIL)
	 ;(BOXES NIL)
	 (CACHED? NIL)
	 ;;flag indicating valid caching.  The old method of checking caused blank rows
	 ;;to call the READER 
	 (CACHED-CHAS NIL)
	 (CACHED-ITEMS NIL)
	 (CACHED-ENTRIES NIL)
	 (CACHED-ELEMENTS NIL))
	(ACTUAL-OBJ-MIXIN   PLIST-MIXIN)
  (:INIT-KEYWORDS :SUPERIOR-BOX :CHAS-ARRAY))

(DEFFLAVOR NAME-ROW
	((CACHED-NAME NIL))
	 ;used for environmental info--a symbol in the boxer users package
	(ROW)
  :INITABLE-INSTANCE-VARIABLES)

(DEFFLAVOR BOX
	((FIRST-INFERIOR-ROW NIL)
	 (CACHED-ROWS NIL)
	 (CACHED-CODE NIL)
	 (PORTS NIL)
	 (DISPLAY-STYLE-LIST (LIST ':NORMAL NIL NIL));A list beginning with :SHRUNK
						;or                    :NORMAL
	 (NAME NIL)
	 (STATIC-VARIABLES-ALIST NIL)
	 (EXPORTS NIL)
	 (LOCAL-LIBRARY NIL)
	 (REGION NIL)
	 (SHRINK-PROOF? NIL)
	 (entry-trigger nil)
	 (exit-trigger nil)
	 (entry-trigger-flag 'disabled)
	 (exit-trigger-flag 'disabled))
	(CHA ACTUAL-OBJ-MIXIN VIRTUAL-COPY-MIXIN PLIST-MIXIN FLAVOR-HACKING-MIXIN)
  :INITABLE-INSTANCE-VARIABLES
  (:INIT-KEYWORDS :SUPERIOR-ROW :TYPE :FIXED-WID :FIXED-HEI))

(DEFFLAVOR DOIT-BOX
	()
	(BOX))

(DEFFLAVOR DATA-BOX
	()
	(BOX))

(DEFFLAVOR LL-BOX
	()
	(BOX POP-UP-BOX-MIXIN))

(DEFFLAVOR PORT-BOX
	()
	(BOX))

;;; Just add a slot for the turtle to a normal box
(defflavor sprite-box 
	((associated-turtle nil))
	(box)
  :gettable-instance-variables
  :initable-instance-variables
  :init-keywords
  :settable-instance-variables) 

;;; I still think these two flavors should become one and only the type 
;;;of screen box should toggle.
(DEFFLAVOR GRAPHICS-BOX
	((GRAPHICS-SHEET NIL))	       ;a leaderless <art-1b> array (no color yet)
	(BOX)
  :INITABLE-INSTANCE-VARIABLES
  (:INIT-KEYWORDS :SUPERIOR-ROW :PICTURE-WID :PICTURE-HEI)
  (:DEFAULT-INIT-PLIST :PICTURE-WID 320 :PICTURE-HEI 200))

(defflavor graphics-data-box
	((graphics-sheet nil))
	(BOX)
  :initable-instance-variables
  (:DEFAULT-INIT-PLIST :FIXED-WID 320 :FIXED-HEI 200))

(DEFFLAVOR INPUT-BOX
	()
	(BOX POP-UP-BOX-MIXIN)
  (:INIT-KEYWORDS :INPUT-TYPE))

(DEFFLAVOR EDITOR-REGION
	((START-BP NIL)
	 (STOP-BP NIL)
	 (ROWS NIL)
	 (BOX NIL)
	 (VISIBILITY NIL)
	 (BLINKER-LIST NIL))
	()
  :INITABLE-INSTANCE-VARIABLES)

;;; Modified by Jeremy to include Draw-mode which can be wrap, window, or fence

(DEFSTRUCT (GRAPHICS-SHEET (:TYPE :NAMED-ARRAY)
			   :CONC-NAME
			   (:CONSTRUCTOR %MAKE-GRAPHICS-SHEET
			    (DRAW-WID DRAW-HEI BIT-ARRAY SUPERIOR-BOX))
			   (:CONSTRUCTOR MAKE-GRAPHICS-SHEET-FROM-FILE
			    (DRAW-WID DRAW-HEI BIT-ARRAY draw-mode))
			   (:PRINT "#<GRAPHICS-SHEET W-~D. H-~D.>"
			    (GRAPHICS-SHEET-DRAW-WID GRAPHICS-SHEET)
			    (GRAPHICS-SHEET-DRAW-HEI GRAPHICS-SHEET)))
  (DRAW-WID *DEFAULT-GRAPHICS-SHEET-WIDTH*)
  (DRAW-HEI *DEFAULT-GRAPHICS-SHEET-HEIGHT*)
  (SCREEN-OBJS NIL)
  (BIT-ARRAY (TV:MAKE-SHEET-BIT-ARRAY *BOXER-PANE*
				      *DEFAULT-GRAPHICS-SHEET-WIDTH*
				      *DEFAULT-GRAPHICS-SHEET-HEIGHT*))
  (OBJECT-LIST NIL)
  (SUPERIOR-BOX NIL)
  (draw-mode ':wrap)
  )

(DEFTYPE-CHECKING-MACROS ROW "a row object")
(DEFTYPE-CHECKING-MACROS NAME-ROW "A Row used to name boxes. ")
(DEFTYPE-CHECKING-MACROS BOX "a box object")
(DEFTYPE-CHECKING-MACROS DOIT-BOX "a Doit Box")
(DEFTYPE-CHECKING-MACROS DATA-BOX "a Data Box")
(DEFTYPE-CHECKING-MACROS LL-BOX "a local library")
(DEFTYPE-CHECKING-MACROS PORT-BOX "a Port Box")
(DEFTYPE-CHECKING-MACROS GRAPHICS-BOX "A Box used for Graphics")
(DEFTYPE-CHECKING-MACROS INPUT-BOX "a box used for input")
(DEFTYPE-CHECKING-MACROS EDITOR-REGION "A Boxer Editor Region")
(deftype-checking-macros Sprite-box "A sprite-box")
(deftype-checking-macros Graphics-data-box "A Graphics-data-box")
(DEFTYPE-CHECKING-MACROS GRAPHICS-SHEET "A Bit Array for Graphics Boxes")


;;;BP's are pointers which are used to move within REAL(that is, ACTUAL) structure
;;;Note that they have nothing to do with SCREEN structure...
;;;The *point* is a BP as is the *mark*
;;;however, operations which move the *point* and the *mark* also update the
;;;global variable's  *current-screen-box* and *marked-screen-box*

(DEFSTRUCT (BP (:TYPE :NAMED-LIST)           ;Easier to Debug
	       (:CONSTRUCTOR MAKE-BP (TYPE))
	       (:CONSTRUCTOR MAKE-INITIALIZED-BP (TYPE ROW CHA-NO))
	       (:CONC-NAME   %BP-)
	       (:ALTERANT    %ALTER-BP))
  (ROW    NIL)
  (CHA-NO 0)
  (SCREEN-BOX NIL)
  (TYPE ':FIXED))

(DEFSUBST BP? (X)
  (AND (LISTP X) (EQ (CAR X) 'BP)))

(DEFMACRO CHECK-BP-ARG (X)
  `(CHECK-ARG ,X BP? "A Boxer Editor Buffer-Pointer (BP)."))

(DEFF BP-ROW        '%BP-ROW)
(DEFF BP-CHA-NO     '%BP-CHA-NO)
(DEFF BP-SCREEN-BOX  '%BP-SCREEN-BOX)
(DEFF BP-TYPE       '%BP-TYPE)


(DEFPROP BP-ROW        ((BP-ROW BP)        SET-BP-ROW BP SI:VAL)        SETF)
(DEFPROP BP-CHA-NO     ((BP-CHA-NO BP)     SET-BP-CHA-NO BP SI:VAL)     SETF)
(DEFPROP BP-SCREEN-BOX ((BP-SCREEN-BOX BP) SET-BP-SCREEN-BOX SI:VAL)    SETF)
(DEFPROP BP-TYPE       ((BP-TYPE BP)       SET-BP-TYPE BP SI:VAL)       SETF)

(DEFSUBST ROW-BPS (ROW) (TELL ROW :BPS))
#-LMITI
(DEFPROP ROW-BPS ((ROW-BPS ROW) TELL ROW :SET-BPS SI:VAL) SETF)
#+LMITI
(DEFSETF ROW-BPS (ROW) (NEW-BPS) `(TELL ,ROW :SET-BPS ,NEW-BPS))

(DEFMACRO MOVE-BP (BP FORM)
  `(MULTIPLE-VALUE-BIND (NEW-ROW NEW-CHA-NO NEW-SCREEN-BOX)
       ,FORM
     (MOVE-BP-1 ,BP NEW-ROW NEW-CHA-NO NEW-SCREEN-BOX)))

(DEFMACRO MOVE-POINT (FORM)
  `(MULTIPLE-VALUE-BIND (NEW-ROW NEW-CHA-NO NEW-SCREEN-BOX)
       ,FORM
     (MOVE-POINT-1 NEW-ROW NEW-CHA-NO NEW-SCREEN-BOX)))

(DEFUN BP-CHA (BP)
  (TELL (BP-ROW BP) :CHA-AT-CHA-NO (BP-CHA-NO BP)))



;;;; FLAVORS HAVING TO DO WITH reDISPLAY.

(DEFFLAVOR REDISPLAYABLE-WINDOW-MIXIN
	((OUTERMOST-SCREEN-BOX NIL))
	()
  )

(DEFSUBST REDISPLAYABLE-WINDOW? (X)
  (TYPEP X 'REDISPLAYABLE-WINDOW-MIXIN))

(DEFFLAVOR ACTUAL-OBJ-MIXIN
	((SCREEN-OBJS NIL)
	 (TICK 1))
	()
  (:ORDERED-INSTANCE-VARIABLES TICK)
  (:OUTSIDE-ACCESSIBLE-INSTANCE-VARIABLES TICK)
  (:DOCUMENTATION :MIXIN
   "Giving an flavor this mixin will allow the the redisplay code to be
    able to display and redisplay that object"))

(DEFTYPE-CHECKING-MACROS ACTUAL-OBJ "an obj with the Actual-Obj-Mixin")


(DEFFLAVOR SCREEN-OBJ
	((ACTUAL-OBJ NIL)
	 (X-OFFSET 0)
	 (Y-OFFSET 0)
	 (WID 0)
	 (HEI 0)
	 (X-GOT-CLIPPED? NIL)
	 (Y-GOT-CLIPPED? NIL)
	 (NEW-WID 0)
	 (NEW-HEI 0)
	 (NEW-X-GOT-CLIPPED? NIL)
	 (NEW-Y-GOT-CLIPPED? NIL)
	 (TICK -1)
	 (NEEDS-REDISPLAY-PASS-2? NIL)
	 (FORCE-REDISPLAY-INFS? NIL))
	()
  :ORDERED-INSTANCE-VARIABLES
  :OUTSIDE-ACCESSIBLE-INSTANCE-VARIABLES
  :SETTABLE-INSTANCE-VARIABLES
  (:REQUIRED-METHODS :REDISPLAY-PASS-1
		     :REDISPLAY-PASS-2))

(DEFTYPE-CHECKING-MACROS SCREEN-OBJ "an object of type Screen-Obj")

(DEFFLAVOR SUPERIOR-SCREEN-OBJ
	()
	(SCREEN-OBJ))

(DEFTYPE-CHECKING-MACROS SUPERIOR-SCREEN-OBJ "an object of type Superior-Screen-Obj")

;;;screen chas are now obselete.  They only exist as a mixin for the box flavor
(DEFFLAVOR SCREEN-CHA
	((SCREEN-ROW NIL))
	(SCREEN-OBJ)
  :SETTABLE-INSTANCE-VARIABLES)

(DEFFLAVOR SCREEN-ROW
	((SCREEN-BOX NIL)
	 (SCREEN-CHAS NIL)
	 (OUT-OF-SYNCH-MARK NIL))
	(SUPERIOR-SCREEN-OBJ)
  :SETTABLE-INSTANCE-VARIABLES)

(DEFTYPE-CHECKING-MACROS SCREEN-ROW "a Screen-Row")

(DEFFLAVOR SCREEN-BOX
	((SCREEN-ROWS NIL)
	 (SCROLL-TO-ACTUAL-ROW NIL)
	 (INF-HOR-SHIFT 0.)
	 (NAME NIL)
	 (BOX-TYPE ':DOIT-BOX)
	 (BPS NIL)
	 (DISPLAY-STYLE-LIST (LIST NIL NIL NIL)) ;NIL means use the information
						 ;in the actual Box, Otherwise
						 ;this (like the actual Box) is
						 ;A list beginning with :SHRUNK
						 ;or                    :NORMAL
						 ;or                    :FIXED
	 (SUPERIOR-SCREEN-BOX NIL))	  ;this stores display info when the box
	(SCREEN-CHA SUPERIOR-SCREEN-OBJ FLAVOR-HACKING-MIXIN)  ;is made into an outermost box
  :SETTABLE-INSTANCE-VARIABLES)

(DEFTYPE-CHECKING-MACROS SCREEN-BOX "a Screen-Box")

(DEFUN CHECK-SCREEN-CHA-ARG (SCREEN-CHA)
  (OR (FIXNUMP SCREEN-CHA)
      (SCREEN-BOX? SCREEN-CHA)))

(DEFFLAVOR GRAPHICS-SCREEN-BOX
	()
	(SCREEN-BOX)
  (:SETTABLE-INSTANCE-VARIABLES))

(DEFTYPE-CHECKING-MACROS GRAPHICS-SCREEN-BOX "A Screen Box used for Graphics")


(DEFFLAVOR REGION-ROW-BLINKER
	((UID NIL))
	(TV:RECTANGULAR-BLINKER)
  (:OUTSIDE-ACCESSIBLE-INSTANCE-VARIABLES UID))

(DEFTYPE-CHECKING-MACROS REGION-ROW-BLINKER "A Boxer Editor Region Blinker")

;;;just in case...

(COMPILER:MAKE-OBSOLETE DO-CHAS "Use (DO-ROW-CHAS (<var> <row>) <body>) instead.")
(COMPILER:MAKE-OBSOLETE DO-ROWS "Use (DO-BOX-ROWS (<var> <box>) <body>) instead.")
(COMPILER:MAKE-OBSOLETE DO-OBJS "Why were you using it anyways??")



;;; Setting up the BOXER-USER package.

;; Boxer stores global definitions in the value cell of the symbol used to name
;; the primitive or variable.  In order to be sure that boxer-functions don't get randomly
;; redefined, we need to be sure that those symbols can't get lambda-bound or
;; have their values set by any code other than boxer-function code.  In order
;; to do this, we set up a special package, the BOXER-USER package, in which we
;; intern all the symbols we use to name boxer-functions.  In addition, this
;; package is set up so that it shadows all symbols. This is done by setting
;; the package's pkg-super-package to nil. Please take a moment to consider
;; the effects of having a package's super package be nil... it means that it
;; will intern all symbols locally, it means that none of the lispms functions
;; or variables are available from that package, it means that if you should
;; manage to bind the value of the variable package to that package you would
;; be in a lot of trouble.  Since I don't expect you to believe this, or even
;; take the time to think about it, I am going to intern the symbols PKG-GOTO,
;; and PKG-USER-PACKAGE in the boxer-user package, this will allow people who
;; manage to get stuck in the boxer-user package to unstick themselves without
;; having to warm-boot (do a (PKG-GOTO PKG-USER-PACKAGE)).

#+MIT
(EVAL-WHEN (LOAD)
  (MAKE-PACKAGE "BOXER-USER"
		':NICKNAMES '(BU BOXER-USERS PKG-BU-PACKAGE PKG-BOXER-USER-PACKAGE)
		':SIZE 1000
		':USE NIL)
  )

#-MIT
(EVAL-WHEN (LOAD)
  (DEFPACKAGE BOXER-USER
    (:NICKNAMES BU BOXER-USERS PKG-BU-PACKAGE PKG-BOXER-USER-PACKAGE)
    (:PREFIX-NAME BU)
    (:USE)
    (:IMPORT SI:PKG-GOTO)
    (:SIZE 1000)))

(EVAL-WHEN (LOAD)
  
  (DEFVAR PKG-BOXER-USER-PACKAGE (PKG-FIND-PACKAGE 'BOXER-USER))
  (DEFVAR PKG-BU-PACKAGE (PKG-FIND-PACKAGE 'BOXER-USER))
  (DEFUN INTERN-IN-BOXER-USER-PACKAGE (SYMBOL)
    (INTERN (STRING SYMBOL) 'BOXER-USER))
  (DEFUN INTERN-IN-BU-PACKAGE (SYMBOL)
    (INTERN (STRING SYMBOL) 'BU))
  
  )

