;;; RCS.LISP (Simple Revision Control System)
;;; Version 1.7, July. '93 
;;; Functions for managing the editing of project code by multiple people.
;;; Hacked by David Neves - neves@ils.nwu.edu
;;;
;;; Changes:
;;; neves  (7/27/93)  Changes to make logical pathname stuff work for a user on the server
;;; jona   (2/2/93)   Wrap menu call to copy-directory in a eval-enqueue.
;;; neves  (1/15/93)  As per Kemi's suggestion, have init-rcs put a call to itself in
;;;                   *lisp-startup-functions*.
;;; neves  (1/14/93)  use *home-directory-o* to store logical pathname of home directory.
;;; jona   (1/6/93)   balloon help and code to better display log file.
;;; neves  (12/10/92) copy-directory now prints out name of file copied. Don't ask if non-text
;;;                   files should be edited when locking them.
;;; neves  (11/12/92) Have button to Forget files be labeled as Forget rather than Unlock.
;;; neves  (10/29/92) Check to see if *server-volume* is mounted.  When locking, don't copy
;;;                   the file over if you already have the most recent version.
;;;                   Other misc changes.
;;; neves  (10/16/92) Add *files-not-to-copy* to prevent RCS bookkeeping files from being
;;;                   copied to a users local disk.  Other misc changes.
;;; neves  (10/15/92) Add help and viewing of log file
;;; neves  (10/15/92) Lock file before copying to the local disk.
;;; neves  (10/14/92) Fix pathname bugs for released MCL 2.0, add copy-directory function
;;; neves  (1/21)     Make a variable to hold folder of server volume on server machine
;;; neves  (1/7/92)   Server now has a separate working directory.
;;; neves  (12/23/91) Updated to MACL 2.0 Beta
;;;
;;; =========================================================================================
;;; Documentation:
;;;   On any large project there is a danger of 2 people editing the same file at the same time.
;;; Most likely one person's changes will be lost.  This software allows someone to "lock" a
;;; file so that no one else can edit it.  When the user is finished editing the file they
;;; can "unlock" the file so that others can edit it.
;;;   Project software is kept on a central server.  Locking a file copies that file to the user's
;;; local hard disk and stores the file name in a list of locked files on the central server. 
;;; When the user unlocks the file, the file is copied back to the server and the file name
;;; is removed from the list of locked files.
;;;   The project directory on the server may be hierarchical. Files copied from it 
;;; will be put in the same relative position on the user hard disk.  
;;; For example, the file server:foo:bar might be copied to
;;; user:foo:bar.  "foo" is a subfolder where bar is located.
;;; =========================================================================================
;;;
;;; User choices from the "lockfile" menu:
;;;   - Lock a file.  This brings up a dialog so that the user can choose a file to lock.  If
;;;     the file is already locked then the user gets an error message.  Locking a file
;;;     copies the file from the server to the local hard disk.  Then the name of the locked
;;;     file is stored in a special file ("locked-file-list") on the server.
;;;   - Unlock a file and copy to server.  This brings up a dialog with all your locked files.  
;;;     Select 1 or more files (with shift-click) to unlock.  The files are copied back to the 
;;;     server and their names are deleted from "locked-file-list".
;;;   - Unlock a file, but don't copy to server.  This is like the choice above but the files
;;;     are not copied to the server.  Useful when the user changes his/her mind about making
;;;     the changes permanent.
;;;   - Copy a newly created file to the server.  The user has just created a file on his/her
;;;     hard disk.  To move it to the server choose this.
;;;   - Update - copy server directory to local disk.  Updates all files.
;;;   - Show all locked files.  Show a list of all the locked files, along with who locked them.
;;;
;;; Hardware needed:
;;;   Each user needs a Macintosh with access to an Appleshare network.
;;;   You need a server machine that can be mounted from other Macs.
;;;
;;; Software needed:
;;;   System 7.0 (or greater) & MACL 2.0 (or greater)
;;;
;;; To install:
;;;    Simply load this file.  The LockFile menu choice will install itself.

;;;;;;
;;; To do:
;;;    from Chung: handle multiple projects
;;;    from Kemi : use apple events to be able to edit other than text files
;;;    It would be nice if this software mounted the server volume.  I don't know how to do this.
;;;
;;; Known bugs:
;;;   None.
;;;
;;; Changes you have to make:
;;;   The only changes you should need to make for your project are to the defparameters below.
;;; Because a person on a server machine cannot mount their own machine
;;; I have a bunch of special case code that allows one to use this software
;;; on a server machine.  
;;; (thus the need for *server-name* & *folder-of-server-volume-on-server*)

(in-package :ccl)

;;; ------------------------------------------------------------------------------------------------
;;; change the following strings for your project.
;(defparameter *server-name* "Chung's Macintosh")
(defparameter *server-name* "neves")  
    ;< only used if someone is using the server machine>
    ;This is the name of the machine that is the server -- the chooser name.
    ;e.g. "neves"


;(defparameter *home-directory-o* "ccl:MOPED;")       
(defparameter *home-directory-o* "ccl:SC-builder;")       
                ;Local home directory where the project files are kept for all users.
                ;This is where a file ends up when locked and copied to the users hard disk.
                ;It must be understood by all user machines so it is recommended it be put under 
                ;ccl: or home:
                ;[[Note use of CL style logical pathname (page 628 of Steele)
                ;with semicolin separating directories.]]
                ;e.g. ccl:myproject;

;(defparameter *server-volume* "Data Storage - AK Lab:MJC backup:MOPED Server:")
(defparameter *server-volume* "SC-builder:test:")
         ;Server folder where the project files are kept.  The first part of it is what users
         ;connect to (i.e. it is the shared folder).
         ;e.g. sc-builder:test: -- users connect to folder sc-builder
        
;(defparameter *folder-of-server-volume-on-server* "MJC backup:")
(defparameter *folder-of-server-volume-on-server* "hd:applications:")
         ;<used only if someone is using the server machine>
         ;This is the folder on the server that contains the *server-volume*.
         ;e.g. hd:serverstuff:
         ;so the path to the directory users will copy from is hd:applications:sc-builder:test

;;; The following two names do not need to be changed.      
(defparameter *filename-locked-file-list-file* "locked-file-list") 
                    ;File for list of locked files
(defparameter *filename-log-file* "logfile")    
                    ;File for documentation on changes made to files
;;; -----------------------------------------------------------------------------------------
(defvar *locked-file-list-file*) ; full pathname of locked-file-list-file
(defvar *log-file*)              ; full pathname of log file
(defvar *home-directory*)        ; set from home-directory-o above
(defvar *expanded-server-volume*); expanded version of server-volume
(defvar *files-not-to-copy*)     ; list of files not to update to local disk from server
(defvar *locked-file-list*)      ; temporary list holding the contents of locked-file-list-file
(defvar *rcs-menu*)              ; lock file menu

(defun on-server-p nil (equal (machine-instance) *server-name*))

(defmacro concat (&rest strings)
  `(concatenate 'string ,@strings))

(defun check-server-p nil
   (if (null (probe-file *expanded-server-volume*))
     (progn
       (message-dialog (concat "Could not find server " *server-volume* ". -- Aborting."))
       nil)
     t))

;;; (expand-host "ccl:foo:bar") --> "HD:MCL 2.0:foo:bar"
;;;  Note that ccl:foo:bar is not a legal logical pathname (no semicolons between directories)
(defun expand-host (path)
  (let ((pos (search ":" path))
        hostname rest)
    (when (null pos) (error "no host for expand-host"))
    (setq hostname (subseq path 0 (1+ pos)))
    (setq rest (subseq path (1+ pos)))
    (concat (namestring (translate-logical-pathname hostname)) rest)))

(defun get-host (path)
  (subseq path 0 (1+ (search ":" path))))
                    

;;; init-rcs is called automatically at the end of this file
(defun init-rcs nil
  (let (server-servers-volume host-server-volume)
    (setq host-server-volume (get-host *server-volume*))
    (if (search ";" *home-directory-o*)
      (setq *home-directory* (namestring (translate-logical-pathname *home-directory-o*)))
      (setq *home-directory* (expand-host *home-directory-o*))) ;non legal pathname.  expand host.
    (setq *expanded-server-volume* *server-volume*)
    (when (on-server-p)
      (setq server-servers-volume (concat (expand-host *folder-of-server-volume-on-server*)
                                           (get-host *server-volume*)))
      (setf (logical-pathname-translations 
             ;; take out the colon at the end of *server-volume*
             (subseq host-server-volume 0 (1- (length host-server-volume))))
            ;; copied right out of steele without understanding it...
            `(("**;*.*.*" ,(concat server-servers-volume "**"))))
      (setq *expanded-server-volume* (expand-host *server-volume*))
      )
    (when (null (check-server-p)) (return-from init-rcs))
;;; check to see if the following two statements do the right thing ZZZ
    (setq *locked-file-list-file* (concat *server-volume* *filename-locked-file-list-file*))
    (setq *log-file* (concat *server-volume* *filename-log-file*))
    (setq *files-not-to-copy* (list *locked-file-list-file* *log-file*))
    (if (find-menu "LockFile") (menu-deinstall *rcs-menu*))
    (setq *rcs-menu* (make-instance 'menu :menu-title "LockFile"))
    (add-menu-items *rcs-menu*
                   (make-instance 'menu-item
                          :menu-item-title "Lock    - (a project file and copy to local disk)"
                          :menu-item-action 'lock-project-file
                          :help-spec 
                          (format nil "Lock a file.  This brings up a dialog so that the ~
                                       user can choose a file to lock. If the file is ~
                                       already locked then the user gets an error message.  ~
                                       Locking a file copies the file from the server to ~
                                       the local hard disk."))
                   (make-instance 'menu-item
                          :menu-item-title "Unlock - (a project file and copy back to server)"
                          :menu-item-action 'unlock-project-file
                          :help-spec
                          (format nil "Unlock a file and copy to server.  This brings up ~
                                       a dialog with all your locked files. Select 1 or ~
                                       more files (with shift-click) to unlock.  The ~
                                       files are copied back to the  server."))
                   (make-instance 'menu-item
                          :menu-item-title "Forget - (Unlock project file but don't copy new version to server)"
                          :menu-item-action 'unlock-file-dont-copy
                          :help-spec 
                          (format nil "Unlock a file, but don't copy to server.  This is ~
                                       like 'Unlock' but the files are not copied to ~
                                       the server. Useful when the user changes his/her ~
                                       mind about making the changes permanent."))
                   (make-instance 'menu-item
                          :menu-item-title "Copy    - (newly created file to server.)"
                          :menu-item-action 'copy-new-file-to-server
                          :help-spec
                          (format nil "Copy a newly created file to the server. The user ~
                                       has just created a file on his/her hard disk.  ~
                                       To move it to the server choose this."))
;                   (make-instance 'menu-item
;                          :menu-item-title "Copy logged files to local disk."
;                          :menu-item-action 'copy-logfiles-to-local-disk)
                   (make-instance 'menu-item
                          :menu-item-title "Update - (files on local disk)"
                          :menu-item-action #'(lambda nil (eval-enqueue
                                                           '(copy-directory-1 *expanded-server-volume* *home-directory*)))
                          :help-spec
                          (format nil "Copy server directory to local disk.  ~
                                       Updates all files on local disk."))
                   (make-instance 'menu-item
                          :menu-item-title "Show   - (all locked files)"
                          :menu-item-action 'find-all-locked-files
                          :help-spec 
                          (format nil "Show a list of all the locked files, ~
                                       along with who locked them."))
                   (make-instance 'menu-item
                          :menu-item-title "Show changes   - (made to project files)"
                          :menu-item-action 'show-log-file
                          :help-spec 
                          (format nil "Show a list of past changes to all files."))
                   ;(make-instance 'menu-item
                   ;       :menu-item-title "Help"
                   ;       :menu-item-action 'show-help)
                   )
  (menu-install *rcs-menu*)
  
  (load-locked-file-list)

  (unless (member 'init-rcs *lisp-startup-functions*)
    (setf *lisp-startup-functions*
          (nconc *lisp-startup-functions* (list 'init-rcs))))
  
  
  ))

(defun server-to-logical-server-name (file)
  (concat *server-volume*
          (strip-left *expanded-server-volume* file)))

;;; lock a file
(defun lock-project-file nil
  (let (fromserverfile from-logical-server-file
        tofile
        tofileyounger
        within
        (server-volume *expanded-server-volume*)
        (default-choose-directory (choose-file-default-directory))
        )
    (when (string-equal (machine-instance) "")
      (message-dialog "Aborted because you have not named your Mac.  Please name your computer in Sharing Setup in Control Panels.")
      (return-from lock-project-file))
    (when (null (check-server-p)) (return-from lock-project-file))
    (setq fromserverfile 
          (catch-cancel 
            (choose-file-dialog :directory *expanded-server-volume*
                                :button-string "Lock file"
                                )))
    (set-choose-file-default-directory default-choose-directory)
    (when (neq fromserverfile :cancel)
      (setq fromserverfile (namestring fromserverfile))
      (setq within (search server-volume fromserverfile :test #'string-equal))
      (when (or (null within) (not (zerop within)))
        (message-dialog 
         (concat "File to be locked was not contained within the server: " server-volume " -- Aborting command."))
        (return-from lock-project-file))
      (setq from-logical-server-file (server-to-logical-server-name fromserverfile))
      (when (is-locked-filep from-logical-server-file)
        (message-dialog (concat from-logical-server-file " is already locked.  Aborting command."))
        (return-from lock-project-file))
      (setq tofile (logicalserver-to-home-name from-logical-server-file))
      (setq tofileyounger (is-youngerp tofile fromserverfile))
      (when (or (not tofileyounger)
                (and tofileyounger
                     (eq t (catch-cancel (y-or-n-dialog
                                          "WARNING!  The file on the local disk is younger than the one on the server.  Should I still copy it?")))))
        (if (probe-file tofile) (unlock-file tofile))
        (update-locked-file-list from-logical-server-file :add)
        (when (null (is-same-age fromserverfile tofile))
          (copy-file fromserverfile tofile
                     :if-exists :overwrite))
        (when
          (and (eq (mac-file-type tofile) :TEXT)
               (y-or-n-dialog  
                (concat fromserverfile " has been copied to your disk and is locked.  To edit the file click on EDIT, otherwise click on OK.")
                :yes-text "EDIT" :no-text "OK" :cancel-text nil))
          (ed tofile))
       ))))

(defun is-youngerp (file1 file2)
  (and (probe-file file1) (probe-file file2) (> (file-write-date file1) (file-write-date file2))))

(defun is-same-age (file1 file2)
  (and (probe-file file1) (probe-file file2) (eql (file-write-date file1) (file-write-date file2))))

;;; format of locked-file-list is ((filename . person) (filename . person) ...)

(defun is-locked-filep (filename)
  (load-locked-file-list)
  (assoc filename *locked-file-list* 
         :test #'string-equal))

(defun load-locked-file-list nil
  (let ((expanded (expand-host *locked-file-list-file*)))
    (if (null (probe-file expanded))
      (with-open-file (stream expanded :direction :output)
        (print nil stream)))
    (setq *locked-file-list*
          (with-open-file  (stream expanded :direction :input)
            (read stream)))))

(defun save-locked-file-list nil
  (let* ((expanded (expand-host *locked-file-list-file*))
         (tempfilename (concat expanded "temp")))
    (with-open-file (stream tempfilename :direction :output :if-exists :supersede)
      (print *locked-file-list* stream))
    (rename-file tempfilename expanded :if-exists :overwrite)))

(defun username nil (machine-instance))

(defun make-pair (&key filename person)
  (cons filename person))
(defun get-person (pair)
  (rest pair))
(defun get-filename (pair)
  (first pair))
 
;;; ------------------------------------------------------------------------------------
;;; unlock a file
(defun unlock-project-file (&optional (dontcopyflag nil))
   (let ((username (machine-instance))
         (homefilename)
         (serverfilenames))
    (when (eql username "")
      (message-dialog "Aborted because you have not named your Mac.  Please name your computer in Sharing Setup in Control Panels.")
      (return-from unlock-project-file))
    (when (null (check-server-p)) (return-from unlock-project-file))
     (setq serverfilenames 
           (catch-cancel 
            (select-item-from-list (find-my-locked-files) :selection-type :disjoint
                                   :default-button-text 
                                   (if dontcopyflag "Forget" "Unlock"))))
     (when (neq serverfilenames :cancel)
      (dolist (serverfilename serverfilenames)
        ;; doncopyflag means unlock the file but don't copy your version to the project directory
        (setq homefilename (logicalserver-to-home-name serverfilename))  ;;JL--removed from the WHEN below     
        (when (null dontcopyflag)
          (if (probe-file homefilename)  
            (copy-to-server-and-update-logfile homefilename serverfilename 
                                               (expand-host serverfilename))
            (format t "You do not have ~a to copy to the project directory~%" homefilename))
        )
     ;;   (let ((window (find-window (pathname-name homefilename))))
     ;;     (when window (window-close window)))         JL--closing the homefile window, if its here
     ;;   (lock-file homefilename)   JL--locking the homefile
        (update-locked-file-list serverfilename :delete)
        ))))

;;; Given a name on the server, construct the corresponding name on the home directory.
(defun logicalserver-to-home-name (filename)
  (concat *home-directory* 
          (strip-left *server-volume* (namestring filename))))

;;; Given a name on the home directory, construct a name for the server
(defun home-to-server-name (filename) 
  (concat *server-volume*
          (strip-left *home-directory*  (namestring filename))))

(defun copy-to-server-and-update-logfile (homefilename serverfilename expandedserverfilename)
  (when (or (null (probe-file expandedserverfilename))
          (>= (file-write-date homefilename) (file-write-date expandedserverfilename))
          (eq t (catch-cancel (y-or-n-dialog
                               "WARNING!  The file on the local disk is older than the one on the server.  Should I still copy it?"))))
   ;;   (when (probe-file expandedserverfilename)
   ;;    (unlock-file expandedserverfilename))     ;;JL--unlock the serverfile if it's there
    (copy-file homefilename expandedserverfilename :if-exists :overwrite)
   ;;  (lock-file serverfilename)     JL--lock the serverfile
   ;; (let ((window (find-window (pathname-name homefilename))))
   ;;   (when window (window-close window)))   JL--close the homefile window if its there
   ;; (lock-file homefilename)      JL--lock the homefile (now that window is closed
      ;; make sure the dates on both files are the same in case clocks are off on
      ;; both machines. 
    (set-file-write-date homefilename (file-write-date expandedserverfilename))
    (update-log-file serverfilename)
      ))

(defun copy-new-file-to-server nil
  (let (homefilename serverfilename expandedserverfilename within)
    (message-dialog "Please select a newly created file to copy to the server.")
    (setq homefilename 
          (catch-cancel (choose-file-dialog :directory *home-directory*
                                )))
    (when (neq homefilename :cancel)
      (setq homefilename (namestring homefilename))
      (setq within (search *home-directory* homefilename :test #'string-equal))
      (when (or (null within) (not (zerop within)))
        (message-dialog 
         (concat "New file was not contained within " *home-directory* " -- Aborting command."))
        (return-from copy-new-file-to-server))
      (setq serverfilename (home-to-server-name homefilename))
      (setq expandedserverfilename (expand-host serverfilename))
      (when (probe-file expandedserverfilename)
        (message-dialog (concat serverfilename " already exists.  Aborting command."))
        (return-from copy-new-file-to-server))
      (copy-to-server-and-update-logfile homefilename serverfilename expandedserverfilename)
      )))
    
(defun update-locked-file-list (file operation)
  (load-locked-file-list)
  (let ((newpair (make-pair :filename file :person (username))))
    (cond
     ((eq operation :add) 
      (pushnew newpair *locked-file-list*))
   ((eq operation :delete) 
    (setq *locked-file-list* 
          (delete newpair *locked-file-list* :test #'equal)))
   (t (error "illegal operation in update-locked-file-list")))
  (save-locked-file-list)))

(defun update-log-file (filename)
  (setq filename (namestring filename))
  (let ((changes))
    (with-open-file (stream (expand-host *log-file*) :direction :output :if-exists :append :if-does-not-exist :create)
      (setq changes (catch-cancel 
                     (get-string-from-user (concat "File " filename " has been copied to the server.  Describe your changes to the file here."))))
      (format stream "~a \"~a\" ~a -- ~a~%" (machine-instance) filename (return-the-date) changes)
      )))

(defun return-the-date nil
  (multiple-value-bind  (second minute hour date month year 
                                day-of-week daylight-saving-timep time-zone)                        
                        (get-decoded-time)
    (declare (ignore second year day-of-week daylight-saving-timep time-zone))
    (format nil "(~a:~2,'0d ~a/~2,'0d)" hour minute month date)))
  
(defun find-my-locked-files nil
  (find-user-locked-files (username)))

(defun find-user-locked-files (user)
  (mapcar 'get-filename
          (remove user *locked-file-list* 
                  :test #'(lambda (user y) (not (equal user (get-person y)))))))
      
(defun find-people-with-locked-files nil
  (let (people)
    (dolist (pair *locked-file-list*)
      (pushnew (get-person pair) people :test #'equal))
    people))

(defun find-all-locked-files nil
  (load-locked-file-list)
  (show-listener)
  (format t "~%--Locked file list--~%")
  (if (null *locked-file-list*) (format t "There are no locked files.")
      (dolist (person (find-people-with-locked-files))
        (format t "Locked files for ~a:~%" person)
        (dolist (file (find-user-locked-files person))
          (format t "   ~a~%" file)))))

(defun show-listener nil
  (window-select (find-window "Listener")))

(defun unlock-file-dont-copy nil
  (unlock-project-file t))

;;; copy a file and make sure the write dates are the same on both files
(defun copy-file-and-set-write-date (fromfile tofile)
  (copy-file fromfile tofile :if-exists :overwrite)
  (set-file-write-date tofile (file-write-date fromfile)))

;;;-----
;;; Copy files from logfile to local disk.  Remove duplicate names in logfile list of files.
;;; BUGS: doesn't check to see if local files are more recent than server files.
;;; This function is currently not being used.
#|
(defun copy-logfiles-to-local-disk nil
    (let (linelist selectlist tofile fromfilelist)
      (with-open-file  (finput *log-file* :direction :input)
        (setq linelist
              (do* ((line (read-line finput nil :eof)(read-line finput nil :eof))
                    (linelist)
                    (pos))
                   ((eq line :eof) linelist)
                (setq pos (position #\" line)) ;kludge for testing for a filename in line
                (if pos
                  (push line linelist)))))
      (setq selectlist
            (catch-cancel 
              (select-item-from-list linelist :selection-type :disjoint)))
      (when (and selectlist (not (eq selectlist :cancel)))
        (show-listener)
        (setq fromfilelist
              (mapcar #'(lambda (line) (read-from-string line nil nil :start (position #\" line)))
                      selectlist))
        (setq fromfilelist (remove-duplicates fromfilelist :test #'string-equal))
        (dolist (fromfile fromfilelist)
          (if (probe-file fromfile)
            (progn
              (setq tofile (server-to-home-name fromfile))
              (format t "~%About to copy file ~a to ~a -- " fromfile tofile)
              (copy-file-and-set-write-date fromfile tofile)
              (format t "DONE"))
            (format t "~%Did not copy file ~a because I could not find it." fromfile))))))
    
|#

(defun rcs-directoryp (string)
  (eql #\: (char string (1- (length string)))))

(defun copy-directory-1 (from to)
  (show-listener)
  (format t "~%About to copy ~s to ~s ~%" from to)
  (copy-directory from to t nil)
  (format t "~%DONE!~%")
  )

;;; copy one directory to another directory

;;; verboseflag,if true, prints out a DOT when a file is read in
;;; purge, if true, deletes the destination directory

(defun copy-directory (from to &optional (verboseflag t) (purge nil))
  (setq from (namestring from)
        to   (namestring to))
  (when verboseflag (show-listener))
  (unless (and (rcs-directoryp from) (probe-file from) (rcs-directoryp to) (not (equal from to)))
    (cond
     ((null (rcs-directoryp from)) (format t "~s is not a directory name, aborted" from))
     ((null (probe-file from)) (format t "Could not find directory ~s, aborted" from))
     ((null (rcs-directoryp to)) (format t "~s is not a directory name, aborted" to))
     ((equal from to) (format t "~s, source and destination directories are the same, aborted")))
    (return-from copy-directory))
  (if (or purge (null (probe-file to))) (create-file to :if-exists nil))
  (dolist (fromfile (list-of-files from))
    (let* ((filename (file-namestring fromfile))
           (tofile (merge-pathnames to filename))
           (tofilepresent (probe-file tofile))
           (fromfilewritedate (file-write-date fromfile))
           (tofilewritedate (and tofilepresent (file-write-date tofile))))
      ;;copy only if no file or new version of file
;      (when verboseflag 
;        (princ ".")
;        (fred-update *TOP-LISTENER*))
      (cond ((member (server-to-logical-server-name (namestring fromfile)) *files-not-to-copy* :test #'string-equal))
            ((or (null tofilepresent) 
                 (< tofilewritedate fromfilewritedate))
             (if tofilepresent (unlock-file tofile))
             (copy-file fromfile tofile :if-exists :overwrite)
;             (lock-file tofile)
             (when verboseflag (format t "~%~a copied." fromfile))
             (set-file-write-date tofile fromfilewritedate))
            ((and tofilewritedate (> tofilewritedate fromfilewritedate))
             (format t "~%Your version of ~a is newer than the server's version so it was left untouched."
                     tofile)))))
  (dolist (dir (directory (concat from "*.*") :directories t :files nil)) ;mac specific
    (let* ((newfromdir (directory-namestring dir))
           (newpartdir (strip-left from newfromdir))
           (newtodir (concat to newpartdir)))
      (copy-directory newfromdir newtodir verboseflag purge))))

      
;;; strip (length sub) characters from the left part of seq
;;; Used to strip off part of a directory from seq
;;; e.g. (strip-left "hd:" "hd:foo:") --> "foo:"
(defun strip-left (sub seq)
  (subseq seq (length sub)))

;;; Return a list of files in directory "dir"
;;; function is probably WRONG
(defun list-of-files (dir)
  (directory (concat dir "*.*")))

(defun show-help ()
 (message-dialog 
"                    User choices from the lockfile menu:
   - Lock a file.  This brings up a dialog so that the user can choose a 
     file to lock.
     If the file is already locked then the user gets an error message.  
     Locking a file copies the file from the server to the local hard disk.

   - Unlock a file and copy to server.  This brings up a dialog with all 
     your locked files.  
     Select 1 or more files (with shift-click) to unlock.  
     The files are copied back to the  server.

   - Forget. Unlock a file, but don't copy to server.  
     This is like the choice above but the files are not copied to the server.  
     Useful when the user changes his/her mind about making the 
     changes permanent.

   - Copy a newly created file to the server.  
     The user has just created a file on his/her hard disk.  
     To move it to the server choose this.

   - Update - copy server directory to local disk.  
     Updates all files on local disk.

   - Show all locked files.  
     Show a list of all the locked files, along with who locked them.

   - Show a list of past changes to files.
"
:size (make-point *screen-width* (- *screen-height* 40))))

(defun show-log-file nil
  (let ((win (make-instance 'fred-window
               :window-title "RCS Change Log"
               :scratch-p t)))
    (buffer-insert-file (fred-display-start-mark win) 
                        (expand-host *log-file*))
    (fred-update win)))

;;; ------------------------------------------------------------------------------
(init-rcs)