(unless (find-package :user)
  (make-package :user))

(in-package :user)
;:::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::::
;;; Flagged Dialog Items

; Defines a class of dialog item which has an associated flag (variable)
; We can set the value of the dialog-item by setting the variable,
; or read the dialog-item value by reading the variable.


; Mick O'Donnell
; University of Sydney
; Email: mick@isi.edu

; tested on for MCL 2.0p

(export '(dialog-item-parent set-flags set-from-flags  
          set-flags-and-exit-action clicked-item))

;1. Define the class

(DEFCLASS flagged-dialog-item (DIALOG-ITEM)
  ((flag :initarg :flag
         :accessor flag)))

(defmethod initialize-instance ((dialog flagged-dialog-item) &rest initargs)
  (apply #'call-next-method dialog initargs)
  (let ((flag-var (getf initargs :flag)))
    (setf (slot-value dialog 'flag) flag-var)
    (eval `(defvar ,flag-var))))

;2. Define some hybrid classes
(defclass flagged-check-box-dialog-item 
  (flagged-dialog-item check-box-dialog-item) nil)
(defclass flagged-radio-button-dialog-item 
  (flagged-dialog-item radio-button-dialog-item) nil)
(defclass flagged-editable-text-dialog-item 
  (flagged-dialog-item editable-text-dialog-item) nil)
(defclass flagged-table-dialog-item 
  (flagged-dialog-item table-dialog-item) nil)
(defclass flagged-sequence-dialog-item
  (flagged-dialog-item sequence-dialog-item) nil)
(defclass flagged-STATIC-TEXT-dialog-item 
  (flagged-dialog-item STATIC-TEXT-dialog-item) nil)
(defclass flagged-button-dialog-item 
  (flagged-dialog-item button-dialog-item) nil)
(defclass flagged-default-button-dialog-item 
  (flagged-dialog-item default-button-dialog-item) nil)
(defclass flagged-pop-up-menu 
  (flagged-dialog-item pop-up-menu) nil)


;3. Call set-flag to set the variable to the field-value of the dialog-item
;   (define your own entry for other dialog-item types)

(defmethod set-flag ((dialog-item flagged-dialog-item))
   (let ((flag (flag dialog-item)))
      (when flag
         (setf (symbol-value flag)
           (typecase dialog-item
               (flagged-check-box-dialog-item
                  (ccl::check-box-checked-p dialog-item))
               (flagged-radio-button-dialog-item 
                  (ccl::radio-button-pushed-p dialog-item))
               (flagged-table-dialog-item 
                  (ccl::table-sequence dialog-item))
               (flagged-sequence-dialog-item
                  (ccl::table-sequence dialog-item))
               (flagged-editable-text-dialog-item
                  (ccl::dialog-item-text dialog-item))
               (flagged-static-text-dialog-item 
                  (ccl::dialog-item-text dialog-item))
               (flagged-pop-up-menu 
                  (my-intern (menu-item-title (ccl::selected-item dialog-item))
                             :keyword))
               (t nil))))))

(defun set-flags (Dialog) 
;; takes values from each dialog-item field, and puts it into the variable named
;; in the flag field for the dialog-item
  (when Dialog
    (dolist (dialog-item (ccl::dialog-items dialog))
      (and (method-exists-p 'user::set-flag dialog-item )
           (user::set-flag dialog-item )))))
  

;4. Call set-from-flag to set the field-value of the dialog-item to the 
;   value of the variable. 
;   (define your own entry for other dialog-item types)

(defmethod set-from-flag ((dialog-item flagged-dialog-item))
   (let* ((flag (flag dialog-item))
          (flag-value (symbol-value flag)))
      (when flag
           (typecase dialog-item
                       (flagged-check-box-dialog-item
                          (if flag-value 
                            (ccl::check-box-check dialog-item) 
                            (ccl::check-box-uncheck dialog-item)))
                       (flagged-radio-button-dialog-item 
                          (if flag-value 
                            (ccl::radio-button-push dialog-item)
                            (ccl::radio-button-unpush dialog-item)))
                       (flagged-sequence-dialog-item
                            (ccl::set-table-sequence dialog-item flag-value))
                       (flagged-editable-text-dialog-item 
                          (ccl::set-dialog-item-text dialog-item (string flag-value)))
                       (flagged-static-text-dialog-item 
                          (ccl::set-dialog-item-text dialog-item (string flag-value)))
                       (flagged-pop-up-menu 
                        (setf (ccl::pop-up-menu-default-item dialog-item) 
                              (1+ (position (string-capitalize flag-value)
                                            (menu-items dialog-item) 
                                            :key #'menu-item-title
                                            :test #'equal))))
                       (t nil)))))


(defun set-from-flags (Dialog) 
;; sets the value of a dialog item from the associated flag variable
  (when Dialog
    (dolist (dialog-item (ccl::dialog-items dialog))
      (and (method-exists-p  'user::set-from-flag dialog-item)
           (user::set-from-flag dialog-item)))))

(defun dialog-item-parent (ditem)
  (view-container ditem))

(defun set-flags-and-exit-action (self)
  " Used by OK button - sets flags then exits"
  (user::set-flags (dialog-item-parent self))
  (return-from-modal-dialog t))

(defun clicked-item (sequence)
  (let ((index (car (ccl:selected-cells sequence))))
    (when index 
      (ccl:cell-contents sequence index))))

(defun my-intern (string package)
  "Interns the string in the package."
  (when (symbolp string) (setq string (string string)))
  (read-from-string (concatenate 'string (string package) "::" string)))

#| EXAMPLE

(setq *data-name* 'rank)
(setq  *data-type* :system)
(setq *graphing-depth* "10")

  (let ((window (MAKE-INSTANCE 'DIALOG
          :WINDOW-TYPE :DOUBLE-EDGE-BOX
          :VIEW-POSITION '(:TOP 59)
          :VIEW-SIZE #@(300 150)
          :CLOSE-BOX-P NIL
          :VIEW-FONT '("Chicago" 12 :SRCOR :PLAIN)
          :window-show nil
          :VIEW-SUBVIEWS 
          (LIST (MAKE-DIALOG-ITEM 'STATIC-TEXT-DIALOG-ITEM
                      #@(53 3) #@(173 19) "Specify Object to Graph" 'NIL)
                (MAKE-DIALOG-ITEM 'STATIC-TEXT-DIALOG-ITEM
                      #@(22 50)  #@(56 16) "Name:" 'NIL)
                (MAKE-DIALOG-ITEM 'flagged-EDITABLE-TEXT-DIALOG-ITEM
                      #@(78 49) #@(106 14) "Unit" 'NIL
                      :flag '*data-name*)
                (MAKE-DIALOG-ITEM 'STATIC-TEXT-DIALOG-ITEM
                      #@(22 83) #@(44 16) "Type:" 'NIL)
                (make-dialog-item 'flagged-pop-up-menu
                      #@(79 82) #@(84 20)  "" 'NIL
                      :dialog-item-text "Data Type"
                      :flag '*data-type*
                      :menu-items
                         (mapcar #'(lambda (type)
                                     (make-instance 'menu-item
                                         :menu-item-title (string-capitalize type)))
                                 '(feature system unit edge)))
                (MAKE-DIALOG-ITEM 'STATIC-TEXT-DIALOG-ITEM
                      #@(22 116) #@(56 16) "Depth:" 'NIL)
                (MAKE-DIALOG-ITEM 'flagged-EDITABLE-TEXT-DIALOG-ITEM
                      #@(89 116) #@(37 16) "100" 'NIL 
                      :flag '*graphing-depth*)
                (MAKE-DIALOG-ITEM 'BUTTON-DIALOG-ITEM
                      #@(220 80) #@(62 16) "Graph"
                      #'set-flags-and-exit-action
                      :DEFAULT-BUTTON T)))))

    ; set the initial dialog-item values to be those of the flags
    (set-from-flags window)
 
   (when (modal-dialog window t)
     ; the "Graph" button sets the variables to the values of the dialog-items
      (print (read-from-string *data-name*))
      (print *data-type*)
      (print (read-from-string *graphing-depth*))))

|#