;;; -*- Mode: Lisp; Package: ccl; -*-

;;; This file contains a simple CLOS class that encapsulates the hair
;;; of getting at information about Mac screens.  Calling (GDevices)
;;; yields a list of such, one for each screen on the Mac that is
;;; found.  The other methods extract the relevant information.  An
;;; alternative way to access GDevice information is part of Michael
;;; S. Engber's "Oodles-of-Utilities" package.

;;; Placed in the public domain 1992, Peter Szolovits, MIT.

(in-package :ccl)

(export '(GDevice GDevices topleft bottomright 
          mainscreen? colorscreen? activescreen?
          pixelres pixelsize))

(defclass GDevice ()
  ((the-mac-GDevice :initarg :GDevice :reader gdevice)))


(defun GDevices (&aux (ans nil))
  "Yields a list of GDevices, one for each screen on this Mac."
  (do ((gdev (#_GetGDevice)
	     (#_GetNextDevice gdev)))
      ((%null-ptr-p gdev) (nreverse ans))
    (push (make-instance 'GDevice :GDevice gdev) ans)))

(defmethod topleft ((gdev GDevice))
  "Yields a point that is the top left of this screen, in the coordinate
   system where the main screen's position is at #@(0 0)."
  (rref (gdevice gdev) :GDevice.gdRect.topleft))

(defmethod bottomright ((gdev GDevice))
  "Yields a point that is the bottom right of this screen."
  (rref (gdevice gdev) :GDevice.gdRect.bottomright))

(defmethod mainscreen? ((gdev GDevice))
  "Predicate to tell whether this is the main screen (with the Menubar)"
  (#_TestDeviceAttribute (gdevice gdev) #$mainScreen))

(defmethod colorscreen? ((gdev GDevice))
  "Prediate to tell whether this screen is a color screen."
  (#_TestDeviceAttribute (gdevice gdev) #$gdDevType))

(defmethod activescreen? ((gdev GDevice))
  "Prediate to tell whether this screen is active."
  ;;  I wish I knew just what "active" meant.
  (#_TestDeviceAttribute (gdevice gdev) #$screenActive))

(defmethod pixelres ((gdev GDevice))
  "Returns the h-v resolution (in pixels/inch) of this screen"
  ;; There is something I don't really understand here.  The hres and
  ;; vres fields both seem to be encoded a points whose h component is
  ;; 0 and whose v component is the actual pixel resolution.  I simply
  ;; return a point whose components are the corresponding
  ;; resolutions.  I hope this is as right as it seems to be.
  (let ((pixmap (rref (gdevice gdev) :gdevice.gdpmap)))
    (make-point (point-v (rref pixmap :pixmap.hres))
                (point-v (rref pixmap :pixmap.vres)))))

(defmethod pixelsize ((gdev GDevice))
  "Returns the number of bits/pixel in the pixmap of the GDevice (=
   color depth)"
  (let ((pixmap (rref (gdevice gdev) :gdevice.gdpmap)))
     (rref pixmap :pixmap.pixelsize)))

(provide :GDevice)
