;;; -*- Mode:Common-Lisp; Package:FILE-SYSTEM; Base:10; Fonts:(CPTFONT CPTFONT CPTFONT CPTFONT CPTFONTB) -*-

;;;  Rich Acuff, Stanford KSL, Feb-92

;;;  Code to do backups of Explorer file servers to other (usually unix) file
;;;  server.

;;;  Main entries are BACKUP-SERVER and MAYBE-BACKUP-SERVER.
;;;  Files being backed up are copied to a directory (call it R) on the
;;;  target file server.  Within R there are directories for files from full
;;;  backups (all files from the source), incremental backups (files from the
;;;  source that were not previously backed up), and log files.  The are
;;;  called FULL, INCR, and LOG respectively.  Within each of these three
;;;  there is a directory for each machine being backed up.  The directory
;;;  structure of the source machine is maintained in the backup copy under
;;;  the FULL or INCR directories.

(defparameter 4*full-backup-interval** 90
  "2Number of days between full backups.*")
(defparameter 4*incr-backup-interval** 1
  "2Number of days between full backups.*")

(defparameter 4*last-full-name** "3DATE-OF-LAST-FULL*")
(defparameter 4*last-incr-name** "3DATE-OF-LAST-INCR*")
(defparameter 4*last-backup-date-file-type** "3LISP*")

(defparameter 4*server-backup-root** (pathname "3expback:/*var3/expback/*")
  "2Files being backed up are stored in this directory.*")

(defparameter 4*server-backup-directory** :wild
  "2Files are backed up from this directory*")
(defparameter 4*server-backup-name** :wild
  "2Files matching this name are backed up.*")
(defparameter 4*server-backup-type** :wild
  "2Files matching this type are backed up.*")
(defparameter 4*server-backup-version** :latest
  "2Files matching this version are backed up.*")

(defparameter 4*report-addresses** '("2acuff@sumex-aim.stanford.edu*")
  "2List of email address strings.  Success or failure reports are mailed to
   these addresses.*")
(defparameter 4*backup-report-from-address**
	      "3From: Server Backup <bug-lispm@sumex-aim>*"
  "2From address for mail messages reporting backup status.*")

(defconstant 4*seconds-per-day** (* 24 60 60))

;;;----------------------------------------------------------------------

(defun 4log-msg *(message)
  (multiple-value-bind (sec min hr)
      (time:decode-universal-time (time:get-universal-time))
    (declare (ignore sec))
    (format *standard-output* "3~&~2,'0D:~2,'0D: ~A~%*" hr min message)))

(defun 4date-string* (&optional (univ-time (time:get-universal-time)))
  "2UNIV-TIME is a universal time value.  Return a string giving the date
   represented by UNIV-TIME in the form \"YY-MM-DD\".*"
  (multiple-value-bind (sec min hr day mon yr)
      (time:decode-universal-time univ-time)
    (declare (ignore sec min hr))
    (format nil "3~2,'0D-~2,'0D-~2,'0D*" (mod yr 100) mon day)))

(defun 4server-backup-root-path* (host subdir)
  "2HOST is the host being backed up.  SUBDIR is a directory within the backup
   target.  Return a pathname with HOST and SUBDIR added to the root.  Eg.,
   if the root is \"EXPBACK:/backup\", HOST is \"x1\", and SUBDIR is \"FULL\"
   then return \"EXPBACK:/backup/full/x1\".*"
  (send *server-backup-root* :new-directory
	(append (pathname-directory *server-backup-root*)
		(list subdir (send (net:parse-host host) :short-name)))))

(defun 4server-backup-root* (host full?)
  "2Where to send files being backed up from an Explorer file server.  Returns
   a pathname to the backup root for HOST, using the FULL direcotory if
   FULL? is non-NIL, and the INCR directory otherwise.*"
  (server-backup-root-path host (if full? "3FULL*" "3INCR*")))

(defun 4server-backup-log-root* (host)
  "2Pathname to the LOG directory.*"
  (server-backup-root-path host "3LOG*"))

(defun 4server-backup-log-path* (host full? &optional (extn ""))
  "2Return a pathname for a backup log file for host HOST for a full backup if
   FULL? is non-NIL, and an incremental backup otherwise.  EXTN is used in
   recursive calls to build alternate pathnames if the there is already a
   file with the pathname we want to use.*"       
  (let ((res (make-pathname :defaults (server-backup-log-root host)
			    :name (format nil "3~A-~A~A*" (date-string)
					  (if full? "3FULL*" "3INCR*")
					  extn)
			    :type "3LOG*")))
    ;1;If this name is taken, try another.*
    (if (probe-file res)
	(server-backup-log-path host full? (1+ (if (numberp extn) extn 1)))
	res)))

(defun 4server-backup-from-path* (host)
  "2Return a pat*hname for the files to be backed up from HOST."
3  (make-pathname :host host*
		3 :directory *server-backup-directory**
		3 :name *server-backup-name**
		3 :type *server-backup-type**
		3 :version *server-backup-version*))*

(defun 4login-to-backup-server* ()
  "2Make sure we can access the backup server.*"
  (file-host-user-id "expback" "expback")
  (store-password-etc "expback" "expback" "exp2back3"))

(defun 4report-successful-backup* (host full? nfiles nkbytes)
  "2Send mail notifying that the backup worked.  HOST is the machine backed
   up, FULL? is non-NIL for full backups, NFILES is the number of files
   backed up, and NKBYTES is the number of kilobytes copied during the
   backup.*"
  (mail:submit-mail
    (format nil "3~A backup copied ~:D files, totalling ~:DkB.*"
	    (if full? "3Full*" "3Incremental*") nfiles nkbytes)
    :to *report-addresses*
    :subject (format nil "3Backup of ~A*" host)
    :other-headers (list *backup-report-from-address*)))

(defun 4report-errors-in-backup* (host message)
  "2Send mail notifying that there was an error during the backup.  HOST
   is the machine that was begin backed up.  MESSAGE is whatever was written
   to *ERROR-OUTPUT*.*"
  (mail:submit-mail
    (format nil "3Error message:~%~A*" message)
    :to *report-addresses*
    :subject (format nil "3ERROR IN BACKUP OF ~A*" host)
    :other-headers (list *backup-report-from-address*)))

(defmacro 4reporting-results* (host full? &body body)
  "2Execute BODY.  If there is an error (ie. anything written to
   *ERROR-OUTPUT*) during BODY, send a failure message, otherwise send a
   success message.  HOST is the machine being backed up and FULL? is non-NIL
   if it's a full backup.*"
  `(let* ((*error-output* (make-string-output-stream))
	  (vals (multiple-value-list (catch-error ,@body))))
     (let ((message (get-output-stream-string *error-output*)))
       (if (zerop (length message))
	   (progn
	     (report-successful-backup ,host ,full? (first vals) (second vals))
	     (write-backup-date-file ,host ,full?))
	   (report-errors-in-backup ,host message)))))

(defprop 4reporting-results* -2 sys::specially-grind)
(defprop 4reporting-results* (2 1) zwei::lisp-indent-offset)

(defun 4ensure-directory* (path)
  "2Make sure the directory specified by PATH exists.*"
  (unless (send path :probe-directory)
    (fs:create-directory path)))

(defmacro 4with-log-file* ((stream path) &body body)
  "2Execute BODY with STREAM writing to the file named by PATH, making sure
   that PATH's directory exists.*"
  `(progn (ensure-directory ,path)
	  (with-open-file (,stream ,path :direction :output)
	    ,@body)))

(defprop 4with-log-file* -1 sys::specially-grind)
(defprop 4with-log-file* (1 1) zwei::lisp-indent-offset)

(defun 4remove-old-files* (host)
  "2Used by full backup to delete files from previous full and incremental
   backups.*"
  (let ((incr (server-backup-root host nil))
	(full (server-backup-root host t)))
    (when (send incr :probe-directory)
      (delete-directory incr :are-you-sure t))
    (when (send full :probe-directory)
      (delete-directory full :are-you-sure t))))

(defun 4write-backup-date-file* (host full?)
  "Writes today's date into a file in HOST's log directory in a format
   acceptable to READ and TIME:PARSE-UNIVERSAL-TIME.  Writes to one of two
   file names, depending on FULL?."
  (with-open-file (s (last-backup-date-pathname host full?) :direction :output)
    (print (time:dd-mmm-yy-string) s)))

(defun 4backup-server-1* (host full?)
  "G2uts of BACKUP-SERVER*"
  (with-log-file (log (server-backup-log-path host full?))
    (let ((fs:*be-quiet-about-already-backed-up-files* t))
      (when full?
	(log-msg "Removing old files")
	(remove-old-files host))
      (log-msg "3Copying files*")
      (backup-files :from-path (server-backup-from-path host)
		    :to-root (server-backup-root host full?)
		    :all-p full?
		    :report-file log
		    :short-report t))))

(defun 4backup-server* (host &optional full?)
  "2Backup files from HOST.  If FULL? is non-NIL this is a full backup and all
   files will be copied.  If FULL? is NIL this is an incremental backup and
   only not-backed-up files will be copied.*"
  (reporting-results host full?
    (sys:with-sys-host-accessible 
      (setf host (net:parse-host host))
      (login-to-backup-server)
      (backup-server-1 host full?))))

(defun 4last-backup-date-pathname* (host full?)
  "Pathname in HOST's log directory where date of last backup is or will be
   stored.  Returns one of two pathnames, depending on FULL?."
  (make-pathname :defaults (server-backup-log-root host)
		 :name (if full? *last-full-name* *last-incr-name*)
		 :type *last-backup-date-file-type*))

(defun 4date-of-last-backup* (host full?)
  "Reads last backup date from the file, if any, on HOST's log directory, and
   returns it as a universal time, or 0 if there was no file."
  (let ((path (last-backup-date-pathname host full?)))
    (if (probe-file path)
	(with-open-file (s (last-backup-date-pathname host full?))
	  (time:parse-universal-time (read s)))
	0)))

(defun 4days-between* (start-ut end-ut)
  "2Returns the number of days between the universal times START-UT and END-UT.*"
  (- (truncate end-ut *seconds-per-day*) (truncate start-ut *seconds-per-day*)))

(defun 4time-to-do-full?* (host)
  "2Non-NIL if it is time to do a full backup of HOST, ie. more than
   *FULL-BACKUP-INTERVAL* days have past since the last backup according to
   DATE-OF-LAST-BACKUP.*"
  (>= (days-between (date-of-last-backup host t) (get-universal-time))
      *full-backup-interval*))

(defun 4time-to-do-incr?* (host)
  "2Non-NIL if it is time to do a*n incremental2 backup of HOST, ie. more than
   **INCR2-BACKUP-INTERVAL* days have past since the last backup according to
   DATE-OF-LAST-BACKUP.*"
  (>= (days-between (date-of-last-backup host nil) (get-universal-time))
      *incr-backup-interval*))

(defun 4maybe-backup-server* (&optional (host (net:parse-host "3LM*")))
  "Determine if a backup of HOST is necessary and do it."
  (let* ((*error-output* (make-string-output-stream)))
    (catch-error
      (sys:with-sys-host-accessible
	(setf host (net:parse-host host))
	(login-to-backup-server)
	(if (time-to-do-full? host)
	    (progn (log-msg (format nil "3Doing full backup of ~A~%*" host))
		   (backup-server host t))
	    (if (time-to-do-incr? host)
		(progn (log-msg (format nil "3Doing incremental backup of ~A~%*" host))
		       (backup-server host nil))
		(log-msg (format nil "3Not doing backup of ~A~%*" host))))))
     (let ((message (get-output-stream-string *error-output*)))
       (unless (zerop (length message))
	 (report-errors-in-backup
	   host
	   (format nil "3Error in MAYBE-BACKUP-SERVER:~%~A*" message))))))

#||
(defparameter *server-backup-root* (pathname "expback:/a/acuff/test/"))
(defparameter *server-backup-directory* "TEMP")

(defun test (full?)
  (unless full? (copy-file "x1:temp;temp.text" "x1:temp;temp.text#>"))
  (backup-server "x1" full?))

(test t)

(maybe-backup-server "x1")

||#
