;;; -*- Mode:Lisp; Package: ILU; Syntax:COMMON-LISP; Base:10 -*-
#|
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.

$Id: ilu.lisp,v 1.68 1994/05/11 01:55:46 janssen Exp $
|#

(cl:in-package :ilu)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Constants
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;;;;;;;;;;;;;;;;;; packet types ;;;;;;;;;;;;;;;;;;;;

(defconstant +packet-type-request+	0	"type value for request packet")
(defconstant +packet-type-reply+	1	"type value for reply packet")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Objects
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun class-or-class-name->type (c-or-c-n)
  (if (or (stringp c-or-c-n)
	  (symbolp c-or-c-n))
      (find-class c-or-c-n)
    c-or-c-n))

(defun class-or-class-name->name (c-or-c-n)
  (if (or (stringp c-or-c-n)
	  (symbolp c-or-c-n))
      c-or-c-n
    (class-name c-or-c-n)))

(defmethod ilu-class-id (something)
  (declare (ignore something))
  nil)

(defmethod ilu-class-name (classname)
  (declare (ignore classname))
  nil)

(defmethod ilu-class-record (classname)
  (declare (ignore classname))
  nil)

(defun get-class-name (class)
  (let ((cp (ilu-class-name (class-or-class-name->name class))))
    (char*-to-string cp)))

(defclass ilu-object ()
  ((ilu-cached-kernel-obj :initform nil :accessor ilu-cached-kernel-obj)	; C-runtime handle
   (ilu-nms-handle 	:initform nil :accessor ilu-nms-handle)
   (ilu-server     	:initform nil :accessor ilu-server)
   (ilu-class	   	:initform nil :accessor ilu-class)
   (ilu-publish-proof	:initform nil :accessor ilu-publish-proof)
   (ilu-instance-id	:initform nil)
   (ilu-string-binding-handle :initform nil)))

(defmethod initialize-instance :after ((self ilu-object) &key &allow-other-keys)
  #+excl
  (excl:schedule-finalization self #'collect-instance)
  )

;; before:  NOT Inside (server, class)
;; after:  if result non-nil, Inside(server, class) of result
(defmethod kernel-obj ((self ilu-object))
  (ilu_enter-server (ilu-server self) (ilu-class self))
  (unless (ilu-cached-kernel-obj self)
    (ilu_exit-server (ilu-server self) (ilu-class self)))
  (ilu-cached-kernel-obj self))

(defmethod instance-id ((self ilu-object))
  (or (slot-value self 'ilu-instance-id)
      (if (null (ilu-cached-kernel-obj self))
	  "(no ilu object!)"
	(let ((kernel-object (kernel-obj self)))
	  (if (null kernel-object)
	      nil
	    ;; now Inside(SERVER,CLASS)
	    (let* ((id-pointer (ilulisp_instance-id kernel-object))
		   (id-string (char*-to-string id-pointer)))
	      (ilu_exit-server (ilu-server self) (ilu-class self))
	      ;; now out of Server
	      (setf (slot-value self 'ilu-instance-id) id-string)
	      id-string))))))

(defmethod print-object ((self ilu-object) stream)
  (format stream "#<~s 0x~x>"
	  (type-of self)
	  (ilu-cached-kernel-obj self)))

(defclass ilu-singleton-object (ilu-object) ())

(defmethod find-ilu-class-name (object)
  (declare (ignore object))
  nil)

(defmethod find-ilu-class-name ((object ilu-object))
  (dolist (class (clos:class-precedence-list (class-of object)))
    (let ((name-pointer (ilu-class-name (class-name class))))
      (when (and name-pointer (> name-pointer 0))
	(return (char*-to-string name-pointer))))))

(defmethod find-ilu-class-id ((object ilu-object))
  (dolist (class (clos:class-precedence-list (class-of object)))
    (let ((id-pointer (ilu-class-id class)))
      (when (and id-pointer (> id-pointer 0))
	(return (char*-to-string id-pointer))))))

(defmethod find-ilu-class-record ((object ilu-object))
  (dolist (class (clos:class-precedence-list (class-of object)))
    (let ((record (ilu-class-record (class-name class))))
      (when (and record (> record 0)) (return record)))))

(defmethod rpc-method ((self ilu-object) putative-class method-id)
  (find-method-by-id putative-class method-id))

(defun id-of-class-record (ilu-class-record)
  (when ilu-class-record
    (let ((p (ilulisp_id-of-class ilu-class-record)))
      (char*-to-string p))))

(defmethod ilu-class-id ((self standard-class))
  (let ((id nil))
    (do* ((class-list (clos:class-precedence-list self) (cdr class-list))
	  (class (car class-list) (car class-list)))
	((or id (null class)) id)
      (setq id (id-of-class-record (ilu-class-record (class-name class)))))))

(defmethod ilu-class-id ((self ilu-object))
  (ilu-class-id (class-of self)))

(defvar *class-table* (make-hash-table) "KEY is kernel class record, VALUE is CLOS class")

(defvar *ilu-initialized* nil "t if the ILU kernel has been initialized")

(defun initialize-ilu-class (class-name)
  (let ((record (ilu-class-record class-name)))
    (when *ilu-initialized*
      (ilu_register-class record))
    (setf (gethash record *class-table*) (find-class class-name))))

(defun initialize-ilu ()
  ;; set up LockTech and WaitTech
  (unless *ilu-initialized*
    (setf *ilu-initialized* t)
    ;;
    (initialize-locking)	;; defined in "ilu-<implementation>"
    ;;
    (maphash #'(lambda (k v)
		 (declare (ignore v))
		 (if (= 0 (ilu_find-class-from-id (ilulisp_id-of-class k)))
		     (ilu_register-class k)))
	     *class-table*)))

(defmacro ensure-initialized ()
    `(unless *ilu-initialized* (initialize-ilu)))

(defmethod singleton-p ((self ilu-object))
  nil)

(defmethod singleton-p ((self ilu-singleton-object))
  t)

;; Inside(server,class)
(defun language-specific-object (obj)
  (let ((index (get-language-specific-object obj)))
    (unless (zerop index)
      (lookup-registered-lisp-object (1- index)))))

;; Inside(server,class)
(defsetf language-specific-object (obj) (lspo)
  `(progn
     (register-language-specific-object ,obj (1+ (register-lisp-object ,lspo)))
     ,lspo))

;; before: Inside(kernel-object->ob_server, kernel-object->ob_class)
;; after:  Main Invariant holds
(defmethod setup-object-links ((lisp-obj ilu-object) kernel-object)
  (setf (language-specific-object kernel-object) lisp-obj)
  (setf (ilu-server lisp-obj) (ilu_ilu-server kernel-object))
  (setf (ilu-class lisp-obj) (ilu_ilu-class kernel-object))
  (setf (ilu-cached-kernel-obj lisp-obj) kernel-object)
  (ilu_exit-server (ilu-server lisp-obj) (ilu-class lisp-obj)))

;; before: Inside(server,class)
;; after:  Main invariant holds
(defun ilu-object->instance (class kernel-obj)
  (let ((lisp-obj (language-specific-object kernel-obj)))
    (if lisp-obj
	(ilu_exit-server (ilu-server lisp-obj) (ilu-class lisp-obj))
      (progn
	(setq lisp-obj (make-instance class)) 
	(setup-object-links lisp-obj kernel-obj)))
    lisp-obj))

;; Main invariant holds
(defun sbh->instance (class-name sbh &optional mstid)
  "Given the CLASS-NAME and STRING-BINDING-HANDLE, return an ilu-object"
  (declare (string sbh mstid))
  (ensure-initialized)
  (let* ((class-record (ilu-class-record class-name))
	 (ilu-obj (object-of-sbh sbh (or mstid (id-of-class-record class-record)) class-record)))
    (if (zerop ilu-obj)
	nil
      ;; now Inside 
      (ilu-object->instance class-name ilu-obj)
      ;; now Main holds
      )))

(defmethod collect-instance ((instance ilu-object))
  (when (ilu-cached-kernel-obj instance)
    (setf (ilu-cached-kernel-obj instance) nil))
  (values))

;; Main invariant holds
(defmethod string-binding-handle ((self ilu-object))
  (or (slot-value self 'ilu-string-binding-handle)
      (let ((kobj (kernel-obj self)))
	(if kobj
	    (let* ((sbh (ilu_sbh-of-object kobj))
		   (lisp-sbh (char*-to-string sbh)))
	      (setf (slot-value self 'ilu-string-binding-handle) lisp-sbh)
	      (ilu_exit-server (ilu-server self) (ilu-class self))
	      lisp-sbh)))))

(defun object-id-read (call pclass discriminator)
  (let* ((class-record-pointer		; pointer to hold second value
	  (make-array 1 :element-type '(unsigned-byte 32)))
	 (id (ilulisp_input-object-id call class-record-pointer pclass discriminator)))
    (declare (type (simple-array (unsigned-byte 32) (1)) class-record-pointer)
	     (dynamic-extent class-record-pointer))
    (values id (aref class-record-pointer 0))))

(defun object-read (class call static-type-name)
  (multiple-value-bind (ilu-obj class-record) (object-id-read call (ilu-class-record static-type-name) 0)
    (if (and ilu-obj (not (zerop ilu-obj)))
	(let ((lisp-obj (ilu-object->instance (gethash class-record *class-table* class) ilu-obj)))
	  lisp-obj)
      (error "Couldn't read kernel object of class ~s with pclass ~s~%" class static-type-name))))

(defmethod object-write (call (self ilu-object) pclass)
  (object-id-write call (kernel-obj self) 0 (ilu-class-record pclass))
  self)

(defun object-id-size (call ilu-object discriminator-p static-type exit-server-p)
  (let ((size (ilu_object-id-size call (kernel-obj ilu-object) discriminator-p static-type)))
    (when exit-server-p
      (ilu_exit-server (ilu-server ilu-object) (ilu-class ilu-object)))
    size))

(defmethod object-size (call (self ilu-object) pclass)
  (object-id-size call self 0 (ilu-class-record pclass) t))

(defmethod publish ((self ilu-object))
  (let ((kobj (kernel-obj self)))
    (when kobj
      (setf (ilu-publish-proof self)
	(let ((proof (ilu_publish-object kobj)))
	  (unless (zerop proof)
	    (char*-to-string proof)))))))

(defmethod withdraw ((self ilu-object))
  (when (stringp (ilu-publish-proof self))
    (let ((kobj (kernel-obj self))
	  (p (string-to-char* (ilu-publish-proof self))))
      (when kobj
	(let ((stat (ilu_withdraw-published-object kobj p)))
	  (free p)
	  (unless (zerop stat)
	    (setf (ilu-publish-proof self) nil))
	  (not (= stat 0)))))))

(defun lookup (type oid)
  (ensure-initialized)
  (let* ((class-record (ilu-class-record (class-or-class-name->name type)))
	 (kobj (ilu_lookup-object-by-oid oid class-record)))
    (when (not (zerop kobj))
      (ilu-object->instance (class-or-class-name->type type) kobj))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Exceptions
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(define-condition rpc-exception (serious-condition)
  ((exception-value :accessor exception-value :initarg :exception-value))
  (:documentation "All exceptions signalled by RPC servers are subtypes of this exception"))

(defgeneric exception-value-read (call exception))
(defmethod exception-value-read (call (exception rpc-exception))
  (declare (ignore call)))

(defgeneric exception-value-size (call exception))
(defmethod exception-value-size (call (exception rpc-exception))
  (declare (ignore call))
  0)

(defgeneric exception-value-write (call exception))
(defmethod exception-value-write (call (exception rpc-exception))
  (declare (ignore call)))

(defmethod initialize-instance :after ((exception rpc-exception)
				       &key (call nil call-p)
				       &allow-other-keys)
  (when call-p
    (setf (exception-value exception)
      (exception-value-read call exception))))

(defun signal-exception (call position e)
  (begin-exception call position (exception-value-size call e))
  (exception-value-write call e)
  (finish-exception call))

(define-condition protocol-error (rpc-exception) ()
  (:documentation "RPC Error"))

;;;;;;;;;;;;;;;;;;;; protocol error values ;;;;;;;;;;;;;;;;;;;;

(defconstant +protocol-exception-success+           0
  "value returned when RPC protocol has succeeded")
(defconstant +protocol-exception-no-such-class+     1
  "value for bad class ID at server")
(defconstant +protocol-exception-version-mismatch+  2
  "value for bad version at server")
(defconstant +protocol-exception-no-such-method+    3
  "value for bad method ID at server")
(defconstant +protocol-exception-bad-arguments+     4
  "value for invalid argument formatting")
(defconstant +protocol-exception-unknown+           5
  "catchall for protocol errors")
(defconstant +protocol-exception-no-connection+     6
  "value for bad or lost connection")
(defconstant +protocol-exception-request-rejected+  7
  "value for server refusing to look at packets")
(defconstant +protocol-exception-request-timeout+   8
  "no answer within timeout period")

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;
;;;  Call primitive functions
;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(defun begin-call (obj)
  (check-type obj ilu-object)
  (let ((call (ilu_begin-call (ilu-server obj))))
    (if (= call 0)
	(error (make-condition
		'protocol-error
		:format-control "protocol error: no connection to server in ~s"
		:format-arguments (list obj)
		:exception-value +protocol-exception-no-connection+))
      call)))

(defmacro begin-reply (call exceptions-p size)
  `(ilu_begin-reply ,call (if ,exceptions-p 1 0) ,size))

(defun wait-for-reply (call)
  (let* ((p1 (make-array 1 :element-type '(unsigned-byte 32)))
	 (protocol-error (ilu_get-reply call p1)))
    (values protocol-error (aref p1 0))))
