/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: tcpascii.c,v 1.4 1994/02/15 20:33:29 janssen Exp $ */
/* Last tweaked by Mike Spreitzer September 9, 1993 4:44 pm PDT */

#include "ilu.h"
#include "iluntrnl.h"

#include "transprt.h"
#include "mooring.h"

#include <sys/types.h>
#include <sys/socket.h>
#include <netinet/in.h>
#include <sys/param.h>		/* for MAXHOSTNAMELEN */
#include <netdb.h>		/* for gethostbyname() */
#include <arpa/inet.h>
#include <assert.h>
#include <sys/ioctl.h>
#include <sys/filio.h>		/* for FIONBIO */
#include <signal.h>
#include <stdlib.h>
#include <string.h>

#include "bffrlist.h"

ilu_TransportClass _ilu_tcpascii_TransportClass(void);

typedef struct tcpparms {
  ilu_string hostname;
  ilu_cardinal port;
  bufferList buffer;
} *TCPParms;

#define TCP_HOSTNAME(a) (((TCPParms)(a))->hostname)
#define TCP_PORT(a) (((TCPParms)(a))->port)
#define TCP_BUFFER(a) (((TCPParms)(a))->buffer)

#define MAXDUMP		10000

ilu_string _ilu_tcpascii_CurrentHostInetName(void);

/***********************************************************************
/***********************************************************************
/***********************************************************************
/**** Some local utilities ************************
/***********************************************************************
/***********************************************************************
/**********************************************************************/

static void FmtSockaddrInet(struct sockaddr_in *sin, char *buf)
{
  sprintf(buf, "IP %d.%d.%d.%d:%d",
	  sin->sin_addr.S_un.S_un_b.s_b1,
	  sin->sin_addr.S_un.S_un_b.s_b2,
	  sin->sin_addr.S_un.S_un_b.s_b3,
	  sin->sin_addr.S_un.S_un_b.s_b4,
	  sin->sin_port);
  return;
}

/***********************************************************************
/***********************************************************************
/***********************************************************************
/**** First, the methods for the TCP Transport ************************
/***********************************************************************
/***********************************************************************
/**********************************************************************/

static ilu_boolean SigPIPEHandler = FALSE;

static void _tcpascii_HandleSigPIPE (void)
{
  /* Ignore SIGPIPEs, since we can occasionally get them.  There
     should be a way to do this that cooperates with other modules
     that want to handle signals.

     [Thanks to Dave Nichols <nichols@parc.xerox.com> for this code.]  */

  struct sigvec sv, osv;

  sv.sv_handler = SIG_IGN;
  sv.sv_mask = 0;
  sv.sv_flags = 0;

  if (OS_SIGVEC(SIGPIPE, &sv, &osv) == 0 && osv.sv_handler != SIG_DFL)
    /* Oops!  Set it back. */
    OS_SIGVEC(SIGPIPE, &osv, &sv);
  SigPIPEHandler = TRUE;
}

static ilu_private _tcpascii_InterpretInfo (ilu_string info)
{
  char hostname[1000];
  ilu_integer port;
  ilu_boolean buffered = FALSE;

  if (*info == 'b')
    buffered = TRUE;
  if ((sscanf (info + (buffered ? 1 : 0), "tcpascii_%[^_]_%u", hostname, &port)) == 2)
    {
      struct tcpparms *new = (struct tcpparms *) malloc(sizeof(struct tcpparms));
      new->hostname = _ilu_Strdup(hostname);
      new->port = port;
      if (buffered)
	{
	  new->buffer = (bufferList) malloc(sizeof(struct bufferlist_s));
	  new->buffer->size = 0;
	  new->buffer->head = NULL;
	  new->buffer->tail = NULL;
	}
      else
	new->buffer = NULL;
      return ((ilu_private) new);
    }
  else
    return (NULL);
}

static void DumpPacket (ilu_byte *packet, ilu_shortcardinal length)
{
  ilu_cardinal dumplength, n;
  ilu_byte c;
  register ilu_integer i, j;

  if (length > MAXDUMP)
    {
      fprintf (stderr, "Request to dump packet of %u bytes.  Only %u bytes being dumped.\n", length, MAXDUMP);
      dumplength = MAXDUMP;
    }
  else
    dumplength = length;
  if (packet == NULL)
    {
      fprintf (stderr, "Attempt to dump NULL packet.\n");
      return;
    }
  fprintf (stderr, "DumpPacket of packet 0x%x, length %u bytes, dumping %u bytes:\n",
	   packet, length, dumplength);
  for (i = 0;  i < dumplength;  i += 16)
    {
      fprintf (stderr, "%6u:  ", i);
      for (j = 0;  j < 16 AND (i + j) < dumplength;  j += 1)
	fprintf (stderr, "%02x%s ", packet[i + j], ((j % 4) == 3) ? " " : "");
      n = (((16-j)/4)*(13)+((16-j)%4)*3)+1;		/* padding before ascii version */
      fprintf (stderr, "% *.*s", n, n, "");
      for (j = 0;  j < 16 AND (i + j) < dumplength;  j += 1)
	{
	  c = packet[i + j];
	  fprintf (stderr, "%c", ((c >= ' ') && (c <= '~')) ? (char) c : '.');
	}
      fprintf (stderr, "\n");
    }
}

static ilu_boolean _tcpascii_DataPresent (ilu_Transport conn)
{
  long count;
  ilu_boolean status;

      status = ((ioctl (conn->tr_fd, FIONREAD, &count) == 0) && (count > 0));
      return (status);
}

static ilu_boolean _tcpascii_WaitForInput (ilu_Transport conn)
{
  fd_set readfds;
  ilu_boolean status;

      _ilu_WaitForInputOnFD(conn->tr_fd);
      return (TRUE);
}

static ilu_boolean _tcpascii_FlushOutput (ilu_Transport self)
{
  bufferList buf;
  bufferElement new, old;
  ilu_boolean status;
  ilu_integer i;

  if (self == NULL)
    return (FALSE);
  
  if ((buf = TCP_BUFFER(transport_data(self))) != NULL && buf->size > 0)
    {
      for (new = buf->head, status = TRUE;  status && buf->size > 0 && new != NULL;  buf->size -= 1)
	{
	  char lenbuf[15];
	  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		(stderr, "_tcpascii_FlushOutput:  writing %d bytes from 0x%x to fd %d.\n", new->size, new->packet, self->tr_fd));

	  sprintf (lenbuf, "%013.13u ", new->size);
	  if ((i = _ilu_Write (self->tr_fd, lenbuf, 14)) < 14)
	    {
	      fprintf (stderr, "_tcpascii_SendMessage:  output to fd %u of %d bytes signalled error \"%s\".\n",
		       self->tr_fd, 14, (i < 0) ? ANSI_STRERROR(errno) : ((i == 0) ? "Connection lost" : "not enough bytes written"));
	      status = FALSE;
	    }
	  else if ((i = _ilu_Write (self->tr_fd, new->packet, new->size)) < new->size)
	    {
	      fprintf (stderr, "_tcpascii_SendMessage:  output to fd %u of %d bytes signalled error \"%s\".\n",
		       self->tr_fd, new->size, (i < 0) ? ANSI_STRERROR(errno) : ((i == 0) ? "Connection lost" : "not enough bytes written"));
	      status = FALSE;
	    }
	  else
	    {
	      old = new;
	      new = new->next;
	      free(old->packet);
	      free(old);
	    }
	}
      if (status)
	{
	  buf->size = 0;
	  buf->head = NULL;
	  buf->tail = NULL;
	}
    }

  return (status);
}

static ilu_boolean _tcpascii_AddPacketToBuffer(ilu_Transport self, ilu_byte *packet, ilu_cardinal size, ilu_boolean eom)
{
  bufferList buf;
  bufferElement new;

  if (self == NULL)
    return (FALSE);

  buf = TCP_BUFFER(transport_data(self));
  new = (bufferElement) malloc(sizeof(struct bufferlist_element_s));

  new->packet = packet;
  new->size = size;
  new->next = NULL;
  new->EOM = eom;
  if (buf->tail == NULL)
    buf->head = buf->tail = new;
  else
    buf->tail->next = new;
  buf->size += 1;

  return (TRUE);
}

static ilu_boolean _tcpascii_SendMessage (ilu_Transport conn, ilu_byte *buffer, ilu_cardinal count)
{
  ilu_boolean status = TRUE;

  if (conn == NULL)
    return (FALSE);

  if (TCP_BUFFER(transport_data(conn)) == NULL || (!_tcpascii_AddPacketToBuffer(conn, buffer, count, FALSE)))
    {
      char lenbuf[15];
      ilu_boolean status;
      ilu_integer i;

      _tcpascii_FlushOutput (conn);

      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	    (stderr, "_tcpascii_SendMessage:  writing %d bytes from 0x%x to fd %d.\n", count, buffer, conn->tr_fd));

      if ((_ilu_DebugLevel & PACKET_DEBUG) != 0)
	DumpPacket(buffer, count);

      status = TRUE;
      sprintf (lenbuf, "%013.13u ", count);
      if ((i = _ilu_Write (conn->tr_fd, lenbuf, 14)) < 14)
	{
	  fprintf (stderr, "_tcpascii_SendMessage:  output to fd %u of %d bytes signalled error \"%s\".\n",
		   conn->tr_fd, 14,
		   (i < 0) ? ANSI_STRERROR(errno) : ((i == 0) ? "Connection lost" : "not enough bytes written"));
	  status = FALSE;
	}
      else
	{
	  if ((i = _ilu_Write (conn->tr_fd, buffer, count)) < count)
	    {
	      fprintf (stderr, "_tcpascii_SendMessage:  output to fd %u of %d bytes signalled error \"%s\".\n",
		       conn->tr_fd, count,
		       (i < 0) ? ANSI_STRERROR(errno) : ((i == 0) ? "Connection lost" : "not enough bytes written"));
	      status = FALSE;
	    }
	  else
	    status = TRUE;
	}
      free(buffer);
    };
  return (status);
}

static ilu_boolean _tcpascii_Connect (ilu_Transport self)
{
  ilu_integer fd;
  struct sockaddr_in sin;
  struct hostent *hp;
  ilu_Transport *new;
  ilu_boolean status = TRUE;
  int one = 1;
  int res;

  _ilu_AutoSetDebugLevel();
  if (!SigPIPEHandler)
    _tcpascii_HandleSigPIPE();

  if (self == NULL)
    return (FALSE);

  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	(stderr, "_tcpascii_Connect:  connecting to host %s, port %d...\n",
	 TCP_HOSTNAME(transport_data(self)),
	 TCP_PORT(transport_data(self))));

  memset((ilu_string ) &sin, 0, sizeof(sin));
  if ((fd = OS_SOCKET(AF_INET, SOCK_STREAM, 0)) < 0)
    {
      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	    (stderr, "_tcpascii_Connect:  socket call failed:  %s.\n",
	     ANSI_STRERROR(errno)));
      status = FALSE;
    }
  else
    {
      (void) OS_SETSOCKOPT(fd, SOL_SOCKET, SO_REUSEADDR,
			   (ilu_string ) NULL, 0);
      (void) OS_SETSOCKOPT(fd, SOL_SOCKET, SO_USELOOPBACK,
			   (ilu_string ) NULL, 0);

/* The following was suggested by Ken Birman of the Cornell ISIS project.
 * On SunOS it prevents the kernel from delaying this packet in hopes
 * of merging it with a quickly-following one.
 */
#ifdef IPPROTO_TCP
#ifdef TCP_NODELAY
      OS_SETSOCKOPT (fd, IPPROTO_TCP, TCP_NODELAY, &one, sizeof(one));
#endif /* TCP_NODELAY */
#endif /* IPPROTO_TCP */     

      sin.sin_family = AF_INET;
      sin.sin_port = TCP_PORT(transport_data(self));

      sin.sin_addr.s_addr = OS_INETADDR( TCP_HOSTNAME(
		transport_data( self)));
      if (sin.sin_addr.s_addr == -1 || sin.sin_addr.s_addr == 0)
	if ((hp = OS_GETHOSTBYNAME(TCP_HOSTNAME(transport_data(self))))
	    != NULL)
	  OS_BCOPY(hp->h_addr,(ilu_string ) &sin.sin_addr, hp->h_length);

      if (sin.sin_addr.s_addr == -1 || sin.sin_addr.s_addr == 0)
	{
	  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		(stderr, "_tcpascii_Connect:  Invalid host name (%s)\n",
		 TCP_HOSTNAME(transport_data(self))));
	  status = FALSE;
	}
      else if (ioctl(fd, FIONBIO, &one) != 0) {
        DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		(stderr, "%s (descriptor 0x%x, host %s) non-blocking.\n",
		 "_tcpascii_Connect:  Failed to set socket",
		 fd, TCP_HOSTNAME(transport_data(self))));
	  status = FALSE;
	}
      else
	{
	  if (OS_CONNECT(fd, (struct sockaddr *) &sin, sizeof(sin)) == 0)
	    {
	      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		    (stderr, "_tcpascii_Connect:  connected to %s[%d]\n",
		     inet_ntoa(sin.sin_addr),
		     TCP_PORT(transport_data(self))));
	      self->tr_fd = fd;
	      status = TRUE;
	    }
	  else if (errno == EINPROGRESS) {
	    struct sockaddr peername;
	    int pnlen = sizeof(peername);
	    _ilu_WaitForOutputOnFD(fd);
	    if (getpeername(fd, &peername, &pnlen) == 0) {
	      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		    (stderr,
		     "_tcpascii_Connect:  eventually connected to %s[%d]\n",
		     inet_ntoa(sin.sin_addr),
		     TCP_PORT(transport_data(self))));
	      self->tr_fd = fd;
	      status = TRUE;
	      }
	    else {
	      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		    (stderr, "%s %s[%d] failed (%s).\n",
		     "_tcpascii_Connect: connect to",
		     OS_INETNTOA(sin.sin_addr),
		     TCP_PORT(transport_data(self)),
		     "no meaningful errno available" ));
	      OS_CLOSE(fd);
	      status = FALSE;
	      }
	    }
	  else
	    {
	      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
		    (stderr, "%s %s[%d] failed, error %d (%s)\n",
		     "_tcpascii_Connect: connect to",
		     OS_INETNTOA(sin.sin_addr),
		     TCP_PORT(transport_data(self)), errno,
		     ANSI_STRERROR(errno)));
	      OS_CLOSE(fd);
	      status = FALSE;
	    }
	}
    }
  return (status);
}

static void _tcpascii_Close (ilu_Transport self)
{
  if (self == NULL)
    return;

  OS_CLOSE (self->tr_fd);
  return;
}

static ilu_boolean _tcpascii_ReadMessage (ilu_Transport self,
				     ilu_bytes *buffer,
				     ilu_cardinal *size)
{
  ilu_byte lenbuf[15];
  ilu_cardinal i, bytesToRead, bytesRead, totalSize = 0;
  ilu_bytes packet = NULL;
  ilu_boolean status = FALSE;
  ilu_boolean lastPacket;

  if (self == NULL)
    return(FALSE);

  do {
    if ((i = _ilu_Read (self->tr_fd, lenbuf, 14)) < 14)
      {
	DEBUG((INCOMING_DEBUG | TCP_DEBUG),
	      (stderr,
	       "%s %d returns %d (errno is %s) %s.\n",
	       "_tcpascii_ReadMessage:  read on fd",
	       self->tr_fd, i, ANSI_STRERROR(errno),
	       "while reading packet length"));
	status = FALSE;
      }
    else
      {
	sscanf (lenbuf, "%u ", &bytesToRead);
	bytesRead = 0;
	if (packet == NULL)
	  packet = (ilu_bytes) malloc(bytesToRead);
	else
	  packet = (ilu_bytes) realloc (packet, bytesToRead + totalSize);
	status = TRUE;
	i = _ilu_Read (self->tr_fd, packet + totalSize + bytesRead,
		       bytesToRead);
	if ((i < 0) || ((i == 0) && (bytesToRead > 0)))
	  {
	    DEBUG((INCOMING_DEBUG | TCP_DEBUG),
		  (stderr, "%s %d returns %d (%s) %s %u, %s %u bytes.\n",
		   "_tcpascii_ReadMessage:  read on fd",
		   self->tr_fd, i, ANSI_STRERROR(errno),
		   "while reading packet of length", bytesToRead,
		   "having already read", bytesRead));
	    status = FALSE;
	  }
	else
	  {
	    DEBUG((INCOMING_DEBUG | TCP_DEBUG),
		  (stderr, "%s %d bytes from fd %d %s %u bytes.\n",
		   "_tcpascii_ReadMessage:  read", i, self->tr_fd,
		   "while asking for", bytesToRead));
	    bytesRead += i;
	    bytesToRead -= i;
	  }
	totalSize += bytesRead;
      }
  } while (status AND (NOT lastPacket));

  if (status)
    {
      *buffer = packet;
      *size = totalSize;
    }
  else
    {
      if (packet != NULL)
	free(packet);
    }

  if (status AND (_ilu_DebugLevel & PACKET_DEBUG) != 0)
    DumpPacket(*buffer, *size);

  return (status);
}

/***********************************************************************
/***********************************************************************
/***********************************************************************
/**** Now the methods for the TCP Mooring ******************************
/***********************************************************************
/***********************************************************************
/**********************************************************************/

static ilu_string _tcpascii_FormHandle (TCPParms parms)
{
  char buf[1000];

  if (parms == NULL)
    return (NULL);
  sprintf (buf, "%stcpascii_%s_%u", (TCP_BUFFER(parms) == NULL) ? "" : "b",
	   TCP_HOSTNAME(parms), TCP_PORT(parms));
  return (_ilu_Strdup(buf));
}

static ilu_Transport _tcpascii_AcceptClient (ilu_Mooring self)
{
  register ilu_integer ns;
  struct sockaddr_in sin;
  ilu_Transport new = NULL;
  long addrlen;
  ilu_TransportClass _ilu_tcpascii_TransportClass(void);
  int one = 1;
  char addrDescr[64];

  _ilu_AutoSetDebugLevel();

  if (self == NULL)
    return (NULL);

  DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	(stderr, "_tcpascii_AcceptClient: mooring file descriptor is %d\n",
	 self->tr_fd));

  addrlen = sizeof(sin);

  if ((ns = OS_ACCEPT(self->tr_fd, &sin, &addrlen)) < 0)
    {
      perror ("_tcpascii_AcceptClient:  accept");
    }
  else
    {
      FmtSockaddrInet(&sin, addrDescr);
      DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	    (stderr,
	     "_tcpascii_AcceptClient: new connection, on fd %d, from %s.\n",
	     ns, addrDescr));

      new = (ilu_Transport) malloc (sizeof(struct _ilu_Transport_s));
      new->tr_class = _ilu_tcpascii_TransportClass();
      new->tr_fd = ns;
      new->tr_data = self->tr_data;
      if (ioctl(ns, FIONBIO, &one) != 0) {
        DEBUG((CONNECTION_DEBUG | TCP_DEBUG),
	      (stderr,
	       "%s (descriptor %d, from %s) non-blocking.\n",
	       "_tcpascii_AcceptClient:  Failed to set socket",
	       ns, addrDescr));
        }
    }
  return (new);
}

static ilu_Mooring _ilu_tcpascii_CreateMooring (TCPParms parms)
{
  struct sockaddr_in sin;
  struct linger linger;
  ilu_integer skt, namelen;
  ilu_integer on = 1;
  ilu_Mooring self;
  
  _ilu_AutoSetDebugLevel();
  if (!SigPIPEHandler)
    _tcpascii_HandleSigPIPE();

  /* setup the new server */
  
  if ((skt = OS_SOCKET(AF_INET, SOCK_STREAM, 0)) < 0) {
    DEBUG((EXPORT_DEBUG | TCP_DEBUG),
	  (stderr, "_tcpascii_CreateMooring: socket failed:  %s.\n",
	   ANSI_STRERROR(errno)));
    return (NULL);
  }
    
  OS_SETSOCKOPT(skt, SOL_SOCKET, SO_REUSEADDR, &on, sizeof(on));
  linger.l_onoff = 0;
  OS_SETSOCKOPT(skt, SOL_SOCKET, SO_LINGER,
		&linger, sizeof(struct linger));

  sin.sin_family = AF_INET;
  sin.sin_port = TCP_PORT(parms);
  sin.sin_addr.s_addr = INADDR_ANY;	/* me, me! */

  if (OS_BIND(skt, &sin, sizeof(sin)) < 0) {
    DEBUG((EXPORT_DEBUG | TCP_DEBUG),
	  (stderr, "_tcpascii_CreateMooring: bind to port %d failed:  %s.\n",
	   TCP_PORT(parms), ANSI_STRERROR(errno)));
    OS_CLOSE (skt);
    return (NULL);
  }

  /* If servicePort was 0 then discover kernel allocated port */
  
  if (sin.sin_port == 0) {
    namelen = sizeof(sin);
    if (OS_GETSOCKNAME(skt, (struct sockaddr *) &sin, &namelen) < 0)
      {
	DEBUG((EXPORT_DEBUG | TCP_DEBUG),
	      (stderr, "_tcpascii_CreateMooring: getsockname failed:  %s.\n",
	       ANSI_STRERROR(errno)));
	OS_CLOSE (skt);
	return (NULL);
      }
  }

  if (OS_LISTEN(skt, 4) < 0) {
    DEBUG((EXPORT_DEBUG | TCP_DEBUG),
	  (stderr,
	   "_tcpascii_CreateMooring:  listen on port %d failed:  %s.\n",
	   sin.sin_port, ANSI_STRERROR(errno)));
    OS_CLOSE (skt);
    return (NULL);
  }

  self = (ilu_Mooring) malloc (sizeof(struct _ilu_Mooring_s));

  self->mo_fd = skt;
  self->mo_transportClass = _ilu_tcpascii_TransportClass();
  self->mo_accept_connection = _tcpascii_AcceptClient;
  TCP_PORT(parms) = sin.sin_port;
  if (TCP_HOSTNAME(parms) != NULL)
    free(TCP_HOSTNAME(parms));
  TCP_HOSTNAME(parms) = _ilu_Strdup(_ilu_tcpascii_CurrentHostInetName());
  self->mo_data = (ilu_private) parms;

  return (self);
}

ilu_TransportClass _ilu_tcpascii_TransportClass (void)
{
  static ilu_TransportClass m = NULL;
  if (m == NULL)
    {
      m = (ilu_TransportClass)
	  malloc(sizeof(struct _ilu_TransportClass_s));
      m->type = ilu_TransportType_TCP;
      m->tc_interpret_info = _tcpascii_InterpretInfo;
      m->tc_form_info = (ilu_string (*)(ilu_refany)) _tcpascii_FormHandle;
      m->tc_connect = _tcpascii_Connect;
      m->tc_close = _tcpascii_Close;
      m->tr_data_present_p = _tcpascii_DataPresent;
      m->tc_wait_for_input = _tcpascii_WaitForInput;
      m->tc_send_message = _tcpascii_SendMessage;
      m->tc_read_message = _tcpascii_ReadMessage;
      m->tc_flush_output = _tcpascii_FlushOutput;
      m->tc_create_mooring = (ilu_Mooring (*)(ilu_private))
			  _ilu_tcpascii_CreateMooring;
    }
  return (m);
}

/***********************************************************************
/***********************************************************************
/***********************************************************************
/**** TCP utilities ****************************************************
/***********************************************************************
/***********************************************************************
/**********************************************************************/

ilu_string _ilu_tcpascii_CurrentHostInetName(void)
{
  char name[100];
  static ilu_string inetname = NULL;
  struct hostent *he;
  ilu_string p;

  if (inetname == NULL)
    {
      if (OS_GETHOSTNAME(name,sizeof(name)) != 0) {
	perror("no hostname for this machine!");
	return (NULL);
      }
  
      he = OS_GETHOSTBYNAME(name);
      p = (ilu_string ) inet_ntoa (he->h_addr_list[0]);
      inetname = _ilu_Strdup (p);
    }
  return (inetname);
}
