/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: spp.c,v 1.12 1994/02/15 20:33:29 janssen Exp $ */

#define _POSIX_SOURCE

#include "ilu.h"
#include "iluntrnl.h"

#include "transprt.h"
#include "mooring.h"

#include <sys/types.h>
#include <sys/param.h>		/* for MAXHOSTNAMELEN */
#include <assert.h>
#include <sys/ioctl.h>
#include <sys/fcntl.h>

#include <tiuser.h>

extern ilu_string sys_errlist[];

#include "bffrlist.h"

typedef struct xnssppparms {
  ilu_byte addr[12];
  bufferList buffer;
  ilu_boolean buffered;		/* buffer is needed to determine when
				   to set EOM mark */
} *XNSSPPParms;

#define XNSSPP_ADDR(a) (((XNSSPPParms)(a->tr_data))->addr)
#define XNSSPP_BUFFER(a) (((XNSSPPParms)(a->tr_data))->buffer)
#define XNSSPP_BUFFERED(a) (((XNSSPPParms)(a->tr_data))->buffered)

#define QUEUE_LENGTH	4	/* number of simultaneous connect requests allowed */

#define SPP_INIT_SIZE	4096	/* initial buffer size when reading a message */
#define SPP_INC_SIZE	2048	/* size of buffer increment when reading a message */

ilu_string xns_addr_to_string (buf, xns_addr)
     ilu_string buf;
     ilu_byte xns_addr[12];
{
  ilu_string b;

  if ((b = buf) == NULL)
    b = (ilu_string ) malloc(36);
  sprintf (b, "%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x",
	   xns_addr[0], xns_addr[1], xns_addr[2], xns_addr[3],
	   xns_addr[4], xns_addr[5], xns_addr[6], xns_addr[7],
	   xns_addr[8], xns_addr[9], xns_addr[10], xns_addr[11]);
  return (b);
}

ilu_byte *xns_string_to_addr (buf, xns_addr)
     ilu_string buf;
     ilu_byte *xns_addr;
{
  ilu_byte *b;

  if ((b = xns_addr) == NULL)
    b = (ilu_byte *) malloc(12);
  if (sscanf (buf, "%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x.%2x",
	      &b[0], &b[1], &b[2], &b[3], &b[4], &b[5],
	      &b[6], &b[7], &b[8], &b[9], &b[10], &b[11]) == 12)
    return (b);
  else if (b != xns_addr)
    free(b);
  return (NULL);    
}

static ilu_private CreateParms (addr, addrinfo, buffered)
     ilu_byte addr[12];
     ilu_string addrinfo;
     ilu_boolean buffered;
{
  struct xnssppparms *new = (struct xnssppparms *)
			    malloc(sizeof(struct xnssppparms));
  if (addr != NULL)
    strncpy (new->addr, addr, sizeof(new->addr));
  else if (addrinfo != NULL)
    xns_string_to_addr(addrinfo, new->addr);
  new->buffered = buffered;
  new->buffer = (bufferList) malloc(sizeof(struct bufferlist_s));
  new->buffer->size = 0;
  new->buffer->head = NULL;
  new->buffer->tail = NULL;
  return ((ilu_private) new);
}

/***********************************************************************
/***********************************************************************
/***********************************************************************
/**** First, the methods for the XNSSPP Transport **********************
/***********************************************************************
/***********************************************************************
/**********************************************************************/

static ilu_private _xnsspp_InterpretInfo (ilu_string info)
{
  char addr[100];
  ilu_boolean buffered = FALSE;

  if (*info == 'b')
    buffered = TRUE;
  if (sscanf (info + (buffered ? 1 : 0), "xnsspp_%s", addr) == 1)
    return (CreateParms(NULL, addr, buffered));
  else
    return (NULL);
}

static ilu_integer _xnsspp_NDataPresent (ilu_Transport conn)
{
  long count;
  ilu_boolean status = ioctl (conn->tr_fd, FIONREAD, &count);
  return ( (status != 0) ? -1 : (count > 0) ? 1 : 0 );
}

static ilu_boolean _xnsspp_WaitForInput(ilu_Transport t,
					ilu_boolean *sure,
					ilu_FineTime *limit)
{
  _ilu_WaitForInputOnFD(t->tr_fd, sure, limit);
  return (TRUE);
}


static ilu_boolean WriteMsg (ilu_integer fd, ilu_byte *packet,
			     ilu_cardinal size, ilu_boolean EOM)
{
  /* assume Write Lock held */
  ilu_integer i;

  DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
	(stderr,
	 "(spp.c:WriteMsg):  writing %d bytes from 0x%x to fd %d.\n",
	 size, packet, fd));
  if ((i = t_snd (fd, packet, size, EOM)) < size)
    {
      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
	    (stderr, "(spp.c:WriteMsg):  error on t_snd on fd %d:  %s.\n",
	     fd,
	     (i < 0) ? sys_errlist[errno]
		     : ((i == 0) ? "Lost connection"
				 : "not enough bytes written")));
      return (FALSE);
    }
  return (TRUE);  
}

static ilu_boolean _xnsspp_FlushOutput (ilu_Transport self)
{
  bufferList buf;
  bufferElement new, old;

  if (self == NULL)
    return (FALSE);

  if ((buf = XNSSPP_BUFFER(self)) != NULL)
    {
      for (new = buf->head; buf->size > 0 && new != NULL; buf->size -= 1)
	{
	  if (!WriteMsg (self->tr_fd, new->packet, new->size, new->EOM))
	    {
	      char buf[100];
	      xns_addr_to_string (buf, XNSSPP_ADDR(self));
	      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		    (stderr,
		     "%s 0x%x, size %u, to fd %u to XNS addr %s.\n",
		     "_xnsspp_FlushOutput:  couldn't write packet",
		     new->packet, new->size, self->tr_fd, buf));
	    }
	  old = new;
	  new = new->next;
	  free(old->packet);
	  free(old);
	}
      buf->size = 0;
      buf->head = NULL;
      buf->tail = NULL;
    };
  return (TRUE);
}

static ilu_boolean _xnsspp_AddPacketToBuffer(ilu_Transport self,
				ilu_byte *packet, ilu_cardinal size,
				ilu_boolean eom)
{
  bufferList buf = XNSSPP_BUFFER(self);
  bufferElement new;
  ilu_boolean status = FALSE;

  if (self == NULL)
    return (FALSE);

  if ((buf = XNSSPP_BUFFER(self)) != NULL)
    {
      new = (bufferElement) malloc(sizeof(struct bufferlist_element_s));

      new->packet = packet;
      new->size = size;
      new->next = NULL;
      new->EOM = eom;
      if (buf->tail == NULL)
	buf->head = buf->tail = new;
      else
	buf->tail->next = new;
      buf->size += 1;
      status = TRUE;
    }
  return;
}

static ilu_boolean _xnsspp_Connect (ilu_Transport self)
{
  ilu_integer fd;
  ilu_Transport *new;
  char xns_addr[12];
  char buf[100];
  struct t_bind tbind;
  struct t_call sndcall, rcvcall;
  ilu_boolean status = FALSE;

  _ilu_AutoSetDebugLevel();

  if (self == NULL)
    return (FALSE);

  DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
	(stderr, "_xnsspp_Connect:  connecting to addr %s...\n",
	 (xns_addr_to_string(buf, XNSSPP_ADDR(self)),
	  buf)));

  if ((fd = t_open("/dev/xs", O_RDWR, NULL)) < 0)
    {
      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
	    (stderr, "_xnsspp_Connect:  t_open call failed:  %s.\n",
	     sys_errlist[errno]));
    }
  else
    {
      tbind.addr.buf = (ilu_string ) XNSSPP_ADDR(self);
      tbind.addr.len = sizeof(XNSSPP_ADDR(self));
      tbind.addr.maxlen = sizeof(XNSSPP_ADDR(self));
      tbind.qlen = 0;

      if (t_bind(fd, NULL, &tbind) < 0)
	{
	  DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		(stderr, "_xnsspp_Connect:  t_bind call failed:  %s.\n",
		 sys_errlist[errno]));
	  t_close(fd);
	}
      else
	{
	  sndcall.addr.buf = (ilu_string ) XNSSPP_ADDR(self);
	  sndcall.addr.len = sizeof(XNSSPP_ADDR(self));
	  sndcall.opt.buf = NULL;
	  sndcall.opt.len = 0;
	  sndcall.udata.buf = NULL;
	  sndcall.udata.len = 0;

	  if (t_connect(fd, &sndcall, &rcvcall) < 0)
	    {
	      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		    (stderr,
		     "_xnsspp_Connect:  t_connect call failed:  %s.\n",
		     sys_errlist[errno]));
	      t_close(fd);
	    }
	  else
	    {
	      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		    (stderr, "_xnsspp_Connect:  connected.\n"));
	      self->tr_fd = fd;
	      status = TRUE;
	    }
	}
    }
  return (status);
}

static void _xnsspp_Close (ilu_Transport self)
{
  if (self != NULL)
    {
      char buf[100];
      xns_addr_to_string(buf, XNSSPP_ADDR(self));
      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
	    (stderr, "_xnsspp_Close:  closing connection to %s, fd %u.\n",
	     buf, self->tr_fd));
      _xnsspp_FlushOutput (self);
      if (t_close (self->tr_fd) < 0)
	{
	  DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		(stderr, "_xnsspp_Close:  error on t_close:  %s.\n",
		 sys_errlist[errno]));
	}
      self->tr_fd = -1;
    }
}

static ilu_boolean _xnsspp_SendMessage (conn, buffer, count)
     ilu_Transport conn;
     ilu_bytes buffer;
     ilu_cardinal count;
{
  ilu_boolean status = FALSE;

  if (conn != NULL && buffer != NULL && count > 0)
    {
      if (XNSSPP_BUFFERED(conn)
	  && !_xnsspp_AddPacketToBuffer(conn, buffer, count, TRUE) )
	status = TRUE;
      else if (WriteMsg (conn->tr_fd, buffer, count, TRUE))
	status = TRUE;
    }
  return (status);
}

static ilu_boolean _xnsspp_ReadMessage (self, packet, size)
     ilu_Transport self;
     ilu_bytes *packet;
     ilu_cardinal *size;
{
  ilu_bytes new;
  long newsize, count, newused;
  ilu_integer flags = 0;
  ilu_boolean status = FALSE;

  if (self != NULL)
    {
      new = (ilu_bytes) malloc(newsize = SPP_INIT_SIZE);
      newused = 0;
      do {
	if (newsize - newused < SPP_INC_SIZE)
	  new = (ilu_bytes) realloc(new,
				    newsize = newsize + SPP_INC_SIZE);
	if ((count = t_rcv(self, new+newused, newsize-newused, &flags))
	    < 0)
	  {
	    DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		  (stderr,
		   "_xnsspp_ReadMessage:  t_rcv call failed:  %s.\n",
		   sys_errlist[errno]));
	    free(new);
	    new = NULL;
	    newused = 0;
	    break;
	  }
	else
	  newused += count;
      } while (flags & T_MORE);
      *packet = new;
      *size = newused;
      status = (new != NULL);
    }
  return (status);
}

/***********************************************************************
/***********************************************************************
/***********************************************************************
/**** Now the methods for the XNSSPP Mooring ******************************
/***********************************************************************
/***********************************************************************
/**********************************************************************/

static ilu_string _xnsspp_FormHandle (ilu_refany parameterblock)
{
  XNSSPPParms parms = (XNSSPPParms) parameterblock;
  char buf[1000];
  char addrs[100];

  if (parms == NULL)
    return (NULL);
  xns_addr_to_string (addrs, parms->addr);
  sprintf (buf, "%sxnsspp_%s", parms->buffered ? "b" : "", addrs);
  return (_ilu_Strdup(buf));
}

static ilu_Transport _xnsspp_AcceptClient (ilu_Mooring self)
{
  ilu_Transport new = NULL;
  struct t_call tcall;
  struct t_bind tbind;
  ilu_byte xns_addr[12];
  ilu_integer newfd;
  XNSSPPParms parms = (XNSSPPParms) mooring_data(self);

  ilu_TransportClass _ilu_xnsspp_TransportClass();

  _ilu_AutoSetDebugLevel();

  if (self != NULL)
    {
      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
	    (stderr,
	     "_xnsspp_AcceptClient: service file descriptor is %u\n",
	     mooring_file_descriptor(self)));

      tcall.addr.buf = (ilu_string ) xns_addr;
      tcall.addr.maxlen = sizeof(xns_addr);
      tcall.opt.buf = NULL;
      tcall.opt.len = 0;
      tcall.udata.buf = NULL;
      tcall.udata.len = 0;

      if (t_listen(mooring_file_descriptor(self), &tcall) != 0)
	{
	  DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		(stderr, "_xnsspp_AcceptClient: error on t_listen:  %s\n",
		 sys_errlist[errno]));
	}
      else
	{
	  char buf[100];
	  xns_addr_to_string(buf, xns_addr);
	  DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		(stderr,
		 "_xnsspp_AcceptClient: new connection request from %s\n",
		 buf));

	  if ((newfd = t_open("/dev/xs", O_RDWR, NULL)) < 0)
	    DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
		  (stderr, "_xnsspp_Connect:  t_open call failed:  %s.\n",
		   sys_errlist[errno]));
	  else
	    {
	      tbind.addr.buf = (ilu_string ) xns_addr;
	      tbind.addr.len = 0;
	      tbind.addr.maxlen = sizeof(xns_addr);
	      tbind.qlen = 0;

	      if (t_bind(newfd, NULL, &tbind) < 0)
		{
		  DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
			(stderr,
			 "%s  %s.\n",
			 "_xnsspp_AcceptClient:  t_bind call failed:",
			 sys_errlist[errno]));
		  t_close(newfd);
		}
	      else
		{
		  if (0 != t_accept(mooring_file_descriptor(self), newfd,
				    &tcall))
		    {
		      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
			    (stderr, "%s:  %s\n",
			     "_xnsspp_AcceptClient: error on t_accept",
			     sys_errlist[errno]));
		      t_close(newfd);
		    }
		  else
		    {
		      DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
			    (stderr, "%s on fd %d\n",
			     "_xnsspp_AcceptClient: request accepted",
			     newfd));

		      new = (ilu_Transport)
			    malloc(sizeof(struct _ilu_Transport_s));
		      new->tr_class = _ilu_xnsspp_TransportClass();
		      new->tr_fd = newfd;
		      new->tr_data = CreateParms(tcall.addr.buf, NULL,
				parms->buffered);
		    }
		}
	    }
	}
    }
  return (new);
}

static ilu_Mooring _xnsspp_CreateMooring (ilu_private mooringInfo)
{
  XNSSPPParms parms = (XNSSPPParms) mooringInfo;
  ilu_Mooring self;
  ilu_integer fd;
  ilu_byte xns_addr[12];
  struct t_bind tbind, retval;
  struct t_call tcall;

  _ilu_AutoSetDebugLevel();

  if ((fd = t_open("/dev/xs", O_RDONLY | O_NDELAY, NULL)) < 0) {
    DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
	  (stderr, "_xnsspp_Connect:  t_open call failed:  %s.\n",
	   sys_errlist[errno]));
    return (FALSE);
  };

  tbind.addr.buf = NULL;
  tbind.addr.len = 0;
  tbind.addr.maxlen = 0;
  tbind.qlen = QUEUE_LENGTH;

  retval.addr.buf = (ilu_string ) xns_addr;
  retval.addr.maxlen = sizeof(xns_addr);

  if (t_bind(fd, &tbind, &retval) < 0) {
    DEBUG((CONNECTION_DEBUG | XNSSPP_DEBUG),
	  (stderr, "_xnsspp_Connect:  t_bind call failed:  %s.\n",
	   sys_errlist[errno]));
    t_close(fd);
    return (FALSE);
  };

  self = (ilu_Mooring) malloc (sizeof(struct _ilu_Mooring_s));

  self->mo_fd = fd;
  self->mo_transportClass = _ilu_xnsspp_TransportClass();
  self->mo_accept_connection = _xnsspp_AcceptClient;
  strncpy (parms->addr, retval.addr.buf, retval.addr.len);
  self->mo_data = (ilu_private) parms;

  return (self);
}

ilu_TransportClass _ilu_xnsspp_TransportClass ()
{
  static ilu_TransportClass m = NULL;
  if (m == NULL)
    {
      m = (ilu_TransportClass)
	  malloc(sizeof(struct _ilu_TransportClass_s));
      m->type = ilu_TransportType_XNSSPP;
      m->tc_interpret_info = _xnsspp_InterpretInfo;
      m->tc_form_info = _xnsspp_FormHandle;
      m->tc_connect = _xnsspp_Connect;
      m->tc_close = _xnsspp_Close;
      m->tc_n_data_present = _xnsspp_NDataPresent;
      m->tc_wait_for_input = _xnsspp_WaitForInput;
      m->tc_send_message = _xnsspp_SendMessage;
      m->tc_read_message = _xnsspp_ReadMessage;
      m->tc_flush_output = _xnsspp_FlushOutput;
      m->tc_create_mooring = _xnsspp_CreateMooring;
    }
  return (m);
}
