/*
Copyright (c) 1991, 1992, 1993 Xerox Corporation.  All Rights Reserved.  

Unlimited use, reproduction, and distribution of this software is
permitted.  Any copy of this software must include both the above
copyright notice of Xerox Corporation and this paragraph.  Any
distribution of this software must comply with all applicable United
States export control laws.  This software is made available AS IS,
and XEROX CORPORATION DISCLAIMS ALL WARRANTIES, EXPRESS OR IMPLIED,
INCLUDING WITHOUT LIMITATION THE IMPLIED WARRANTIES OF MERCHANTABILITY
AND FITNESS FOR A PARTICULAR PURPOSE, AND NOTWITHSTANDING ANY OTHER
PROVISION CONTAINED HEREIN, ANY LIABILITY FOR DAMAGES RESULTING FROM
THE SOFTWARE OR ITS USE IS EXPRESSLY DISCLAIMED, WHETHER ARISING IN
CONTRACT, TORT (INCLUDING NEGLIGENCE) OR STRICT LIABILITY, EVEN IF
XEROX CORPORATION IS ADVISED OF THE POSSIBILITY OF SUCH DAMAGES.
*/
/* $Id: mainloop.c,v 1.25 1994/03/19 00:52:51 spreitze Exp $ */
/* Last tweaked by Mike Spreitzer March 17, 1994 10:59 pm PST */

#define _BSD_SOURCE 1	/* tell gcc that we're using BSD select() */

#include "ilu.h"
#include "iluntrnl.h"

#include <unistd.h>	/* for read() and write() */
#include <stdio.h>	/* I/O defs (including popen and pclose) */
#include <string.h>	/* for memset(), used by FD_ZERO() */
#include <sys/types.h>
#include <sys/time.h>
#include <errno.h>
#include <sys/errno.h>
#include <math.h>

/* ================ UNIX ilu_FineTime ================ */
/*L1, L2, Main unconstrained*/

ilu_cardinal ilu_FineTimeRate = 1000000;

ilu_FineTime ilu_FineTime_Now()
{
  struct timeval tv;
  ilu_FineTime ans;
  ASSERT(gettimeofday(&tv, NULL) == 0, buf,
	 (buf, "UNIX time.c:gettimeofday failed, errno=%d", errno));
  ans.ft_s = tv.tv_sec;
  ans.ft_t = tv.tv_usec;
  return ans;
}

ilu_FineTime ilu_FineTime_Add(ilu_FineTime a, ilu_FineTime b)
{
  ilu_FineTime c;
  c.ft_s = a.ft_s + b.ft_s;
  c.ft_t = a.ft_t + b.ft_t;
  if (c.ft_t >= 1000000) {
      c.ft_t -= 1000000;
      c.ft_s += 1;
    }
  return c;
}

ilu_FineTime ilu_FineTime_Sub(ilu_FineTime a, ilu_FineTime b)
{
  ilu_FineTime c;
  c.ft_s = a.ft_s - b.ft_s - 1;
  c.ft_t = a.ft_t + 1000000 - b.ft_t;
  if (c.ft_t >= 1000000) {
      c.ft_t -= 1000000;
      c.ft_s += 1;
    }
  return c;
}

ilu_FineTime ilu_FineTime_Mul(ilu_FineTime a, float b)
{
  double as  = a.ft_s;
  double at  = a.ft_t;
  double amd = as + (at/1.0E6);
  double amp = amd * b;
  return ilu_FineTime_FromDouble(amp);
}

ilu_FineTime ilu_FineTime_FromDouble(double seconds)
{
  double usecs = seconds * 1.0E6;
  double csd = floor(usecs/1.0E6);
  double ctd = usecs - (csd * 1.0E6);
  int icsd = csd;
  int ictd = ctd;
  ilu_FineTime c;
  c.ft_s = icsd;
  c.ft_t = ictd;
  _ilu_Assert(0 <= ictd && ictd < 1000000, "ilu_FineTime_FromDouble");
  return (c);
}

ilu_integer ilu_FineTime_Cmp(ilu_FineTime a, ilu_FineTime b)
{
   if (a.ft_s != b.ft_s)
        return (a.ft_s - b.ft_s);
   else return ( ((ilu_integer) a.ft_t) - ((ilu_integer) b.ft_t) );
}

ilu_cardinal ilu_rescale(ilu_cardinal n, ilu_cardinal dfrom,
					 ilu_cardinal dto)
{
  if (dfrom == dto)
      return n;
  else {
      double from = dfrom ? dfrom : (1.0 + (double) (ilu_cardinal) -1);
      double to   = dto   ? dto   : (1.0 + (double) (ilu_cardinal) -1);
      double ans = floor(to * n / from);
      int ians = ans;
      return (ians);
    }
}

/* ================ Main Loop ================ */

struct io_reg {
  /*L1, L2, Main unconstrained --- single-threaded*/
  int fd, input;
  ilu_private rock;
  /*For calling: Main Invariant holds; L2 otherwise unconstrained*/
  void (*proc)(int fd, ilu_private rock);
};

#define IOTABSZ		256

/*L1, L2, Main unconstrained*/

static struct io_reg IOTab[IOTABSZ];
static int nIdx = 0, lastIdx = 0;
static fd_set readfds, writefds, excnfds;

static ilu_MainLoop *theMainLoop = NULL;
static int mlPhase = 0;	/* !=0 => can't change theMainLoop */

void ilu_SetMainLoop(ilu_MainLoop *ml)
{
  theMainLoop = ml;
  _ilu_Assert(mlPhase == 0, "ilu_SetMainLoop");
  mlPhase = 1;
  _ilu_gcoAlarm = ilu_CreateAlarm();
  _ilu_gccAlarm = ilu_CreateAlarm();
  _ilu_ioTimeoutAlarm = ilu_CreateAlarm();
  _ilu_grAlarm = ilu_CreateAlarm();
}

ilu_boolean ilu_RegisterInputSource (int fd,
	void (*proc)(int fd, ilu_private rock),
	ilu_private rock)
{
  mlPhase = 1;
  if (theMainLoop != NULL)
    return ( (*theMainLoop->ml_register_input)(fd, proc, rock) );
  if (nIdx < IOTABSZ) {
      IOTab[nIdx]. fd  = fd;
      IOTab[nIdx].proc = proc;
      IOTab[nIdx].rock = rock;
      IOTab[nIdx].input = 1;
      nIdx++;
      return (ilu_TRUE);
    }
  return (ilu_FALSE);
}

ilu_boolean ilu_UnregisterInputSource (int fd)
{
  register int i;
  mlPhase = 1;
  if (theMainLoop != NULL)
    return ( (*theMainLoop->ml_unregister_input)(fd) );
  for (i = 0;  i < nIdx;  i += 1)
    if ((IOTab[i].fd == fd) && IOTab[i].input) {
	nIdx--;
	if (i != nIdx) {
	    IOTab[i]. fd   = IOTab[nIdx].fd;
	    IOTab[i].input = IOTab[nIdx].input;
	    IOTab[i].proc  = IOTab[nIdx].proc;
	    IOTab[i].rock  = IOTab[nIdx].rock;
	}
	return (ilu_TRUE);
    }
  return (ilu_FALSE);
}

ilu_boolean ilu_RegisterOutputSource (int fd,
	void (*proc)(int fd, ilu_private rock),
	ilu_private rock)
{
  mlPhase = 1;
  if (theMainLoop != NULL)
    return ( (*theMainLoop->ml_register_output)(fd, proc, rock) );
  if (nIdx < IOTABSZ) {
      IOTab[nIdx]. fd  = fd;
      IOTab[nIdx].proc = proc;
      IOTab[nIdx].rock = rock;
      IOTab[nIdx].input = 0;
      nIdx++;
      return (ilu_TRUE);
    }
  return (ilu_FALSE);
}

ilu_boolean ilu_UnregisterOutputSource (int fd)
{
  register int i;
  mlPhase = 1;
  if (theMainLoop != NULL)
    return ( (*theMainLoop->ml_unregister_output)(fd) );
  for (i = 0;  i < nIdx;  i += 1)
    if ((IOTab[i].fd == fd) && !IOTab[i].input) {
	nIdx--;
	if (i != nIdx) {
	    IOTab[i]. fd   = IOTab[nIdx].fd;
	    IOTab[i].input = IOTab[nIdx].input;
	    IOTab[i].proc  = IOTab[nIdx].proc;
	    IOTab[i].rock  = IOTab[nIdx].rock;
	}
	return (ilu_TRUE);
    }
  return (ilu_FALSE);
}

/* timu == ilu_daimu == mxamu */

typedef struct {
  /*L1 >= {daimu} for access*/
  
  ilu_Alarmette_s ae;
  /*for invoking: Main Invariant holds, L2 otherwise unconstrained*/
  void (*proc)(ilu_private rock);
  ilu_private rock;
} DefaultAlarm_s, *DefaultAlarm;

/*L1 = {daimu};
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu})*/
static void daInvoke(ilu_Alarmette a);

/*L1 >= {daimu}; L2, Main unconstrained*/

static void daUrset(ilu_FineTime t);
static void daUrcancel(void);

static ilu_FineTime alarmTime = {0, 0};
static ilu_boolean alarmSet = 0;
static ilu_Alarmette_s alarmHead = {&alarmHead, &alarmHead, FALSE, {0,0}};
static ilu_AlarmRep dar = {&alarmHead, daInvoke, daUrset, daUrcancel};

static DefaultAlarm_s gcoDefaultAlarm
	= {{NULL, NULL, FALSE, {0, 0}}, NULL, NULL};
static DefaultAlarm_s gccDefaultAlarm
	= {{NULL, NULL, FALSE, {0, 0}}, NULL, NULL};
static DefaultAlarm_s iotDefaultAlarm
	= {{NULL, NULL, FALSE, {0, 0}}, NULL, NULL};
static DefaultAlarm_s grDefaultAlarm
	= {{NULL, NULL, FALSE, {0, 0}}, NULL, NULL};

ilu_refany _ilu_gcoAlarm = &gcoDefaultAlarm;
ilu_refany _ilu_gccAlarm = &gccDefaultAlarm;
ilu_refany _ilu_ioTimeoutAlarm = &iotDefaultAlarm;
ilu_refany _ilu_grAlarm = &grDefaultAlarm;

/*L1 = {daimu};
  forall conn: (L2 >= {conn.iomu}) => (L2 >= {conn.callmu})*/
static void daInvoke(ilu_Alarmette a)
{
  DefaultAlarm da = (DefaultAlarm) a;
  _ilu_ReleaseMutex(ilu_daimu);
  (*da->proc)(da->rock);
  _ilu_AcquireMutex(ilu_daimu);
  return;
}

static void daUrset(ilu_FineTime t)
{
  alarmTime = t;
  alarmSet = TRUE;
}

static void daUrcancel(void)
{
  alarmSet = FALSE;
}

/*L1_sup < timu*/

ilu_refany ilu_CreateAlarm(void)
{
  mlPhase = 1;
  if (theMainLoop != NULL)
      return ( (*theMainLoop->ml_create_alarm)() );
  else {
      DefaultAlarm da = (DefaultAlarm) malloc(sizeof(DefaultAlarm_s));
      ilu_Alarmette_s ae = {NULL, NULL, FALSE, {0, 0}};
      da->ae = ae;
      da->proc = NULL;
      da->rock = NULL;
      return da;
    }
}

void ilu_SetAlarm(ilu_refany alarm, ilu_FineTime t,
		  /*for invoking: Main Invariant holds*/
		  void (*proc)(ilu_private rock),
		  ilu_private rock)
{
  mlPhase = 1;
  if (theMainLoop != NULL)
      (*theMainLoop->ml_set_alarm)(alarm, t, proc, rock);
  else {
      DefaultAlarm da = (DefaultAlarm) alarm;
      _ilu_AcquireMutex(ilu_daimu);
      da->proc = proc;
      da->rock = rock;
      ilu_MXASet(&dar, &da->ae, t);
      _ilu_ReleaseMutex(ilu_daimu);
      return;
    }
}

void ilu_UnsetAlarm(ilu_refany alarm)
{
  mlPhase = 1;
  if (theMainLoop != NULL)
      (*theMainLoop->ml_unset_alarm)(alarm);
  else {
      DefaultAlarm da = (DefaultAlarm) alarm;
      _ilu_AcquireMutex(ilu_daimu);
      ilu_MXAClear(&dar, &da->ae);
      _ilu_ReleaseMutex(ilu_daimu);
      return;
    }
}

/*Main Invariant holds; L2 otherwise unconstrained*/
void ilu_RunMainLoop(int *stop)
{
  mlPhase = 1;
  if (theMainLoop != NULL) {
      (*theMainLoop->ml_run)(stop);
      return;
    }
  /* else use the default impl: */

  *stop = 0;
  while (! *stop)
    { /* Find one thing to do, and do it. */
      struct timeval tv, *ptv;
      int width, stat;
      register int i;
      ilu_FineTime t;
      
      /* Does the default alarm impl need processing? */
      _ilu_AcquireMutex(ilu_daimu);
      if (alarmSet) {
          t = ilu_FineTime_Now();
          if (ilu_FineTime_Cmp(t, alarmTime) >= 0) {
              alarmSet = 0;
              ilu_MXAProc(t, &dar);
              _ilu_ReleaseMutex(ilu_daimu);
              goto next;	/* find next thing to do */
            }
        }
      else
	{
	  t.ft_s = 0;
	  t.ft_t = 0;
	}
      t = ilu_FineTime_Sub(alarmTime, t);
      ptv = (alarmSet) ? &tv : NULL;
      _ilu_ReleaseMutex(ilu_daimu);
      tv.tv_sec  = t.ft_s;
      tv.tv_usec = ilu_rescale(t.ft_t, ilu_FineTimeRate, 1000000);

      /* Try the results of the previous call on select */
      while (lastIdx > 0) {
	--lastIdx;
	if (   (FD_ISSET(IOTab[lastIdx].fd, &excnfds))
	    || (FD_ISSET(IOTab[lastIdx].fd,
			 (IOTab[lastIdx].input
			  ? &readfds : &writefds))))
	  {
	    (*IOTab[lastIdx].proc)(IOTab[lastIdx].fd,
				   IOTab[lastIdx].rock);
	    goto next;
	  }
      }

      /* Make a new call on select */
      width = 0;
      FD_ZERO(&readfds);
      FD_ZERO(&writefds);
      FD_ZERO(&excnfds);
      for (i = 0;  i < nIdx;  i++)
	{
	  FD_SET(IOTab[i].fd, (IOTab[i].input ? &readfds : &writefds));
	  FD_SET(IOTab[i].fd, &excnfds);
	  if (IOTab[i].fd >= width)
	    width = IOTab[i].fd + 1;
	}
      lastIdx = nIdx;

      stat = select (width, &readfds, &writefds, &excnfds, ptv);
      ASSERT(stat >= 0 || errno == EINTR, buf,
	     (buf, "UNIX mainloop.c:select failed, errno=%d=%s",
	      errno, ANSI_STRERROR(errno) ));
      if (stat < 0)
          lastIdx = 0;
     next:
      stat = 0;		/* harmless; useful for breakpoint setting */
    }
  return;
}

/*L1, L2 unconstrained*/

void ilu_ExitMainLoop(int *stop)
{
  mlPhase = 1;
  if (theMainLoop != NULL)
      (*theMainLoop->ml_exit)(stop);
  else
      *stop = 1;
  return;
}


/* ================ Other Concurrent IO Stuff ================ */


typedef struct wait_frame_s WaitFrame;
struct wait_frame_s {			/* A chained stack frame */
  /*L1, L2, Main unconstrained --- single-threaded*/

  ilu_Alarmette_s wake;		/* for timing out */
  WaitFrame *fd_next;		/* next in chain */
  WaitFrame *hotter, *cooler;	/* stack is doubly linked */
  int fd;			/* half of chain key */
  int input;			/* half of chain key */
  int stop;			/* the stacked value */
  ilu_boolean sure;		/* was this innermost? */
};

/*L1, L2, Main unconstrained --- single-threaded*/

static WaitFrame *wfs = NULL;	/* A chain of stacks */

static void TakeTimeout(ilu_private rock);

static void TAInvoke(ilu_Alarmette a)
{
  WaitFrame *wf = (WaitFrame *) a;
  while (wf->cooler != NULL)
      wf = wf->cooler;
  for (wf = wf; wf != NULL; wf = wf->hotter) {
      wf->sure = 0;
      ilu_ExitMainLoop(&(wf->stop));
    }
  return;
}

static void TASet(ilu_FineTime t)
{
  ilu_SetAlarm(_ilu_ioTimeoutAlarm, t, TakeTimeout, NULL);
}

static void TACancel(void)
{
  ilu_UnsetAlarm(_ilu_ioTimeoutAlarm);
}

static ilu_Alarmette_s timeHead = {&timeHead, &timeHead, FALSE, {0, 0}};
static ilu_AlarmRep timeAlarm = {&timeHead, TAInvoke, TASet, TACancel};

static void TakeTimeout(ilu_private rock)
{
  ilu_MXAProc(ilu_FineTime_Now(), &timeAlarm);
}

static void FoundFD(int fd, ilu_private rock)
{
  WaitFrame *wf = (WaitFrame *) rock; /* coolest frame in the stack */
  _ilu_Assert(wf->cooler == NULL, "mainloop.c:FoundFD");
  for (wf = wf; wf != NULL; wf = wf->hotter) {
      wf->sure = (wf->hotter == NULL) && (wf->stop == 0);
      ilu_ExitMainLoop(&(wf->stop));
    }
  return;
}

static ilu_WaitTech *nsWT = NULL;
static int wtPhase = 0;

void ilu_SetWaitTech(ilu_WaitTech *wt)
{
  nsWT = wt;
  _ilu_Assert(wtPhase == 0, "SetWaitTech");
  wtPhase = 1;
}

/*Main Invariant holds; L2 otherwise unconstrained*/

static void IOWait(int fd, int input, ilu_boolean *sure,
		   ilu_FineTime *limit)
{
  WaitFrame this, **pp;
  ilu_boolean bottom;
  this.hotter = NULL;
  for (pp = &wfs; (*pp) != NULL; pp = &((*pp)->fd_next) ) {
      if ( ((*pp)->fd == fd) && ((*pp)->input == input) ) {
          _ilu_Assert((*pp)->hotter == NULL,
		      "mainloop.c:IOWait (*pp)->hotter != NULL");
          this.cooler = *pp;
          (*pp)->hotter = &this;
          this.fd_next = (*pp)->fd_next;
          *pp = &this;
          bottom = FALSE;
          goto redy;
        }
    }
  this.cooler = NULL;
  this.fd_next = wfs;
  wfs = &this;
  if (input)
       ilu_RegisterInputSource (fd, FoundFD, (ilu_private) &this);
  else ilu_RegisterOutputSource(fd, FoundFD, (ilu_private) &this);
  bottom = TRUE;
redy:
  this.fd = fd;
  this.input = input;
  if (limit != NULL)
      ilu_MXASet(&timeAlarm, &this.wake, *limit);
  ilu_RunMainLoop(&this.stop);
  *sure = this.sure;
  if (limit != NULL)
      ilu_MXAClear(&timeAlarm, &this.wake);
  if ( bottom ) {
      _ilu_Assert(wfs == &this, "IOWait: pop new");
      (input ? ilu_UnregisterInputSource:ilu_UnregisterOutputSource)(fd);
      wfs = this.fd_next;
    }
  else {
      _ilu_Assert(this.cooler != NULL, "IOWait: this.cooler == NULL");
      _ilu_Assert(this.fd_next == this.cooler->fd_next,
		  "IOWait: pop old");
      *pp = this.cooler;
      (*pp)->hotter = NULL;
    }
  return;
}

void _ilu_WaitForInputOnFD(int fd, ilu_boolean *sure,
			   ilu_FineTime *limit)
{
  wtPhase = 1;
  if (nsWT != NULL) {
      (*nsWT->wt_read_wait)(fd, sure, limit);
    }
  else
      IOWait(fd, 1, sure, limit);
  return;
}

void _ilu_WaitForOutputOnFD(int fd, ilu_boolean *sure,
			    ilu_FineTime *limit)
{
  wtPhase = 1;
  if (nsWT != NULL) {
      (*nsWT->wt_write_wait)(fd, sure, limit);
      *sure = TRUE;
    }
  else
      IOWait(fd, 0, sure, limit);
  return;
}

/*Main Invariant holds*/
/*L2 >= {fd's connection's callmu, iomu}*/

int _ilu_Read(int fd, unsigned char *buf, int nbytes)
{
  int now, sofar=0;
  ilu_boolean sure;
  while (sofar < nbytes) {
      _ilu_WaitForInputOnFD(fd, &sure, NULL);
      now = read(fd, buf+sofar, nbytes-sofar);
      if (now < 0 && errno == EAGAIN)	/* would have blocked -- so why did WaitForInputOnFD return? */
	continue;
      if (now == 0) return sofar;
      if (now < 0) return now;
      sofar += now;
    }
  return sofar;
}

int _ilu_Write(int fd, unsigned char *buf, int nbytes)
{
  int now, sofar=0;
  ilu_boolean sure;
  while (sofar < nbytes) {
      _ilu_WaitForOutputOnFD(fd, &sure, NULL);
      now = write(fd, buf+sofar, nbytes-sofar);
      _ilu_Assert(now != 0 || !sure, "mainloop.c:_ilu_Write");
      if (now < 0) return now;
      sofar += now;
    }
  return sofar;
}

