;;; -*- Mode: Lisp -*-

;;; priority-queues.lisp --
;;; Priority Queues as Binary Heaps in the CLR style (Cormen,
;;; Leiserson and Rivest, "Introduction to Algorithms", ppgg.
;;; 140--152, MIT Press).
;;; Array implementation.
;;;
;;; Author: Marco Antoniotti
;;; Address: Robotics Laboratory
;;;          Courant Institute of Mathematical Science
;;;          New York University
;;;          New York, NY, 10012
;;;
;;; Copyright (c) 1992. All rights reserved.
;;;
;;; Version: 1.0 gamma
;;;
;;; Tested in CMU CL 16c.


;;;============================================================================
;;; General License Agreement and Lack of Warranty
;;;
;;; This software is distributed in the hope that it will be useful (both
;;; in and of itself and as an example of lisp programming), but WITHOUT
;;; ANY WARRANTY. The author(s) do not accept responsibility to anyone for
;;; the consequences of using it or for whether it serves any particular
;;; purpose or works at all. No warranty is made about the software or its
;;; performance. 
;;; 
;;; Use and copying of this software and the preparation of derivative
;;; works based on this software are permitted, so long as the following
;;; conditions are met:
;;; 	o  The copyright notice and this entire notice are included intact
;;; 	   and prominently carried on all copies and supporting documentation.
;;; 	o  No fees or compensation are charged for use, copies, or
;;; 	   access to this software. You may charge a nominal
;;; 	   distribution fee for the physical act of transferring a
;;; 	   copy, but you may not charge for the program itself. 
;;; 	o  If you modify this software, you must cause the modified
;;; 	   file(s) to carry prominent notices (a Change Log)
;;; 	   describing the changes, who made the changes, and the date
;;; 	   of those changes.
;;; 	o  Any work distributed or published that in whole or in part
;;; 	   contains or is a derivative of this software or any part 
;;; 	   thereof is subject to the terms of this agreement. The 
;;; 	   aggregation of another unrelated program with this software
;;; 	   or its derivative on a volume of storage or distribution
;;; 	   medium does not bring the other program under the scope
;;; 	   of these terms.
;;; 	o  Permission is granted to manufacturers and distributors of
;;; 	   lisp compilers and interpreters to include this software
;;; 	   with their distribution. 
;;; 
;;; This software is made available AS IS, and is distributed without 
;;; warranty of any kind, either expressed or implied.
;;; 
;;; In no event will the author(s) or their institutions be liable to you
;;; for damages, including lost profits, lost monies, or other special,
;;; incidental or consequential damages arising out of or in connection
;;; with the use or inability to use (including but not limited to loss of
;;; data or data being rendered inaccurate or losses sustained by third
;;; parties or a failure of the program to operate as documented) the 
;;; program, even if you have been advised of the possibility of such
;;; damanges, or for any claim by any other party, whether in an action of
;;; contract, negligence, or other tortious action.
;;;
;;;
;;; The current version of this software and a variety of related
;;; utilities may be obtained by anonymous ftp from ftp.cs.cmu.edu
;;; (128.2.206.173) or any other CS machine in the directory 
;;;       /afs/cs.cmu.edu/user/mkant/Public/Lisp/
;;; You must cd to this directory in one fell swoop, as the CMU
;;; security mechanisms prevent access to other directories from an
;;; anonymous ftp. For users accessing the directory via an anonymous
;;; ftp mail server, the file README contains a current listing and
;;; description of the files in the directory. The file UPDATES describes
;;; recent updates to the released versions of the software in the directory.
;;; The file COPYING contains the current copy of this license agreement.
;;; Of course, if your site runs the Andrew File System and you have
;;; afs access, you can just cd to the directory and copy the files directly.
;;; 
;;; If you wish to be added to the CL-Utilities@cs.cmu.edu mailing list, 
;;; send email to CL-Utilities-Request@cs.cmu.edu with your name, email
;;; address, and affiliation. This mailing list is primarily for
;;; notification about major updates, bug fixes, and additions to the lisp
;;; utilities collection. The mailing list is intended to have low traffic.


;;;============================================================================
;;; Documentation (initial)
;;;
;;; *STANDARD-HEAP-ALLOCATION-SIZE* 16                               [CONSTANT]
;;;    The standard allocation block size for the underlying priority queue
;;;    rep. 
;;;
;;; *STANDARD-EXTEND-SIZE* 16                                        [CONSTANT]
;;;    The standard size by which the underlying priority queue rep. is
;;;    augmented.
;;;
;;; PQ (heap key-fun comp-fun extend-size name modify-map           [STRUCTURE]
;;;     adjustable original-size)
;;;
;;; EMPTY-ERROR (pq)                                                [CONDITION]
;;;    Error signaled when an operation is tried on an empty priority queue
;;;
;;; UNEQUAL-KEYS (pq-name new-key element-key)                      [CONDITION]
;;;    Error signaled by 'modify-key' when the new key to be associated
;;;    to an element and the element key itself are unequal
;;;
;;; OVERFLOW (pq)                                                   [CONDITION]
;;;    Condition signaled by 'insert' when tried on a full and non
;;;    adjustable priority queue
;;;
;;; DUPLICATE-KEY (pq key pq-name)                                  [CONDITION]
;;;    Error signaled by 'insert' when the key is already present
;;;
;;; PQPRINT (pq strm k)                                              [FUNCTION]
;;;    Print function for the internal priority queue data structure.
;;;
;;; MAKE-PRIORITY-QUEUE (&key (key #'identity) (test #'>)            [FUNCTION]
;;;                      (size *standard-heap-allocation-size*)
;;;                      (extend-by *standard-extend-size*)
;;;                      (name "") (element-type t)
;;;                      (key-type 'number) (support-modify t)
;;;                      (adjustable nil))
;;;    Creates a priority queue internal structure.
;;;    Arguments:
;;;    &key
;;;    :key  : the function used for access the 'key' of an element
;;;           (default #'identity)
;;;    :test : the comparison function (default #'>)
;;;    :size : the initial size of the heap (default 16)
;;;    :extend-by : the resize value (default 16)
;;;    :name : a name for the priority queue (default "")
;;;    :element-type : the type of the elements in the queue (default t)
;;;    :key-type  : the type of the 'key' used in comparison (default
;;;                 number)
;;;    :support-modify : whether the priority queue supports the
;;;                      'modify-key' operation (default t)
;;;    :adjustable: controls whether care must be taken to shrink the heap
;;;                 upon 'extract-head' (default nil)
;;;
;;; ELEMENT-TYPE (pq)                                                [FUNCTION]
;;;
;;; PRIORITY-QUEUE-ALLOCATION (pq)                                   [FUNCTION]
;;;
;;; PARENT (i)                                                       [FUNCTION]
;;;
;;; RIGHT (i)                                                        [FUNCTION]
;;;
;;; LEFT (i)                                                         [FUNCTION]
;;;
;;; SHRINK-PHYSICAL-HEAP (new-heap-size phys-heap pq)                [FUNCTION]
;;;
;;; PPRINT-HEAP (pq)                                                 [FUNCTION]
;;;
;;; EMPTY-P (pq)                                                     [FUNCTION]
;;;
;;; SIZE (pq)                                                        [FUNCTION]
;;;
;;; HEAD (pq)                                                        [FUNCTION]
;;;    Returns the 'head' of the priority queue.
;;;
;;; INSERT (elem pq &aux (true-heap (pq-heap pq)) (keyfun            [FUNCTION]
;;;         (pq-key-fun pq)) (compfun (pq-comp-fun pq)) (modmap
;;;         (pq-modify-map pq)))
;;;    Inserts an element in the priority queue
;;;    Arguments:
;;;    elem : the element to be inserted
;;;    pq   : the priority queue
;;;
;;; EXTRACT-HEAD (pq &aux (true-heap (pq-heap pq)) (keyfun           [FUNCTION]
;;;               (pq-key-fun pq)) (modmap (pq-modify-map pq)))
;;;    Removes the head element from the priority queue and readjusts its
;;;    contents. Arguments:
;;;    pq : the priority queue
;;;
;;; DELETE-BY-KEY (elem pq)                                          [FUNCTION]
;;;    Deletes the element by accessing its key and 'position' in the
;;;    priority queue. The priority queue is returned after restructuring.
;;;    Arguments:
;;;    elem : the element to be removed (must have a field accessible by the
;;;           key function associated with the priority queue).
;;;    pq   : the priority queue.
;;;
;;; HEAPIFY (pq index)                                               [FUNCTION]
;;;
;;; MODIFY-KEY (elem old-key-value new-key-value pq)                 [FUNCTION]
;;;    Modifies the key of an element by giving it a new value and
;;;    readjusts the priority queue. The new key value must be compatible
;;;    with respect to the comparison function associated with the priority
;;;    queue. Arguments:
;;;    elem : the element with the new key value
;;;    old-key-value : the old value (it must be passed for consistency and
;;;                   efficiency reasons)
;;;    new-key-value : the new value for the element key
;;;    pq : the priority queue.


;;;============================================================================
;;; History:
;;; 12.28.1992: interface modified
;;; 12.23.1992: check for duplicated keys introduced
;;; 12.23.1992: completed with support for 'modify-key'
;;; 12.21.1992: released.


;;;============================================================================
;;; Notes:
;;; 12.23.1992: Refine conditions and restarts. (CMUCL has not conditions
;;;             integrated with CLOS yet).
;;; 12.23.1992: still missing: good condition treatment and modify-key
;;;             with no need for 'old-key-value' (this is really low
;;;             priority).
;;; 12.21.1992: modify-key requires keeping information around. A hash
;;;             table should do it, but this requires rewriting insert
;;;             and heapify.


;;;============================================================================
;;; Bugs:
;;; None known (what a hubris :))
;;;
;;; Send bug reports, notes and suggestions to the author
;;;
;;;	marcoxa@cs.nyu.edu


;;;============================================================================
;;; Prologue

(eval-when #+CLtL2 (:compile-toplevel :load-toplevel)
	   #-CLtL2 (compile load)
  (unless (find-package "PRIORITY-QUEUES")
    (load "priority-queues-package")))

(in-package "PRIORITY-QUEUES")


;;;============================================================================
;;; Global definitions

;;; version
;;; => string
;;; => integer
;;; => integer
;;; => string

(defun version ()
  (values "PRIORITY QUEUES: version 1.0 gamma"
	  1
	  0
	  "gamma"
	  ))
;;; version


;;; +standard-heap-allocation-size+ +standard-extend-size+ --

(defconstant +standard-heap-allocation-size+ 16
  "The standard allocation block size for the underlying priority queue rep.")

(defconstant +standard-extend-size+ 16
  "The standard size by which the underlying priority queue rep. is
augmented.")
;;; +standard-heap-allocation-size+ +standard-extend-size+ --


;;; pq structure --
;;;
;;; Notes:
;;; 12.23.1992: add CLtL2 style constructor and remove make-priority-queue.

(defstruct (pq (:print-function pqprint))
  (heap (make-array +standard-heap-allocation-size+ :fill-pointer 0)
	:type (array t *))
  (key-fun #'identity)
  (comp-fun #'>)
  (extend-size +standard-extend-size+	:type integer)
  (name nil		:type symbol)
  (modify-map nil	:type (or null hash-table))
  (adjustable nil)
  (original-size +standard-heap-allocation-size+
		 :type integer)
  )
;;; pq --


;;; empty-error condition --

(define-condition empty-error (simple-error)
  (pq)

  (:report
   (lambda (cnd strm)
     (format strm
	     ">> PRIORITY QUEUES: empty priority queue~:[ ~A.~;.~]"
	     (string= (pq-name (empty-error-pq cnd)) "")
	     (pq-name (empty-error-pq cnd)))))
  (:documentation
   "Error signaled when an operation is tried on an empty priority queue")
  )
;;; empty-error --


;;; unequal-keys --

(define-condition unequal-keys (simple-error)
  (pq-name new-key element-key)
  (:documentation
   "Error signaled by 'modify-key' when the new key to be associated
to an element and the element key itself are unequal"))
;;; unequal-keys --


;;; overflow --

(define-condition overflow (storage-condition)
  (pq)
  (:report
   (lambda (cnd strm)
     (format strm
	     ">> PRIORITY QUEUES: overflow~:[ in priority queue ~A.~;.~]"
	     (string= (pq-name (overflow-pq cnd)) "")
	     (pq-name (overflow-pq cnd)))))
  (:documentation
   "Condition signaled by 'insert' when tried on a full and non adjustable
priority queue"))
;;; overflow --


;;; duplicate-key --

(define-condition duplicate-key (simple-error)
  (pq key pq-name)
  (:report
   (lambda (cnd strm)
     (format strm
	     ">> PRIORITY QUEUES: Key ~S already used~
              ~:[ in priority queue ~A.~;.~]"
	     (duplicate-key-key cnd)
	     (string= (duplicate-key-pq-name cnd) "")
	     (duplicate-key-pq-name cnd)
	     )))
  (:documentation
   "Error signaled by 'insert' when the key is already present"))
;;; duplicate-key --


;;;============================================================================
;;; Functions

;;;----------------------------------------------------------------------------
;;; Constructor and type related functions.

;;; pqprint pq stream integer --

(defun pqprint (pq strm k)
  "Print function for the internal priority queue data structure."
  (print-unreadable-object
   (pq strm :type nil :identity t)
   (format strm
	   "PRIORITY QUEUE ~:[~A with ~
            size ~D, element type ~S, modifiable keys ~S~;~
            ~A~]:"
	   (and (not (null *print-level*)) (>= k *print-level*))
	   (pq-name pq)
	   (fill-pointer (pq-heap pq))
	   (array-element-type (pq-heap pq))
	   (not (null (pq-modify-map pq))))))
;;; pqprint --


;;; make-priority-queue --
;;; => pq

(defun make-priority-queue (&key (key #'identity)
				 (test #'>)
				 (size +standard-heap-allocation-size+)
				 (extend-by +standard-extend-size+)
				 (name "")
				 (element-type t)
				 (key-type 'number)
				 (support-modify t)
				 (adjustable nil)
				 )
  "Creates a priority queue internal structure.
Arguments:
&key
:key  : the function used for access the 'key' of an element
        (default #'identity)
:test : the comparison function (default #'>)
:size : the initial size of the heap (default 16)
:extend-by : the resize value (default 16)
:name : a name for the priority queue (default \"\")
:element-type : the type of the elements in the queue (default t)
:key-type  : the type of the 'key' used in comparison (default number)
:support-modify : whether the priority queue supports the 'modify-key'
                  operation (default t)
:adjustable: controls whether care must be taken to shrink the heap
             upon 'extract-head' (default nil)"

  (declare (ignore key-type))
  ;; Ignored for the time being.
 
  (let ((pq (make-pq :key-fun key
		     :comp-fun test
		     :extend-size extend-by
		     :name name
		     :heap (make-array size
				       :element-type element-type
				       :adjustable adjustable
				       :fill-pointer 0)
		     :adjustable adjustable
		     :original-size size
		     )))
    (when support-modify
      (setf (pq-modify-map pq) (make-hash-table :test #'equal :size size)))
    pq))
;;; make-priority-queue --


;;; make-heap ... => pq
;;; An alias for 'make-priority-queue'

(setf (documentation 'make-heap 'function)
      (documentation 'make-priority-queue 'function)
      (symbol-function 'make-heap) #'make-priority-queue)

;;; make-heap --


;;; priority-queue-p pq => (member t nil)
;;; heap-p pq => (or t nil)
;;; Predicates for type checking (aliases for pq-p).

(setf (symbol-function 'priority-queue-p) #'pq-p)

(setf (symbol-function 'heap-p) #'pq-p)
;;; priority-queue-p pq, heap-p --


;;; element-type pq => Type Specifier

(defun element-type (pq)
  (array-element-type (pq-heap pq)))
;;; element-type --


;;; adjustable-priority-queue-p pq
;;; adjustable-heap-p pq
;;; priority-queue-name pq
;;; heap-name pq
;;; Aliases for heap inspecting functions.

(setf (symbol-function 'adjustable-priority-queue-p) #'pq-adjustable)

(setf (symbol-function 'adjustable-heap-p) #'pq-adjustable)

(setf (symbol-function 'priority-queue-name) #'pq-name)

(setf (symbol-function 'heap-name) #'pq-name)
;;; adjustable-priority-queue-p, adjustable-heap-p, priority-queue-name
;;; heap-name


;;; priority-queue-allocation pq => integer
;;; The allocation for a non extensible heap is always 1+ the maximum
;;; heap 'size', since I am not using the first element of the
;;; underlying array.

(defun priority-queue-allocation (pq)
  (first (array-dimensions (pq-heap pq))))

(setf (symbol-function 'heap-allocation) #'priority-queue-allocation)
;;; priority-queue-allocation --


;;;----------------------------------------------------------------------------
;;; Auxiliary functions

;;; left right parent -- Auxiliary functions for the array implementation.

(declaim (inline left right parent shrink-physical-heap))

(defun parent (i)
  (declare (integer i))
  (floor (/ i 2)))

(defun right (i)
  (declare (integer i))
  (1+ (* 2 i)))

(defun left (i)
  (declare (integer i))
  (* 2 i))
;;; left right parent --


;;; shrink-physical-heap integer (array t *) pq => (array t *)
;;; "Physically" shrink the size of the heap if allowed, but
;;; never below the originally specified size.

(defun shrink-physical-heap (new-heap-size phys-heap pq)
  (declare (integer new-heap-size))
  (declare (type (array t *) phys-heap))
  ;; (assert (and (eq phys-heap (pq-heap pq))
  ;;              (= new-heap-size
  ;;                 (fill-pointer phys-heap))))
  (when (and (pq-adjustable pq)
	     (> new-heap-size (pq-original-size pq))
	     (<= new-heap-size
		 (floor (/ (first (array-dimensions phys-heap)) 4))))
    (adjust-array phys-heap
		  (floor (/ (first (array-dimensions phys-heap)) 2)))
    )
  phys-heap)
;;; shrink-physical-heap --


;;; pprint-heap pq => nil
;;; This function is here only for debugging purposes. It should be removed.

(defun pprint-heap (pq)
  (format *standard-output* "[")
  (if (empty-p pq)
      (format *standard-output* "]~%")
    (dotimes (i
	      (1+ (fill-pointer (pq-heap pq)))
	      (format *standard-output* "]~%"))
      (format *standard-output* "~d " (aref (pq-heap pq) i)))
    ))
;;; pprint-heap --


;;;----------------------------------------------------------------------------
;;; External interface functions.

;;; empty-p pq => (member t nil)
;;; size pq => (integer 0 *)

(defun empty-p (pq) (< (fill-pointer (pq-heap pq)) 1))

(defun size (pq) (fill-pointer (pq-heap pq)))
;;; empty-p size --


;;; head pq => t
;;; This is actually the first time I use the restart feature. Hope it is OK.

(defun head (pq)
  "Returns the 'head' of the priority queue."
  (restart-case
   (if (empty-p pq)
       (error 'empty-error :pq pq)
       (aref (pq-heap pq) 1))
   (use-value (pq-h)
	      :report
	      (lambda (strm)
		(format strm "Specify a value to be returned (it will not be~
                              evaluated)."))
	      :interactive
	      (lambda ()
		(format *query-io* "Value: ")
		(list (read *query-io*)))
	      pq-h)
   ))
;;; head --


;;; insert t pq => t
;;; For simplicity it is better not to use the first element of the array.
;;; See CLR.

(defun insert (elem pq
		    &aux
		    (true-heap (pq-heap pq))
		    (keyfun (pq-key-fun pq))
		    (compfun (pq-comp-fun pq))
		    (modmap (pq-modify-map pq))
		    )
  "Inserts an element in the priority queue
Arguments:
elem : the element to be inserted
pq : the priority queue"

  ;; Check for duplicate keys
  (when (and modmap (gethash (funcall keyfun elem) modmap))
    (cerror "Remove key and old element from priority queue."
	    'duplicate-key
	    :pq pq
	    :key (funcall keyfun elem)
	    :pq-name (pq-name pq))
    (delete-by-key elem pq)
    )

  ;; Actually increment the size of the heap via 'incf'.
  (let ((fp (incf (fill-pointer true-heap))))
    (declare (integer fp))
    
    ;; Extend the heap if allowed.
    (when (= fp (first (array-dimensions true-heap)))
      (if (pq-adjustable pq)
	  ;; I did not use '(adjustable-array-p true-heap)' because I
	  ;; want more control on the adjustability of the heap. I
	  ;; believe my implementeation is consistent.
	  (adjust-array true-heap (+ fp (pq-extend-size pq)))
	  (progn
	    ;; fix fill-pointer, hence heap size
	    (decf (fill-pointer true-heap))
	    (error 'overflow :pq pq))))
    (do ((i fp (parent i)))
	((or (<= i 1)
	     (funcall compfun
		      (funcall keyfun (aref true-heap (parent i)))
		      (funcall keyfun elem)))
	 (setf (aref true-heap i) elem)
	 ;; Update the modification map if needed.
	 (when modmap (setf (gethash (funcall keyfun elem) modmap) i))
	 elem
	 )
      (declare (integer i))
      (setf (aref true-heap i)
	    (aref true-heap (parent i)))
      ;; Update the modification map if needed.
      (when modmap
	(setf (gethash (funcall keyfun (aref true-heap i)) modmap) i))
      )))
;;; insert --


;;; extract-head pq => t

(defun extract-head (pq &aux
			(true-heap (pq-heap pq))
			(keyfun (pq-key-fun pq))
			(modmap (pq-modify-map pq)))
  "Removes the head element from the priority queue and readjusts its contents.
Arguments:
pq : the priority queue"
  
  (if (empty-p pq)
      (error 'empty-error :pq pq)
      (let ((result (head pq)))
	(setf (aref true-heap 1)
	      (aref true-heap (fill-pointer true-heap)))
	;; Update the entry in the modify map for the element just
	;; swapped and remove the entry for the element being extracted.
	(when modmap
	  (setf (gethash (funcall keyfun (aref true-heap 1))
			 modmap)
		1)
	  (remhash (funcall keyfun result) modmap))
	;; Shrink the size of the heap.
	(shrink-physical-heap (decf (fill-pointer true-heap))
			      true-heap
			      pq)

	(heapify pq 1)
	result)))
;;; extract-head --


;;; delete-by-key elem pq => pq

(defun delete-by-key (elem pq)
  "Deletes the element by accessing its key and 'position' in the
priority queue. The priority queue is returned after restructuring.
Arguments:
elem : the element to be removed (must have a field accessible by the
       key function associated with the priority queue).
pq   : the priority queue."
  (if (null (pq-modify-map pq))
      (error ">> PRIORITY QUEUE ~A: 'modify-key' and 'delete-by-key' ~
                 not supported."
	     (pq-name pq))
      (let* ((true-heap (pq-heap pq))
	     ;; (compfun (pq-comp-fun pq))
	     (keyfun (pq-key-fun pq))
	     (modmap (pq-modify-map pq))
	     (elem-heap-pos (gethash (funcall keyfun elem) modmap))
	     )
	(declare (integer elem-heap-pos))
	(declare (type (array t *) true-heap))

	(setf (aref true-heap elem-heap-pos)
	      (aref true-heap (fill-pointer true-heap)))

	;; Update modify map and remove entry for deleted element from
	;; modify map.
	(setf (gethash (funcall keyfun (aref true-heap elem-heap-pos))
		       modmap)
	      elem-heap-pos)
	(remhash (funcall keyfun elem) modmap)

	;; Shrink the heap if allowed. The call to 'decf' is the
	;; "logical" shrinking of the heap.
	(shrink-physical-heap (decf (fill-pointer true-heap))
			      true-heap
			      pq)
	
	(heapify pq elem-heap-pos)
	pq)))
;;; delete-by-key --


;;; heapify pq index => nil
;;; Readjusts the heap property (see CLR).

(defun heapify (pq index)
  (declare (integer index))
  (let ((true-heap (pq-heap pq))
	(keyfun (pq-key-fun pq))
	(compfun (pq-comp-fun pq))
	(modmap (pq-modify-map pq))
	)
    ;; It might be more efficient. It is with CMULISP 16c.
    (labels
	((do-heapify (index)
	   (let ((l (left index))
		 (r (right index))
		 (largest 0)		; 0 is very convenient!
		 )
	     (declare (integer l r largest))

	     ;; Figure out which "son" is the largest -- 'largest'
	     ;; will contain its index in the array/heap.
	     (if (and (<= l (fill-pointer true-heap))
		      (funcall compfun
			       (funcall keyfun (aref true-heap l))
			       (funcall keyfun (aref true-heap index))))
		 (setq largest l)
		 (setq largest index))
	     (when (and (<= r (fill-pointer true-heap))
			(funcall compfun
				 (funcall keyfun (aref true-heap r))
				 (funcall keyfun (aref true-heap largest))
				 ))
	       (setq largest r))

	     ;; Exchange elements if necessary -- 'rotatef' does this.
	     ;; and update the info in the modify map
	     (when (/= largest index)
	       (rotatef (aref true-heap index) (aref true-heap largest))
	       (when modmap
		 (setf (gethash (funcall keyfun (aref true-heap index))
				modmap)
		       index
		       (gethash (funcall keyfun (aref true-heap largest))
				modmap)
		       largest))
	       (do-heapify largest)
	       )))			; do-heapify
	 )
      (do-heapify index))))
;;; heapify --


;;; modify-key elem old-key-value new-key-value pq
;;; => elem

(defun modify-key (elem old-key-value new-key-value pq)
  "Modifies the key of an element by giving it a new value and
readjusts the priority queue. The new key value must be compatible
with respect to the comparison function associated with the priority queue.
Arguments:
elem : the element with the new key value
old-key-value : the old value (it must be passed for consistency and
                efficiency reasons)
new-key-value : the new value for the element key
pq : the priority queue."
  (cond ((null (pq-modify-map pq))
	 (error ">> PRIORITY QUEUE ~A: 'modify-key' not supported."
		(pq-name pq)))
	((not (equal (funcall (pq-key-fun pq) elem) new-key-value))
	 (error 'unequal-keys
		:pq-name (pq-name pq)
		:new-key new-key-value
		:element-key (funcall (pq-key-fun pq) elem)))
	(t (let ((true-heap (pq-heap pq))
		 (compfun (pq-comp-fun pq))
		 (keyfun (pq-key-fun pq))
		 (modmap (pq-modify-map pq))
		 )

	     ;; From here on it is pretty much as insert. (I shouild
	     ;; have implemented the whole thing with the 'sift-up',
	     ;; 'sift-down' procedures).
	     (if (funcall compfun new-key-value old-key-value)
		 (do ((i (gethash old-key-value modmap) (parent i)))
		     ((or (<= i 1)
			  (funcall compfun
				   (funcall keyfun (aref true-heap (parent i)))
				   new-key-value))
		      (setf (aref true-heap i) elem
			    (gethash new-key-value modmap) i)
		      (remhash old-key-value modmap)
		      elem
		      )
		   (declare (integer i))
		   (setf (aref true-heap i)
			 (aref true-heap (parent i)))

		   (setf (gethash (funcall keyfun (aref true-heap i))
				  modmap)
			 i)
		   )			; end do --
		 (aref true-heap (gethash old-key-value modmap))
		 ))
	   )))
;;; modify-key --


;;; end of file -- priority-queues.lisp --
