
;;;
;;; Code for generating random csp's.
;;; assume you have n variables, where each variable has
;;; a descrete domain of m values (1 to m). The probability of
;;; a constraint existing between a pair of variables is p1
;;; and if there is a constraint between i and j, then p2
;;; is the probability that a pair of values is in conflict.
;;;
;;; therefore we can define csp as csp(n,m,p1,p2)
;;; See the code below !!
;;;

(in-package :user)

(defvar *random-out* nil)
(defvar *csp-id* nil)
(defvar *p1* nil)
(defvar *p2* nil)
(defvar *conflicts* 0)

(proclaim '(optimize (compilation-speed 0) (safety 1) (speed 3)))

(defun consistent? (conflicts)
  #'(lambda (x y) (null (member (list x y) conflicts :test #'equal))))
;;;
;;; we can create a constraint function between 2 variables by
;;; specifying the conflicting pairs of values. For example
;;; assume that X and Y have a domain of the 1st 5 integers '(1 2 3 4 5)
;;; and the conflicts are when x=1 and y=3, and x=4 and y=2. We can
;;; synthetise this function f as follows
;;; (setq f (consistent? '((1 3) (4 2))))
;;; We can then check a constraint as follows
;;; (funcall f 2 5) which should be true
;;;


(defun make-random-constraint (p1 p2 vi vj m)
  (cond ((>= p1 (/ (random 101) 100.00)) ;constraint exists
	 (let ((conflicts-ij nil)
	       (conflicts-ji nil))
	   (loop for xi from 1 to m do
		 (loop for xj from 1 to m do
		       (cond ((>= p2 (/ (random 101) 100.00)) ;conflict if vi=xi and vj=xj
			      (setq *conflicts* (+ 2 *conflicts*))
			      (push (list xi xj) conflicts-ij)
			      (push (list xj xi) conflicts-ji)))))
	   (make-constraint vi (consistent? conflicts-ij) vj)
	   (make-constraint vj (consistent? conflicts-ji) vi)))))
;;;
;;; Given the name of variable i (vi) and the name of variable j (vj)
;;; where vi and vj have domains 1...m
;;; A constraint exists between vi and vj with probability p1
;;; and a conflict exists between instantiations with probability p2
;;;

(defun make-random-csp (n m p1 p2)
  (map nil #'delete-object *vars*)
  (map nil #'delete-object *constraints*)
  (setq *p1* 0 *p2* 0 *conflicts* 0)
  (let ((d nil))
    (loop for i from 1 to n do
	  (setq d nil)
	  (loop for j from m downto 1 do
		(push j d))
	  (make-var i d))
    (loop for i from 1 to (- n 1) do
	  (loop for j from (+ i 1) to n do
		(make-random-constraint p1 p2 i j m))))
  (init)
  (reset)
  (setq *p1* (/ (round (* 100 (/ (length *constraints*) (* *n* (- *n* 1) 1.0)))) 100.0))
  (cond ((> (length *constraints*) 0)
	 (setq *p2* (/ (round (* 100 (/ *conflicts* (* 1.0 m m (length *constraints*))))) 100.0)))))

(defun save-random-constraint (p1 p2 vi vj m f)
  (cond ((>= p1 (/ (random 101) 100.00)) ;constraint exists
	 (let ((conflicts-ij nil)
	       (conflicts-ji nil))
	   (loop for xi from 1 to m do
		 (loop for xj from 1 to m do
		       (cond ((>= p2 (/ (random 101) 100.00)) ;conflict if vi=xi and vj=xj
				    (push (list xi xj) conflicts-ij)
				    (push (list xj xi) conflicts-ji)))))
	   (write (list vi conflicts-ij vj) :stream f)
	   (write (list vj conflicts-ji vi) :stream f)))))


(defun save-csp (csp-id n m p1 p2 f-out)
  (write (list 'start-of-csp csp-id) :stream f-out)
  (write (list n m p1 p2) :stream f-out)
  (loop for i from 1 to (- n 1)
	do (loop for j from (+ i 1) to n
		 do (save-random-constraint p1 p2 i j m f-out)))
  (write (list 'end-of-csp csp-id) :stream f-out))


(defun save-csps (quantity n m p1 p2 directory)
  (let* ((data-file (format nil "~A/CSP-~A-~A-~A-~A-~A.data" directory quantity n m p1 p2))
	 (f-out (open data-file :direction :output :if-exists :rename)))
    (loop for i from 1 to quantity do
	  (save-csp i n m p1 p2 f-out))
    (write (list 'end-of-csps) :stream f-out)
    (close f-out)))


(defun load-next-csp (f-in)
  (map nil #'delete-object *vars*)
  (map nil #'delete-object *constraints*)
  (let* ((datum (read f-in))
	 (n (first datum))
	 (m (second datum))
	 (d nil))
    (loop for i from 1 to n do
	  (setq d nil)
	  (loop for j from m downto 1 do
		(push j d))
	  (make-var i d))
    (setq datum (read f-in))
    (loop until (equal (first datum) 'end-of-csp) do
	  (make-constraint (first datum) (consistent? (second datum)) (third datum))
	  (setq datum (read f-in)))
    (init)
    (reset)))

(defun skip-csp (f-in)
  (let ((datum (read f-in)))
    (loop until (equal (first datum) 'end-of-csp)
	  do (setq datum (read f-in)))))
		       

(defun get-csp (id data-file)
  (let* ((f-in (open data-file :direction :input))
	 (datum (read f-in))
	 (found nil))
    (loop until (or (equal (first datum) 'end-of-csps)
		    (setq found (equal id (second datum)))) do
	  (skip-csp f-in)
	  (setq datum (read f-in)))
    (cond (found
	   (load-next-csp f-in)
	   (close f-in)
	   'created)
	  (t  (close f-in)
	      'not-found))))


(defun random-csp-tests (functions filename)
  (let* ((data-file (format nil "~A.data" filename))
	 (results-file (format nil "~A.results" filename))
	 (f-in (open data-file :direction :input))
	 (f-out (open results-file :direction :output :if-exists :rename))
	 (datum (read f-in))
	 (csp-id nil)
	 (start-f nil) (end-f nil) (cpu-time nil)
	 (status nil) (current-status nil)
	 (solution nil) (current-solution nil))
    (setq *trials* 0)
    (loop until (equal (first datum) 'end-of-csps) do
	  (setq csp-id (second datum))
	  (load-next-csp f-in)
	  (setq status nil)
	  (setq *bandwidth* (get-bandwidth)
		*width* (get-width)
		*induced-width* (get-induced-width))
	  (setq *trials* (+ 1 *trials*))
	  (loop for f in functions do
		(reset)
		(setq *f* f)
		(setq start-f (get-internal-run-time))
		(setq current-status (funcall f))
		(setq end-f (get-internal-run-time))
		(setq cpu-time (/ (* 1.0 (- end-f start-f)) internal-time-units-per-second))
		(cond ((null status)
		       (setq status current-status
			     solution (get-solution))))
		(setq current-solution (get-solution))
		(cond ((not (equal status current-status))
		       (print (list "Error: different status"  current-status *trials* *f* csp-id))))
		(cond ((and (equal current-status 'solution)
			    (not (equal-lists solution current-solution)))
		       (print (list "Error: different solution" *trials* *f* csp-id))))
		(write (list csp-id f checks nodes cpu-time
			     *bandwidth* *width* *induced-width* current-status) :stream f-out)
		(terpri f-out))
	  (setq datum (read f-in)))
    (close f-in)
    (close f-out)))


(defun random-csp-experiments (functions n m p1-s p2-s q filename)
  (let* ((results-file (format nil "~A.results" filename))
	 (f-out (open results-file :direction :output :if-exists :rename))
	 (start-f nil) (end-f nil) (cpu-time nil)
	 (status nil) (current-status nil)
	 (solution nil) (current-solution nil))
    (setq *random-out* f-out)
    (setq *trials* 0)
    (loop for p1 in p1-s do
	(loop for p2 in p2-s do
	      (loop for i from 1 to q do
		    (make-random-csp n m p1 p2)
		    (setq status nil
			  *bandwidth* (get-bandwidth)
			  *width* (get-width)
			  *induced-width* (get-induced-width)
			  *csp-id* (list i n m p1 p2
				       *p1* *p2*
				       *bandwidth* *width* *induced-width*))
		    (setq *trials* (+ 1 *trials*))
		    (loop for f in functions do
			  (reset)
			  (setq *f* f)
			  (setq start-f (get-internal-run-time))
			  (setq current-status (funcall f))
			  (setq end-f (get-internal-run-time))
			  (setq cpu-time (/ (* 1.0 (- end-f start-f)) internal-time-units-per-second))
			  (cond ((null status)
				 (setq status current-status
				       solution (get-solution))))
			  (setq current-solution (get-solution))
			  (cond ((not (equal status current-status))
				 (print (list "Error: different status"
					      current-status *trials* *f* *csp-id*)) (abort)))
			  (cond ((and (equal current-status 'solution)
				      (not (equal-lists solution current-solution)))
				 (print (list "Error: different solution" *trials* *f* *csp-id*))
				 (print solution) (print current-solution) (abort)))
			  (write (list *csp-id* f checks nodes cpu-time current-status) :stream f-out)
			  (terpri f-out)))))
    (write "end of data" :stream f-out)
    (close f-out)))

;;;
;;; Note: the random problems that are created above "may" be classified by 
;;; the parameters <n,m,p1,p2>. However, we should compute the actual values
;;; of p1 and p2 for the problems actually created (rather than the values
;;; of p1 and p2 passed to the creation algorithm). We can compute p1 as
;;; the number of constraints in G (length *constraints*) over the maximum
;;; allowable constraints in G (n(n-1)). We can also compute p2. We know that
;;; the number of pairs of instantiations between vi and vj is m*m. We can
;;; then sum the number of conflicts created, and divide this by the total number
;;; of possible pair-wise conflicts in G, namely m*m*(length *constraints*)
;;;
;;;
