;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: svd.lisp
;;;  Author: Chichilnisky
;;;  Description: singular-value-decomposition
;;;               Interface to Linpack SVD. Various SVD-based utilities.
;;;  Creation Date: August, 1992
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

(in-package 'obvius)
(export '(singular-value-decomposition svd singular-values with-static-svd with-svd
	  principal-components row-space row-null-space col-space col-null-space
	  singular-values left-singular-matrix right-singular-matrix
	  solve-eigenvalue matrix-inverse condition-number determinant
	  quadratic-decomposition))

(defmacro svd (&rest args)
  `(singular-value-decomposition  ,@args))

(defun left-singular-matrix (matrix)
  (multiple-value-bind (d u v) (svd matrix)
    (declare (ignore d v))
    u))

(defun right-singular-matrix (matrix)
  (multiple-value-bind (d u v) (svd matrix)
    (declare (ignore d u))
    (with-static-arrays ((vt (copy v)))
      (matrix-transpose vt :-> v))))

;;; Implemented with dummy q and r keywords to avoid breaking code
;;; that relies on teh qr decomposition.
;;; *** need a better way to handle SVD arguments, here and other places
;;; *** note that SVD ignores sign of determinant!!
(defmethod determinant ((matrix array) &key (ignore-zeros nil) q r u v
                        suppress-warning)
  (declare-matrices (matrix) nil)
  (when (and (or q r) (not suppress-warning))
    (warn "QR decomposition not used now, using SVD and ignoring q,r keywords. "))
  (let ((case (cond ((not (square-p matrix)) 0)
                    ((not ignore-zeros) (row-dim matrix))
                    (t t))))
    (case case
      (0 0.0)
      (1 (aref matrix 0 0))
      (2 (- (* (aref matrix 0 0) (aref matrix 1 1))
            (* (aref matrix 1 0) (aref matrix 0 1))))
      (3 (- (+ (* (aref matrix 0 0) (aref matrix 1 1) (aref matrix 2 2))
               (* (aref matrix 1 0) (aref matrix 2 1) (aref matrix 0 2))
               (* (aref matrix 2 0) (aref matrix 0 1) (aref matrix 1 2)))
            (+ (* (aref matrix 0 2) (aref matrix 1 1) (aref matrix 2 0))
               (* (aref matrix 1 2) (aref matrix 2 1) (aref matrix 0 0))
               (* (aref matrix 2 2) (aref matrix 0 1) (aref matrix 1 0)))))
      (t (unless suppress-warning
           (warn "Bug in determinant: absolute value of determinant returned"))
         (let* ((s (svd matrix :u u :v v))
                (val 1.0)
                elt)
           (dotimes (i (length s))
             (setq elt (aref s i))
             (when (or (> (abs elt) short-float-epsilon) (not ignore-zeros))
               (setq val (* val elt))))
           val)))))
#|TEST:
(determinant (make-matrix '((1 2) (2 1))))
(determinant (make-matrix '((1 2) (2 1))) :ignore-zeros t)
(determinant (make-matrix '((1 0) (0 1))))
(determinant (make-matrix '((-1 0) (0 1))))
(determinant (make-matrix '((1 0 0) (0 1 0))))
(determinant (make-matrix '((1 0) (0 0) (1 0))))
(determinant (randomize (make-array '(4 4) :element-type 'single-float) 1.0))
|#

(defmacro with-svd ((s u v) matrix . body)
  `(let* ((rows (row-dim ,matrix))
	  (cols (col-dim ,matrix)))
    (with-static-arrays ((,s (similar ,matrix :dimensions (min rows cols)))
			 (,u (similar ,matrix :dimensions (list rows rows)))
			 (,v (similar ,matrix :dimensions (list cols cols))))
      (svd ,matrix :s ,s :u ,u :v ,v)
      (unwind-protect (progn ,@body)))))

(defmacro with-static-svd ((s u v) matrix . body)
  `(let* ((rows (row-dim ,matrix))
	  (cols (col-dim ,matrix)))
    (with-static-arrays ((,s (similar ,matrix :static t :dimensions (min rows cols)))
			 (,u (similar ,matrix :static t :dimensions (list rows rows)))
			 (,v (similar ,matrix :static t :dimensions (list cols cols))))
      (svd ,matrix :s ,s :u ,u :v ,v)
      (unwind-protect (progn ,@body)))))

(defun singular-values (matrix)
  (with-svd (s u v) matrix
    (copy s :-> (similar matrix :dimensions (dimensions s)))))

(defun singular-value-decomposition (matrix &key u v s)
  
  (let ((rows (row-dim matrix))
	(cols (col-dim matrix))
	first-good-value)
    
    ;; Make space for the results
    (unless u (setq u (similar matrix :dimensions (list rows rows))))
    (unless v (setq v (similar matrix :dimensions (list cols cols))))
    (unless s (setq s (similar matrix :dimensions (min rows cols))))

    (checktype-matrices (u s v))
    (unless (and (= (row-dim u) (col-dim u) rows)
		 (= (row-dim v) (col-dim v) cols)
		 (= (length s) (min rows cols)))
      (error "Dimensions of result arrays passed to SVD are invalid"))
    
    (with-static-arrays ((u-t (allocate-array (reverse (dimensions u)) :element-type 'single-float))
			 (v-t (allocate-array (reverse (dimensions v)) :element-type 'single-float))
			 (matrix-t (allocate-array (list cols rows)  :element-type 'single-float))
			 (tmp-col (allocate-array rows :element-type 'single-float))
			 (tmp-row (allocate-array cols :element-type 'single-float))
			 (tmp-s (allocate-array (+ 1 (length s))  :element-type 'single-float)))
      
      (matrix-transpose matrix :-> matrix-t)
			  
      ;; Call the C routine with all the transposed tmp arrays.
      (setq first-good-value (internal-svd matrix-t rows cols tmp-s u-t v-t tmp-col tmp-row))
      (matrix-transpose u-t :-> u)
      (matrix-transpose v-t :-> v)
      (copy (vectorize tmp-s :size (length s)) :-> s)

      (if (zerop first-good-value)
	  (values s u v)
	  (warn "SVD only good beyond ~ath singular value, NIL returned" first-good-value)))))

#|
(defun singular-value-decomposition (matrix &key u v s)
  
  (let ((rows (row-dim matrix))
	(cols (col-dim matrix))
	first-good-value)
    
    ;; Setup and error-checking
    (unless u (setq u (similar matrix :dimensions (list rows rows))))
    (unless v (setq v (similar matrix :dimensions (list cols cols))))
    (unless s (setq s (similar matrix :dimensions (min rows cols))))

    (checktype-matrices (matrix u v s))
    (unless (and (= (row-dim u) (col-dim u) rows)
		 (= (row-dim v) (col-dim v) cols)
		 (= (length s) (min rows cols)))
      (error "Dimensions of working arrays passed to SVD are invalid"))
    
    ;; The real work
    (setq first-good-value (internal-svd matrix rows cols u s v))
    
    (if (zerop first-good-value)
	(values s u v)
	(warn "SVD only good beyond ~ath singular value, nil returned" first-good-value))))
|#
#|
;;; Test SVD

(setq A (make-array '(10 5) :element-type 'single-float))
(randomize A 1.0 :-> A)

(time (multiple-value-setq (W U V) (singular-value-decomposition A :u nil :v nil)))
(time (multiple-value-setq (W U V) (singular-value-decomposition A)))

;;; rows and cols of V and U should be orthogonal
(progn
  (assert (identity-p (matrix-transpose-mul V V)))
  (assert (identity-p (matrix-mul-transpose V V)))
  (assert (identity-p (matrix-transpose-mul U U)))
  (assert (identity-p (matrix-mul-transpose U U))))

;;; compare original array with decomposition
(setq Wmat (make-array (dimensions A) :element-type 'single-float :initial-element 0.0))
(loop for i from 0 below (x-dim W) do
      (setf (aref Wmat i i) (aref W i)))
(range (sub A (matrix-mul-transpose (matrix-mul U Wmat) V)))

;;; Test of standard svd bug
(setq A (make-array '(3 3) :element-type 'single-float
		    :initial-contents '((0.0 1.0 0.0)
					(0.0 1.0 1.0)
					(0.0 0.0 0.0))))
(multiple-value-setq (W U V) (singular-value-decomposition A))
(setq Wmat (make-array '(3 3) :element-type 'single-float :initial-element 0.0))
(loop for i from 0 below (array-dimension W 0) do
      (setf (aref Wmat i i) (aref W i)))
(range (sub A (matrix-mul-transpose (matrix-mul U Wmat) V)))


(setq A (make-array '(3 3) :element-type 'single-float
		    :initial-contents '((1.0 2.0 3.0)
					(4.0 5.0 6.0)
					(7.0 8.0 9.0))))
(multiple-value-setq (W U V) (singular-value-decomposition A))
(setq Wmat (make-array '(3 3) :element-type 'single-float :initial-element 0.0))
(loop for i from 0 below (array-dimension W 0) do
      (setf (aref Wmat i i) (aref W i)))
(range (sub A (matrix-mul-transpose (matrix-mul U Wmat) V)))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Utilities that make use of SVD:

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
#|
(defmethod row-null-space ((m array) &key ((:-> result)
					   (similar m :dimensions
						    (list (- (col-dim m) (row-dim m)) (col-dim m)))))
  (unless (> (col-dim m) (row-dim m))
    (error "Row null-space only defined for short, fat matrices"))
  (crop (right-singular-matrix m) :y (row-dim m) :y-dim (row-dim result) :-> result))

(defmethod row-space ((m array) &key ((:-> result) (similar m)))
  (crop (right-singular-matrix m) :y 0 :y-dim (row-dim result) :-> result))

(defmethod col-null-space ((m array) &key ((:-> result)
					   (similar m :dimensions
						    (list (row-dim m) (- (row-dim m) (col-dim m))))))
  (unless (> (row-dim m) (col-dim m))
    (error "Column null-space only defined for tall, thin matrices"))
  (crop (left-singular-matrix m) :x (col-dim m) :x-dim (col-dim result) :-> result))

(defmethod col-space ((m array) &key ((:-> result) (similar m)))
  (crop (left-singular-matrix m) :x 0 :x-dim (col-dim result) :-> result))

|#

(defmethod row-null-space ((m array) &key ((:-> result)) (singular-value-limit 0.0))
  (with-static-svd (s u v) m
   (let* ((test #'(lambda(x) (<= x singular-value-limit)))
	  (row-dim (or (position-if test s) (row-dim m)))
	  (row-null-dim (- (col-dim m) row-dim)))
     (unless (plusp row-null-dim) (error "Matrix has full row rank"))
     (unless result
	(setq result (similar m :dimensions (list row-null-dim (col-dim m)))))
     (with-static-arrays ((vt (copy v)))
       (matrix-transpose vt :-> v)
       (crop v :y row-dim :y-dim row-null-dim :-> result)))))

(defmethod row-space ((m array) &key ((:-> result)) (singular-value-limit 0.0))
  (with-static-svd (s u v) m
   (let* ((test #'(lambda(x) (<= x singular-value-limit)))
	  (row-dim (or (position-if test s) (row-dim m))))
     (unless (plusp row-dim) (error "Matrix is degenerate"))
     (unless result
       (setq result (similar m :dimensions (list row-dim (col-dim m)))))
     (with-static-arrays ((vt (copy v)))
       (matrix-transpose vt :-> v)
       (crop v :y 0 :y-dim row-dim :-> result)))))

(defmethod col-space ((m array) &key ((:-> result)) (singular-value-limit 0.0))
  (with-static-svd (s u v) m
   (let* ((test #'(lambda(x) (<= x singular-value-limit)))
	  (col-dim (or (position-if test s) (col-dim m))))
     (unless (plusp col-dim) (error "Matrix is degenerate"))
     (unless result
       (setq result (similar m :dimensions (list (row-dim m) col-dim))))
     (crop u :x 0 :x-dim col-dim :-> result))))


(defmethod col-null-space ((m array) &key ((:-> result)) (singular-value-limit 0.0))
  (with-static-svd (s u v) m
   (let* ((test #'(lambda(x) (<= x singular-value-limit)))
	  (col-dim (or (position-if test s) (col-dim m)))
	  (col-null-dim (- (row-dim m) col-dim)))
     (unless (plusp col-null-dim) (error "Matrix has full column rank"))
     (unless result
       (setq result (similar m :dimensions (list (row-dim m) col-null-dim))))
     (crop u :x col-dim :x-dim col-null-dim :-> result))))


#|
(print-values (setq foo (make-matrix '((1 0 0) (0 1 0)))))
(print-values (row-space foo))
(print-values (row-null-space foo))
(print-values (setq foo (make-matrix '((1 0 0) (1 0 0) (0 1 0)))))
(print-values (row-space foo))
(print-values (row-null-space foo))
(print-values (setq foo (make-matrix '((1 0 0) (0 0 1) (0 1 0)))))
(print-values (row-space foo))
(print-values (row-null-space foo))
(print-values (setq foo (make-matrix '((2 0 0 0) (0 3 0 0)))))
(print-values (row-space foo))
(print-values (row-null-space foo))

(print-values (setq foo (matrix-transpose (make-matrix '((1 0 0) (0 1 0))))))
(print-values (col-space foo))
(print-values (col-null-space foo))
(print-values (setq foo (matrix-transpose (make-matrix '((1 0 0) (1 0 0) (0 1 0))))))
(print-values (col-space foo))
(print-values (col-null-space foo))
(print-values (setq foo (matrix-transpose (make-matrix '((1 0 0) (0 0 1) (0 1 0))))))
(print-values (col-space foo))
(print-values (col-null-space foo))
(print-values (setq foo (matrix-transpose (make-matrix '((2 0 0 0) (0 3 0 0))))))
(print-values (col-space foo))
(print-values (col-null-space foo))

|#

(defmethod condition-number ((matrix array))
  (with-static-svd (s u v) matrix
   (sqrt (div (minimum s) (maximum s)))))

;;; Find eigenvector corresponding to the smallest eigenvalue of A.
;;; This is equivalent to: "Find unit vector x such that x minimizes x A
;;; x, for square symmetric matrix A".
(defun solve-eigenvalue (matrix
			 &key
			 (S (make-array (x-dim matrix) :element-type 'single-float))
			 (U (make-array (dimensions matrix) :element-type 'single-float))
			 (V (make-array (dimensions matrix) :element-type 'single-float))
			 ((:-> e-vector) (make-array (x-dim matrix) :element-type 'single-float)))
  (multiple-value-setq (S U V)
    (singular-value-decomposition matrix :u U :v V :s S))
  (multiple-value-bind (min-ev min-ev-pos) (minimize S)
    ;;Fill the e-vector with elements from the pos-of-min column of U.
    (dotimes (i (array-total-size e-vector))
      (setf (aref e-vector i) (aref U i min-ev-pos))))
  (values e-vector S))

;;; Return minimum-value item and its position in the sequence.
(defun minimize (seq &key key)
  (if (consp seq)
      (if key
	  (loop with min-pos = 0
		with min-val = (funcall key (car seq))
		for item in (cdr seq)
		for val = (funcall key item)
		for pos from 1
		do (when (< val min-val) (setq min-pos pos  min-val val))
		finally (return (values min-val min-pos )))
	  (loop with min-pos = 0
		with min-val = (car seq)
		for val in (cdr seq)
		for pos from 1
		do (when (< val min-val) (setq min-pos pos  min-val val))
		finally (return (values min-val min-pos ))))
      (if key
	  (loop with min-pos = 0
		with min-val = (funcall key (aref seq 0))
		for pos from 1 below (array-total-size seq)
		for val = (funcall key (aref seq pos))
		do (when (< val min-val) (setq min-pos pos  min-val val))
		finally (return (values min-val min-pos )))
	  (loop with min-pos = 0
		with min-val = (aref seq 0)
		for pos from 1 below (array-total-size seq)
		for val = (aref seq pos)
		do (when (< val min-val) (setq min-pos pos  min-val val))
		finally (return (values min-val min-pos ))))))

#| old code: bad arg names, doesn't use U, V, W, minimization code is slow and conses...
(defun solve-eigenvalue (matrix
			 &key
			 (U-matrix (make-array (dimensions matrix) :element-type 'single-float))
			 (V-matrix (make-array (dimensions matrix) :element-type 'single-float))
			 (W-vector (make-array (x-dim matrix) :element-type 'single-float))
			 ((:-> e-vector) (make-array (x-dim matrix) :element-type 'single-float)))
  (multiple-value-setq (W-vector U-matrix V-matrix) 
    (singular-value-decomposition matrix))
  (let* ((temp-list (concatenate 'list W-vector))
	 (min-of-temp-list (apply 'min temp-list))
	 ;;Position of minimum element.
	 (pos-of-min (position min-of-temp-list temp-list)))
    ;;Fill the e-vector with elements from the pos-of-min column of U-matrix.
    (loop for i from 0 below (length e-vector) do
	  (setf (aref e-vector i) (aref U-matrix i pos-of-min))))
  (values e-vector W-vector))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Principal Components:

;;; Assumes the data to be explained lie in the rows.
;;; *** Is it right to weight the components by the singular values?
(defun principal-components (matrix &key (dimension (min (row-dim matrix) (col-dim matrix)))
					scale)
  (with-svd (s u v) matrix
    (when (> dimension (length s))
      (error "Too many principal components requested"))

    (let ((principal-components (similar matrix :dimensions (list dimension (col-dim matrix))))
	  (principal-values (similar matrix :dimensions (list dimension (row-dim matrix))))
	  (singular-values (similar matrix :dimensions dimension))
	  (diagonal (when scale (diagonal-matrix (vectorize s :size dimension)))))

      ;; Calculate the principal components
      (crop (matrix-transpose v) :y-dim dimension :-> principal-components)
      (when scale (with-static-arrays ((product (matrix-mul diagonal principal-components)))
		    (copy product :-> principal-components)))

      ;; Calculate the principal values
      (crop u :y-dim dimension :-> principal-values)
      (when scale
	(with-static-arrays ((product (matrix-mul principal-values diagonal)))
	  (copy product :-> principal-values)))

      ;; Calculate the shortened singular values
      (crop s :x-dim dimension :-> singular-values)
      
      (values principal-components principal-values singular-values))))

#|
;; Test the principal components code
;; Set up a matrix with timecourse in rows
(setf dimensions (list 6 30))
(dimensions (setf matrix (make-array dimensions :element-type 'single-float)))

;; Set up a simple signal
(dimensions (setf signal (make-array (col-dim matrix) :element-type 'single-float)))
(loop with length = (length signal)
      for i from 0 below length
      with scale = (/ (* 3 2-pi) length)
      do (setf (aref signal i) (cos (* i scale))))
(image-from-array signal)

;; Add the signal into each row of the matrix
(dimensions (zero! matrix))
(dimensions (add-rows matrix signal :-> matrix))
(image-from-array (row 0 matrix))
(image-from-array (row 1 matrix))

;; Add noise to the matrix
(dimensions (randomize matrix 1.0 :-> matrix))
(image-from-array (row 0 matrix))
(image-from-array (row 1 matrix))

(null (multiple-value-setq (principal-components principal-values singular-values)
	(row-principal-components matrix :scale t)))

;; Look at the principal components
(mapcar 'display (mapcar 'image-from-array (displaced-rows principal-components 0 3)))
(image-from-array singular-values)
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Given symmetric matrix M, want to get a matrix T so that Tt M T = I.
;;; To do this, write M = A At, take the SVD, and use what you get back.
#|
;; Is this different from quadratic-decomp? 
(defun symmetric-decomposition (matrix &key ((:-> result) (similar matrix)))
  (unless (symmetric-p matrix)
    (error "Symmetric decomposition only possible on symmetric matrices"))
  (multiple-value-bind (s u v) (svd matrix)
    (declare (ignore v))
    (square-root s :-> s)
    (div 1.0 s :-> s :zero-val 0.0)
    (matrix-mul u (diagonal-matrix s) :-> result)))
|#

#|
(pprint (setf foo (make-matrix '((1 2 3) (4 5 6) (7 9 13)))))
(pprint (setf cov (matrix-mul-transpose foo foo)))
(pprint (setf sim (symmetric-decomposition cov)))
(assert (identity-p (setf result (matrix-transpose-mul sim (matrix-mul cov sim)))))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Given a matrix M, returns a matrix A such that:
;;; M = A * A^t
(defmethod quadratic-decomposition ((matrix array) &key ((:-> result) (similar matrix)))
  (declare-matrices (matrix) nil)
  (multiple-value-bind (s u v) (svd matrix)
    (declare-matrices (u v) (s))
    (unless (almost-equal u v)
      (error "Quadratic decomposition is impossible"))
    (square-root s :-> s)
    (matrix-mul v (diagonal-matrix s) :-> result)))

#|
(setf matrix (make-matrix '((1 2) (3 4))))
(setf product (matrix-mul-transpose matrix matrix))
(setf decomposition (quadratic-decomposition product))
(matrix-mul-transpose decomposition decomposition)
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Calculate the Moore-Penrose left pseudo-inverse using the SVD.
;;; SVD returns (s u v) such that m = u s v^t.  This function returns
;;; m^# = v 1/s u^t.  Allows the calling function to restrict the
;;; dimensions of the pseudo-inverse.  If dimension-limit is non-nil
;;; (it must be a whole number N), throw away all but the first N
;;; dimensions (set the singular values to 0).  If
;;; condition-number-limit or singular-value-limit is non-nil
;;; (must be a numbers), zero those singular-values that are too small.
;;; TODO
;;;    tmp matrices should be keywords

(defmethod matrix-inverse ((matrix array)
			   &key dimension-limit condition-number-limit singular-value-limit
			   suppress-warning
			   ((:-> result) (make-array (reverse (dimensions matrix))
						     :element-type (array-element-type matrix))))
  (declare-matrices (matrix result) nil)
  (unless (>= (row-dim matrix) (col-dim matrix))
    (error "Left pseudo-inverse doesn't make sense for short, fat matrices"))
  (with-static-arrays (s u v)
    (multiple-value-setq (s u v) (svd matrix))
    (with-static-arrays
	((diagonal (similar matrix :dimensions (list (col-dim v) (col-dim u)) :initial-element 0.0))
	 (tmp-arr (similar matrix :dimensions (list (col-dim v) (row-dim u)))))
    
      ;; throw away smallest singular values
      (when dimension-limit
	(decf dimension-limit)		; Decrement since the user specifies dimensions from 1...n
	(dotimes (index (length s))
	  (when (> index dimension-limit)
	    (setf (aref s index) 0.0))))
      (when condition-number-limit
	(dotimes (index (length s))
	  (when (> (sqrt (/ (aref s 0) (aref s index))) condition-number-limit)
	    (setf (aref s index) 0.0))))
      (when singular-value-limit
	(dotimes (index (length s))
	  (when (< (aref s index) singular-value-limit)
	    (setf (aref s index) 0.0))))

      ;; Compute the inverse
      (div 1.0 s :zero-value 0.0 :suppress-warning suppress-warning :-> s)
      (diagonal-matrix s :-> diagonal)
      (matrix-mul v (matrix-mul-transpose diagonal u :-> tmp-arr) :-> result))))

#|
;;; Test the matrix-inverse code
(setf *print-array* t)

(setf matrix (make-array '(3 3) :element-type 'single-float
			 :initial-contents '((1.0 0.0 0.0) (0.0 2.0 0.0) (0.0 0.0 3.0))))
(setf matrix-inverse (matrix-inverse matrix))
(print-values (matrix-mul matrix matrix-inverse))
(setf matrix (make-array '(3 3) :element-type 'single-float
			 :initial-contents '((1.0 2.0 3.0) (-1.0 2.0 3.0) (1.0 -2.0 3.0))))
(setf matrix-inverse (matrix-inverse matrix))
(print-values (matrix-mul matrix matrix-inverse))
(setf matrix (make-array '(4 3) :element-type 'single-float
			 :initial-contents '((1.0 2.0 3.0) (-1.0 2.0 3.0) (1.0 -2.0 3.0) (1.0 1.0 1.0))))
(setf matrix-inverse (matrix-inverse matrix))
(print-values (matrix-mul matrix-inverse matrix))

(setf matrix (make-array '(3 3) :element-type 'single-float
			 :initial-contents '((1.0 2.0 3.0) (4.0 5.0 6.0) (7.0 8.0 9.0))))
(setf matrix-inverse (matrix-inverse matrix :singular-value-limit 1e-6))
(print-values (matrix-mul matrix-inverse matrix))
|#

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; Multivariate gaussian noise:

;;; Generate gaussian noise given a covariance matrix.
;;; If Y is a (row) vector with entries distributed iid N(0,1),
;;; and if Z is a (row) vector, Z = Y A  (some matrix A)
;;; then the covariance matrix of Z is At*A.
;;; Hence, given a covariance matrix M for Z (desired covariance),
;;; we do this:
;;; (0) Decompose M = A At
;;; (1) Generate Y
;;; (2) Return Z = Y At
(defmethod gaussian-noise ((mean vector) (covariance array) &key ((:-> result) (similar mean)))
  (let* ((temp-arr (similar result))
         (temp-vec (vectorize temp-arr))
         (transform (matrix-transpose (quadratic-decomposition covariance))))
    (dotimes (index (length temp-vec))
      (setf (aref temp-vec index) (gaussian-noise 0.0 1.0)))
    (matrix-mul temp-arr transform :-> result)
    (add-rows result mean :-> result))
  result)

#|
;;; Will work one vector at a time or on rows.
(dimensions (setf noise (make-array '(500 2) :element-type 'single-float)))
(setf mean (make-matrix '(10.0 10.0)))
(setf transform (make-matrix '((20.0 10.0) (-3.0 15.0))))
(setf covariance (matrix-transpose-mul transform transform))

;;; Make one/many observations, check them against desired
(print-values (gaussian-noise mean covariance))
(dimensions (gaussian-noise mean covariance :-> noise))
(print-values (div (sub (row-covariance noise) covariance) covariance))
(print-values (div (sub (row-mean noise) mean) mean))

;;; Check the circularizing transform
(print-values (row-sample-covariance (matrix-mul noise (matrix-inverse transform))))
|#



;;; Local Variables:
;;; buffer-read-only: t 
;;; End:



