;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: steer.lisp
;;;  Author: David Heeger and Bill Freeman
;;;  Description: steerable filters and quadrature-steerable filters
;;;  Creation Date: summer '89
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,  
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; revisions history:

;;; Feb. 2, 1991  Freeman added steer-filter-list.
;;; March 26, 1991  freeman added make-avg-steerable-filters and *avg-steerable-filters*.
;;; July 11, 1991  freeman commented-out *avg-steerable-filters*, because I don't use
;;;   it and it slowed-down load-in.
;;; Oct. 8, 1991  Freeman allowed make-quadrature... to take existing even and odd
;;;   steerable basis.

;;; Sept 16, 1992 Simoncelli ported from obvius-1.2 to obvius-2.2.  Changes:
;;;   Added :startx key to sample-1d.
;;;   Added my-gauss to bottleneck this fn.
;;;   Removed steer-even, steer-odd, sum-even, sum-odd
;;;   Wrote generic-steer method: Steers any comuted from separable or directional
;;;      derivatives.  It is slightly slower than the hand-coded version.
;;;   Changed names magnitude -> directional-magnitude,
;;;      squared-magnitude -> directional-energy, complex-phase -> directional-phase.
;;;   Wrote separable-to-directional and directional-to-separable conversion functions, but
;;;      these can't be used unless we make a new class or add a basis-type slot.
;;;   **** Would be nice to provice the following:
;;;      (make-directional-steerable-filters dims :order :base-function :delx :dely)

(in-package 'obvius)
(export '(*default-even-steerable-filters* *default-odd-steerable-filters*
	  make-g1-steerable-filters
	  make-g2-steerable-filters make-h2-steerable-filters
	  make-g4-steerable-filters make-h4-steerable-filters
	  make-avg-steerable-filters
	  steerable-basis steerable-basis-p separable-steerable-basis-p
	  make-steerable-basis even-steerable-basis odd-steerable-basis
	  filter-list image-list order
	  dimensions x-dim y-dim minimum maximum range
	  steer generic-steer
	  quadrature-steerable-basis make-quadrature-steerable-basis 
	  steer-even steer-odd sum-even sum-odd
	  directional-magnitude directional-energy directional-phase
	  average-energy steer-filter-list
	  sample-1d sample-2d))

(obv-require :matrix)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; DEFAULT STEERABLE FILTERS
;;; 2nd derivs of Gaussians 
;;; and their Hilbert transforms.

;;; sample a function in 2-d and make a filter out of the result.
(defun sample-2d (function  &key (y-window-size 9) (x-window-size 9) 
			    (delx 1.0) (dely 1.0)
			    (startx (- (* (1- x-window-size) delx 0.5)))
			    (starty (- (* (1- y-window-size) dely 0.5))))
  (let ((kernel (make-array (list y-window-size x-window-size) :element-type 'single-float)))
    (loop for j from 0 below y-window-size
	  for y = starty then (+ y dely)
 	  do
 	  (loop for i from 0 below x-window-size
		for x = startx then (+ x delx)
 		do
 		(setf (aref kernel j i) (float (funcall function y x )))))
    kernel))

;;; return a vector of samples of the function. window-size is
;;; dimension of vector returned.  delx is the space between samples.
;;; startx is the position of the first sample.
;;; *** Args not consistent with make-synthetic-image or make-discrete-function.
(defun sample-1d (function &key (window-size 9) (delx 1.0)
			   (startx (- (* (1- window-size) delx 0.5))))
  (let ((kernel (make-array (list window-size) :element-type 'single-float)))
    (loop for i from 0 below window-size
	  for x = startx then (+ x delx)
	  do
	  (setf (aref kernel i) (funcall function x)))
    kernel))

;;; Standard gaussian: **** NOTE: this is not univariate!
(defun my-gauss (x) (exp (* x x -1.0)))

(defun g2.gauss (x) (my-gauss x))

;;; 2nd derv of gaussian filters.   The polynomial functions:
;;; What is the 0.92132?
(defun g2.poly (x)
  (* (my-gauss x)
     0.92132
     (- (* 2.0 (expt x 2)) 1.0)))

;;; What is the normalizer?
(defun g2.diag (x)
  (* (my-gauss x)
     1.35744
     x))

;;; hilbert transforms of 2nd deriv of gaussian filters.  Polynomial forms.
(defun h3.hila (x)
  (* (my-gauss x)
     0.97796 
     (+ (* -2.2544 x)
	(expt x 3))))

(defun h3.hilb (x) (my-gauss x))

(defun h3.minusx (x)
  (* (my-gauss x)
     x))

(defun h3.minusy (x)
  (* (my-gauss x)
     0.97796
     (+ (expt x 2) -0.751465)))

(defun make-g2-steerable-filters ()
  (let ((n2dgpoly (make-filter (sample-1d 'g2.poly :window-size 9 :delx 0.67)))
	(n2dggauss (make-filter (sample-1d 'g2.gauss :window-size 9 :delx 0.67)))
	(n2diag (make-filter (sample-1d 'g2.diag :window-size 9 :delx 0.67))))
    (list (make-separable-filter n2dggauss n2dgpoly :edge-handler :reflect1)
	  (make-separable-filter n2diag n2diag :edge-handler :reflect1)
	  (make-separable-filter n2dgpoly n2dggauss :edge-handler :reflect1))))

(defun make-h2-steerable-filters ()
  (let ((n3hila (make-filter (sample-1d 'h3.hila :window-size 9 :delx 0.67)))
	(n3hilb (make-filter (sample-1d 'h3.hilb :window-size 9 :delx 0.67)))
	(n3minusy (make-filter (sample-1d 'h3.minusy :window-size 9 :delx 0.67)))
	(n3minusx (make-filter (sample-1d 'h3.minusx :window-size 9 :delx 0.67))))
    (list (make-separable-filter n3hilb n3hila :edge-handler :reflect1)
	  (make-separable-filter n3minusx n3minusy :edge-handler :reflect1)
	  (make-separable-filter n3minusy n3minusx :edge-handler :reflect1)
	  (make-separable-filter n3hila n3hilb :edge-handler :reflect1))))

;;; 2nd derivative of Gaussian filters  (normalized)
(defvar *default-even-steerable-filters* (make-g2-steerable-filters))

;;; Hilbert transform of 2nd derivative of Gaussian filters  (normalized)
(defvar *default-odd-steerable-filters* (make-h2-steerable-filters))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; STEERABLE 1ST DERIVATIVE OF GAUSSIAN
;;; Sept. 27, 1989

;;; (Ix Iy)
(defun make-g1-steerable-filters ()
  (let ((n1dgpoly (make-filter (sample-1d 'g1.poly :window-size 9 :delx 0.67)))
	(n1dggauss (make-filter (sample-1d 'g1.gauss :window-size 9 :delx 0.67))))
    (list (make-separable-filter n1dggauss n1dgpoly :edge-handler :reflect1)
	  (make-separable-filter n1dgpoly n1dggauss :edge-handler :reflect1)))) 
;;; from loading  ~/mtca/defns.m  into mathematica, and evaluating N[normderivgauss2d[x,y,1]],
;;; we find the normalization required for the integral of 1st deriv of Gaussian to equal 1.
(defun g1.poly (x)
  (* (my-gauss x)
     (* x -1.59577)))

(defun g1.gauss (x)
  (my-gauss x))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; DEFINE STEERABLE 4TH DERIVATIVE OF GAUSSIAN, AND STEERABLE 5TH ORDER FIT TO
;;; ITS HILBERT TRANSFORM
;;; August 27, 1989 Bill Freeman created from ~freeman/lisp/steer/steerables.lisp,
;;;                              and ~freeman/mtca/sepout.  See also Vision Science
;;;                              Technical Report #118.
;;; x-y separable Gaussian derivatives, order 4
(defun g4.4y (y)
  (* 1.24585
     (my-gauss y)
     (+ 0.75 (* -3.0 y y) (expt y 4))))
(defun g4.4x (x)
  (my-gauss x))

(defun g4.3x (x)
  (* 1.24585 x (my-gauss x)))
(defun g4.3y (y)
  (* (my-gauss y) (+ (* -1.5 y) (expt y 3))))

(defun g4.2x (x)
  (* 1.116176 (my-gauss x) (+ (* x x) -0.5)))
(defun g4.2y (y)
  (* 1.116176 (my-gauss y) (+ (* y y) -0.5)))

(defun g4.1y (y)
  (* 1.24585 y (my-gauss y)))
(defun g4.1x (x)
  (* (my-gauss x) (+ (* x -1.5) (expt x 3))))

(defun g4.0x (x)
  (* 1.24585
     (my-gauss x)
     (+ 0.75 (* -3.0 x x) (expt x 4))))
(defun g4.0y (y)
  (my-gauss y))

;;; x-y separable order 5 Hilbert transforms of order 4 Gaussian derivatives

(defun h4.5y (y)
  (* 0.39752
     (my-gauss y)
     (+ (expt y 5) (* (expt y 3) -7.5014) (* y 7.1891))))
(defun h4.5x (x)
  (my-gauss x))

(defun h4.4y (y)
  (* 0.39752
     (my-gauss y)
     (+ (expt y 4) (* (expt y 2) -4.501) 1.4378)))
(defun h4.4x (x)
  (* x (my-gauss x)))

;;; ;; non-separable version of h4.3
;;; (defun h4.3yx (y x)(* 0.39752 (exp (+ (* x x -1.0) (* y y -1.0))) 
;;; 		      (+
;;; 		       (* 1.4378 y)
;;; 		       (* x x y -2.25043)
;;; 		       (* (expt y 3) -.750143)
;;; 		       (* (expt y 3) (expt x 2)))))
;;; 

;; separable (approximate) version of h4.3
(defun h4.3y (y)
  (* 0.39752
     (my-gauss y)
     (+ (expt y 3) (* y -2.225))))
(defun h4.3x (x)
  (* (my-gauss x)
     (+ (expt x 2) -0.6638)))

;;; ;; non-separable version of h4.2
;;; (defun h4.2yx (y x)(* 0.39752 (exp (+ (* y y -1.0) (* x x -1.0))) 
;;; 		      (+
;;; 		       (* 1.4378 x)
;;; 		       (* y y x -2.25043)
;;; 		       (* (expt x 3) -.750143)
;;; 		       (* (expt x 3) (expt y 2)))))
;;; 

;;; approximately separable version of h4.2
(defun h4.2x (x)
  (* 0.39752
     (my-gauss x)
     (+ (expt x 3) (* x -2.225))))
(defun h4.2y (y)
  (* (my-gauss y)
     (+ (expt y 2) -0.6638)))

(defun h4.1x (x)
  (* 0.39752
     (my-gauss x)
     (+ (expt x 4) (* (expt x 2) -4.501) 1.4378)))
(defun h4.1y (y)
  (* y (my-gauss y)))

(defun h4.0x (x)
  (* 0.39752
     (my-gauss x)
     (+ (expt x 5) (* (expt x 3) -7.5014) (* x 7.1891))))
(defun h4.0y (y)
  (my-gauss y))

(defun make-g4-steerable-filters ()
  (list 
   (make-separable-filter 
    (make-filter (sample-1d 'g4.0y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.0x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'g4.1y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.1x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'g4.2y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.2x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'g4.3y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.3x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'g4.4y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'g4.4x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)))

(defun make-h4-steerable-filters ()
  (list 
   (make-separable-filter 
    (make-filter (sample-1d 'h4.0y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.0x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'h4.1y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.1x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'h4.2y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.2x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'h4.3y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.3x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'h4.4y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.4x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)
   (make-separable-filter 
    (make-filter (sample-1d 'h4.5y :window-size 13 :delx 0.5))
    (make-filter (sample-1d 'h4.5x :window-size 13 :delx 0.5))
    :edge-handler :reflect1)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;; this is for a set of steerable filters which average along a particular direction.

;;; function chooses the cosine power for the filter response based on 
;;;  the number of filters chosen for the steerable basis.
;;;  Normalize this so that, ignoring the sampling effects, the function will
;;;  have a unity DC value.  See p. 262 wtf loose red notebook.
;;;  integral cos^N theta   exp(-r^2 / (2 sigma^2)  r  d theta  d r
;;;  =  (using abramowitz and stegun, p. 77 4.3.127, and p. 302, 7.4.5)
;;;  2 pi 1/2  3/4  5/6 ...  N-1/N     times  sigma^2.
;;; In practise, norm was within 3%, for a 9x9 filter.  I think that's ok.
(defun make-avg-steerable-filters (&key 
				   (n-filts 5)
				   (sigma 2.0)
				   (del 1.0)
				   (size 9))
  (let* ((cos-power (1- n-filts))
	 (norm (norm-avg-integral cos-power sigma)))
    (loop for i from 0 below n-filts collect
	  (make-filter
	   (sample-2d #'(lambda (y x) 
			  (/ (avg-filter (* i (/ pi n-filts)) cos-power sigma y x) norm))
		      :x-window-size size :y-window-size size
		      :delx del :dely del)
	   :edge-handler :reflect1))))

;;; a steerable filter to perform angularly adaptive local averaging
;;; returns:  Cos(theta - theta-offset)^cos-power  exp(-r^2/(2 sigma^2)),
;;; where r = sqrt(x^2 + y^2), and theta = the usual arctan(y/x).
;;; Assumes cos-power is an even integer, so it takes abs to fix a numerical problem.
;;; Note:  play with symmetries (like atan y x  vs  atan x y) until get the
;;;        desired starting angle, and direction of rotation.
(defun avg-filter (theta-offset cos-power sigma y x &key (debug nil))
  (let ((r2 (+ (* x x) (* y y)))
	(theta (if (and (= 0.0 x) (= 0.0 y))
		   0.0
		   (atan x y)))
	(out nil))
    (setq out (* (expt (abs (cos (- theta theta-offset))) cos-power)
		 (exp (/ r2 (* -2 sigma sigma)))))
    (when debug
      (format debug "y ~d  x ~d  theta ~d  r^2 ~d~%" y x theta r2)
      (format debug "out ~d ~%" out))
    out))

;;; an auxiliary function to compute normalization for steerable blurring filters.
(defun norm-avg-integral (order sigma)
  (cond ((< order 0) (error "Negative order argument."))
	((= order 1) (error "Even orders only for now."))
	((= order 2) (* pi sigma sigma))
	(t (/ (* (1- order) (norm-avg-integral (- order 2) sigma)) order))))

;;; (defvar *avg-steerable-filters* (make-avg-steerable-filters))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; STEERABLE BASIS OBJECT

;;; *** Why do we need to hang on to the filter-list?  It is only used
;;; to determine if the basis is separable.  So either make a new
;;; basis-type slot, or a new object.
(def-simple-class steerable-basis (image-sequence)
  (filter-list)
  (:default-initargs
      :filter-list *default-even-steerable-filters*
      :display-type 'pasteup))

(defmacro steerable-basis-p (obj)
  `(typep ,obj 'steerable-basis))

(defmacro separable-steerable-basis-p (obj)
  `(and (typep ,obj 'steerable-basis)
        (every #'(lambda (x) (separable-filter-p x))
	       (filter-list ,obj))))
		      
;;; the input to this can be either an image, or a list of images
(defun make-steerable-basis
    (image &rest initargs
	   &key 
	   (filter-list *default-even-steerable-filters*)
	   display-type name ->)
  (declare (ignore name display-type))
  (when -> (setf (getf initargs :name) ->))
  (unless (getf initargs :image-list)
    (setf (getf initargs :image-list)
	  (cond ((image-p image)
		 (mapcar #'(lambda (f) (apply-filter f image)) filter-list))
		((listp image)  image)
		(t (error "First arg must be either an image or a list or images")))))
  (with-result ((result nil)
		`(:class steerable-basis ,@initargs)
		'apply 'make-steerable-basis image initargs)
    result))

(defmethod inferiors-of ((steerable steerable-basis))
  (image-list steerable))

(defmethod order ((steerable steerable-basis))
  ;(1- (length (filter-list steerable)))
  (1- (sequence-length steerable)))

;;; *** Should we define this on image-sequence?
(defmethod sum ((steerable steerable-basis) &key ->)
  (with-result ((res ->) (aref (data steerable) 0 0)
		'sum steerable)
    (loop for im in (image-list steerable) do
	  (add im res :-> res))
    res))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; NEW GENERIC STEERING CODE.  Written 9/92 by EPS.

;;; Interpolate responses of filters at the given angle.  ANGLE can be
;;; a number or an image.  Call hand-coded more efficient versions for
;;; specific order filters.  Otherwise, call generic version.
(defmethod steer ((steerable steerable-basis) angle &key ->)
  (with-result ((result ->)
		(aref (data steerable) 0 0)
		'steer steerable angle)
    (if (separable-steerable-basis-p steerable)
	(case (order steerable)		; separable steerable
	  (1 (steer-separable-1 (image-list steerable) angle result))
	  (2 (steer-separable-2 (image-list steerable) angle result))
	  (3 (steer-separable-3 (image-list steerable) angle result))
	  (4 (steer-separable-4 (image-list steerable) angle result))
	  (5 (steer-separable-5 (image-list steerable) angle result))
	  (otherwise (warn "Using generic steering code")
	      (generic-steer steerable angle result)))
	(case (order steerable)		; non-separable steerable
	  (1 (steer-separable-1 (image-list steerable) angle result))
	  (3 (steer-3 (image-list steerable) angle result))
	  (4 (steer-4 (image-list steerable) angle result))
	  (otherwise (warn "Using generic steering code")
	      (generic-steer steerable angle result))))
    result))

;;; In other functions, we want to be able to steer images,
;;; and have it just copy the result to the output.  The angle is irrelevant.
(defmethod steer ((im image) angle &key ->)
  (with-result ((res ->)
		im
		'steer im angle)
    (copy im :-> res)))

;;; Assumes basis contains either separable derivative images (x^N,
;;; x^N-1y, ... y^N), or directional derivative images oriented at
;;; (npi/N), n=0..N.  ANGLE can be a number or an image.
(defmethod generic-steer ((basis steerable-basis) angle &key ->)
  (with-result ((result ->)
		(aref (data basis) 0 0)
		'steer basis angle)
    (with-local-viewables
	((sep-steer-vect (separable-to-directional-weights angle (order basis))))
      (if (or (separable-steerable-basis-p basis) (= (order basis) 1))
	  (dot-product basis sep-steer-vect :-> result)
	  (with-local-viewables
	      ((dir-to-sep (directional-to-separable-mtx (order basis)))
	       (dir-steer-vect (matrix-mul sep-steer-vect dir-to-sep)))
	    (dot-product basis dir-steer-vect :-> result))))))


#|  TO WRITE THIS, MUST add a basis-type slot, or make new classes:
(defun separable-to-directional (basis &key ->)
  (unless (steerable-basis-p basis)
    (error "Argument is not a steerable basis: %a" basis))
  (with-result ((res ->)
		basis
		'separable-to-directional basis)
    (if (separable-steerable-basis-p basis)
	(matrix-mul (separable-to-directional-mtx (order basis))
		    basis
		    :-> res)
	(copy basis :-> res))))

(defun directional-to-separable (basis &key ->)
  (unless (steerable-basis-p basis)
    (error "Argument is not a steerable basis: %a" basis))
  (with-result ((res ->)
		basis
		'directional-to-separable basis)
    (if (separable-steerable-basis-p basis)
	(copy basis :-> res)
	(matrix-mul (directional-to-separable-mtx (order basis))
		    basis
		    :-> res))))
|#

;;; Compute an (order+1)x(order+1) matrix, that converts from
;;; directional basis to a separable one.  
(defun directional-to-separable-mtx (order)
  (matrix-inverse (separable-to-directional-mtx order)))

;;; Compute separable-to-directional weights for a set of
;;; uniformly distributed directions, and paste them into the rows of
;;; a square matrix.
(defun separable-to-directional-mtx (order)
  (let ((res (make-array (list (1+ order) (1+ order))
			 :element-type 'single-float)))
    (loop for index from 0 to order
	  for angle = (* index (/ pi (1+ order)))
	  for row = (displaced-row index res)
	  do
	  (separable-to-directional-weights angle order :-> row))
    res))

;;; Computes the coefficients in the polynomial (x cos + y sin)^order.
;;; *** Currently hardwired for 2D.  Should be generalized.
(defmethod separable-to-directional-weights
    ((angle number) order &key
     ((:-> res) (make-array (1+ order) :element-type 'single-float)))
  (unless (and (typep res '(array single-float *))
	       (= (total-size res) (1+ order)))
    (error "Result arg must be a single-float array of size ~A" (1+ order)))
  (let* ((cos (cos angle))
	 (sin (sin angle))		;negated below!
	 (order-factorial  (factorial order)))
    (setq sin (- sin))
    (loop for index from (- order 1) downto 0
	  for c^n = cos then (* c^n cos)
	  do (setf (aref res index) c^n))
    (loop for index from 1 to order
	  for weight = (/ order-factorial 
			  (* (factorial index) (factorial (- order index))))
	  for s^n = sin then (* s^n sin)
	  do
	  (if (< index order)
	      (setf (aref res index) (* weight (aref res index) s^n))
	      (setf (aref res index) s^n)))
    res))

(defmethod separable-to-directional-weights ((angle image) order &key ->)
  (with-result ((res ->)
		`(:class image-sequence
		  :length ,(1+ order)
		  :dimensions ,(dimensions angle))
		'separable-to-directional-weights angle order)
    (with-local-viewables ((cos (cos. angle))
			   (sin (sin. angle)) ;this is negated below!
			   (order-factorial  (factorial order))
			   (temp (similar cos)))
      (-. sin :-> sin)
      (loop for index from (- order 1) downto 0
	    for c^n = (copy cos :-> (aref (data res) 0 index))
	    then (*. c^n cos :-> (aref (data res) 0 index)))
      (loop for index from 1 to order
	    for weight = (/ order-factorial 
			    (* (factorial index) (factorial (- order index))))
	    for s^n = sin then (*. s^n sin :-> temp)
	    do
	    (if (< index order)
		(*. weight (aref (data res) 0 index) s^n :-> (aref (data res) 0 index))
		(*. weight s^n :-> (aref (data res) 0 index))))
      res)))


;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; HAND-CODED STEERING FUNCTIONS FOR EACH DIFFERENT ORDER

;;; this is the original steering code: harder to read, but a little
;;; bit faster and memory-efficient

;;;;;;;; steering functions for images made with x-y separable filters:  ;;;;;;;;;

(defun steer-separable-1 (image-list angle result)
  (with-local-viewables ((a (make-image (dimensions (car image-list))))
			 (b (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (-. a :-> a)
	   (cos. angle :-> b)
	   (mul (nth 0 image-list) b :-> b)      ;;; cos * image
	   (mul (nth 1 image-list) a :-> a)      ;;; sin * image
	   (add a b :-> result))
	  (t 
	   (mul (nth 0 image-list) (cos angle) :-> a)  ;;; cos * image
	   (mul (nth 1 image-list) (- (sin angle)) :-> b)  ;;; sin * image
	   (add a b :-> result)))))

(defun steer-separable-2 (image-list angle result)
  (with-local-viewables ((a (make-image (dimensions (car image-list))))
			 (b (make-image (dimensions (car image-list))))
			 (c (make-image (dimensions (car image-list))))
			 (d (make-image (dimensions (car image-list))))
			 (e (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (square b :-> c);; cos-sqr
	   (mul (nth 0 image-list) c :-> result);; h-part
	   (square a :-> c);; sin-sqr
	   (mul (nth 2 image-list) c :-> d);; v-part
	   (add d result :-> result);; h-part + v-part
	   (mul a b :-> c);; sin-cos
	   (mul c -2.0 :-> c);; -2 sin-cos
	   (mul c (nth 1 image-list) :-> d);; d-part
	   (add result d :-> result))
	  (t 
	   (mul (nth 0 image-list) (expt (cos angle) 2) :-> a);; h-part
	   (mul (nth 1 image-list) (* -2.0 (sin angle) (cos angle)) :-> b);;d-part
	   (mul (nth 2 image-list) (expt (sin angle) 2):-> c);; v-part
	   (add a c :-> result)
	   (add b result :-> result)))))

(defun steer-separable-3 (image-list angle result)
  (with-local-viewables ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list))))
		      (f (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (square b :-> c);; cos-sqr
	   (mul c b :-> f);; cos-cube
	   (mul c a :-> d);; cos-sqr-sin
	   (mul (nth 1 image-list) d :-> e);; d2-part
	   (square a :-> c);; sin-sqr
	   (mul b c :-> d);; cos-sin-sqr
	   (mul (nth 2 image-list) d :-> d);; d1-part
	   (mul (sub d e :-> d) 3.0 :-> result)
	   (mul (nth 0 image-list) f :-> f);; h-part
	   (mul c a :-> c);; sin-cube
	   (mul (nth 3 image-list) c :-> c);; v-part
	   (sub f c :-> f);; "sub h-part v-part :-> h-part" 
	   (add f result :-> result))
	  (t 
	   (mul (nth 0 image-list) (expt (cos angle) 3) :-> a)  ;; h-part
	   (mul (nth 3 image-list) (expt (sin angle) 3) :-> b)  ;; v-part
	   (mul (nth 2 image-list) (* 3.0 (expt (sin angle) 2) (cos angle)) :-> c) ;;d1-part 
	   (mul (nth 1 image-list) (* 3.0 (sin angle) (expt (cos angle) 2)) :-> d) ;; d2-part
	   (add (sub a b :-> a) 
		(sub c d :-> c) :-> result)))))


;;; see vision science technical report for the steering formulas
(defun steer-separable-4 (image-list angle result)
  (with-local-viewables ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list))))
		      (f (make-image (dimensions (car image-list))))
		      (g (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> e)
	   (square a :-> b)   ;; sin^2
	   (mul b a :-> c)    ;; sin^3
	   (square b :-> d)   ;; sin^4          d
	   (mul e c :-> c)    ;; sin^3 cos      c
	   (square e :-> f)   ;; cos^2
	   (mul f b :-> b)    ;; cos^2 sin^2    b
	   (mul e f :-> g)    ;; cos^3
	   (mul g a :-> a)    ;; cos^3 sin      a
	   (square f :-> g)   ;; cos^4          g
	   (mul a -4.0 :-> a) ;; -4sincos^3       a
	   (mul b 6.0 :-> b)  ;; 6sin^2 cos^2     b
	   (mul c -4.0 :-> c) ;; -4sin^3 cos      c
	   (mul (nth 0 image-list) g :-> g);;      g
	   (mul (nth 1 image-list) a :-> a);;      a
	   (mul (nth 2 image-list) b :-> b);;      b
	   (mul (nth 3 image-list) c :-> c);;      c
	   (mul (nth 4 image-list) d :-> d);;      d
	   (add g a :-> e);; cos^4 + 4 cos^3 sin
	   (add b c :-> f);; 6 sin^2 cos^2 -4 sin^3 cos
	   (add e f :-> a);; cos^4 + 4 cos^3 sin + 6 sin^2 cos^2 -4 sin^3 cos
	   (add a d :-> result));; cos^4 + 4 cos^3 sin + 6 sin^2 cos^2 -4 sin^3 cos + sin^4
	  (t 
	   (mul (nth 0 image-list) (expt (cos angle) 4) :-> a)
	   (mul (nth 1 image-list) (* -4.0 (sin angle) (expt (cos angle) 3)) :-> b)
	   (mul (nth 2 image-list) (* 6.0 (sqr (cos angle)) (sqr (sin angle))) :-> c)
	   (mul (nth 3 image-list) (* -4.0 (cos angle) (expt (sin angle) 3)) :-> d)
	   (mul (nth 4 image-list) (expt (sin angle) 4) :-> e)
	   (add a b :-> f)
	   (add c d :-> g)
	   (add f g :-> a)
	   (add e a :-> result)))))

;;; see vision science technical report for the steering formulas
(defun steer-separable-5 (image-list angle result)
  (cond ((image-p angle)      
	 (with-local-viewables ((a (make-image (dimensions (car image-list))))
			     (b (make-image (dimensions (car image-list))))
			     (c (make-image (dimensions (car image-list))))
			     (d (make-image (dimensions (car image-list))))
			     (e (make-image (dimensions (car image-list))))
			     (f (make-image (dimensions (car image-list))))
			     (g (make-image (dimensions (car image-list)))))
	   (sin. angle :-> a) 
	   (cos. angle :-> b)  ;; cos-angle
	   (square a :-> c)   ;; sin^2
	   (mul c a :-> d)    ;; sin^3
	   (square c :-> e)   ;; sin^4          
	   (mul c d :-> f)    ;; sin^5          f
	   (mul b e :-> e)    ;; sin^4 cos      e
	   (square b :-> g)   ;; cos^2
	   (mul g d :-> d)    ;; cos^2 sin^3    d
	   (mul g b :-> g)    ;; cos^3
	   (mul g c :-> c)    ;; cos^3 sin^2    c
	   (mul g b :-> g)    ;; cos^4          
	   (mul g a :-> a)    ;; cos^4 sin      a
	   (mul g b :-> g)    ;; cos^5          g
	   (mul a -5.0 :-> a) ;; -5 cos^4 sin
	   (mul c 10.0 :-> c) ;; 10 cos^3 sin^2
	   (mul d -10.0 :-> d) ;; -10 cos^2 sin^3
	   (mul e 5.0 :-> e)  ;; 5 cos sin^4
	   (mul f -1.0 :-> f) ;; -sin^5
	   (mul (nth 0 image-list) g :-> g);;      g
	   (mul (nth 1 image-list) a :-> a);;      a
	   (mul (nth 2 image-list) c :-> c);;      c
	   (mul (nth 3 image-list) d :-> d);;      d
	   (mul (nth 4 image-list) e :-> e);;      e
	   (mul (nth 5 image-list) f :-> f);;      f
	   (add g a :-> b)            ;; cos^5 - 5 cos^4 sin
	   (add c d :-> result)       ;; 10 cos^3 sin^2 + -10 cos^2 sin^3
	   (add b result :-> result)  ;; cos^5 - 5 cos^4 sin + 10 cos^3 sin^2 + -10 cos^2 sin^3
	   (add e f :-> b)            ;; 5 cos sin^4 - sin^5
	   (add b result :-> result)));; cos^5 - 5 cos^4 sin + 10 cos^3 sin^2 + 
                                      ;;    -10 cos^2 sin^3 + 5 cos sin^4 - sin^5
	(t 
	 (with-local-viewables ((a (make-image (dimensions (car image-list))))
			     (b (make-image (dimensions (car image-list))))
			     (c (make-image (dimensions (car image-list)))))
	   (mul (nth 0 image-list) (expt (cos angle) 5) :-> a)
	   (mul (nth 1 image-list) (* -5.0 (sin angle) (expt (cos angle) 4)) :-> b)
	   (add a b :-> c)
	   (mul (nth 2 image-list) (* 10.0 (expt (cos angle) 3) (sqr (sin angle))) :-> a)
	   (mul (nth 3 image-list) (* -10.0 (sqr (cos angle)) (expt (sin angle) 3)) :-> b)
	   (add a b :-> result)
	   (add c result :-> result)
	   (mul (nth 4 image-list) (* 5.0 (cos angle) (expt (sin angle) 4)) :-> a)
	   (mul (nth 5 image-list) (* -1.0 (expt (sin angle) 5)) :-> b)
	   (add a b :-> c)
	   (add c result :-> result)))))


;;;;;;;; steering functions for images made with x-y non-separable filters:  ;;;;;;;;;

;;; non-separable, 3rd order polynomial filters.  See ~freeman/mtca/nonsepsteer.m
;;; for the formulae.
(defun steer-3 (image-list angle result)
  (with-local-viewables ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (mul (square b :-> c) b :-> c);; cos cubed
	   (sub (mul c 2 :-> c) b :-> e);; coefficient of 0th image
	   (mul (nth 0 image-list) e :-> result) 
	   (mul (add a b :-> c) .7071 :-> c)  ;;; 0.707107*Cos[theta] + 0.707107*Sin[theta] 
	   (mul c (square c :-> d) :-> d)  ;;; above, cubed
	   (sub (mul d 2.0 :-> e) c :-> e);; coefficieint of 1st image
	   (add (mul (nth 1 image-list) e :-> e) result :-> result);; running total
	   (mul (square a :-> c) a :-> c);; sin cubed
	   (sub (mul c 2 :-> c) a :-> e);; coefficient of 2nd image
	   (add (mul (nth 2 image-list) e :-> e) result :-> result);; running total
	   (mul (sub a b :-> c) .7071 :-> c)  ;;; 0.707107*Sin[theta] - 0.707107*Cos[theta] 
	   (mul c (square c :-> d) :-> d)  ;;; above, cubed
	   (sub (mul d 2.0 :-> e) c :-> e);; coefficieint of 3rd image
	   (add (mul (nth 3 image-list) e :-> e) result :-> result));; result
	  (t 
	   (mul (nth 0 image-list) 
		(- (* (expt (cos angle) 3) 2.0) (cos angle)) :-> result)
	   (mul (nth 1 image-list) 
		(- (* (expt (cos (- angle (/ pi 4))) 3) 2.0)
		   (cos (- angle (/ pi 4))))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 2 image-list) 
		(- (* (expt (cos (- angle (/ pi 2))) 3) 2.0)
		   (cos (- angle (/ pi 2))))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 3 image-list) 
		(- (* (expt (cos (- angle (* 3 (/ pi 4)))) 3) 2.0)
		   (cos (- angle (* 3 (/ pi 4)))))
		:-> a)
	   (add a result :-> result)))))

(defun steer-4 (image-list angle result)
  (with-local-viewables ((a (make-image (dimensions (car image-list))))
		      (b (make-image (dimensions (car image-list))))
		      (c (make-image (dimensions (car image-list))))
		      (d (make-image (dimensions (car image-list))))
		      (e (make-image (dimensions (car image-list))))
		      (f (make-image (dimensions (car image-list)))))
    (cond ((image-p angle)      
	   (sin. angle :-> a)
	   (cos. angle :-> b)
	   (square b :-> c);; cos squared
	   (square c :-> d);; cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d)
;;; - 2.4*Cos[theta]^2 + 3.2*Cos[theta]^4
	   (add d 0.2 :-> e);; coefficient of 0th image
	   (mul (nth 0 image-list) e :-> result) 

	   (add (mul b .809017 :-> d) (mul a .587785 :-> c) :-> f);; cos + sin
	   (square f :-> c);; sin + cos squared
	   (square c :-> d);; sin + cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d) 
	   (add d 0.2 :-> e);; coefficient of 1st image
	   (add (mul (nth 1 image-list) e :-> e) result :-> result)

	   (add (mul b 0.309017 :-> d) (mul a 0.951057 :-> c) :-> f);; cos + sin
	   (square f :-> c);; sin + cos squared
	   (square c :-> d);; sin + cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d) 
	   (add d 0.2 :-> e);; coefficient of 2nd image
	   (add (mul (nth 2 image-list) e :-> e) result :-> result)

	   (add (mul b -0.309017 :-> d) (mul a 0.951057 :-> c) :-> f);; cos + sin
	   (square f :-> c);; sin + cos squared
	   (square c :-> d);; sin + cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d) 
	   (add d 0.2 :-> e);; coefficient of 2nd image
	   (add (mul (nth 3 image-list) e :-> e) result :-> result)

	   (add (mul b -0.809017 :-> d) (mul a 0.587785 :-> c) :-> f);; cos + sin
	   (square f :-> c);; sin + cos squared
	   (square c :-> d);; sin + cos 4th
	   (add (mul c -2.4 :-> c) (mul d 3.2 :-> d) :-> d) 
	   (add d 0.2 :-> e);; coefficient of 2nd image
	   (add (mul (nth 4 image-list) e :-> e) result :-> result))
	  (t 
	   (mul (nth 0 image-list) 
		(+ 0.2 (* -2.4 (expt (cos angle) 2)) (* 3.2 (expt (cos angle) 4)))
		:-> result)
	   (mul (nth 1 image-list) 
		(+ 0.2 (* -2.4 (expt (cos (- angle (/ pi 5))) 2)) 
		   (* 3.2 (expt (cos (- angle (/ pi 5))) 4)))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 2 image-list) 
		(+ 0.2 (* -2.4 (expt (cos (- angle (* 2 (/ pi 5)))) 2)) 
		   (* 3.2 (expt (cos (- angle (* 2 (/ pi 5)))) 4)))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 3 image-list) 
		(+ 0.2 (* -2.4 (expt (cos (- angle (* 3 (/ pi 5)))) 2)) 
		   (* 3.2 (expt (cos (- angle (* 3 (/ pi 5)))) 4)))
		:-> a)
	   (add a result :-> result)
	   (mul (nth 4 image-list) 
		(+ 0.2 (* -2.4 (expt (cos (- angle (* 4 (/ pi 5)))) 2)) 
		   (* 3.2 (expt (cos (- angle (* 4 (/ pi 5)))) 4)))
		:-> a)
	   (add a result :-> result)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; QUADRATURE STEERABLE BASIS OBJECT

(DEF-simple-class quadrature-steerable-basis (viewable)
  (even-steerable-basis
   odd-steerable-basis)
  (:default-initargs :display-type nil))

;;; note: no display type for these guys for now
;;; look at the even- or odd-steerable-basis inferiors 
;;; to see the basis images.
;;; "image" can be an image, or a list of even and odd steerable bases.
(defun make-quadrature-steerable-basis 
    (image &rest initargs
	   &key (even-filters *default-even-steerable-filters*)
	   (odd-filters *default-odd-steerable-filters*)
	   name ->)
  (declare (ignore name))
  (when -> (setf (getf initargs :name) ->))
  (let* ((even-steerable (if (listp image)
			     (car image)
			     (make-steerable-basis image :filter-list even-filters)))
	 (odd-steerable (if (listp image)
			    (cadr image)
			    (make-steerable-basis image :filter-list odd-filters))))
    (with-result ((result nil)
		  `(:class quadrature-steerable-basis
		    :even-steerable-basis ,even-steerable
		    :odd-steerable-basis ,odd-steerable
		    ,@initargs)
		  'apply 'make-quadrature-steerable-basis initargs)
      result)))

(defmethod initialize-instance ((basis quadrature-steerable-basis) &rest initargs)
  (let ((even (getf initargs :even-steerable-basis))
	(odd  (getf initargs :odd-steerable-basis)))
    (unless (and even odd) (error "must provide even and odd steerable bases"))
    (call-next-method)
    (push basis (superiors-of even))
    (push basis (superiors-of odd))))

(defmethod inferiors-of ((basis quadrature-steerable-basis))
  (list (even-steerable-basis basis) (odd-steerable-basis basis)))

(defmethod dimensions ((steerable quadrature-steerable-basis))
  (dimensions (even-steerable-basis steerable)))

(defmethod order ((steerable quadrature-steerable-basis))
  (min (order (even-steerable-basis steerable))
       (order (odd-steerable-basis steerable))))

(defmethod x-dim ((steerable quadrature-steerable-basis))
  (x-dim (even-steerable-basis steerable)))

(defmethod y-dim ((steerable quadrature-steerable-basis))
  (y-dim (even-steerable-basis steerable)))

#| These seem unnecessary:
(defmethod steer-even ((steerable quadrature-steerable-basis) angle &key ->)
  (with-result ((res ->) (aref (data steerable) 0 0)
		'steer-even steerable angle)
    (steer (even-steerable-basis steerable) angle :-> res)))

(defmethod steer-odd ((steerable quadrature-steerable-basis) angle &key ->)
  (with-result ((res ->) (aref (data steerable) 0 0)
		'steer-odd steerable angle)
    (steer (odd-steerable-basis steerable) angle :-> res)))
|#

(defmethod steer ((steerable quadrature-steerable-basis) angle &key ->)
  (with-result ((result ->)
		(list :class 'image-pair
		      :image-list
		      (list (steer (odd-steerable-basis steerable) angle) 
			    (steer (even-steerable-basis steerable) angle)))
		'steer steerable angle)
    result))

#| Unnecessary
(defmethod sum-even ((steerable quadrature-steerable-basis) &key ((:-> result)))
  (sum (even-steerable-basis steerable) :-> result))

(defmethod sum-odd ((steerable quadrature-steerable-basis) &key ((:-> result)))
  (sum (odd-steerable-basis steerable) :-> result))
|#
    
(defmethod directional-magnitude ((steerable quadrature-steerable-basis) angle  &key ->)
  (with-result ((result ->) (aref (data (even-steerable-basis steerable)) 0 0)
		'directional-magnitude steerable angle)
    (with-local-viewables ((pair (steer steerable angle)))
      (magnitude pair :-> result))))

(defmethod directional-energy ((steerable quadrature-steerable-basis) angle &key ->)
  (with-result ((result ->) (aref (data (even-steerable-basis steerable)) 0 0)
		'directional-energy steerable angle)
    (with-local-viewables ((pair (steer steerable angle)))
      (square-magnitude pair :-> result))))

(defmethod directional-phase ((steerable quadrature-steerable-basis) angle &key ->)
  (with-result ((result ->) (aref (data (even-steerable-basis steerable)) 0 0)
		'directional-phase steerable angle)
    (with-local-viewables ((pair (steer steerable angle)))
      (complex-phase pair :-> result))))

;;; gives the integral over all angles of the orientated energy
(defmethod average-energy  ((qsb quadrature-steerable-basis) &key ->)
  (with-result ((result ->) (aref (data qsb) 0 0)
		'average-energy qsb)
    (cond ((and (= (order (even-steerable-basis qsb)) 2) 
		(= (order (odd-steerable-basis qsb)) 3)) 
	   (get-average-energy-2-3 (image-list (even-steerable-basis qsb))
				   (image-list (odd-steerable-basis qsb)) result))
	  (t (error "Only written for 2nd order basis")))
    result))

(defun get-average-energy-2-3 (even-list odd-list result)
  (with-local-viewables ((a (mul (nth 2 even-list) (nth 2 even-list)))
			 (b (mul (nth 0 even-list) (nth 0 even-list)))
			 (c (add a b))
			 (total (mul c 0.375)))  
    (mul (nth 0 odd-list) (nth 0 odd-list) :-> a)
    (mul (nth 3 odd-list) (nth 3 odd-list) :-> b)
    (add a b :-> c)
    (mul c 0.3125 :-> a)
    (add a total :-> total)
    (mul (nth 1 odd-list) (nth 1 odd-list) :-> a)
    (mul (nth 2 odd-list) (nth 2 odd-list) :-> b)
    (add a b :-> c)
    (mul c 0.5625 :-> a)
    (add a total :-> total)
    (mul (nth 0 odd-list) (nth 2 odd-list) :-> a)
    (mul (nth 1 odd-list) (nth 3 odd-list) :-> b)
    (add a b :-> c)
    (mul c 0.375 :-> a)
    (add a total :-> total)
    (mul (nth 1 even-list) (nth 1 even-list) :-> a)
    (mul a 0.5 :-> b)
    (add b total :-> total)
    (mul (nth 0 even-list) (nth 2 even-list) :-> a)
    (mul a 0.25 :-> b)
    (add b total :-> result)))

#|
;;; *** Bill, please write this ***  
(defmethod total-energy ((steerable quadrature-steerable-basis) &key ->)
  (with-result ((result ->) (aref (data steerable) 0 0)
		      'total-energy steerable)
    stuff))

;;; *** we also need to write these

(defmethod energy-vs-angle ((steerable quadrature-steerable-basis) j i
			    &key ->)
  ;; returns a discrete function
  )

(defmethod energy-peak ((steerable quadrature-steerable-basis)
			&key ->)
  ;; returns image-pair with vector-field display-type
  )
|#

;;; a steering method for filters.  This is more efficient sometimes than 
;;; applying each filter to an image, and then steering the images.  (if you
;;; only have one angle to which you are going to steer the results).
;;; This converts the filter-list to an image, steers the image, and then
;;; converts the result back to a filter.  
;;; It seems that convolutions applied with the same image size as the filter
;;; give spurious results.  You need a buffer of one pixel on a side.  The
;;; proper things is to go in and fix the filtering code, but I don't want to 
;;; touch that.  So make a bigger temporary image, and then crop it back down.
;;; Steer the filter to pi more than the actual angle, because you want to 
;;; transpose the filter when you apply it.
(defun steer-filter-list (filter-list angle &key ->)
  (with-result ((result ->)
		(car filter-list)
		'steer-filter-list)
    (let* ((dim (dimensions (car filter-list)))
	   (ydim (+ 2 (car dim)))
	   (xdim (+ 2 (cadr dim))))
      (with-local-viewables ((imp (make-impulse (list ydim xdim)))
			     (sb (make-steerable-basis imp :filter-list filter-list))
			     (filter-image (steer sb (+ pi angle)))
			     (cropped (crop filter-image :y 1 :x 1 :y-dim (- ydim 2) :x-dim (- xdim 2))))
	(copy (obvius::data cropped) :-> (kernel result))))
    result))

