;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;;  File: lucid-defsys.lisp
;;;  Author: Simoncelli/Heeger
;;;  Description: System definition file for OBVIUS in Lucid Common Lisp.
;;;  Creation Date:  Spring, 1988
;;;  Modified:  Fall, 1989
;;;  ----------------------------------------------------------------
;;;    Object-Based Vision and Image Understanding System (OBVIUS),
;;;      Copyright 1988, Vision Science Group,  Media Laboratory,
;;;              Massachusetts Institute of Technology.
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;; 1) Edit the global variables in the file site-paths.lisp.
;;; 2) Edit pathnames in the Makefile's in the <c-source> directory.
;;; 3) Edit site-init.lisp file to pre-load desired modules.
;;; 4) Start from a Lucid Lisp image, v4.0 or higher, with CLOS and
;;;    LispView loaded.  If you are in Emacs, do M-x load-file
;;;    "<obv>/emacs-source/cl-obvius", and then M-x run-cl.  Otherwise,
;;;    just run Lucid Common Lisp from the shell.
;;; 5) Load this file into your lisp world: (load "<obv>/lucid-defsys").
;;; 6) Before using OBVIUS, we recommend compiling all the files.  To do this,
;;;    execute (obv::load-obvius).  Then execute (quit), start up a new lisp,
;;;    and re-load lucid-defsys.
;;; 6) To run OBVIUS, execute (obvius::run-obvius).
;;; 7) To save OBVIUS as an executable lisp image, run (obvius::make-obvius). 
;;; 6) You may wish to create .obvius and/or .obvius-windows files in your
;;;    home directory for personal modifications.  The defaults are
;;;    <obv>/lisp-source/obvius-init.lisp and
;;;    <obv>/lisp-source/obvius-window-init.lisp.

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Package handling.

;;; Make sure this is LUCID Common Lisp, version 4.0
#-LCL4.0 
(eval-when (load compile eval)
  (error "This file is meant to be run in Lucid Lisp 4.0 only"))

;;; Create the OBVIUS package, making it use lisp, clos and lucid.
(in-package 'obvius 
	    :nicknames '(OBV)
	    :use
	    #+Sun '(LUCID-COMMON-LISP SYSTEM LISP CLOS)
	    #+Vax '(LUCID SYSTEM LISP CLOS))

;;; Export a few symbols that are defined in this file:
(export '(compile-load obv-compile-load obv-source-load obv-require *obvius-version*))

(when (find-package "FLAVORS")
      (unuse-package "FLAVORS"))   ;get rid of flavors 
(when (find-package "WINDOWS")
      (unuse-package "WINDOWS"))   ;get rid of Sun's window interface

(provide 'OBVIUS)		   ;Add OBVIUS to the *modules* list
(pushnew :OBVIUS *features*)	   ;Add OBVIUS to the *features* list

(use-package 'OBVIUS 'USER)	   ;USER package "uses" the OBVIUS package
(use-package 'CLOS 'USER)	   ;USER package "uses" the CLOS package

(defconstant *obvius-version* 2.2
  "OBVIUS version number.")
	     
;;; This value should be set appropriately for the current source
;;; code.  That is, it should hold a value one greater than the last
;;; patch file that has been incorporated into the source tree.  When
;;; the patches are loaded, this variable will be incremented to one
;;; greater than the number of the last patch file.  Note that this
;;; heppens when you make a new lisp world AND at runtime.  This
;;; allows available patches to be compiled into the world, and newer
;;; patches to be loaded at runtime.
(defvar *starting-patch-file* 1
  "This variable holds the current patch level (or, equivalently, the
number of the next patch file to load) of the OBVIUS environment in
which it is evaluated.  Note that this value is incremented by
load-obvius-patches."
)

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Lucid environment variables, and compiler extensions

;;#-PQC
;;(load (merge-pathnames "prodcomp" *lucid-path*))

;;; *** where do we do this? 
;(set-signal 8 (foreign-address-of 'fp-error)) ;catch floating point errors
  
;;; Production mode compiler optimizations.  These are used when
;;; compiling OBVIUS source.  NOTE: It seems that we MUST COMPILE
;;; LUCID-FFI.lisp in production mode for it to work!
(defvar *obvius-compiler-optimizations*
  '((COMPILATION-SPEED 0) (SPEED 3) (SAFETY 0) (SPACE 0)))

(defun current-compiler-optimizations ()
  (mapcar #'(lambda (pair) (list (car pair) (cdr pair)))
	  lucid::*compiler-optimizations*))

;;; Similar to Lucid's with-compiler-options, but works on the
;;; optimizations that can be passed to (proclaim (optimize ...))
;;; instead of on the keyword options to the compiler.  Evaluate
;;; LUCID::COMPILER-OPTIMIZATION-NAMES) for a list of these.  For
;;; example: '((speed 3) (compilation-speed 0)).  Note that
;;; compilation-speed=0 causes the production compiler to be used.
;;; Optimizations can also be nil, in which case this macro does
;;; nothing.
(defmacro with-compiler-optimizations (optimizations &body body)
  (let ((old-optimizations (gensym))
	(new-optimizations (gensym)))
    `(let ((,new-optimizations ,optimizations)
	   ,old-optimizations)
      (if ,new-optimizations
	  (unwind-protect
	       (progn
		 (setq ,old-optimizations (current-compiler-optimizations))
		 (proclaim (cons 'optimize ,new-optimizations))
		 ,@body)
	    (proclaim (cons 'optimize ,old-optimizations)))
	  (progn ,@body)))))

(def-foreign-function (set-umask (:language :c)
				 (:name "_umask")
				 (:return-type :signed-32bit))
    (number :signed-32bit))
(load-foreign-libraries '() '("-lc"))

;;; Set file write permissions.  Easiest to specify as octal (eg,
;;; #o002).  Nil arg means don't modify.
(defmacro with-umask (umask &body body)
  (let ((new-mask (gensym)) (old-mask (gensym)))
    `(let ((,new-mask ,umask)
	   ,old-mask)
      (if ,new-mask			;if nil, leave umask alone.
	  (unwind-protect
	       (progn (setq ,old-mask (set-umask ,new-mask))
		      ,@body)
	    (set-umask ,old-mask))
	  (progn ,@body)))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Stuff for loading and compiling OBVIUS files

;;; Load pathnames for this site:
(load (merge-pathnames "site-paths.lisp" lucid::*source-pathname*))

(defvar *obvius-features* nil
  "List of modules loaded using obv-require")

;;; Require a module for obvius.  Works like the Common Lisp require
;;; function, except it looks for pathnames in *obvius-module-plist*
;;; (defined below) and fills in missing directories in filenames with
;;; obvius source-path.  *** This is a little broken -- If user loads
;;; the file(s) manually, they will not go on the *obvius-features*
;;; list.
(defun obv-require (module-name
		    &key
		    (pathnames (or (getf *obvius-module-plist* module-name)
				   (string-downcase module-name)))
		    (initialize t))
  ;; Fill in default directory: look in *lisp-source-path*
  (setq pathnames
	(mapcar #'(lambda (path)
		    (merge-pathnames path *lisp-source-path*))
		(if (listp pathnames) pathnames (list pathnames))))
  (when (not (member module-name *obvius-features*))
    (dolist (file pathnames)
      (obv-compile-load file))
    (setq *obvius-features* (pushnew module-name *obvius-features*))
    (when initialize (run-initialization-functions))
    t))					;return t => files loaded

;;; A function to compile-load obvius source code that knows which
;;; directory to look in for source code, and which directory to
;;; compile into.  Also sets optimizations appropriately.  Also, bypasses
;;; the global *source-file-suffix*, which may not be correct for the
;;; OBVIUS source code.
(defun obv-compile-load (basename)
  (let ((*record-source-files* t))
    (declare (special *record-source-files*))
    (compile-load
     (merge-pathnames (merge-pathnames basename *lisp-source-path*) ".lisp")
     :output-file (merge-pathnames *binary-file-suffix*
				   (merge-pathnames *binary-path* basename))
     :umask #o000			;unprotected 
     :optimizations *obvius-compiler-optimizations*)))

;;; Call compile-if-necessary on OBVIUS source file, compiling into
;;; correct directory.  Also sets optimizations appropriately.
(defun obv-compile (basename)
  (let ((*record-source-files* t))
    (declare (special *record-source-files*))
    (compile-if-necessary
     (merge-pathnames (merge-pathnames basename *lisp-source-path*) ".lisp")
     :output-file (merge-pathnames *binary-file-suffix*
				   (merge-pathnames *binary-path* basename))
     :umask #o000			;unprotected
     :optimizations *obvius-compiler-optimizations*)))

(defun obv-source-load (basename)
  (let ((*record-source-files* t)) (declare (special *record-source-files*))
    (load (merge-pathnames
	   (merge-pathnames basename *lisp-source-path*) ".lisp"))))

(defun obv-binary-load (basename)
  (load (merge-pathnames *binary-file-suffix*
			 (merge-pathnames basename *binary-path*))))

;;; Load compiled version of source file, recompiling if the binary
;;; (compiled version) is not up-to-date or if it does not exist.
;;; Adds *source-file-suffix* to source-pathname if it is missing a
;;; suffix.  By default, assumes the binary lives in the same
;;; directory, with suffix matching *binary-file-suffix*.
(defun compile-load (source-pathname &rest compiler-options)
  "Load the compiled version of the source, compiling if the binary is out of
date or doesn't exist."
  (multiple-value-bind
	(path action)
      (apply 'compile-if-necessary source-pathname compiler-options)
    (values (load path) action)))

;;; This compiles source files if the binary (compiled version) is not
;;; up-to-date or if it does not exist.  Takes any kwywords that the
;;; compile-file function takes.  Additional keywords: 1) :umask
;;; keyword allows you to set the read/write/execute protection mask
;;; on the binary file, 2) :optimizations keyword that allow you to
;;; pass options that you would pass to (proclaim (optimize ...)), 3)
;;; :brief-optimize-message keyword that overrides the standard
;;; :optimize-message keyword and prints a one line message.  The
;;; function returns binary file pathname, and a non-nil second value
;;; if the file had to be compiled.  NOTE: We can't use Lucid:load
;;; since it doesn't allow separate binary pathname specification and
;;; wouldn't allow us to modify optimizations or umask.
(defun compile-if-necessary (source-pathname &rest compiler-options
					     &key
					     (output-file source-pathname)
					     (umask nil)
					     (optimizations nil)
					     (brief-optimize-message optimizations)
					     &allow-other-keys
					     &aux action)
  ;; Remove additional keywords, rest are passed to compile-file
  (remf compiler-options :umask)
  (remf compiler-options :optimizations)
  (remf compiler-options :brief-optimize-message)
  (when brief-optimize-message
    (setf (getf compiler-options :optimize-message) nil))
  #+(and Sun (not Sparc)) (setf (getf compiler-options :target) 'sun-68881)
  ;; Add suffix to source-pathname if not already there.
  (setq source-pathname (merge-pathnames source-pathname *source-file-suffix*))
  ;; Force output-file suffix to be *binary-file-suffix*, fill in any
  ;; holes from source-pathname.
  (setq output-file
	(merge-pathnames *binary-file-suffix*
			 (merge-pathnames output-file source-pathname)))
  (when (not (probe-file source-pathname))
    (error "Source file ~S does not exist" source-pathname))
  (when (or (not (probe-file output-file))
	    (> (file-write-date source-pathname)
	       (file-write-date output-file)))
    (setq action :compiled)
    (format t "~%;;; Compiling file ~S:~%" source-pathname)
    (with-compiler-optimizations optimizations
      (when brief-optimize-message
	(format t ";;; Compiler optimizations: ~A~%"
		(current-compiler-optimizations)))
      (with-umask umask
	(apply #'compile-file source-pathname :output-file output-file
	       compiler-options)))
    (format t ";;; Done compiling file ~S~%~%" source-pathname))
  (values output-file action))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Definition of the OBVIUS system and subsystems.

(defvar *obvius-system-files* 
  (list  #-Lucid "cl-loop"		;loop macro
	 #+Lucid "source-file-extensions" ;extensions for recording Method source files
	 "repl"
	 "clos-extensions" "lucid-ffi"
	 "generic-fns" "misc" "gaussian-noise"
	 "memory" "coord-xforms"
	 "viewable" "picture" "pane"
	 "array-ops" "matrix" "list-ops"
	 "lucid-image-loops"
	 "image" "bit-image"
	 "viewable-matrix" "viewable-sequence" "image-pair"
	 "discrete-function"
	 "filter"
	 ;;"param-filter"
	 "generic-ops" "user-macros" "imops" "fft" "warp" "synth"
	 "gray" "drawing"  "graph" "surface-plot"
	 "flipbook" "pasteup" "overlay"
	 "arrayio-c" "fileio" "hardcopy"
	 ))

(defvar *obvius-X-files*  
  (list
   "lv-image-patch"     ;LispView patch for faster image blt'ing (but it's still slow!)
   "lv-bucky-patch"	;LispView patch to catch bucky events for mouse doc
   "lv-color-patch"	;LispView patch to keep track of allocated colors
   "x-screen"
   "x-window"
   "x-blt"
   "x-draw"
   "x-mouse-utilities"
   "x-mouse"
   "x-dialogs"
   ))

;;; P-list of module symbols and filenames.  Note that filename can be
;;; a single file or a list of files.  Directory defaults to
;;; *lisp-source-path*.
(defvar *obvius-module-plist*
  (list :X-windows *obvius-X-files*
	:X-control-panel "x-control-panel"
	:color '("color-image" "color-picture")
	:gaussian-pyramid "gaussian-pyramid"
	:overlay "overlay"
	:contour-plot "contour-plot"
	:matrix '("svd" "qr" "regress" "row-ops")
	:statistics '("numerical-recipes" "statistics")
	:stepit "stepit"
	:simplex "simplex"
	:steerable '("steer" "steer-orient")
;;***	:qmf-pyramid "qmf-pyramid"
;;***   :kalman
;;***   :Canny
;;***   :HIPS  ;Mike Landy's image processing package
	))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Foreign (C) libraries.

;;; Runs a "make" on the foreign code.  Returns a value which indicates 
;;; whether the libraries were remade and need to be loaded again.
;;; this results in new libraries and executables which are placed at 
;;; their respective places.

(defun update-C-libraries ()
  (let ((dir (LCL:pwd)))
    (LCL:cd *c-source-path*)
    (multiple-value-bind (x y return-value)
	#+HP  (LCL:run-program "make" :arguments '("-f" "Makefile-hp" "all"))
	#+Vax (LCL:run-program "make" :arguments '("-f" "Makefile-vax"))
	#+(and Sun Sparc) (LCL:run-program "make" :arguments '("-f" "Makefile-sun4"))
	#+(and Sun (not Sparc)) (LCL:run-program "make" :arguments '("-f" "Makefile-sun3"))
      (declare (ignore x y))
      (LCL:cd dir)
      (not (= return-value 0)))))

;;; This is put on the *initialization-functions* list to be executed at startup.
;;; See lucid-ffi.lisp.
(defun load-C-libraries ()
  (let ((dir (LCL:pwd))
	(libs (if (probe-file "/dev/fpa") 
		  (list "fpaobv.a" "-lm" "-lc") 
		  (list "obv.a" "-lm" "-lc"))))
    (LCL:cd *binary-path*)
    (LCL:load-foreign-files nil libs)
    (LCL:cd dir)))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Loading and Running OBVIUS starting from an empty Lisp World.

;;; Starting from an empty lisp world, this loads all of the obvius code,
;;; compiling when necessary, and then initializes the system.
(defun run-obvius ()
  (load-obvius)				;load code, compiling if necessary
  (optimize-clos-calls)
  (initialize-obvius))			;initialize obvius

;;; This function loads all of the obvius lisp code, compiling when
;;; necessary.  It also compiles the C code into a library which will
;;; be loaded by initialize-obvius at run-time.  It does not
;;; initialize OBVIUS!
(defun load-obvius ()
  (format t "~%;;; Loading OBVIUS system files ...~%")
  (unless (probe-file *binary-path*)
    (LCL:run-program "mkdir" :arguments
		     (list (string-right-trim '(#\/) obv::*binary-path*))))    
  (update-C-libraries)
  (mapc 'obv-compile-load *obvius-system-files*)
  (let ((site-init (merge-pathnames "site-init.lisp" *obvius-directory-path*)))
    (if (probe-file site-init)
	(load site-init)
	(warn "Cannot find file ~S" site-init)))
  (format t ";;; Done loading OBVIUS system files.~%~%"))

;;; Load files from the patches directory with names of the form
;;; patch-<num>.lisp in numerical order (according to <num>), starting
;;; with the one designated by the variable *starting-patch-file*.
;;; Increments this variable to be one more than the last patch number
;;; loaded.  *** wasteful re-parsing here....
(defun load-obvius-patches ()
  (declare (special *obvius-directory-path* *source-file-suffix*
		    *starting-patch-file*))
  (let* ((dir (pathname (concatenate 'string *obvius-directory-path* "patches/")))
	 (prefix "patch-")
	 (len (length prefix))
	 (patch-files (directory (merge-pathnames dir *source-file-suffix*)))
	 (*redefinition-action* nil))	;get rid of redefinition-warnings
    (declare (special *redefinition-action*))
    ;; delete filenames that are not of the form "patch-<num>.lisp", where
    ;; <num> is greater than *starting-patch-file*.
    (setq patch-files
	  (delete-if-not
	   #'(lambda (name &aux num)
	       (and (> (length name) len)
		    (string= name prefix :end1 len)
		    (setq num (parse-integer name :start len :junk-allowed t))
		    (>= num *starting-patch-file*)))
	   patch-files
	   :key #'pathname-name))
    ;; sort files according to <num>:
    (setq patch-files
	  (sort patch-files #'<
		:key #'(lambda (p)
			 (parse-integer (pathname-name p) :start len :junk-allowed t))))
    ;; Compile and load the patch files:
    (dolist (file patch-files)
      (obv-compile-load file)
      (setq *starting-patch-file*
	    (1+ (parse-integer (pathname-name file) :start len :junk-allowed t))))
    *starting-patch-file*))

(defun compile-obvius-modules ()
  (loop for plist = *obvius-module-plist* then (cddr plist)
	until (null plist)
	for files = (cadr plist)
	do
	(if (consp files)
	    (mapc #'(lambda (f) (obv-compile f)) files)
	    (obv-compile files))))

(defun optimize-clos-calls ()
  (format t "~%;;; Compiling CLOS generic-functions and dispatch code ...~%")
  (with-compiler-optimizations *obvius-compiler-optimizations*
    (lcl::precompile-generic-functions)
    ;; ** Used to give lucid::STRUCTURE-IS-STANDARD-OBJECT-P not bound errors:
    (lcl::compile-all-dispatch-code)))

;;; This function must be called to initialize obvius.  It loads patch
;;; files that are not already loaded, loads the foreign (C)
;;; libraries, initializes the window system, loads the user's startup
;;; file and starts the OBVIUS read-eval-print loop (repl).  It is
;;; called by run-obvius (if running from an empty world) and
;;; startup-internal (if running from a world containing OBVIUS).
;;; NOTE: patches and the window-init files are loaded BEFORE
;;; initializing the window system.  We load the C libraries at run
;;; time because otherwise it is hard to patch changes to the C code.
;;; *** Should use the lisp function user-homedir-pathname instead of
;;; "~/".
(defun initialize-obvius ()
  (format t "~%;;; >>>---  Initializing OBVIUS version ~A  ---<<<~%~%" *obvius-version*)
  (load-C-libraries)   ;*** should this be done as an initialization from lucid-ffi.lisp?
  (load-obvius-patches)
  (let ((site-patches (merge-pathnames "site-patches.lisp" *obvius-directory-path*)))
    (if (probe-file site-patches)
	(obv-compile-load site-patches)
	(warn "Cannot find file ~S" site-patches)))
  (cond ((probe-file "~/.obvius-windows") (load "~/.obvius-windows"))
	((probe-file "~/obvius-window-init.lisp") (load "~/obvius-window-init.lisp"))
	(t  (obv-source-load "obvius-window-init")))
  (run-initialization-functions)	;window system initializations
  (push-onto-eval-queue
   (cond ((probe-file "~/.obvius") '(load "~/.obvius"))
	 ((probe-file "~/obvius-init.lisp") '(load "~/obvius-init.lisp"))
	 (t '(obv-source-load "obvius-init"))))
  (push-onto-eval-queue '(run-initialization-functions)) ;Other initializations
  (repl))

;;; Code in various files puts functions (or symbols) on this list.
;;; They will be eval'ed by initialize-obvius at startup.
;;; *** should probably do this with a lexical closure (hide the list).
(defvar *initialization-functions* nil)

;;; Run the functions on the *initialization-functions* list in
;;; reverse order (assume that they were added to the list using
;;; push or pushnew).
(defun run-initialization-functions ()
  (let (list-o-functions)
    ;; Do this to make sure multiple processes don't bash...
    (psetq list-o-functions *initialization-functions*
	   *initialization-functions* nil)
    (mapc #'(lambda (func) (funcall func)) (nreverse list-o-functions))))

;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;;; Building and saving a Lisp World containing OBVIUS.

;;; Save a lisp-world containing obvius.  Note that existing patch
;;; files are loaded before saving the world.  Also compiles all
;;; modules, so that they can be quickly loaded at runtime.
(defun make-obvius (&optional filename)
  (unless filename
    (setq filename (format nil "~Aobvius-~A" *obvius-image-path* *obvius-version*)))
  (load-obvius)
  (load-obvius-patches)
  (optimize-clos-calls)
  (save-lisp-world filename)
  (compile-obvius-modules))

;;; This function saves the current lisp world into the given pathname.
;;; *** get rid of the magic numbers!
(defun save-lisp-world (pathname)
  (in-package 'user)			;make sure it's saved in 'user
  (format t "~%;;; ---- Saving lisp executable ~A ---- ;;;" pathname)
  (when (probe-file pathname)
    (format t "~%;;; ---- It will overwrite the previous executable ---- ;;;"))
  (if (y-or-n-p "~%;;; Do you want to continue? ")
      (disksave pathname
		:full-gc t
		:reserved-free-segments 128  ;8 Meg for static heaps
		:dynamic-free-segments  128   ;8 Meg for dynamic space
		:verbose T)
      (format t "~%;;; ---- ABORTED ---- ;;;")))

	
;;;; Define Lisp Startup hook to initialize obvius:

(defvar user::*enter-top-level-hook* nil)
(defvar *pre-obvius-start-hook* user::*enter-top-level-hook*)
(setq user::*enter-top-level-hook* 'obvius-top-level-hook)

;;; Set hook so that initialize-obvius is called automatically when
;;; the lisp world is loaded.  Note that the function bound to
;;; *enter-top-level-hook* must be compiled to work properly.
(defun obvius-top-level-hook ()
  (when *pre-obvius-start-hook*
    (funcall *pre-obvius-start-hook*))
  (initialize-obvius))

(unless (compiled-function-p (symbol-function 'obvius-top-level-hook))
  (compile 'obvius-top-level-hook))


;;; Local Variables:
;;; buffer-read-only: t 
;;; End:
