function zoomline(ga,newx,newy)
%ZOOMLINE Replace "zoomed" line in ZOOMTOOL.
%       ZOOMLINE(H,YDATA) replaces the "zoomed" line with YDATA in
%       the ZOOMTOOL active in the axes pointed to by the handle
%       H.  The horizontal scaling remains the same, starting
%       from zero.
%
%       ZOOMLINE(H,XDATA,YDATA) replaces the "zoomed" line with
%       YDATA using XDATA for the horizontal scaling.
%
%       This function make low-level graphic calls to replace
%       the 'zoomed' line in ZOOMTOOL without destroying the
%       cursors and other objects which maybe contained in the
%       data.  There should only be the 'zoomed' line in the
%       ZOOMTOOL (see Toggle property of ZOOMTOOL).  The low-
%       level nature of this function saves is quicker and
%       smoother than successive calls to ZOOMCLR, PLOT, and
%       ZOOMTOOL which would accomplish the same task.
%
%       All the necessary changes are made to adapt ZOOMTOOL to
%       the new line are made dependent upon the calling form.
%       For example, when only the YDATA is given, the data
%       displyed in the axes limits is still visible even more
%       points may have been added or some taken away as happens
%       with cut and past operations.  When called with both
%       XDATA and YDATA, ZOOMTOOL is essentially reinitialized.
%
%       See also ZOOMTOOL, WORKMENU, ZOOMCLR, ZOOMEDIT

%       Dennis W. Brown 1-31-94
%       Copyright (c) 1994 by Dennis W. Brown
%       May be freely distributed.
%       Not for use in commercial products.

% arg check
if nargin < 2 | nargin > 3,
	error('zoomline: Invalid number of input arguments.');
end;

% parent figure
gf = get(ga,'Parent');

% get handles to zoomtool uicontrols
handles = get(finduitx(gf,'zoomtool'),'UserData');
xfactor = handles(28);
xmin = handles(33);
xxlim = get(ga,'XLim');

% compute indices currently at the limits
xind = round((xxlim - xmin) / xfactor) + 1;

% cursor indices before
cv1 = get(handles(24),'XData'); cv1 = cv1(1);
cv2 = get(handles(26),'XData'); cv2 = cv2(1);
i1 = round((cv1(1) - xmin) / xfactor) + 1;
i2 = round((cv2(1) - xmin) / xfactor) + 1;

if nargin == 2,

	% create new ydata from old
	newy = newx;
	newx = (0:(length(newy)-1))' * xfactor + xmin;

	% get length now
	ylen = length(newy);

else,

	% get length now
	ylen = length(newy);

	if ylen ~= length(newx),
		error('zoomline: New XData and YData not equal length.');
	end;
end;


% redraw line
set(handles(39),'XData',newx,'YData',newy);

% new handle data
handles(33) = newx(1);
handles(34) = newx(length(newx));
handles(28) = (handles(34) - handles(33)) / (ylen - 1);
handles(35) = min(newy);
handles(36) = max(newy);
set(finduitx(gf,'zoomtool'),'UserData',handles);

% set limits so that what was at the limits remains as close as possible 
% to that as before
xindbefore = xind;
if xfactor ~= handles(28),
	% lengths changed
	xind(1) = 1;
	xind(2) = ylen;
end;
xlim = newx(xind);
set(ga,'XLim',xlim);

% adjust readouts and move cursors 
if i1 > ylen, i1 = xind(1)+fix((xind(2)-xind(1))/4); end;
if i2 > ylen, i2 = xind(1)+3*fix((xind(2)-xind(1))/4); end;
cv1 = newx(i1);
ch1 = newy(i1);
cv2 = newx(i2);
ch2 = newy(i2);
set(handles(18),'String',num2str(cv1));
set(handles(19),'String',num2str(ch1));
set(handles(20),'String',num2str(abs(cv2-cv1)));
set(handles(21),'String',num2str(abs(ch2-ch1)));
set(handles(22),'String',num2str(cv2));
set(handles(23),'String',num2str(ch2));
set(handles(24),'XData',[cv1 cv1]);
set(handles(25),'YData',[ch1 ch1]);
set(handles(26),'XData',[cv2 cv2]);
set(handles(27),'YData',[ch2 ch2]);

% set Y limits
ymax = max(newy(xind(1):xind(2)));
ymin = min(newy(xind(1):xind(2)));
del = (ymax - ymin) * 0.05;
if del == 0; del = 1; end;
ylim = [ymin-del ymax+del];
set(ga,'YLim',ylim);

if xindbefore(1) ~= xind(1) | xindbefore(2) ~= xind(2),

	% different lengths than before

	% invalidate any prior zoom locations and adjust full
	set(handles(2),'UserData',[handles(33) handles(34)]);
	set(handles(3),'UserData',[handles(33) handles(34)]);
	del = (handles(36) - handles(35)) * 0.05;
	if del == 0, del == 1; end;
	set(handles(5),'UserData',[handles(35)-del handles(36)+del]);
	set(handles(6),'UserData',[handles(35)-del handles(36)+del]);
end;

% adjust cursor lengths
set(handles(24),'YData',ylim);
set(handles(25),'XData',xlim);
set(handles(26),'YData',ylim);
set(handles(27),'XData',xlim);

