/*
 * $Id: segmentlist.c,v 1.17 1993/05/07 18:12:22 johans Exp $
 *
 */

/* standard C library include file directives */
#include <sys/types.h>
#include <stdio.h>
#include <stdlib.h>
#include <fcntl.h>
#include <malloc.h>
#include <string.h>

/* speech library include file directive */
#include <speech.h>

/* special segmenter labels */
#define Unknown         "UNK"
#define Head            "HEAD"

#define BIGBUF		(1024)

extern char *sys_errlist[];
extern int sys_nerr;
extern int errno;

/* local module header declarations */

/*
 * ReadSegmentList( file, seg, ms )
 *
 * 
 * parameters:
 *	file - charater pointer to a NULL terminated string
 *	seg - pointer to a pointer to a segment list
 *	ms - poitner to a float.
 *
 * returns:
 *	a positive numer on suxxess, otherwise a
 *	negative number is returner and ErrorString is set.
 *
 */

int ReadSegmentList( char *file, SegmentList **seg, float *ms )
{
  double atof();
  AsciiHdr *hdr;
  SegmentList *tsg;
  char *tmp;
  float tms;

  if( ReadSegmentListAndHeader( file, &tsg, &hdr ) < 0 ) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Read Segment List failed";

    return( -1 );
  }

  /* extract time resolution */
  tmp = FindHeaderVal( hdr, "MillisecondsPerFrame" );
  if( tmp == NULL ) {
    tms = 3.0;	/* assume 3.0 ms frames */
  } else {
    tms = (float)atof( tmp );
  }
  
  FreeHeader( hdr );
  
  *seg = tsg;
  *ms = tms;
  
  return( 1 );
}


/*
 * ReadSegmentListAndHeader( file, seg, hdr )
 *
 *
 * parameters:
 *	file - charater pointer to a NULL terminated string
 *	seg - pointer to a pointer to a segment list
 *	hdr - pointer to a pointer to a header
 *
 * returns:
 *	a positive number on success, otherwise a
 *	negative number is returned and ErrorString is set.
 *
 */

int ReadSegmentListAndHeader( char *file, SegmentList **seg, AsciiHdr **hdr )
{
  AsciiHdr *thdr;
  char tlabel[BIGBUF];
  char buf[BIGBUF];
  char oneline[80];
  SegmentList *tmp;
  SegmentList *sgl;
  int begin;
  int end;
  int c;
  FILE *fp;

  /* open lola file */
  if( ( fp = fopen( file, "r" ) ) == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "fopen failed";
    }
    return( -1 );
  }

  /* read lola header */
  thdr = ReadHeader( fp );
  if( thdr == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "Not a Lola file";
    }
    return( -1 );
  }

  /* form segmentation linked list */
  sgl = NewSegmentList();
  if( sgl == NULL ) {
    FreeHeader( thdr );
    fclose( fp );
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "not a lola file";
    }
    return( -1 );
  }

  /* scan through file and extract labels */
  while( fgets( buf, sizeof(buf), fp ) != NULL ) {
    
    c = sscanf( buf, "%d %d %s", &begin, &end, tlabel );
    if( c == 2 ) {
      
      strcpy( tlabel, "<no label>" );
      
    } else if( c != 3 ) {
      ErrorString = "lola file format error";
      FreeHeader( thdr );
      FreeSegmentList( sgl );
      fclose( fp );
      return( -1 );
    }
    
    if( AddSegment( sgl, begin, end, tlabel ) < 0 ) {
      FreeHeader( thdr );
      FreeSegmentList( sgl );
      fclose( fp );
      return( -1 );
    }
    
  }
  
  fclose( fp );
  
  *hdr = thdr;
  *seg = sgl;
  
  return( 1 );
}


/*
 * WriteSegmentList( file, seg, ms )
 *
 *
 * parameters:
 *	file - charater pointer to a NULL terminated string
 *	seg - pointer to segment list
 *	ms - float (time in msec per frame)
 *
 * returns:
 *	a positive number if write was successful, otherwise
 *	a negative number is returned and ErrorString is set.
 *
 */

int WriteSegmentList( char *file, SegmentList *seg, float ms )
{
  AsciiHdr hdr[2];
  float ftmp;
  
  ftmp = ms;
  hdr[0].ah_key = "MillisecondsPerFrame";
  hdr[0].ah_vlen = 1;
  hdr[0].ah_value = FloatArray2StrArray( &ftmp, 1 );
  
  hdr[1].ah_key = NULL;
  hdr[1].ah_vlen = 0;
  hdr[1].ah_value = NULL;
  
  
  return( WriteSegmentListAndHeader( file, seg, hdr ) );
  
}


/*
 * WriteSegmentListAndHeader( file, seg, hdr )
 *
 *
 * parameters:
 *	file - charater pointer to a NULL terminated string
 *	seg - pointer to segment list
 *	hdr - pointer to header
 *
 * returns:
 *	a positive number if write was successful, otherwise
 *	a negative number is returned and ErrorString is set.
 */

int WriteSegmentListAndHeader( char *file, SegmentList *seg, AsciiHdr *hdr )
{
  SegmentList *tmp;
  mode_t oumask;
  FILE *fp;
  int fd;
  

  fd = open( file, O_WRONLY | O_CREAT | O_TRUNC, 0200 );
  
  if( fd < 0 ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "open failed";
    }
    return( -1 );
  }
  
  if( ( fp = fdopen( fd, "w" ) ) == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "fdopen failed";
    }
    return( -1 );
  }
  
  if( WriteHeader( fp, hdr ) < 0 ) {
    fclose( fp );
    return( -1 );
  }
  
  tmp = seg->sl_forward;

  while( tmp != NULL ) {
    
    if( fprintf( fp, "%d %d %s\n", tmp->sl_begin, 
		tmp->sl_end, tmp->sl_label ) == EOF ) {
      
      if( errno < sys_nerr ) {
	ErrorString = sys_errlist[errno];
      } else {
	ErrorString = "fprintf failed";
      }
      fclose( fp );
      return( -1 );
    }

    tmp = tmp->sl_forward;
  }
  
  if( fclose( fp ) == EOF ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "fclose failed";
    }
    return( -1 );
  }
  
  oumask = umask(022);
  umask(oumask);
  if( chmod( file, 0666 & ~ oumask) == -1 ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "chmod failed";
    }
    return( -1 );
  }
  
  
  return( 1 );
}


/*
 * NewSegmentList()
 *
 *   Creates and initalizes a new segment list.
 *
 *  parameters:
 *	
 *  returns:
 *	a pointer to and empty segment list if successful,
 *	otherwise NULL and ErrorString is set.
 */

SegmentList *NewSegmentList()
{
	SegmentList *tmp;

	
	tmp = NewSegment();
	if( tmp == NULL ) {
		return( NULL );
	}
	LabelSegment(tmp,"HEAD");

	return( tmp );
}

/*
 * FreeSegmentList( seg )
 *
 *  Frees space used by seg.
 *
 * parameters:
 *	seg - pointer to a segment list.
 *
 * returns:
 *
 */

void FreeSegmentList( SegmentList *seg )
{
	SegmentList *cur;
	SegmentList *tmp;

	cur = seg->sl_forward;

	while( cur != NULL ) {
		tmp = cur;
		cur = cur->sl_forward;
		FreeSegment( tmp );
	}

	free( seg );

}


/*
 * AddSegment( seg, begin, end, label )
 *
 *
 * parameters:
 *	seg - pointer to segment list
 *	begin - begining frame index
 *	end - ending frame index
 *	label - character pointer to a null terminated string.
 *
 * returns:
 *	a positive number if add was successful, 
 *	otherwise a negitive number is returned 
 *	and ErrorString	is set.
 */

int AddSegment( SegmentList *seg, int begin, int end, char *label )
{
	SegmentList *slot;
	SegmentList *tmp;

	slot = FindSegment( seg, begin );

	tmp = NewSegment();

	tmp->sl_label = (char *)malloc( strlen(label) + 1 );
	if( tmp->sl_label == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		return( -1 );
	}
	strcpy( tmp->sl_label, label );

	tmp->sl_begin = begin;
	tmp->sl_end = end;

	InsertSegment( slot, tmp );

	return( 1 );
}


/*
 * DeleteSegment( seg );
 *
 * parameters:
 *	seg - pointer to segment to delete;
 *
 * returns:
 *
 */

void DeleteSegment( SegmentList *seg )
{
	SegmentList *tmp;


	seg->sl_backward->sl_forward = seg->sl_forward;

	/* 
	 * tring to delete the end element?? 
	 */
	if( seg->sl_forward != NULL ) {
		seg->sl_forward->sl_backward = seg->sl_backward;
	}

	FreeSegment( seg );
}


/*
 * FindLastSegment( seg )
 *
 *  returns a pointer to the last segment.
 *
 * parameters:
 *	seg - pointer to segmentlist
 *
 * returns:
 *	a pointer to the last segment on success, otherwise
 *	a NULL is returned if the segmentlist is empty.
 */

SegmentList *FindLastSegment( SegmentList *seg )
{
	SegmentList *tmp;

	
	if( seg->sl_forward == NULL ) {
		return( NULL );
	}

	tmp = seg;

	while( tmp->sl_forward != NULL ) {
		tmp = tmp->sl_forward;
	}

	return( tmp );
}


/*
 * LabelSegment( seg, label )
 *
 *  sets seg segment label to label.
 *
 * parameters:
 *	seg - pointer to segment
 *	label - pointer to NULL terminated character string
 *
 * returns:
 *	a positive number if successful, otherwise
 *	a negative number is returned and ErrorString()
 *	is set.
 */

int LabelSegment( SegmentList *seg, char *label )
{
	
	if( seg->sl_label != NULL ) {
		free( seg->sl_label );
	}

	seg->sl_label = (char *)malloc( strlen( label ) + 1 );

	if( seg->sl_label == NULL ) {
		if( errno < sys_nerr ) {
			ErrorString = sys_errlist[errno];
		} else {
			ErrorString = "malloc failed";
		}
		return( -1 );
	}

	strcpy( seg->sl_label, label );

	return( 1 );
}


/*
 * SegmentListHead( seg )
 *
 *   given a pointer to any segment in the list, return the
 *   head of the segment list.
 *
 * parameters:
 *	seg - pointer in to the segment list.
 *	
 * returns:
 *	poitner to head of the segmentlist or NULL
 *	in segment list is empty.
 */

SegmentList *SegmentListHead( SegmentList *seg )
{

	if( seg == NULL ) {
		return( NULL );
	}

	while( seg->sl_backward != NULL ) {
		seg = seg->sl_backward;
	}

	return( seg );
}


static void InsertSegment( SegmentList *slot, SegmentList *seg )
{
	if(slot->sl_forward != NULL ) {
		seg->sl_forward = slot->sl_forward;
		slot->sl_forward->sl_backward = seg;
	}

	slot->sl_forward = seg;
	seg->sl_backward = slot;
}


/*
 * FindSegment( seg, begin )
 *
 *  find segment closest to begin frame.
 *
 * parameters:
 *	seg - pointer to segment list
 *	begin - int (frame number)
 *
 * returns: 
 *	a pointer to the nearest segment or NULL 
 *	is segmentlist is empty.
 */

SegmentList *FindSegment( SegmentList *seg, int begin )
{
	SegmentList *cur;
	SegmentList *last;

	cur = seg->sl_forward;
	last = seg;

	if( cur == NULL ) {
		return( seg );
	}

	while( cur != NULL ) {

		if( cur->sl_begin > begin ) {
			break;
		}
		last = cur;
		cur = cur->sl_forward;
	}

	return( last );

}


/* 
 * FindNearestSegment( seg, frame )
 *
 *   Returns a pointer to the nearest segment to frame.
 *   Nearest is defined as nearest border.    
 *   This function is used in Lyre when grabbing a boundary..
 *
 * parameters:
 *	seg - pointer to segment list.
 *	frame - int.
 *
 * returns:
 *	a pointer to a segment or NULL if
 *	segment list if empty.
 */

SegmentList *FindNearestSegment( SegmentList *seg, int frame )
{
  SegmentList *tsl;
  SegmentList *ret;
  float tmp, BestSoFar = 1000000.;
  
  if( seg->sl_forward == NULL )
    {
      return( NULL );
    }
  
  tsl = seg->sl_forward;
  
  while( tsl != NULL )
    {
      
      tmp = abs( frame - tsl->sl_begin );
      if(frame < tsl->sl_begin) tmp += 0.1; /* bias tie away from this */
      if( tmp <= BestSoFar )
	{
	  ret = tsl;
	  BestSoFar = tmp;
	}
      
      tmp = abs( frame - tsl->sl_end );
      if(frame > tsl->sl_end) tmp += 0.1; /* bias tie away from this */
      if( tmp <= BestSoFar )
	{
	  ret = tsl;
	  BestSoFar = tmp;
	}
      
      tsl = tsl->sl_forward;
    }
  
  
  return( ret );
  
}


/*
 * SegmenList *NewSegment()
 *
 * Create A new segment in the linked list
 *
 */

static SegmentList *NewSegment() 
{
  SegmentList *tmp;


  tmp = (SegmentList *)malloc( sizeof(SegmentList) );
  
  if( tmp == NULL ) {
    if( errno < sys_nerr ) {
      ErrorString = sys_errlist[errno];
    } else {
      ErrorString = "malloc failed";
    }
    return( NULL );
  }
  
  tmp->sl_label = NULL;
  tmp->sl_begin = -1;
  tmp->sl_end = -1;
  tmp->sl_forward = NULL;
  tmp->sl_backward = NULL;
  
  return( tmp );
}


/*
 * FreeSegment(seg)
 *
 * Free allocated memory for particular segment
 *
 * seg (in): pointer to segment in linked list
 *
 */

static void FreeSegment( SegmentList *seg )
{
	if( seg->sl_label != NULL ) {
		free( seg->sl_label );
	}
	free( seg );
}
