/*
 * melcep.c
 *
 * This module computes the mel frequency cepstra of a speech signal
 * based on the following algorithm.
 *
 * Given a window width of 256 Samples, Compute
 *
 * 1. The magnitude of 128 DFT coefficients
 * 2. Compute 40 weighted averages using the triangle function
 *    spaced according to the Mel Scale.
 * 3. Compute the log of the result.
 * 4. Compute the first 12 inverse discrete cosine transform
 *    DCT coefficients.
 *
 */

/* include file directives */
#include <stdio.h>
#include <stdlib.h>
#include <math.h>
#include <speech.h>

#define sqr(x) ((x) * (x))


/* external variable declarations */
extern int errno, sys_nerr;
extern char *sys_errlist[];


/*
 * void CalculateMelBands(fstart, fend, NumBands, Bands)
 *
 * Calculates the energy bands according to the mel scale
 * frequency representation.
 *
 * fstart   (in): frequency to start calculation
 * fend     (in): end point of mel scale frequencies
 * NumBands (in): Number of inputs bands in scale transformation
 * Bands   (out): pointer to Mel Scale frequency bands (of type float)
 *
 */

void CalculateMelBands(float fstart, float fend, int NumBands, float *Bands)
{
  float fmelstart, fmelend, step, factor;
  int   i;

  fmelstart = (float) 2595.0*log10(1.0 + (double) (fstart/700.0));
  fmelend   = (float) 2595.0*log10(1.0 + (double) (fend/700.0));
  step      = (fmelend - fmelstart)/((float)NumBands);

  for (i = 0; i <= NumBands; i++) {
    factor  = fmelstart + i*step;
    factor /= 2595.0;
    factor = (float) (pow(10.0, (double) factor));
    Bands[i] = (factor - 1.0)*700.0;
  }
}


/*
 * void MelIntBands(NumBands, Samples, Bands, IntBands)
 *
 * This procedure calculates the integer values in the FFT amplitude
 * spectrum which corresponds to the frequencies in the mel bands.
 * 
 * NumBands (in): Number of bands in the mel scale
 * Samples  (in): Number of samples in FFT ampitude spectrum
 * Bands    (in): Pointer to Frequency bands calculated previously
 * IntBands(out): Poitner to output integer band index values
 *
 */

void MelIntBands(int NumBands, int Samples, float *Bands, int *IntBands)
{
  float Range, step;
  int   i;

  Range = Bands[NumBands] - Bands[0];
  step  = ((float)Samples)/Range;

  for (i=0; i<= NumBands; i++) 
    IntBands[i] = (int) ((Bands[i] - Bands[0])*step + 0.5);
}

    
/*
 * void TriangleWeights(NumBands, Samples, IntBands, Weightings)
 *
 * This procedure calculates the mel scale triangle filter weighting
 * values, used for the spectrak averaging process.
 *
 * NumBands (in): Total number of melscale frequency bands
 * Samples  (in): Corresponding number of samples (size of weightings)
 * IntBands (in): Pointer to index position of mel bands
 * Weigtings(out): Pointer to array of triangular spectral weights
 *
 */

void TriangleWeights(int NumBands, int Samples, int *IntBands, 
		     float *Weightings)
{
  float gradient;
  int   start, stop, middel;
  int   i, j, k;

  for (i=0; i<NumBands-1; i++) {
    start = IntBands[i];
    stop  = IntBands[i+1];
    middel = (stop + start)/2 + 1;
    gradient = (1.0)/((float) (middel - start));
    for (j=start; j<middel; j++) 
      Weightings[j] = gradient*((float) (j-start));
    for (j=middel; j<stop; j++) 
      Weightings[j] = gradient*(stop-j);
  }
}
    

/*
 * SquareWeights(NumBands, Samples, IntBands, Weightings)
 *
 * Initializes the weighting array for a square filter
 *
 * NumBands (in): Total number of melscale frequency bands
 * Samples  (in): Corresponding number of samples (size of weightings)
 * IntBands (in): Pointer to index position of mel bands
 * Weigtings(out): Pointer to array of triangular spectral weights
 *
 */

void SquareWeights(int NumBands, int Samples, int *IntBands, float *Weightins)
{
  int i;

  for (i=0; i<Samples; i++)
    Weightins[i] = 1.0;
}


/*
 * void Normalize_Vector(vector, size)
 *
 * Normalizes incoming vector according to maximum value in vector.
 *
 * vector (in/out): vector of length (size)
 * size       (in): length of vector
 *
 */

void Normalize_Vector(float *vector, int size)
{
  int   i;
  float max;

  max = -1.0;
  for (i=0; i<size; i++) 
    if (vector[i]>max) 
      max = vector[i];

  for (i=0; i<size; i++)
    vector[i] /= max;
}
      

/* 
 * int MelCepstrum(speech, length, window, step, fSamp, Nfilters
 *                 MelCep, n)
 *
 * This function determines the first n mel cepstral coefficients
 * based on the mel cepstal analyses using trianlge filter function
 *
 * speech (in): pointer to incoming speech signal of type short (16 bits)
 * length (in): number of samples in speech signal.
 * window (in): length of sampling window in milliseconds
 * step   (in): stepsize of feature window in milliseconds
 * fSamp  (in): sampling frequency in Hz
 * Nfilter(in): number of traingle filter for frequency weighting
 * MelCep(out): output mel cepstral frequency coefficients
 * n      (in): number of mel frequenct coefficients requested
 *
 * On exits returns the number of frames calculated, or (-1) for
 * error.
 *
 */

int MelCepstrum(short *speech, int length, float window, float step,
		float fSamp, int Nfilter, float ***MelCep, int n)
{
  float   *signal, *preemp, *hamm;         /* speech signal frame per frame */
  float   *MelBands, *weights;             /* Mel Scale filter values       */
  float   *cepstral;                       /* Mel cepstral values           */
  float   *dftsig;                         /* fft signal for 2^N point fft  */
  float   *dftmag;                         /* magnitude of dft              */
  Complex *dft;                            /* complex fft output array      */
  float   cepval, filter;                  /* melcepstral value and filter  */
  int     *IntBands;                       /* Mel scale index values        */
  int     success;                         /* return call from FloatFFT     */
  int     win, st, dftsize, dftlen;        /* window sizes for calculations */
  int     numWin;                          /* total number of feature wins  */
  int     i, j, k, l, m;                   /* loop counter variables        */
  int     start, stop;                     /* start and stop indices filter */

  win = (int) ((window / 1000.0) * fSamp);
  st  = (int) ((step / 1000.0) * fSamp);
  numWin = length / st;
  dftsize = (int) floor((log((double) win)/(log(2.0)))) + 1;
  dftlen  = (int) pow(2.0, (double) dftsize);

  /* allocate memory for speech windows */
  if (!(MelBands = (float *) malloc((Nfilter + 1) * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:MelBands - malloc failed";
    return (-1);
  }
    
  if (!(cepstral = (float *) malloc((Nfilter + 1) * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:cepstral - malloc failed";
    return (-1);
  }
    
  if (!(IntBands = (int *) malloc((Nfilter + 1) *  sizeof(int)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:MelBands - malloc failed";
    return (-1);
  }
    
  if (!(signal = (float *) malloc(win * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:signal - malloc failed";
    return (-1);
  }

  if (!(preemp = (float *) malloc(win * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:preemp - malloc failed";
    return (-1);
  }

  if (!(hamm = (float *) malloc(win * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:hamm - malloc failed";
    return (-1);
  }

  if (!(dftsig = (float *) calloc(dftlen, sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:dftsig - malloc failed";
    return (-1);
  }

  if (!(dftmag = (float *) malloc(dftlen * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:dftmag - malloc failed";
    return (-1);
  }

  if (!(weights = (float *) malloc(dftlen * sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:weightings - malloc failed";
    return (-1);
  }

  if (!(dft = (Complex *) malloc(dftlen * sizeof(Complex)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:hamm - malloc failed";
    return (-1);
  }

  /* allocate memory for output mel cepstrum coefficients */
  if (!((*MelCep) = (float **) Alloc2d (numWin, n, sizeof(float)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "MelCepstrum:MelCep - malloc failed";
    return (-1);
  }

  /* Calcuate Mel Scale pertinent values */
  CalculateMelBands(0.0, fSamp/2.0, Nfilter, MelBands);
  MelIntBands (Nfilter, dftlen/2, MelBands, IntBands);
/*  TriangleWeights(Nfilter, dftlen/2, IntBands, weights); */
  SquareWeights(Nfilter, dftlen/2, IntBands, weights);

  /* do calculations */
  for (i=st, k=0; i<length; i+= st, k++) {
    for (j=0; j<win; j++)
      signal[j] = (float) speech[i-st+j];

    PreEmphasize(signal, preemp, win);
    Hamming(preemp, hamm, win);

    for (l=0; l<win; l++)                /* save to dft signal array   */
      dftsig[l] = hamm[l];               /* rest is auto padded with 0 */

    success = FloatFFT(dftsig, dft, dftsize);
    if (success < 0) {                  /* free allocated memory before exit */
      free ((char *) MelBands);
      free ((char *) cepstral);
      free ((char *) IntBands);
      free ((char *) weights);
      free ((char *) signal);
      free ((char *) preemp);
      free ((char *) hamm);
      free ((char *) dftsig);
      free ((char *) dftmag);
      free ((char *) dft);
      Free2d((char **) (*MelCep));
      return (-1);
    }

    for (l=0; l<dftlen/2; l++)          /* magnitude of fft spectrum        */
      dftmag[l] = (float) sqrt((double) (sqr(dft[l].c_real) +  
				      sqr(dft[l].c_imag)));
/*    Normalize_Vector(dftmag, dftlen/2); */
      
    for (l=0; l<Nfilter; l++) {        /* melscale triangular filtering     */
      cepval = 0.0;
      start = IntBands[l];
      stop  = IntBands[l+1];
      for (m=start; m<stop; m++)
	cepval += weights[m]*dftmag[m];
      cepval /= (float) (stop-start);
      cepstral[l] = log(cepval);
    }

    for (l=0; l<n; l++) {               /* discrete cosine tranform          */
      cepval = 0.0;
      for (m=0; m<Nfilter; m++)
	cepval += cepstral[m] * cos (l * (m - 0.5) * M_PI/(float)Nfilter);
      (*MelCep)[k][l] = cepval;
    }
  }

  /* free allocated memory before function exits */
  free ((char *) MelBands);
  free ((char *) cepstral);
  free ((char *) IntBands);
  free ((char *) weights);
  free ((char *) signal);
  free ((char *) preemp);
  free ((char *) hamm);
  free ((char *) dftsig);
  free ((char *) dftmag);
  free ((char *) dft);

  return (numWin);
}
      
      
      
      

    
  


    
