/*
 * $Id: wavextract.c,v 1.8 1993/05/12 00:51:18 johans Exp johans $
 *
 *  wavextract.c
 *
 *
 */

/* Standard C library include file directives */
#include <stdio.h>
#include <malloc.h>
#include <string.h>

/* Speech Libary include file directives */
#include <ad.h>
#include <speech.h>

#define TRUE	(1)
#define FALSE	(0)

struct node {
  int n_samples;
  short *n_wav;
  struct node *n_next;
};


/* Local module header declarations */
static void usage();
static int ExtractWave( char *file, int start, int stop);
static int WriteWave( char *file );
static int AddWave(int samples, short *wav );
static int GetTotalSamples();
static int NumberOfNodes();
static int AddWave();
static struct node *NewNode();

static struct node WavList = { NULL, NULL };
static struct node *TailWavList = &WavList;

static int SampleRate = 0;	/* in Hz */
static int PauseSamples = 0;	/* samples in the pause between waveforms */
static int PauseLength = 0;	/* in milliseconds */

main(int argc, char *argv[])
{
  char buf[1024];
  char *file, *c;
  int start, stop;

  if( argc != 3 ) {
    usage();
    exit( 1 );
  }

  PauseLength = atoi( argv[2] );

  while( fgets( buf, sizeof( buf ), stdin ) != NULL ) {

		file = buf;
		c = strchr( buf, ' ' );
		*c = NULL;
		c++;
		start = atoi( c );
		c = strchr( c, ' ' );
		stop = atoi( c );

		if( ExtractWave( file, start, stop ) == FALSE ) {

		  fprintf( stderr, "wavextract: could not extract wave form from \'$s\': %s\n", file, ErrorString );
		  exit( 1 );
		}

	}

	if( WriteWave( argv[1] ) == FALSE ) {
		fprintf( stderr, "wavextract: could not write wavfile: %s\n",
								ErrorString );
		exit( 1 );
	}

	exit( 0 );

}

static void usage()
{
	fprintf( stderr, "usage: wavextract file pause_length\n" );

}

static int ExtractWave( char *file, int start, int stop)
{
	short *wav = NULL;
	int samples;
	short *s, *twav;
	float rate;
	int i;

	if( ( samples = SpeechRead( file, &wav, &rate ) ) < 0) {
		return( FALSE );
	}

	/*
	 * This assumes that all of the
	 * waveforms extracted have the same
	 * sample rate.
	 */
	SampleRate = (int)rate;
	PauseSamples = (SampleRate/1000) * PauseLength;

	samples = (stop - start) + 1;

	s = (short *)malloc( sizeof(short) * samples );

	if( s == NULL ) {
		perror( "wavextract: malloc:" );
		return( FALSE );
	}

	twav = wav;
	twav += start;

	memcpy( (char *)s, (char *)twav, samples * 2 );

	AddWave( samples, s );

	free( wav );

	return( TRUE );
}

static int WriteWave( char *file )
{
	struct node *tmp;
	int total_samples = 0;
	short *wav;
	short *t;
	int i;


	total_samples = GetTotalSamples();
	total_samples += ( NumberOfNodes() * PauseSamples );

	wav = (short *)malloc( sizeof(short) * total_samples );
	
	if( wav == NULL ) {
		perror( "wavextract: malloc" );
		return( FALSE );
	}

	t = wav;

	tmp = WavList.n_next;

	while( tmp != NULL ) {

		memcpy( (char *)t, (char *)tmp->n_wav, tmp->n_samples * 2 );
		t += tmp->n_samples;

		/*
 		 * add in pause
		 */
		for( i = 0; i < PauseSamples; i++ ) {
			*t++ = 1;
		}

		tmp = tmp->n_next;

	}

	if( WavWrite( file, wav, total_samples, SampleRate, 0 ) < 0 ) {
		return( FALSE );
	}

	return( TRUE );
}

/*
 * AddWave()
 *
 *  adds a waveform to the end of the list.
 *
 */

static int AddWave(int samples, short *wav )
{
	struct node *tmp;

	tmp = NewNode();

	tmp->n_samples = samples;
	tmp->n_wav = wav;

	TailWavList->n_next = tmp;
	TailWavList = tmp;

	return( TRUE );
}

static int GetTotalSamples()
{
	struct node *tmp;
	int t_samples = 0;


	tmp = WavList.n_next;

	while( tmp != NULL ) {
		t_samples += tmp->n_samples;
		tmp = tmp->n_next;
	}

	return( t_samples );

}

static int NumberOfNodes()
{
	struct node *tmp;
	int number = 0;

	tmp = WavList.n_next;

	while( tmp != NULL ) {
		number++;
		tmp = tmp->n_next;
	}

	return( number );

}

static struct node *NewNode()
{
	struct node *tmp;
	
	tmp = (struct node *)malloc( sizeof(struct node) );

	if( tmp == NULL ) {
		perror( "wavextract: malloc:" );
		exit( 2 );
	}

	tmp->n_samples = -1;
	tmp->n_wav = NULL;
	tmp->n_next = NULL;

	return( tmp );

}
