/*
 * phonebase.c
 *
 * This module implements the main data base routines
 *
 */

/* Standard C library include file directives */
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>

/* Speech tools include file directives */
#include <speech.h>

/* data base include file directives */
#include <phonebase.h>

/* external varaible declaration */
extern char machineorder;                   /* byte ordering of machine  */
extern int  errno, sys_nerr;
extern char *sys_errlist[];
extern int  CREATE_DATABASE;                /* add new data flag         */

/* local module variables */
int phonemecount;                           /* count of valid phonemes   */
int buffersize = 256;                       /* initial index buffer size */


/*
 * float SWAPF(inval)
 *
 * Swaps the byte ordering for floating point values from little to
 * big endian and vica versa.
 *
 * inval (in): input floating point value
 *
 * On exit returns the swapped value
 *
 */

float SWAPF (float inval)
{
  union {
    unsigned int intval;
    float floatval;
  } swap;

  swap.floatval = inval;
  swap.intval = SWAPL(swap.intval);

  return(swap.floatval);
}
 

/*
 * int Read_Index_File(FileName, PhonemeIndex, PreviousLink)
 *
 * This function attempts to read an Index file. If the file
 * does not exists, it will create and empty file, else it will
 * abort. Index files are assumed always to be less than 32767
 * possible phonemes.
 *
 * FileName      (in): Name of Index file
 * PhonemeIndex (out): pointer to array of record of type phoneme rec
 * PreviousLink (out): pointer to array of previous links in data base
 *
 * On Error returns -1, else 1
 *
 */

int Read_Index_File(char *FileName, PhonemeRec **PhonemeIndex, int **PreviousLink)
{
  int   i;                       /* loop counter variable                */
  FILE  *indexfile;              /* index to contextual data base        */
  PhonemeRec *Index;             /* phone index pointer                  */
  int        *Link;              /* pointer to vector of previous links  */
  char  byteswap;
  
  /* open index file for read or try to create */
  if (!(indexfile = fopen(FileName, "rb"))) {
    /* file does not exist - thus create     */
    if (CREATE_DATABASE) {
      fprintf(stderr, 
	      "Assuming file does not exist --> attempting to create\n");
      if (!(indexfile = fopen(FileName, "wb"))) {
	if (errno < sys_nerr )
	  ErrorString = sys_errlist[errno];
	else
	  ErrorString = "Error creating Index File";
	return (-1);
      }
	
      /* data base has now been created */
      /* allocate memory for initial buffer size */
      buffersize = BUFFER_MARGIN;
      if (!(*PhonemeIndex = (PhonemeRec *) malloc(buffersize * 
						  sizeof(PhonemeRec)))) {
	if ( errno < sys_nerr )
	  ErrorString = sys_errlist[errno];
	else
	  ErrorString = "Read_Index_File: PhonemeIndex -- malloc failed";
	fclose (indexfile);
	return (-1);
      }
      
      /* memory for index buffer has now been allocated */
      /* write empty index to disk */
      Index = *PhonemeIndex;
      for (i=0; i<buffersize; i++) {
	Index[i].headorder = machineorder;
	Index[i].startrec  = -1;
	Index[i].endrec    = -1;
	strcpy(Index[i].phoneme, "");

	if (fwrite(&Index[i], sizeof(PhonemeRec), 1, indexfile)==0) {
	  if ( errno < sys_nerr )
	    ErrorString = sys_errlist[errno];
	  else
	    ErrorString = "Read_Index_File -- write file io error";
	  fclose(indexfile);
	  return (-1);
	}
      }
    } else {
      if (errno < sys_nerr) 
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "Could not open Index File for read";
      return (-1);
    }
  }

  /* file is now open for read       */
  /* read in phoneme index from file */
  else {
    fseek(indexfile, 0, SEEK_END);
    phonemecount = ftell(indexfile)/sizeof(PhonemeRec);
    fseek(indexfile, 0, SEEK_SET);
    buffersize   = phonemecount;
    
    /* allocate memory for phoneme index buffer */
    if (!(*PhonemeIndex = (PhonemeRec *) malloc(buffersize * 
						sizeof(PhonemeRec)))) {
      if (errno < sys_nerr )
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "Read_Index_FIlee: PhonemeIndex -- malloc failed";
      fclose(indexfile);
      return (-1);
    }
    
    Index = *PhonemeIndex;
    for (i=0; i<phonemecount; i++) {
      if (fread(&Index[i], sizeof(PhonemeRec), 1, indexfile)==0) {
	if ( errno < sys_nerr )
	  ErrorString = sys_errlist[errno];
	else
	  ErrorString = "Read_Index_File -- read file io error";
	fclose(indexfile);
	return (-1);
      }
      
      byteswap = (((Index[i].headorder) && (!machineorder)) ||
		  ((!Index[i].headorder) && (machineorder)));
      if (byteswap) {
	Index[i].startrec = SWAPL(Index[i].startrec);
	Index[i].endrec   = SWAPL(Index[i].endrec);
      }
    }
    for (i=phonemecount; i<buffersize; i++) {
      Index[i].headorder = machineorder;
      Index[i].startrec  = -1;
      Index[i].endrec    = -1;
      strcpy(Index[i].phoneme, "");
    }
  }

  /* allocate memory for previous link vector */
  if (!(*PreviousLink = (int *) malloc (buffersize * sizeof(int)))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Read_Index_File: PreviousLink -- malloc failed";
    fclose(indexfile);
    return (-1);
  }
  
  Link = *PreviousLink;
  for (i=0; i<buffersize; i++) {
    if (strlen(Index[i].phoneme)) 
      Link[i] = Index[i].endrec;
    else
      Link[i] = -1;
  }
  
  fclose(indexfile);
  return (1);
}


/*
 * int Write_Index_File(FileName, PhonemeIndex)
 *
 * This function writes the array of index values out to the index 
 * file.
 *
 * FileName    (in): name of index file (output file name)
 * PhomemeIndex(in): pointer to array of index values
 *
 * On error returns -1, and sets ErrorString, else returns 1
 *
 */

int Write_Index_File(char *FileName, PhonemeRec *PhonemeIndex)
{
  int   i, flag;                 /* loop counter variable                */
  FILE  *indexfile;              /* index to contextual data base        */

  if (!(indexfile = fopen(FileName, "wt"))) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Could not open Index File for Update";
    return (-1);
  }

  flag = 1;
  for (i=0; i<buffersize; i++) {
    PhonemeIndex[i].headorder = machineorder;
    if (fwrite(&PhonemeIndex[i], sizeof(PhonemeRec), 1, indexfile)==0) {
      if ( errno < sys_nerr )
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "Write_Index_File -- write file io error";
      return (-1);
    }
    if ((flag) && (strlen(PhonemeIndex[i].phoneme)==0)) {
      phonemecount = i;
      flag = 0;
    }
  }

#ifdef DEBUG
  printf("buffersize: %d -- phonemecount: %d\n", buffersize, phonemecount);
#endif

  fclose(indexfile);

  return (1);
}


/*
 * int Retrieve_Phoneme_Index(InputPhoneme, PhonemeIndex, PreviousLink)
 *
 * This function determines the index value of the Input Phoneme
 * If this phoneme does not exist, it places it in the index file
 * if it can. It assumes the current file pointer points to
 * the position where the record will appear. If it can not
 * add to the index file it will abort and return (-1)
 *
 * InputPhoneme   (in): string contaning the new or old phoneme
 * PhonemIndex (in/out): pointer to index for retrieval of information
 * PreviousLink(in/out): pointer to vector of previous links of phoneme
 *
 * On error returns -1, else +1
 *
 */

int Retrieve_Phoneme_Index(char *InputPhoneme, PhonemeRec **PhonemeIndex, int **PreviousLink)
{
  int i, ph_curr;                        /* loop counter variable        */
  int flag = 0;                   
  PhonemeRec *Index;                     /* pointer for phoneme index    */
  int        *Link;                      /* pointer of previous links    */
  int        newbuffersize;              /* size of new buffer           */

  Index = *PhonemeIndex;
  for (i=0; i<buffersize; i++) {
    if (strcmp(InputPhoneme, Index[i].phoneme) == 0) {
      flag = 1;
      break;
    }
    if (strcmp(Index[i].phoneme,"") == 0) {
      flag = 2;
      break;
    }
  }

  if ((flag == 0) && (phonemecount==32767)) {
#ifdef DEBUG
    for (i=0; i<127; i++)
      printf("%s ", Index[i].phoneme);
    printf("\n\n INDEX FILE FULL....\n");
#endif
    fprintf(stderr, "MAJOR ERROR, ABORTING EVERYTHING -- INDEX FILE FULL\n");
    exit(1);
  }
  
  if (flag == 1) 
    return (i+1);                     /* found index value                 */

  /* add additional new phoneme to index file */
  ph_curr = i;
  if (CREATE_DATABASE) {
    if (i >= buffersize) {                      /* reallocate memory      */
      newbuffersize = buffersize + BUFFER_MARGIN;
#ifdef DEBUG
      printf("Stretching size of index buffer...\n");
#endif

      /* stretch buffer using realloc */
      if (!(*PhonemeIndex = (PhonemeRec *) 
	    realloc(*PhonemeIndex, newbuffersize * sizeof(PhonemeRec)))) {
	if (errno < sys_nerr) 
	  ErrorString = sys_errlist[errno];
	else
	  ErrorString = "Retrieve_Phoneme_Index:PhonemeIndex - realloc failed";
	return (-1);
      }
      
      /* stretch previous link buffer */
      if (!(*PreviousLink = (int *) 
	    realloc(*PreviousLink, newbuffersize * sizeof(int)))) {
	if (errno < sys_nerr)
	  ErrorString = sys_errlist[errno];
	else
	  ErrorString = "Retrieve_Phoneme_Index:PreviousLink - realloc failed";
	return(-1);
      }
      
      /* set remaining new memory */
      Index = *PhonemeIndex;
      Link  = *PreviousLink;
      for (i=buffersize; i<newbuffersize; i++) {
	Index[i].headorder = machineorder;
	Index[i].startrec  = -1;
	Index[i].endrec    = -1;
	strcpy(Index[i].phoneme, "");
	Link[i] = -1;
      }
      buffersize = newbuffersize;
    }

    Index[ph_curr].headorder = machineorder;
    strcpy(Index[ph_curr].phoneme, InputPhoneme);
    Index[ph_curr].startrec = -1;
  } else {
    ErrorString = "Phoneme does not exist in given phoneme index file";
    return (-1);
  }

  return(ph_curr+1);
}


/*
 * Open_Contextual_Base(contextfile, FileName)
 *
 * This procedure opens the Contextual data base file, and sets 
 * global pointer positions to the record numbers.
 *
 * contextfile(in): pointer to file stream 
 * FileName   (in): name of contextual database file
 *
 * On Error return -1, else +1
 *
 */

int Open_Contextual_Base(FILE **contextfile, char *FileName)
{
  int  filepointer;

  if (!(*contextfile = fopen(FileName, "r+b"))) {
    if (CREATE_DATABASE) {
      fprintf(stderr, 
	      "Assuming database does not exist --> attempting to create\n");
      if (!(*contextfile = fopen(FileName, "w+b"))) {
	if (errno < sys_nerr)
	  ErrorString = sys_errlist[errno];
	else
	  ErrorString = "error creating contextual data base";
	return (-1);
      }
    }
    else {
      if (errno < sys_nerr)
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "Could not open data base.";
      return (-1);
    }
  }
  
  /* data base is now open for use */
  /* set file pointer to end of file */
  fseek(*contextfile, 0, SEEK_END);
  filepointer = (int) ftell(*contextfile);
  
  return (1);
}


/*
 * Close_Contextual_Base(contextfile)
 *
 * This procedure closes the contextual data base
 *
 */

void Close_Contextual_Base(FILE **contextfile)
{
  fclose(*contextfile);
}


/*
 * int Read_Context_Record(contextfile, Phoneme, recordno)
 *
 * This function reads one phoneme record from the contextual
 * data base.
 *
 * contextfile (in): pointer to the file stream
 * Phoneme    (out): pointer to the contextual record in file
 * recordno    (in): number of record to retrieve
 *
 * On error returns -1, and sets ErrorString else returns +1
 *
 */

int Read_Context_Record(FILE *contextfile, ContextRec *Phoneme, int recordno)
{
  char byteswap, headorder;

  fseek(contextfile, recordno * sizeof(ContextRec), SEEK_SET);
  if (fread(Phoneme, sizeof(ContextRec), 1, contextfile)==0) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Read_Context_Record -- read file io error";
    return (-1);
  }

  headorder = Phoneme->headorder;
  byteswap = (((headorder) && (!machineorder)) ||
	      ((!headorder) && (machineorder)));

  if (byteswap) {
    Phoneme->from_file_id = SWAPL(Phoneme->from_file_id);
    Phoneme->prevlink     = SWAPL(Phoneme->prevlink);
    Phoneme->nextlink     = SWAPL(Phoneme->nextlink);
    Phoneme->startloc     = SWAPL(Phoneme->startloc);
    Phoneme->endloc       = SWAPL(Phoneme->endloc);
    Phoneme->prev_start   = SWAPL(Phoneme->prev_start);
    Phoneme->next_end     = SWAPL(Phoneme->next_end);
    Phoneme->ph_prevlink  = SWAPL(Phoneme->ph_prevlink);
    Phoneme->ph_nextlink  = SWAPL(Phoneme->ph_nextlink);
    Phoneme->ph_curr      = SWAPW(Phoneme->ph_curr);
    Phoneme->ph_prev      = SWAPW(Phoneme->ph_prev);
    Phoneme->ph_next      = SWAPW(Phoneme->ph_next);
  }

  return (1);
}


/*
 * int Write_Context_Record(contextfile, Phoneme, recordno)
 *
 * This function writes one record to the file, at the specified
 * location.
 *
 * contextfile (in): pointer to file stream
 * Phoneme     (in): contextual data base record to write
 * recordno    (in): record to write to file
 *
 * On error returns -1, and sets ErrorString else returns +1
 *
 */

int Write_Context_Record(FILE *contextfile, ContextRec Phoneme, int recordno)
{
  fseek(contextfile, recordno * sizeof(ContextRec), SEEK_SET);
  Phoneme.headorder = machineorder;

  if (fwrite(&Phoneme, sizeof(ContextRec), 1, contextfile)==0) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Write_Context_Record -- write file io error";
    return (-1);
  }
  
  return (1);
}


/*
 * int Update_Context_Record(contextfile, previous_link, current_link)
 *
 * This function reads in a record from the previous link
 * value and updates its next link value.
 *
 * contextfile   (in): pointer to file stream
 * previous_link (in): record number in contextual data base
 * current_link  (in): current record in contextual data base
 *
 * On error returns -1, and sets ErrorString else returns +1
 *
 */

int Update_Context_Record(FILE *contextfile, int previous_link,
			  int current_link)
{
  ContextRec ContextPhoneme;               /* one record in data base */

  /* update record in data base */
  if (Read_Context_Record(contextfile, &ContextPhoneme, previous_link)<0)
    return (-1);
  
  ContextPhoneme.nextlink = current_link;

#ifdef DEBUG
  printf("---> %d  %d  %d -- %d\n", 
	 ContextPhoneme.ph_prev, 
	 ContextPhoneme.ph_curr,
	 ContextPhoneme.ph_next,
	 ContextPhoneme.nextlink);
#endif

  if (Write_Context_Record(contextfile, ContextPhoneme, previous_link)<0)
    return (-1);

  /* reset file pointer value */
  fseek(contextfile, current_link * sizeof(ContextRec), SEEK_SET);
  
  return (1);
}


/*
 * Print_Context_Record(Phoneme)
 *
 * This procedure prints one phoneme contextual record on the
 * screen.
 *
 * Phoneme (in): Contextual record in the data base
 *
 */

void Print_Context_Record(ContextRec Phoneme)
{
  fprintf(stderr, "headorder   : %d\n", Phoneme.headorder);
  fprintf(stderr, "from_file_id: %d\n", Phoneme.from_file_id);
  fprintf(stderr, "prevlink    : %d\n", Phoneme.prevlink);
  fprintf(stderr, "nextlink    : %d\n", Phoneme.nextlink);
  fprintf(stderr, "ph_curr     : %d\n", Phoneme.ph_curr);
  fprintf(stderr, "startloc    : %d\n", Phoneme.startloc);
  fprintf(stderr, "endloc      : %d\n", Phoneme.endloc);
  fprintf(stderr, "ph_prev     : %d\n", Phoneme.ph_prev);
  fprintf(stderr, "prev_start  : %d\n", Phoneme.prev_start);
  fprintf(stderr, "ph_next     : %d\n", Phoneme.ph_next);
  fprintf(stderr, "next_end    : %d\n", Phoneme.next_end);
  fprintf(stderr, "ph_prevlink : %d\n", Phoneme.ph_prevlink);
  fprintf(stderr, "ph_nextlink : %d\n", Phoneme.ph_nextlink);
  fprintf(stderr, "\n");
}

    
/*
 * int Open_FileID_Base(idfile, FileName)
 *
 * This procedure opens the FileID data base file, and sets the global
 * file id pointer to the current record number (end of file )
 *
 * idfile   (in): pointer to file stream
 * FileName (in): name of file id data base
 *
 * On error returns -1, else 1
 *
 */

int Open_FileID_Base(FILE **idfile, char *FileName)
{
  int filepointer;
  
  if (!(*idfile = fopen(FileName, "r+b"))) {
    if (CREATE_DATABASE) {
      fprintf(stderr, 
	      "Assuming ID File does not exist --> attempting to create\n");
      if (!(*idfile= fopen(FileName, "w+b"))) {
	if (errno < sys_nerr)
	  ErrorString = sys_errlist[errno];
	else
	  ErrorString = "error creating id data base.";
	exit(1);
      }
    } else {
      if (errno < sys_nerr)
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "Could not open ID File.";
      return (-1);
    }
  }

  /* set file pointer to end of file */
  fseek(*idfile, 0, SEEK_END);
  filepointer = (int) ftell(*idfile);

  return(1);
}


/*
 * Close_FileID_Base(idfile)
 *
 * Closes the from file id data base
 *
 */

void Close_FileID_Base(FILE **idfile)
{
  fclose(*idfile);
}


/*
 * int Read_FileID_Record(idfile, FileID, recordno)
 *
 * This function reads one record frm the id file, which describes
 * where the specific sounds can be found on disc.
 *
 * idfile  (in): pointer to file stream
 * FileID (out): one record in data base (pointer)
 * recordno(in): record numebr to retrieve
 *
 * On error returns -1, and sets ErrorString else returns +1
 *
 */

int Read_FileID_Record(FILE *idfile, FileIDRec *FileID, int recordno)
{
  char byteswap, horder;

  fseek(idfile, recordno * sizeof(FileIDRec), SEEK_SET);
  if (fread(FileID, sizeof(FileIDRec), 1, idfile)==0) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Read_FileID_Record -- read file io error";
    return (-1);
  }
  
  horder = FileID->headorder;
  byteswap = (((horder) && (!machineorder)) ||
	      ((!horder) && (machineorder)));

  if (byteswap) {
    FileID->start_point = (int) SWAPL((unsigned int) FileID->start_point);
    FileID->ms          = SWAPF(FileID->ms);
  }

  return(1);
}


/*
 * int Write_FileID_Record(idfile, FileID, recordno)
 *
 * This function writes one FileID record to disc
 *
 * idfile   (in): pointer to file stream
 * FileID   (in): one record in the data base
 * recordno (in): record number where to write in file
 *
 * On error returns -1, and sets ErrorString else return +1
 *
 */

int Write_FileID_Record(FILE *idfile, FileIDRec FileID, int recordno)
{
  fseek(idfile, recordno * sizeof(FileIDRec), SEEK_SET);

  FileID.headorder = machineorder;
  if (fwrite(&FileID, sizeof(FileIDRec), 1, idfile)==0) {
    if ( errno < sys_nerr )
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Write_FileID_Record -- write file io error";
    return (-1);
  }
}


/* 
 * Print_FileID_Record(idrec)
 *
 * This procedure prints one file id record on screen
 *
 * idrec (in): File ID record in file id database
 *
 */

void Print_FileID_Record(FileIDRec idrec)
{
  fprintf(stderr, "directory  : %s\n", idrec.wavfilename);
  fprintf(stderr, "filename   : %s\n", idrec.phnfilename);
  fprintf(stderr, "headorder  : %d\n", idrec.headorder);
  fprintf(stderr, "start_point: %d\n", idrec.start_point);
}


/*
 * int Return_FileID(idfile, FileID)
 *
 * This function searches the FileID data base for the required FileID
 * 
 * idfile (in): pointer to file stream
 * FileID (in): record needed for match
 *
 */

int Return_FileID(FILE *idfile, FileIDRec FileID)
{
  int i, flag = 0;
  FileIDRec OneRec;

  /* rewind file */
  fseek(idfile, 0, SEEK_SET); 
  if (feof(idfile)) 
    return (-1);

  i = 0;
  while (!(feof(idfile))) {
    fread(&OneRec, sizeof(FileIDRec), 1, idfile);
    flag = ((strcmp(OneRec.phnfilename, FileID.phnfilename) == 0) &&
	    (strcmp(OneRec.wavfilename, FileID.wavfilename) == 0));
    if (flag) 
      return (i+1);

    i++;
  }

  return (-1);
}
