/*
 * packbase.c
 *
 * The packbase utility recreates the speech data base, using
 * the chains as defined in the input data base. Used to optimize the
 * search chains and index files in the data base.
 *
 */

/* Standard C file inclue file directives */
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* Speech tools include file directives */
#include <speech.h>


/* data base include file directives */
#include <phonebase.h>


/* Global variable declaration */
int        CREATE_DATABASE = 1;
int        *PreviousLink;            /* array of most recent previous links */
PhonemeRec *PhonemeIndex;            /* phoneme for index file              */

FILE       *contextfile;             /* contextual phoneme data base        */
FILE       *newcontextfile;          /* new contextual data base stream     */
FILE       *idfile;                  /* from file id data base              */
FILE       *newidfile;               /* new from file id data base          */

char       headorder, machineorder;  /* byte ordering of record and machine */
char       byteswap;                 /* must i do byte swapping flag        */


/* global commannd line parameters */
char       contextfilename[200];     /* name of contextual database file    */
char       newcontextfilename[200];  /* name of new contextual database     */
char       indexfilename[200];       /* name of index file for contextual   */
char       newindexfilename[200];    /* name of new context index file      */
char       fileidfilename[200];      /* name of file id data base           */
char       newfileidfilename[200];   /* name of new from file id databse    */

char       dbname[200]="";           /* name of data base to search         */
char       newdbname[200]="";        /* name of new data base to search     */


/*
 * Usage();
 * 
 * print out the usage parameter of the buildlola program
 *
 */

void Usage()
{
  fprintf(stderr, "Usage: buildlola {dbname} {newdbname}\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "dbname   : name of data base to pack\n");
  fprintf(stderr, "newdbname: name of new data base, result of pack\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "\t -h Help. This help message\n");
  exit(1);
}


/*
 * get_comline()
 *
 * read an interpret the command line parameters
 *
 * argc, argv (in): command line variables
 *
 */

void get_comline (int argc, char **argv)
{
  int c;
  extern int  optind;
  extern char *optarg;

  while ( (c = getopt (argc, argv, "h")) != -1) {
    switch (c) 
      {
	
      case 'h':
      default:
	Usage();
      }
  }

  if ((argc-optind) != 2)
    Usage();

  strcpy(dbname, argv[optind]);
  strcpy(newdbname, argv[optind+1]);

  /* create data base and index file name */
  strcpy(contextfilename, dbname);
  strcat(contextfilename, CONTEXTBASE);
  strcpy(newcontextfilename, newdbname);
  strcat(newcontextfilename, CONTEXTBASE);

  strcpy(indexfilename, dbname);
  strcat(indexfilename, INDEX);
  strcpy(newindexfilename, newdbname);
  strcat(newindexfilename, INDEX);

  strcpy(fileidfilename, dbname);
  strcat(fileidfilename, IDBASE);
  strcpy(newfileidfilename, newdbname);
  strcat(newfileidfilename, IDBASE);
}


/*
 * Rebuild_Data_Base(contextfile, newcontextfile, idfile, newidfile, 
 *                   PhonemeIndex, PreviousLink)
 *
 * This procedure loops through the data in the old data base and
 * creates the new data base, with horizontal and vertical links
 * optimized.
 *
 * contextfile    (in): pointer to old contextual data base file stream
 * newcontextfile (in): new contextual data base (packed) file stream
 * idfile         (in): pointer to from file id data base (old)
 * newidfile      (in): pointer to new from file id data stream
 * PhonemeIndex   (in): contextual data base phoneme index.
 * PreviousLink   (in): pointer to listing of previous links in database
 *
 */

void Rebuild_Data_Base (FILE *contextfile, FILE *newcontextfile,
			FILE *idfile, FILE *newidfile,
			PhonemeRec *PhonemeIndex, int *PreviousLink)
{
  int i, j;                            /* loop counter variables          */
  ContextRec OneContext;               /* One Record in Contextual Base   */
  FileIDRec  IDFile;                   /* One IDFile record               */
  int        record;                   /* current record number           */
  int        newrec;                   /* new record number               */
  int        ph_prevlink, ph_nextlink; /* previous and next links in chain*/
  int        ph_curr;                  /* current phoneme value           */

  i = 0;
  newrec = 0;
  do {                                 /* all files in from file ID index */
    if (Read_FileID_Record (idfile, &IDFile, i)<0)
      break;
    
    record = IDFile.start_point;
    
    /* update new from file id values */
    printf("%s\n", IDFile.phnfilename);
    IDFile.headorder   = machineorder;
    IDFile.start_point = newrec;
    fwrite(&IDFile, sizeof(FileIDRec), 1, newidfile);
    
    /* loop until end of lola chain */
    do {
      if (Read_Context_Record (contextfile, &OneContext, record)<0)
	break;

      /* build new horizontal links */
      if (record == IDFile.start_point) 
	OneContext.ph_prevlink = -1;
      else
	OneContext.ph_prevlink = newrec -1;
      
      if (OneContext.ph_nextlink > 0)
	OneContext.ph_nextlink = newrec + 1;
      else 
	OneContext.ph_nextlink = -1;

      ph_curr = OneContext.ph_curr;
      record  = OneContext.ph_nextlink;
      
      /* build new vertical links */
      if ((OneContext.prevlink = PreviousLink[ph_curr-1]) > 0)
	Update_Context_Record (newcontextfile, OneContext.prevlink, newrec);
      else
	PhonemeIndex[ph_curr-1].startrec = newrec;
      
      PreviousLink[ph_curr-1] = newrec;
      PhonemeIndex[ph_curr-1].endrec = newrec;

      /* save record to new data base */
      OneContext.headorder = machineorder;
      
      fwrite(&OneContext, sizeof(ContextRec), 1, newcontextfile);
      
      newrec++;
    } while (record>0);
    
    i++;
  } while (!(feof(idfile)));
}
  
  

/*
 * master magician code
 *
 */

main (int argc, char **argv)
{
  int       i;                          /* loop counter variables    */

  /* process command line directives */
  get_comline(argc, argv);

  machineorder = LittleIndian();

  /* do old data base setup */
  if (Read_Index_File (indexfilename, &PhonemeIndex, &PreviousLink) < 0) {
    fprintf(stderr, "Read_Index_File: %s\n", ErrorString);
    exit(1);
  }
  
  if (Open_Contextual_Base (&contextfile, contextfilename) < 0) {
    fprintf(stderr, "Open_Contextual_Base: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    exit(1);
  }

  if (Open_FileID_Base (&idfile, fileidfilename) < 0) {
    fprintf(stderr, "Open_FileIDBase: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    Close_Contextual_Base(&contextfile);
    exit(1);
  }

  /* assume new data bases and index file does not exist */
  if (Write_Index_File (newindexfilename, PhonemeIndex) < 0) {
    fprintf(stderr, "Write_Index_File: %s\n", ErrorString);
    exit(1);
  }
  
  Open_Contextual_Base (&newcontextfile, newcontextfilename);
  Open_FileID_Base (&newidfile, newfileidfilename);


  /* rebuild data base */
  for (i=0; i<256; i++) 
    PreviousLink[i] = -1;
  
  Rebuild_Data_Base(contextfile, newcontextfile, idfile, newidfile,
		    PhonemeIndex, PreviousLink);

  /* do sytem cleanup */
  Close_Contextual_Base(&contextfile);
  Close_FileID_Base(&idfile);
  
  if (Write_Index_File(newindexfilename, PhonemeIndex) < 0) 
    fprintf(stderr, "Write_Index_File: %s\n", ErrorString);
    
  Close_Contextual_Base(&newcontextfile);
  Close_FileID_Base(&newidfile);

  free((char *) PhonemeIndex);
  free((char *) PreviousLink);
}


  

 
