/*
 * find_phone.c
 *
 * The find_phone program search the data base for instances of all or
 * only certain phonemes, in certain required phoneme context.
 *
 */

/* standard C library include file directives */
#include <stdio.h>
#include <stdlib.h>
#include <string.h>


/* Speech tools include file directives */
#include <speech.h>


/* data based include file directives */
#include <phonebase.h>

/* external module headers */
int CreateDFT(char *InFileName, char *OutFileName);


/* global constant definition */
#define   LOGEXT    ".log"          /* log file for mergin process          */
#define   LOLAEXT   ".lola"         /* lola file extension                  */
#define   WAVEXT    ".wav"          /* wave file name extension             */
#define   DFTEXT    ".dft"          /* dft file name extension              */


/* external variable declaration */
extern int  errno, sys_nerr;
extern char *sys_errlist[];

/* global variable declaration */
int        CREATE_DATABASE = 0;
int        *PreviousLink;           /* array of most recent previous links  */
PhonemeRec *PhonemeIndex;           /* phoneme for index file               */

FILE       *contextfile;            /* contextual phoneme data base         */
FILE       *idfile;                 /* from file id data base               */
FILE       *lolafile;               /* listing of lola files to build base  */
FILE       *wavfile;                /* waveform output file stream          */
FILE       *logfile;                /* logfile output stream                */

char       headorder, machineorder; /* byte ordering of record and machine  */
char       byteswap;                /* must i do byte swapping flag         */


/* global command line parameter variables */
char       contextfilename[200];    /* name of contextual database file     */
char       indexfilename[200];      /* name of index file for contextual    */
char       fileidfilename[200];     /* name of file id data base            */
char       lolafilename[200];       /* name of lola listing of files        */

char       logfilename[200]="";     /* log file name for merging process    */
char       lolafilename[200]="";    /* lola file name for output lola       */
char       wavfilename[200]="";     /* wave file name (speech data)         */
char       dftfilename[200]="";     /* dft (Spectrogram) output file        */
int        createwav = 0;           /* create waveform file                 */
int        createdft = 0;           /* create dft file flag                 */
int        numsegments = 1000;      /* retrieve all segments                */
int        pause = 60;              /* pause inbetween segments             */
int        post = 0;                /* post segments are not saved          */
int        pre  = 0;                /* pre segments are saved               */
int        seq  = 0;                /* number of previous segments to skip  */
float      Sf   = 8000.0;           /* sampling frequency                   */
int        totalspeech = 0;         /* total number of speech samples       */

char       dbname[200]="";          /* name of data base to search          */
char       pattern[200]="";         /* search pattern                       */


/*
 * Usage()
 *
 * print out the usage parameters of the find phone program
 *
 */

void Usage()
{
  fprintf(stderr, "Usage: find_phone [options] {dbname} {search pattern}\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "dbname        : name of data base to search\n");
  fprintf(stderr, "search pattern: phone search pattern to use for searching\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "\t -f [log filename] Sets search log output file.     [dbname.log]\n");
  fprintf(stderr, "\t -l [lola filename] Sets the lola output file name. [dbname.lola]\n");
  fprintf(stderr, "\t -w [wave filename] Sets the waveform output file.  [dbname.wav]\n");
  fprintf(stderr, "\t -s [spectogram file] Sets the spectogram output.   [dbname.dft]\n");
  fprintf(stderr, "\t -W Waveform file is created                        [No Create]\n");
  fprintf(stderr, "\t -S Spectrogram file is created                     [No Create]\n");
  fprintf(stderr, "\t -c [number] Saves [number] of segments.            [all]\n");
  fprintf(stderr, "\t -p [number] Sets the pause in milliseconds between\n");
  fprintf(stderr, "\t             found phonemes.                        [60ms]\n");
  fprintf(stderr, "\t -t Saves the segment occuring after target segment [do not]\n");
  fprintf(stderr, "\t -e Saves the segment occuring before target        [do not]\n");
  fprintf(stderr, "\t -q [number] Count [number] segments before saving  [0]\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "\t -h Help. This help message\n");
  exit(1);
}


/*
 * get_comline (argc, argv)
 *
 * read an interpret the command line parameters
 *
 * argc, argv (in): command line variables 
 *
 */

void get_comline(int argc, char **argv)
{
  int c;
  extern int optind;
  extern char *optarg;

  while ( (c = getopt (argc, argv, "f:l:w:s:WSc:p:teq:h")) != -1) {
    switch (c) 
      {
      case 'f': /* log file name */
	strcpy(logfilename, optarg);
	strcat(logfilename, LOGEXT);
	break;

      case 'l': /* lola file name */
	strcpy(lolafilename, optarg);
	strcat(lolafilename, LOLAEXT);
	break;

      case 'w': /* waveform output file name */
	strcpy(wavfilename, optarg);
	strcat(wavfilename, WAVEXT);
	break;

      case 's': /* dft output file name */
	strcpy(dftfilename, optarg);
	strcat(dftfilename, DFTEXT);
	break;

      case 'W': /* Create waveform file flag */
	createwav = 1;
	break;

      case 'S': /* create dft file flag */
	createdft = 1;
	break;

      case 'c': /* number of segments to save to disk */
	numsegments = atoi(optarg);
	break;

      case 'p': /* pause between segments */
	pause = atoi(optarg);
	break;

      case 't': /* post segment must be saved */
	post = 1;
	break;

      case 'e': /* pre segment must be saved */
	pre = 1;
	break;

      case 'q': /* number of occurences to skip before saving */
	seq = atoi(optarg);
	break;
	
      case 'h':
      default:
	Usage();
      }
  }

  if ((argc - optind) != 2)
    Usage();

  strcpy(dbname, argv[optind]);
  strcpy(pattern, argv[optind+1]);

  /* create default options */
  if (strlen(logfilename)==0) {
    strcpy(logfilename, dbname);
    strcat(logfilename, LOGEXT);
  }
  if (strlen(lolafilename)==0) {
    strcpy(lolafilename, dbname);
    strcat(lolafilename, LOLAEXT);
  }
  if (strlen(wavfilename)==0) {
    strcpy(wavfilename, dbname);
    strcat(wavfilename, WAVEXT);
  }
  if (strlen(dftfilename)==0) {
    strcpy(dftfilename, dbname);
    strcat(dftfilename, DFTEXT);
  }

  /* create data base and index file names */
  strcpy(contextfilename, dbname);
  strcat(contextfilename, CONTEXTBASE);

  strcpy(indexfilename, dbname);
  strcat(indexfilename, INDEX);

  strcpy(fileidfilename, dbname);
  strcat(fileidfilename, IDBASE);
}


/*
 * Strip_White_Begin(char *string)
 *
 * This procedure strips the white space from the beginning of the
 * string
 *
 * string (in/out): string to strip
 *
 */

char *Strip_White_Begin(char *string)
{
  int i, k;

  k = strlen(string);

  for (i=0; i<k; i++)
    if (string[i]!=32)
      break;

  return (string + i);
}


/*
 * Parse_Phoneme_List(pattern, phonlist)
 *
 * This procedure parses a list of phonemesm for the search 
 * procedure.
 *
 * pattern ( in): pattern to parse (list of phonemes)
 * phonlist(out): list of phonemes in parsed string
 * 
 */

void Parse_Phoneme_List(char *pattern, char **phonlist)
{
  char patternstring[100];
  char *token;
  int  length;
  int  count=0;

  strcpy(patternstring, pattern);
  length = strlen(patternstring);
  
  /* parse list of phonemes */
  if (length>0) {
    strcpy(phonlist[count], strtok(patternstring, " "));
    count++;
    do {
      token = strtok(NULL, " ");
      if (token == NULL)
	break;

      strcpy(phonlist[count], token);
      count++;
    } while ((int) (token-patternstring) < length);
  }

  strcpy(phonlist[count], "");
}


/*
 * Parse_Pattern(pattern, prephon, centerphon, postphon)
 *
 * This procedure parses the search pattern string in a set
 * phonemes used for the match.
 *
 * pattern (in): search pattern to parse
 * prephon    (out): pre phoneme in search
 * centerphon (out): center phoneme in search
 * postphon   (out): post phoneme in search
 *
 */

void Parse_Pattern(char *pattern, char **prephon, char **centerphon,
		   char **postphon)
{
  char prestring[100], *prestr;
  char centerstring[100], *centerstr;
  char poststring[100], *poststr;
  char patternstring[100];
  char substring[100];
  char *token;
  int  length;
  int  count = 0;

  strcpy(patternstring, " ");
  strcat(patternstring, pattern);
  length = strlen(patternstring);

  /* parse pre center and post phoneme lists */
  if (strlen(patternstring)>0) {
    strcpy(prestring, strtok(patternstring, ","));
    count++;
    do {
      token = strtok(NULL, ",");
      if (token == NULL)
	break;

      if (count==1)
	strcpy(centerstring, token);
      else if (count==2)
	strcpy(poststring, token);
      else
	break;

      count++;
    } while ((int) (token-patternstring) < length);
  }
  
  /* check which exist */
  if (count==1) {
    strcpy(centerstring, prestring);
    strcpy(prestring, "");
    strcpy(poststring, "");
  } else if (count==2)
    strcpy(poststring, "");

  prestr    = Strip_White_Begin(prestring);
  centerstr = Strip_White_Begin(centerstring);
  poststr   = Strip_White_Begin(poststring);

  /* parse lists of each phoneme */
  Parse_Phoneme_List(prestr, prephon);
  Parse_Phoneme_List(centerstr, centerphon);
  Parse_Phoneme_List(poststr, postphon);
}


/*
 * int Count_Strings(strlist)
 *
 * This functions calculates the string listing, ending in
 * null string.
 *
 * strlist (in): string listing 
 *
 */

int Count_Strings(char **strlist)
{
  int i;

  i = 0;
  while (strlen(strlist[i])) 
    i++;

  return (i);
}


/*
 * int Find_Context_Phoneme(contextfile, idfile, prephon, centerphon, postphon,
 *                          ContextPhon, FileID)
 *                          
 * This function search the contextual and file id data bases for
 * the occurrence of a specific sound with context, as described by
 * the prephone, centerphon, postphon strings. If any of the
 * strings do not exist a warning messages is returns in error string
 * 
 * contextfile (in): pointer to the file stream for the contextual input
 * idfile      (in): pointer to the file id data base stream
 * prephon     (in): string contaninng the pre contextual phoneme
 * centerphon  (in): the string containing the main search chain phoneme
 * postphon    (in): the string containing the post contextual phoneme
 * ContextPhon(out): the output is a record in the contextual data base
 * FileIDRec  (out): The from file id record, containing the file names etc
 *
 * Returns (-1) on error, or else 1
 *
 */

int Find_Context_Phoneme(FILE *contextfile, FILE *idfile, 
			 char *prephon, char *centerphon, char *postphon,
			 ContextRec *ContextPhon, FileIDRec  *FileID)
{
  ContextRec OneContext;                 /* One Record in context database */
  FileIDRec  OneWave;                    /* one record in fileid database  */
  PhonemeRec PhonIndex;                  /* index of phoneme record        */
  short      currphoneme;                /* char value of phoneme          */
  char       ph_prev[PHON_WIDTH];        /* prev phoneme in context        */
  char       ph_curr[PHON_WIDTH];        /* current phoneme in context     */
  char       ph_next[PHON_WIDTH];        /* next phoneme in context        */
  int        IDFile, found;              /* found the occurance            */
  int        current, next;              /* file record pointers           */
  static int flag = 1;                   /* search chain flags             */
  static int start, end, curlink;        /* current file pointers          */
  static short prevphoneme=0;            /* previous phoneme               */

  /* retrieve phoneme index */
  if ((currphoneme = Retrieve_Phoneme_Index(centerphon, &PhonemeIndex, 
					    &PreviousLink)) < 0) {
    fprintf(stderr, "Find_Context_Phoneme -> 1: %s\n", ErrorString);
    ErrorString = "Could not find phoneme in given index";
    return (-1);
  }
  
  if (currphoneme != prevphoneme) {
    flag = 1;
    prevphoneme = currphoneme;
  }

  /* retrieve phoneme chain file pointers */
  if (flag) {
    start = PhonemeIndex[currphoneme-1].startrec;
    end   = PhonemeIndex[currphoneme-1].endrec;
    fseek (contextfile, start * sizeof(ContextRec), SEEK_SET);
    curlink = start;
    flag = 0;
  }

  if (curlink == -1) {
    ErrorString = "At end of phoneme chain";
    return(-1);
  }
  
  /* search the data base */
  do {
    found = 1;
    if (Read_Context_Record(contextfile, ContextPhon, curlink) < 0) {
      fprintf(stderr, "Find_Context_Phoneme -> 2: %s\n", ErrorString);
      found = 0;
      break;
    }
      
    if (ContextPhon->ph_prev-1 >= 0) 
      strcpy(ph_prev, PhonemeIndex[ContextPhon->ph_prev-1].phoneme);
    else
      strcpy(ph_prev,"");

    if (ContextPhon->ph_curr-1 >= 0)
      strcpy(ph_curr, PhonemeIndex[ContextPhon->ph_curr-1].phoneme);
    else
      strcpy(ph_curr,"");

    if (ContextPhon->ph_next-1 >= 0)
      strcpy(ph_next, PhonemeIndex[ContextPhon->ph_next-1].phoneme);
    else
      strcpy(ph_next,"");
    
    if (strlen(prephon)>0)
      found = found && (strcmp(prephon, ph_prev) == 0);
    if (strlen(centerphon)>0)
      found = found && (strcmp(centerphon, ph_curr) == 0);
    if (strlen(postphon)>0)
      found = found && (strcmp(postphon, ph_next) == 0);
    if (found) {
      fprintf(logfile, "%d\n", curlink);
      if (Read_FileID_Record(idfile, FileID, ContextPhon->from_file_id-1)<0) {
	fprintf(stderr, "Find_Context_Phoneme -> 3: %s\n", ErrorString);
	found = 0;
	break;
      }
#ifdef DEBUG
      printf("%s  -- %s  -- %s -- %d -- %d\n", ph_prev, ph_curr, ph_next,
	     curlink, ContextPhon->nextlink); 
      printf("%d -- %d -- %d -- %d\n",
	     ContextPhon->prev_start, ContextPhon->startloc,
	     ContextPhon->endloc, ContextPhon->next_end);
#endif
    }

    curlink = ContextPhon->nextlink;
  } while ((!found) && (curlink != -1));

  if (found==0) 
    return (-1);

  return (1);
}


/*
 * SegnentList *Add_Segment(lolasegment, begin, end, phoneme)
 *
 * This procedure add a segment to the end of the segment list
 * and then returns to the end fo the segment list.
 *
 * lolasegment (in): the segment list to use
 * begin       (in): the starting location
 * end         (in): ending location in units
 * phoneme     (in): pointer to phone label
 *
 * on error returns a NULL pointer
 *
 */

SegmentList *Add_Segment(SegmentList *lolasegment, int begin, int end,
			 char *phoneme)
{
  SegmentList *tmp;
  SegmentList *pointer;

  pointer = lolasegment;
  if (pointer->sl_forward != NULL) {
    do {
      pointer = pointer->sl_forward;
    } while (pointer->sl_forward);
  }

  if (end>begin) {                       /* add only non-null segments   */
    if (!(tmp = (SegmentList *) malloc(sizeof(SegmentList)))) {
      if (errno < sys_nerr)
	ErrorString = sys_errlist[errno];
      else
	ErrorString = " SegmentList tmp --> malloc failed";
      fprintf(stderr, "Add_Segment -> 1: %s\n", ErrorString);
      return (NULL);
    }

    tmp->sl_label = (char *) malloc(strlen(phoneme) + 1);
    if (!(tmp->sl_label)) {
      if (errno < sys_nerr)
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "tmp->sl_label --> malloc failed";
      fprintf(stderr, "Add_Segment -> 2: %s\n", ErrorString);
      return (NULL);
    }
    
    tmp->sl_begin = begin;
    tmp->sl_end   = end;
    strcpy(tmp->sl_label, phoneme);
    tmp->sl_forward =  NULL;
    tmp->sl_backward = pointer;
    pointer->sl_forward = tmp;
  } else
    tmp = lolasegment;

  return (tmp);
}


/*
 * int RetrieveSpeech(speechfilename, begin, end, units)
 *
 * This function retrieve the speech and stores it in a continually
 * growing array of speech.
 *
 * speechfilename (in): name of speech file 
 * begin          (in): start index in units
 * end            (in): end index in units
 * units          (in): scale for units 
 *
 * On error returns -1, else +1
 *
 */

int RetrieveSpeech(char *speechfilename, int begin, int end, float units)
{
  int   length;                        /* number of samples in speech signal */
  int   start, stop;                   /* boundaries of speech signal        */
  static char filename[200]="";        /* filename of current speech         */
  static short *speech=NULL;

  if (createwav) {
    if (strcmp(filename, speechfilename) != 0) {
      if (speech!=NULL)
	free((char *) speech);
      if ((length = SpeechRead(speechfilename, &speech, &Sf))<0) {
	fprintf(stderr, "RetrieveSpeech -> 1: %s -- %s\n", speechfilename,
		ErrorString);
	return (-1);
      }
      strcpy(filename, speechfilename);
    }
    
    start = (int) (((float) begin)*units*Sf/1000.0);
    stop  = (int) (((float) end)*units*Sf/1000.0);
    
    fwrite(speech + start, sizeof(short), stop-start, wavfile);
    totalspeech += stop-start;
  }

  return (1);
}


/*
 * AddSilence()
 * 
 * This procedure adds the silence parts to the speech file
 *
 * on error returns -1, else +1
 *
 */

int AddSilence()
{
  short *silence;
  int   length;

  if (createwav) {
    length = (int) (((float) pause)*(Sf/1000.0));
    if (!(silence = (short *) calloc(length, sizeof(short)))) {
      if (errno < sys_nerr)
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "silence --> calloc failed";
      fprintf(stderr, "AddSilence -> 1: %s", ErrorString);
      return (-1);
    }
    fwrite(silence, sizeof(short), length, wavfile);
    totalspeech += length;
    free((char *) silence);
  }

  return (1);
}
    

/*
 * int Build_SegmentList(lolasegment, contextfile, idfile,
 *                       prephon, centerphon, postphon)
 *
 * Build the data base search segment list
 *
 * lolasegment(out): pointer to the segment list being created
 *                   must be at the head of segment list
 * contextfile (in): contextual data base file
 * idfile      (in): from file id data base file stream
 * prephon     (in): pre contextual phoneme label
 * centerphon  (in): center contextual phoneme label
 * postphon    (in): post contextual phoneme label
 *
 * on error returns -1, else +1
 */

int Build_SegmentList(SegmentList **lolasegment, 
		      FILE *contextfile, FILE *idfile, 
		      char *prephon, char *centerphon, char *postphon)
{
  int i, j, k;                      /* loop counter variables         */
  int endsegment;                   /* index of previous segment      */
  ContextRec OneContext;            /* retrieved context record in DB */
  FileIDRec  OneWave;               /* from file name in fileid DB    */
  SegmentList *tmp;                 /* new addition to segment list   */
  char prestr[80];
  char censtr[80];
  char poststr[80];                 /* phoneme labels                 */
  float scale;                      /* scaling factor for lola files  */
  
  for (i=0; (i<numsegments+seq); i++) {
    if (Find_Context_Phoneme(contextfile, idfile, 
			     prephon, centerphon, postphon,
			     &OneContext, &OneWave) < 0) {
      fprintf(stderr, "Build_SegmentList -> 1: %s\n", ErrorString);
      break;
    }

    if (OneWave.ms != 1.0) {
      scale = OneWave.ms/1.0;
      OneContext.prev_start = (int) (((float) OneContext.prev_start)*scale);
      OneContext.startloc   = (int) (((float) OneContext.startloc)*scale);
      OneContext.endloc     = (int) (((float) OneContext.endloc)*scale);
      OneContext.next_end   = (int) (((float) OneContext.next_end)*scale);
    }
    
    if (OneContext.ph_prev>0) 
      strcpy(prestr, PhonemeIndex[OneContext.ph_prev-1].phoneme);
    else 
      strcpy(prestr, "");

    if (OneContext.ph_curr>0)
      strcpy(censtr, PhonemeIndex[OneContext.ph_curr-1].phoneme);
    else
      strcpy(censtr, "");

    if (OneContext.ph_next>0)
      strcpy(poststr, PhonemeIndex[OneContext.ph_next-1].phoneme);
    else
      strcpy(poststr, "");

    /* add pre contextual segment */
    if ((pre) && (i>=seq)) {
      if ((*lolasegment = Add_Segment(*lolasegment, OneContext.prev_start,
				 OneContext.startloc, prestr)) == NULL) {
	fprintf(stderr, "Build_SegmentList -> 2: %s\n", ErrorString);
	return (-1);
      }
      if (RetrieveSpeech(OneWave.wavfilename, OneContext.prev_start, 
			 OneContext.startloc, OneWave.ms) < 0) {
	fprintf(stderr, "Build_SegmentList -> 3: %s\n", ErrorString);
	return (-1);
      }
      endsegment = OneContext.startloc;
    }
    
    /* add center contextual segment */
    if (i>=seq) {
      if ((*lolasegment = Add_Segment(*lolasegment, OneContext.startloc,
				      OneContext.endloc, censtr)) == NULL) {
	fprintf(stderr, "Build_SegmentList -> 4: %s\n", ErrorString);
	return (-1);
      }
      if (RetrieveSpeech(OneWave.wavfilename, OneContext.startloc, 
			 OneContext.endloc, OneWave.ms) < 0) {
	fprintf(stderr, "Build_SegmentList -> 5: %s\n", ErrorString);
	return (-1);
      }
      endsegment = OneContext.endloc;
    }

    /* add post contextual segment */
    if ((post) && (i>=seq)) {
      if ((*lolasegment = Add_Segment(*lolasegment, OneContext.endloc,
				      OneContext.next_end, poststr)) == NULL) {
	fprintf(stderr, "Build_SegmentList -> 6: %s\n",ErrorString);
	return (-1);
      }
      if ((RetrieveSpeech(OneWave.wavfilename, OneContext.endloc, 
			  OneContext.next_end, OneWave.ms) < 0)) {
	fprintf(stderr, "Build_SegmentList -> 7: %s\n", ErrorString);
	return (-1);
      }
      endsegment = OneContext.next_end;
    }

    if (i>=seq) {
      if (AddSilence() < 0) {
	fprintf(stderr, "Build_SegmentList -> 8: %s\n", ErrorString);
	return (-1);
      }
      if ((*lolasegment = Add_Segment(*lolasegment, endsegment,
				      endsegment+pause, "notting")) == NULL) {
	fprintf(stderr, "Build_SegmentList -> 8: %s\n", ErrorString);
	return (-1);
      }
    }
  }
}
		

/*
 * Fix_Segment_List(SegmentList *lolasegment)
 *
 * this procedure sweeps through the segment list
 * and sets the borders correct for start and end mgn
 *
 * lolasegment (in/out): the segment list to align
 *
 */

void Fix_Segment_List(SegmentList *lolasegment)
{
  int start, end;
  SegmentList *segpointer;

  segpointer = lolasegment->sl_forward;
  start = segpointer->sl_begin;
  end   = segpointer->sl_end-start;
  start = 0;
  
  while (segpointer) {
    segpointer->sl_begin = start;
    segpointer->sl_end   = end;

    start = segpointer->sl_end;
    segpointer = segpointer->sl_forward;
    if (segpointer) 
      end = start + (segpointer->sl_end - segpointer->sl_begin);
  }
}


/*
 * WriteWaveFile()
 *
 * This procedure writes the waveform file to disk in NIST
 * format
 *
 * On error returns -1, else +1
 *
 */

int WriteWaveFile()
{
  short *speech;
  FILE  *infile;

  if (!(speech = (short *) malloc(totalspeech * sizeof(short)))) { 
    if (errno < sys_nerr)
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "WriteWavFile: speech --> malloc failed";
    fprintf(stderr, "WriteWavFile -> 1: %s\n", ErrorString);
    return (-1);
  }

  if (!(infile = fopen(wavfilename, "r"))) {
    if (errno < sys_nerr)
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Could not access waveform data";
    fprintf(stderr, "WriteWavFile -> 2: %s\n", ErrorString);
    free ((char *) speech);
    return (-1);
  }

  fread(speech, sizeof(short), totalspeech, infile);
  fclose(infile);

  if (WavWrite(wavfilename, speech, totalspeech, Sf, 0) < 0) {
    fprintf(stderr, "Write_WavFile -> 3: %s\n", ErrorString);
    free ((char *) speech);
    return (-1);
  }

  free((char *) speech);
  return (1);
}
    

/*
 * master magician code
 *
 */

main (int argc, char **argv)
{
  char **prephonemes;                  /* previous context phonemes   */
  char **centerphonemes;               /* center context phonemes     */
  char **postphonemes;                 /* post context phonemes       */
  int  prenum;                         /* number of pre contextual ph */
  int  centernum;                      /* number of center contextual */
  int  postnum;                        /* number of center contextual */
  SegmentList *contextlola;            /* contextual lola file        */
  SegmentList *curr_lola;              /* pointer for segment list    */

  int  flag = 1, success;
  int  i, j, k;

  /* process command line directives */
  get_comline(argc, argv);

  machineorder = LittleIndian();

  /* do data bases setup phase */
  if (Read_Index_File(indexfilename, &PhonemeIndex, &PreviousLink) < 0) {
    fprintf(stderr, "Read_Index_File: %s\n", ErrorString);
    exit(1);
  }
  
  if (Open_Contextual_Base(&contextfile, contextfilename) < 0) {
    fprintf(stderr, "Open_Contextual_Base: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    exit(1);
  }
  
  if (Open_FileID_Base(&idfile, fileidfilename) < 0) {
    fprintf(stderr, "Open_FileID_Base: %s\n", ErrorString);
    free((char *) PhonemeIndex);
    free((char *) PreviousLink);
    Close_Contextual_Base(&contextfile);
    exit(1);
  }

  /* allocate memory for phoneme listings */
  if (!(prephonemes = (char **) Alloc2d(255, PHON_WIDTH, sizeof(char)))) {
    if (errno < sys_nerr) 
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "prephonemes --> Alloc2d failed";
    fprintf(stderr, "main: %s\n", ErrorString);
    exit(1);
  }
  if (!(centerphonemes = (char **) Alloc2d(255, PHON_WIDTH, sizeof(char)))) {
    if (errno < sys_nerr) 
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "centerhonemes --> Alloc2d failed";
    fprintf(stderr, "main: %s\n", ErrorString);
    exit(1);
  }
  if (!(postphonemes = (char **) Alloc2d(255, PHON_WIDTH, sizeof(char)))) {
    if (errno < sys_nerr) 
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "posthonemes --> Alloc2d failed";
    fprintf(stderr, "main: %s\n", ErrorString);
    exit(1);
  }
  
  Parse_Pattern(pattern, prephonemes, centerphonemes, postphonemes);
  
  prenum     = Count_Strings(prephonemes);
  centernum  = Count_Strings(centerphonemes);
  postnum    = Count_Strings(postphonemes);

  /* start search log */
  if (!(logfile = fopen(logfilename, "w"))) {
    if (errno < sys_nerr) 
      ErrorString = sys_errlist[errno];
    else
      ErrorString = "Could not create log file";
    fprintf(stderr, "logfile error: %s\n", ErrorString);
    exit(1);
  }

  fprintf(logfile, "%d  %d  %d\n", prenum, centernum, postnum);
  fprintf(logfile, "%d  %d\n", pre, post);
  
  for (i=0; i<prenum; i++)
    fprintf(logfile, "%s ", prephonemes[i]);
  fprintf(logfile, "\n");
  for (i=0; i<centernum; i++)
    fprintf(logfile, "%s ", centerphonemes[i]);
  fprintf(logfile, "\n");
  for (i=0; i<postnum; i++)
    fprintf(logfile, "%s ", postphonemes[i]);
  fprintf(logfile, "\n\n");

  if (createwav)
    if (!(wavfile = fopen(wavfilename, "w"))) {
      if (errno < sys_nerr)
	ErrorString = sys_errlist[errno];
      else
	ErrorString = "Could not create waveform file";
      fprintf(stderr, "wavfile error: %s\n", ErrorString);
      exit(1);
    }

  /* create segment list */
  contextlola = NewSegmentList();
  curr_lola   = contextlola;
  Sf = 8000.0;
  curr_lola   = Add_Segment(curr_lola, 0, pause, "notting");
  AddSilence();
  
  i = 0;
  while (strlen(centerphonemes[i])) {
    j = 0;
    while (strlen(prephonemes[j])) {
      k = 0;
      while (strlen(postphonemes[k])) {
	if (Build_SegmentList(&curr_lola, contextfile, idfile, 
	      prephonemes[j], centerphonemes[i], postphonemes[k]) < 0) {
	  fprintf(stderr, "Build_SegmentList: %s\n", ErrorString);
	  break;
	}
	  
	k++;
      }
      if (k==0) 
	if (Build_SegmentList(&curr_lola, contextfile, idfile, 
			      prephonemes[j], centerphonemes[i], "") < 0) {
	  fprintf(stderr, "Build_SegmentList: %s\n", ErrorString);
	  break;
	}
      j++;
    }

    if (j==0) {
      k = 0;
      while (strlen(postphonemes[k])) {
	if (Build_SegmentList(&curr_lola, contextfile, idfile, 
			  "", centerphonemes[i], postphonemes[k]) < 0) {
	  fprintf(stderr, "Build_SegmentList: %s\n", ErrorString);
	  break;
	}
	k++;
      }
      if (k==0) 
	if (Build_SegmentList(&curr_lola, contextfile, idfile, 
			  "", centerphonemes[i], "") < 0) {
	  fprintf(stderr, "Build_SegmentList: %s\n", ErrorString);
	  break;
	}
    }
    i++;
  }

  Fix_Segment_List(contextlola); 
  if (WriteSegmentList(lolafilename, contextlola, 1.0) < 0) 
    fprintf(stderr, "WriteSegmentList: %s\n", ErrorString);

  /* save all files and exit */
  fclose(logfile);

  if (createwav) {
    fclose(wavfile);
    WriteWaveFile();
  }

  if ((createdft) && (createwav)) 
    if (CreateDFT(wavfilename, dftfilename) < 0) 
      fprintf(stderr, "CreateDFT: %s\n", ErrorString);

  Close_Contextual_Base(&contextfile);
  Close_FileID_Base(&idfile);

  free((char *) PhonemeIndex);
  free((char *) PreviousLink);
}


