/*
 * $Id: makedft.c,v 2.2 1993/06/10 17:14:34 johans Exp $
 *
 * makedft.c
 *
 */

/* Standard C library include file directives */
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <string.h>
#include <math.h>

/* Speech Libary include file directives */
#include <speech.h>


/* Local module header declarations */
static void WriteDFT(FILE *fp, short **dft, int rows, int cols, 
		     float winsize_ms, float inc_ms, float rate, 
		     double start_time );


/* Global constant defintions */
#define DEF_WINDOW_SIZE		(10.0)	/* in ms */
#define DEF_INCREMENT		(3.0)	/* in ms */
#define DEF_COEFFS		(64)

#define DEF_ALPHA		(0.94)

#define TRUE		(1)
#define FALSE		(0)

/* global variable declarations */
char InFileName[200]="-";                 /* Input file name / stdin      */
char OutFileName[200]="-";                /* Output file name / stdout    */
int  numcoeffs=64;                       /* number of dft coefficients   */
double start_time = 0.0;                 /* starting time in ms          */
double end_time = 0.0;                   /* ending time                  */
double winsize_ms = 10.0;                /* window size in milliseconds  */
double inc_ms = 3.0;                     /* window increment in millisec */
float alpha = 0.94;                      /* preemphasis filter factor    */


/*
 * usage()
 *
 * display usage paramters on screen
 *
 */

void usage() 
{
  fprintf(stderr, "Usage: <stdin> | makedft [options] | <stdout>\n");
  fprintf(stderr, "\n");
  fprintf(stderr, "Where options can be any of the following:\n");
  fprintf(stderr, "-f input file name [stdin]\n");
  fprintf(stderr, "-o output file name [stdout]\n");
  fprintf(stderr, "-s start time [0.0]\n");
  fprintf(stderr, "-e end time [length of file]\n");
  fprintf(stderr, "-w window size in milli seconds [10.0]\n");
  fprintf(stderr, "-i increment in milli seconds [3.0]\n");
  fprintf(stderr, "-n number of dft coefficients [64]\n");
  fprintf(stderr, "-a pre-emphasis filter factor [0.94]\n");
  fprintf(stderr, "-p activate pre-emphasis filter\n");
  fprintf(stderr, "-h This help message\n");
  exit(1);
}


/*
 * get_comline(int argc, char **argv)
 *
 * read and interpret command line
 *
 */

void get_comline(int argc, char **argv)
{
  int c;
  extern int optind;
  extern char *optarg;

  while( (c = getopt(argc, argv, "f:o:s:e:w:i:n::a:p")) != EOF ) {
    switch( c ) {
    case 'f': /* name of outputfile */
      strcpy(InFileName, optarg);
      break;
	
    case 'o': /* name of outputfile */
      strcpy(OutFileName, optarg);
      break;
	
    case 's': /* starting time option */
      start_time = atof( optarg );
      break;

    case 'e': /* required ending time */
      end_time = atof( optarg );
      break;

    case 'w': /* window size in milliseconds */
      winsize_ms = atof( optarg );
      break;
		
    case 'i': /* increment of window in milliseconds */
      inc_ms = atof( optarg );
      break;

    case 'n': /* number of coefficients to generate */
      numcoeffs = atoi( optarg );
      break;
	
    case 'a': /* alpha factor for preemphasis */
      alpha = (float) atof( optarg );
      break;
    
    case 'p': /* switch to preemphasis */
      alpha = DEF_ALPHA;
      break;
	
    case 'h':
    default:
      usage();
      exit( 1 );
    }
  }
}


main(int argc, char **argv)
{
  short **dft;
  short *data;
  int start, end;
  int len;

  FILE *fp;
  int winsize;
  float rate;
  int inc;
  int numframes;
  int c;
  int i;

  alpha = 0.0;
  get_comline(argc, argv);
  if( ( len = SpeechRead( InFileName, &data, &rate ) )  < 0 ) {
    fprintf( stderr, "makedft: %s: ", ErrorString );
    exit( 1 );
  }

  /*
   * Find increment and window size
   */
  inc = (int) (((double)(rate/1000.0)) * inc_ms);
  if(((double)inc) != (((double)(rate/1000.0)) * inc_ms)) {
    fprintf( stderr, "makedft: increment not and \n" );
    exit( 1 );
  }

  winsize = (int) (((double)(rate/1000.0)) * winsize_ms);
  if(((double)winsize) != 
     (((double)(rate/1000.0))*winsize_ms)){
    fprintf( stderr, "makedft: window size not and \n" );
    exit( 1 );
  }

  if( start_time == 0.0 ) {
    start = 0;
  } else {
    start = (int) (((double)(rate/1000.0)) * start_time);
  }
  
  if( end_time == 0.0 ) {
    end = len;
  } else {
    end = (int) (((double)(rate/1000.0)) * end_time);
  }
  
  if( end > len ) {
    end = len;
  }

  len = end - start;

  dft = ComputeDFT( data, end, start, winsize, inc, numcoeffs, alpha, 
		   &numframes);
  if( strcmp(OutFileName, "-") == 0 ) {
    fp = stdout;
  } else {
    if( ( fp = fopen( OutFileName, "w" ) ) == NULL ) {
      fprintf( stderr, "makedft: " );
      perror( NULL );
      exit( 1 );
    }
  }

  WriteDFT( fp, dft, numcoeffs, len / inc, winsize_ms, 
	   inc_ms, rate, start_time );
  
  fclose( fp );
  free((char *) data);
  Free2d((char **) dft);
  exit( 0 );
}


static void WriteDFT(FILE *fp, short **dft, int rows, int cols, 
		     float winsize_ms, float inc_ms, float rate, 
		     double start_time)
{
  AsciiHdr header[20];
  float *hz;
  int tmp;
  float ftmp;
  int i;

  hz = (float *) calloc(rows, sizeof(float));

  if( hz == NULL ) {
    fprintf( stderr, "makedfT: WriteDFT:" );
    perror( NULL );
    exit( 1 );
  }

  for( i = 0; i < rows; i++ ) {
    *(hz+i) = (((float)i)/((float)(rows-1))) * 
      (rate/2.0);
  }

  i = 0;
  header[i].ah_key = "Rows";
  header[i].ah_vlen = 1;
  header[i++].ah_value = IntArray2StrArray( &rows, 1 );

  header[i].ah_key = "Columns";
  header[i].ah_vlen = 1;
  header[i++].ah_value = IntArray2StrArray( &cols, 1 );

  header[i].ah_key = "Type";
  header[i].ah_vlen = 1;
  header[i++].ah_value = Str2StrArray( "short" );

  header[i].ah_key = "RowLabels";
  header[i].ah_vlen = rows;
  header[i++].ah_value = FloatArray2StrArray( hz, rows );

  ftmp = rate;
  header[i].ah_key = "SamplesPerSecond";
  header[i].ah_vlen = 1;
  header[i++].ah_value = FloatArray2StrArray( &ftmp, 1 );

  ftmp =  inc_ms;
  header[i].ah_key = "Increment";
  header[i].ah_vlen = 1;
  header[i++].ah_value = FloatArray2StrArray( &ftmp, 1 );

  ftmp = winsize_ms;
  header[i].ah_key = "WindowWidth";
  header[i].ah_vlen = 1;
  header[i++].ah_value = FloatArray2StrArray( &ftmp, 1 );

  ftmp = (float)start_time;
  header[i].ah_key = "StartTime";
  header[i].ah_vlen = 1;
  header[i++].ah_value = FloatArray2StrArray( &ftmp, 1 );

  header[i].ah_key = NULL;
  header[i].ah_vlen = 0;
  header[i++].ah_value = NULL;

  if( fTdatWrite( fp, (char **) dft, header ) < 0 ) {
    fprintf( stderr, "makedft: %s\n", ErrorString );
    exit( 1 );
  }
}
