static char *rcsident = "$Header: /projects/cslu/speech/work/src/bin/auto_lyre/RCS/Lola.c,v 4.4 1993/05/28 21:27:40 johans Exp $";

/*
 * Lola.c - Lola widget
 *------------------------------------------------------------*
 * Copyright 1988, Fil Alleva and Carnegie Mellon University
 *------------------------------------------------------------*
 * HISTORY
 * 14-Aug-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	Added in window scrolling. If edit proceeds beyond the window
 *	boundry then the display will scroll.
 *
 * 28-Jun-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	- Fixed boundry condition on redisplay after segment deletion.
 *	- Fixed bug in MakeList() that failed to NULL terminate the new
 *	  list.
 *
 *  7-Feb-89  Fil Alleva (faa) at Carnegie-Mellon University
 *
 * 13-Apr-93 Johan Schalkwyk at Oregon Graduate Institue changed
 *       to ANSI style header declarations
 *
 */

/* Standard C library include file directives */
#include <c.h>
#include <sys/file.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>

/* Speech library include file directives */
#include <speech.h>

/* X library include file directives */
#include <X11/Xos.h>
#include <X11/cursorfont.h>
#include <X11/IntrinsicP.h>
#include <X11/StringDefs.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xaw/List.h>
#include <X11/Shell.h>
#include <X11/Xaw/AsciiText.h>

/* Lyre include file directives */
#include <LolaP.h>
#include <Scale.h>
#include <LyreDisp.h>
#include <ToolUtil.h>


/*
 *  RESCOURCES
 */

static int Zero = 0;
static Boolean defTRUE = TRUE;

#define offset(field) XtOffset(LolaWidget, lola.field)
#define Coffset(field) XtOffset(LolaWidget, core.field)

static XtResource resources[] = { 
   {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
        offset(foreground_pixel), XtRString, "Black"},
   {XtNfont, XtCFont, XtRFont, sizeof(Font),
	offset (font), XtRString, "Fixed"},
   {XtNinputSeg, XtNinputSeg, XtRPointer, sizeof(caddr_t),
	offset (input_seg), XtRPointer, (caddr_t) NULL},
   {XtNseg, XtCParameter, XtRPointer, sizeof(caddr_t),
	offset (seg), XtRPointer, (caddr_t) NULL},
   {XtNhscaleWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	offset (hscale_widget), XtRPointer, NULL},
   {XtNvscaleWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	offset (vscale_widget), XtRPointer, NULL},
   {XtNlabelFileName, XtCParameter, XtRString, sizeof(String),
	offset (labelFileName), XtRString, "Lyre.Label"},
};
#undef offset


/* Private procedure header declarations */
static void Initialize (Widget request_in, Widget new_in, ArgList arglist, 
			Cardinal *num_args);
static void Realize (Widget w_in, XtValueMask *valueMask, 
		     XSetWindowAttributes *attrs);
static void Redisplay (Widget w_in, XEvent *expose_in, Region region);
static void Resize (Widget gw);
static void Destroy (Widget gw);
static Boolean SetValues (Widget current_in, Widget request_in, Widget new_in,
			  ArgList arglist, Cardinal *num_args);

static void ClearSegment (LolaWidget w, SegmentList *seg, int expose);
static void AddSeg (LolaWidget w, XButtonEvent *event, String *params,
		    Cardinal *num_params);
static void ContinueEdit (LolaWidget w, XButtonEvent *event, String *params,
			  Cardinal *num_params);
static void DeleteSeg (LolaWidget w, XKeyEvent *event, String *params,
		       Cardinal *num_params);
static void LabelSeg (LolaWidget w, XButtonEvent *event, String *params,
		      Cardinal *num_params);
static void StartEdit (LolaWidget w, XButtonEvent *event, String *params,
		       Cardinal *num_params);
static void KeyInput (LolaWidget w, XKeyEvent *event, String *params,
		      Cardinal *num_params);
static void DeleteChar (LolaWidget w, XKeyEvent *event, String *params,
			Cardinal *num_params);
static void NextSeg (LolaWidget w, XKeyEvent *event, String *params,
		     Cardinal *num_params);
static void PrevSeg (LolaWidget w, XKeyEvent *event, String *params,
		     Cardinal *num_params);
static void SetCurEdit (LolaWidget w, XButtonEvent *event, String *params,
			Cardinal *num_params);
static void NoAction();
static char **MakeList (char *file);
static char *AddChar (char *str, char c);


static char defaultTranslations[] = "\
	<Key>BackSpace:	TrackMouse(l) DeleteChar() TrackMouse(e) \n\
	<Key>Delete:	TrackMouse(l) DeleteChar() TrackMouse(e) \n\
	<Key>Return:	TrackMouse(l) NextSeg() TrackMouse(e) \n\
	<Key>Linefeed:	TrackMouse(l) NextSeg() TrackMouse(e) \n\
	<Key>Right:	TrackMouse(l) NextSeg() TrackMouse(e) \n\
	<Key>Left:	TrackMouse(l) PrevSeg() TrackMouse(e) \n\
	Ctrl<Key>D:	TrackMouse(l) DeleteSeg() TrackMouse(e) \n\
	:<Key>:		TrackMouse(l) KeyInput() TrackMouse(e) \n\
        <Btn2Down>:	TrackMouse(l) SetCurEdit() TrackMouse(e) \n\
	<Btn1Down>:     TrackMouse(l) StartEdit() \n\
	<Btn1Motion>:   ContinueEdit() \n\
	<Btn1Up>:	TrackMouse(e) \n\
	<Btn3Down>:     TrackMouse(l) AddSeg() StartEdit() \n\
	Shift<Btn3Down>:  TrackMouse(l) AddSeg() StartEdit() \n\
	<Btn3Motion>:   ContinueEdit() \n\
	<Btn3Up>:	TrackMouse(e) \n\
	<Enter>:       	TrackMouse(e) \n\
	<Leave>:       	TrackMouse(l) \n\
	<Motion>: 	TrackMouse(m) \n\
";

static XtActionsRec actions[] = {
	{"TrackMouse",	(XtActionProc) LyreDispTrackMouse},
	{"AddSeg",	(XtActionProc) AddSeg},
	{"DeleteSeg",	(XtActionProc) DeleteSeg},
	{"LabelSeg",	(XtActionProc) LabelSeg}, 
	{"ContinueEdit",(XtActionProc) ContinueEdit},
	{"StartEdit",	(XtActionProc) StartEdit},
	{"KeyInput",    (XtActionProc) KeyInput},
	{"DeleteChar",  (XtActionProc) DeleteChar},
	{"NextSeg",     (XtActionProc) NextSeg},
	{"PrevSeg",     (XtActionProc) PrevSeg},
	{"NoAction",    (XtActionProc) NoAction},
	{"SetCurEdit",  (XtActionProc) SetCurEdit},
	{NULL,NULL}
};

LolaClassRec lolaClassRec = {
  {
    (WidgetClass) &lyreDispClassRec,	/* superclass		  */	
    "Lola",				/* class_name		  */
    sizeof(LolaRec),			/* size			  */
    NULL,				/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    Realize,				/* realize		  */
    actions,				/* actions		  */
    XtNumber(actions),			/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    (TRUE | XtExposeGraphicsExpose),	/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    Destroy,				/* destroy		  */
    Resize,				/* resize		  */
    Redisplay,				/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    defaultTranslations,		/* tm_table		  */
    NULL,				/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL,				/* extension		  */
  },  /* CoreClass fields initialization */
  {
    0,		                 	/* unused */
  },  /* LolaClass fields initialization */
};

WidgetClass lolaWidgetClass = (WidgetClass) &lolaClassRec;

/*
 * Initialize (request, new)
 *
 * Initialize a lola widget, label creating and marking widget
 *
 */

static void Initialize (Widget request, Widget new, ArgList arglist,
			Cardinal *num_args)
{
    LolaWidget          w = (LolaWidget) new;
    XtGCMask            valuemask;
    XGCValues           myXGCV;

    valuemask = GCForeground | GCBackground | GCLineWidth | GCFont;

    myXGCV.foreground = w->lola.foreground_pixel;
    myXGCV.background = w->core.background_pixel;
    myXGCV.line_width = 0;
    myXGCV.font = w->lola.font;
    w->lola.lolaGC = XtGetGC ((Widget) w, valuemask, &myXGCV);

    w->lola.font_info = XQueryFont (XtDisplay (w), w->lola.font);

    w->lola.seg = 0;
    w->lola.input_seg = 0;
    w->lola.cur_edit = 0;
    w->lola.shell = NULL;
    w->lola.labelList = NULL;

    SetScale((LyreDispWidget)w->lola.hscale_widget, w->lyreDisp.secs_ppix);
    InitList (w);
}


/*
 * Realize (w, valueMask, attrs)
 *
 * After lola widget has been placed, and text added, reliaze, display
 * full widget.
 *
 */

static void Realize (Widget w_in, XtValueMask *valueMask, 
		     XSetWindowAttributes *attrs)
{
    LolaWidget w = (LolaWidget) w_in;
    attrs->cursor = XCreateFontCursor( XtDisplay(w), XC_xterm );
    *valueMask |= CWCursor;

    XtCreateWindow ((Widget) w, InputOutput, (Visual *) CopyFromParent,
		    *valueMask, attrs);
    /*
     * Popup the label shell only if we made one !
     */
    if (w->lola.shell)
	XtPopup (w->lola.shell, XtGrabNone);
    Resize ((Widget) w);
}


/*
 * Resize (gw) 
 *
 * Request for resizing of lola label widget
 *
 */

static void Resize (Widget gw)
{
    LolaWidget         w = (LolaWidget) gw;

    /* don't do this computation if window hasn't been realized yet. */
    if (XtIsRealized ((Widget) w)) {
      
    }
}


/* 
 * Destroy (gw)
 *
 * Destroy a lola label widget
 *
 */

static void Destroy (Widget gw)
{
     LolaWidget w = (LolaWidget) gw;
/*   XFreeFontInfo(w->lola.font_info); */
     XtDestroyGC (w->lola.lolaGC);
}


/*
 * Redisplay (w, expose, region)
 *
 * Redisplay loal label widgets after and expose event
 *
 */

#define FRM2PIX(x)	((int)((x)/w->lyreDisp.units_ppix))

static void Redisplay (Widget w_in, XEvent *expose_in, Region region)
{
    LolaWidget         w = (LolaWidget) w_in;
    XExposeEvent *expose = (XExposeEvent *) expose_in;
    SegmentList *sl;
    register int end_x;
    register int start_x;
    register int b_pix, e_pix;
    int  virt_x = 0;
    int  raise;


    if( w->lola.seg == NULL ) {
	return;
    }
    virt_x = w->lyreDisp.virtual_x;

    end_x = expose->x + expose->width + virt_x;
    start_x = expose->x + virt_x;

    raise = 0;			/* raise = 1 means move label up */
    sl = w->lola.seg->sl_forward;

    while( sl != NULL ) {
	b_pix = FRM2PIX(sl->sl_begin);
	e_pix = FRM2PIX(sl->sl_end);
	if ((e_pix >= start_x) && (b_pix <= end_x))
	  {
	    if( sl == w->lola.cur_edit ) {
		    RedisplayCurEdit(w, sl,raise);
	    } else {
		    redisplay_seg (w, sl,raise);
	    }
	  }
	raise = 1 - raise;
	sl = sl->sl_forward;
    }
}


/*
 * redisplay_seg (w, sl, raise)
 *
 */

redisplay_seg (LolaWidget w, register SegmentList *sl, int raise)
{
    register int        box_x;
    int                 label_y;
    int                 box_width;
    int                 virt_x = 0;

    virt_x = w->lyreDisp.virtual_x;

    label_y = w->core.height - 4 - w->lola.font_info->descent;

   /*
    * Draw the box 
    */
    box_width = FRM2PIX(sl->sl_end - sl->sl_begin);
    box_x = FRM2PIX(sl->sl_begin) - virt_x;
    XDrawRectangle (XtDisplay (w), XtWindow (w), w->lola.lolaGC,
			box_x, 2, box_width, w->core.height - 4);

    if (sl->sl_label)
	XDrawString (XtDisplay (w), XtWindow (w), w->lola.lolaGC,
		     box_x + 2,
		     label_y - (raise * (w->lola.font_info->ascent + 
			w->lola.font_info->descent)),
		     sl->sl_label, strlen (sl->sl_label));
}

/*
 * Boolean SetValues (current, request, new)
 *
 */

#define MyXtSetArg(args,i,arg,val)	{XtSetArg (args[i], arg, val); i++;}

static Boolean SetValues (Widget current_in, Widget request_in, Widget new_in,
			  ArgList arglist, Cardinal *num_args)
{
    LolaWidget current = (LolaWidget) current_in;
    LolaWidget request = (LolaWidget) request_in;
    LolaWidget new     = (LolaWidget) new_in;

    SegmentList *tsl;
    Boolean     redisplay = FALSE;

    if (new->lola.input_seg) {
	if( new->lola.seg != NULL ) {
		FreeSegmentList( new->lola.seg );
	}
	new->lola.seg = new->lola.input_seg;
	new->lola.input_seg = 0;
	new->lola.cur_edit = 0;
	redisplay = TRUE;
    }
    if ((new->lyreDisp.secs_ppix != current->lyreDisp.secs_ppix) ||
	(new->lyreDisp.units_psec != current->lyreDisp.units_psec)) {
        new->lyreDisp.units_ppix = new->lyreDisp.secs_ppix * 
				   new->lyreDisp.units_psec;
	redisplay = TRUE;
    }

    if (new->lyreDisp.secs_ppix != current->lyreDisp.secs_ppix) {
        SetScale ((LyreDispWidget)new->lola.hscale_widget, 
		  new->lyreDisp.secs_ppix);
	redisplay = TRUE;
    }

    if (new->lola.seg) {
	tsl = FindLastSegment(new->lola.seg);
	if( tsl != NULL ) {
		new->lyreDisp.virtual_width = tsl->sl_end
					/ new->lyreDisp.units_ppix;
	}
    }

    if (current->lyreDisp.virtual_width != new->lyreDisp.virtual_width)
	XtCallCallbacks ((Widget) new, XtNwidthProc, 
			 (XtPointer) new->lyreDisp.virtual_width);

    return (redisplay);
}


/*
 * InitList (w)
 * 
 */
 
InitList (LolaWidget w)
{
    int                 arg_cnt;
    Arg                 args[10];
    int                 longest;
    int			count;
    char		**list = 0;
    XtTranslations      listTransTab = XtParseTranslationTable
			("<BtnDown>,<BtnUp>:	Set() Notify()");

    list =  MakeList (w->lola.labelFileName);
    if (list == NULL)
	return;
    longest = FindLongest (list);
    count = CountList (list);

    arg_cnt = 0;
    w->lola.shell = XtCreatePopupShell ("label", topLevelShellWidgetClass,
					(Widget) w, args, arg_cnt);

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNlist, list);
    MyXtSetArg (args, arg_cnt, XtNnumberStrings, count);
    MyXtSetArg (args, arg_cnt, XtNdefaultColumns, 4);
    MyXtSetArg (args, arg_cnt, XtNtranslations, listTransTab);
    w->lola.labelList = XtCreateManagedWidget ("list", listWidgetClass,
					       w->lola.shell,
					       args, arg_cnt);
}


/*
 * char **MakeList()
 *
 */

static char **MakeList (char *file)
{
    char              **list;
    char                str[1024];
    int                 count = 0;
    FILE               *fs;

    fs = fopen( file, "r");

    if (fs == NULL) {
#if 0
	XtAlert ("Could not find %s along $LPATH", file);
#endif
	return (0);
    }
    list = (char **) XtMalloc (sizeof (char *) * 64);
    while (fgets (str, sizeof (str), fs)) {
	str[strlen (str) - 1] = 0;	/* Nuke the new line from fgets */
	list[count++] = (char *) salloc (str);
	if ((count % 64) == 0)
	    list = (char **) XtRealloc ((char *) list, 
					sizeof (char *) * (count + 64));
    }
    list[count] = 0;
    fclose (fs);
    return list;
}


/*
 * int CountList (list)
 *
 */

int CountList (char **list)
{
    register int count = 0;
    while (*list) {
      list++; count++;
    }
    return count;
}


/*
 * int FindLongest (list)
 *
 */

int FindLongest (char **list)
{
    register int longest = 0;
    int tmp;

    while (*list) {
      tmp = strlen (*list);
      longest = MAX(tmp, longest);
      list++;
    }
    return longest;
}

#define PIX2FRM(w,x) ((x+w->lyreDisp.virtual_x) * w->lyreDisp.units_ppix)


/*
 * AddSeg (w, event, params, num_params)
 *
 */

static void AddSeg(LolaWidget w, XButtonEvent *event, String *params, 
		   Cardinal *num_pamams)
{
    SegmentList *seg = w->lola.seg;
    int frm = PIX2FRM(w,event->x);

    if( seg == NULL ) {
	seg = NewSegmentList();
	w->lola.seg = seg;
    }
    AddSegment( seg, frm, frm+5, "" );
    seg = FindSegment( seg, frm );

    if( seg ) {
	ClearSegment( w, seg, TRUE );
    }
}


/*
 * ContinueEdit(w, event, params, num_params)
 *
 */
 
static void ContinueEdit (LolaWidget w, XButtonEvent *event, String *params, 
			  Cardinal *num_pamams)
{
    SegmentList *seg = w->lola.cur_edit;
    int                 frm;
    int                 e_frm, b_frm;

    if (event->x < 0) {
	LScroll (w->lyreDisp.scroll_widget, 0.01 * event->x);
	event->x = 0;
    }
    else if (event->x > (int) w->core.width) {
	LScroll (w->lyreDisp.scroll_widget, 0.01 * ((float) event->x - 
						    (float) w->core.width));
	event->x = w->core.width;
    }

    frm = PIX2FRM (w, event->x);

    if( seg ) {

	ClearSegment( w, seg, FALSE );

	b_frm = seg->sl_begin;
	e_frm = seg->sl_end;
	if (abs (seg->sl_begin - frm) < abs (seg->sl_end - frm)) {
	    seg->sl_begin = frm;
	    b_frm = MIN (frm, b_frm);
	}
	else {
	    seg->sl_end = frm;
	    e_frm = MAX (frm, e_frm);
	}

	ClearSegment( w, seg, TRUE );

/*
	XClearArea (XtDisplay (w), XtWindow (w),
		    FRM2PIX (b_frm) - w->lyreDisp.virtual_x, 0,
		    FRM2PIX (e_frm - b_frm) + 1,
		    w->core.height, TRUE);
*/
    }
}


/*
 * DeleteSeg(w, event, params, num_params)
 *
 */

static void DeleteSeg (LolaWidget w, XKeyEvent *event, String *params, 
		       Cardinal *num_pamams)
{
    SegmentList *tmp;
    int frm = PIX2FRM(w,event->x);


    if( w->lola.seg == NULL ) {
	return;
    }

    tmp = FindNearestSegment( w->lola.seg, frm );

    /*
     * Any segments to delete??
     */
    if( tmp == w->lola.seg ) {
	return;
    }

    if( tmp == w->lola.cur_edit ) {
	w->lola.cur_edit = NULL;
    }

    if( tmp ) {
      if( tmp == w->lola.cur_edit ) {
	w->lola.cur_edit = NULL;
      }
      ClearSegment( w, tmp, TRUE );
      DeleteSegment( tmp );
    }

}


/*
 * LabelSeg(w, event, params, num_params)
 *
 */

static void LabelSeg (LolaWidget w, XButtonEvent *event, String *params, 
		      Cardinal *num_pamams)
{
    SegmentList *seg;
    int frm = PIX2FRM(w,event->x);
    XawListReturnStruct *item;

    /*
     * If there is no labelList widget then we cannot label the segment
     */
    if (w->lola.labelList == NULL)
	return;

    seg = FindNearestSegment( w->lola.seg, frm );

    /*
     * Must have a segment to label
     */
    if( seg ) {
      item = XawListShowCurrent (w->lola.labelList);
      if (item->list_index == XAW_LIST_NONE)
	return;

      LabelSegment( seg, item->string );
      ClearSegment( w, seg, TRUE );
    }
}


/*
 * StartEdit(w, event, params, num_params)
 *
 */

static void StartEdit (LolaWidget w, XButtonEvent *event, String *params, 
		       Cardinal *num_pamams)
{
    SegmentList *seg;
    int frm = PIX2FRM(w,event->x);
    int e_frm, b_frm;


    if( w->lola.seg == NULL ) {
	return;
    }

    seg = FindNearestSegment( w->lola.seg, frm );

    if( w->lola.cur_edit ) {
	    ClearSegment( w, w->lola.cur_edit, TRUE );
    }
    w->lola.cur_edit = seg;

    if( seg ) {

	ClearSegment( w, seg, FALSE );

	b_frm = seg->sl_begin;
	e_frm = seg->sl_end;
	if (abs(seg->sl_begin - frm) < abs(seg->sl_end - frm)) {
	    seg->sl_begin = frm;
	    b_frm = MIN (frm, b_frm);		
        }
	else {
	    seg->sl_end = frm;
	    e_frm = MAX (frm, e_frm);
   	}

	ClearSegment( w, seg, TRUE );
    }

}


/*
 * KeyInput (w, event, params, num_params)
 *
 */

static void KeyInput( LolaWidget w, XKeyEvent *event, String *params, 
		     Cardinal *num_pamams )
{
	SegmentList *tmp;
	int frm = PIX2FRM(w,event->x);
	char buf[10];
	int l;


	/*
	 * A segment to Edit??
	 */
	if( w->lola.cur_edit == 0 ) {
		return;
	}

	buf[0] = 0;

	l = XLookupString( event, buf, 10, NULL, NULL );

	if( isprint(buf[0]) ) {
		w->lola.cur_edit->sl_label = 
				AddChar(w->lola.cur_edit->sl_label,*buf);
		ClearSegment( w, w->lola.cur_edit, TRUE );
	}
}


/*
 * DeleteChar( w, event, params, num_params)
 *
 */

static void DeleteChar( LolaWidget w, XKeyEvent *event, String *params, 
		       Cardinal *num_pamams )
{
	SegmentList *seg;
	char buf[10];
	char *tmp;
	int len;

	/*
	 * A segment to Edit??
	 */
	if( w->lola.cur_edit == 0 ) {
		return;
	}

	seg = w->lola.cur_edit;

	tmp = seg->sl_label;

	len = strlen( tmp );

	if( len > 0 ) {
		ClearSegment( w, seg, TRUE );
		*(tmp+len-1) = NULL;
		ClearSegment( w, seg, TRUE );
	}
}


/*
 * SetCurEdit (w, event, params, num_params)
 *
 */

static void SetCurEdit( LolaWidget w, XButtonEvent *event, String *params, 
		       Cardinal *num_pamams )
{
	SegmentList *seg;
	int frm = PIX2FRM(w,event->x);

	if( w->lola.seg == NULL ) {
	  return;
	}

	if( w->lola.cur_edit ) {
		ClearSegment( w, w->lola.cur_edit, TRUE );
	}

	seg = FindNearestSegment( w->lola.seg, frm );

	w->lola.cur_edit = seg;

	if( seg ) {
		ClearSegment( w, seg, TRUE );
	}
}


/*
 * NextSeg (e, event, params, num_params)
 *
 */

static void NextSeg( LolaWidget w, XKeyEvent *event, String *params, 
		    Cardinal *num_pamams )
{
	if( w->lola.cur_edit == 0 ) {
		return;
	}

	ClearSegment( w, w->lola.cur_edit, TRUE );

	w->lola.cur_edit = w->lola.cur_edit->sl_forward;

	if( w->lola.cur_edit ) {
		ClearSegment( w, w->lola.cur_edit, TRUE );
	}
}


/*
 * PrevSeg (w, event, params, num_params)
 *
 */

static void PrevSeg( LolaWidget w, XKeyEvent *event, String *params, 
		    Cardinal *num_pamams )
{
	if( w->lola.cur_edit == NULL ) {
		return;
	}

	ClearSegment( w, w->lola.cur_edit, TRUE );

	w->lola.cur_edit = w->lola.cur_edit->sl_backward;

	if( w->lola.cur_edit->sl_backward ) {
		ClearSegment( w, w->lola.cur_edit, TRUE );
	} else {
		w->lola.cur_edit = NULL;
	}
}


/*
 * ClearSegment(w, seg, expose)
 *
 */

static void ClearSegment( LolaWidget w, SegmentList *seg, int expose )
{
	int tl;
	int x1;
	int wd;

	if( seg->sl_label[0] != NULL ) {
		tl = XTextWidth( w->lola.font_info, seg->sl_label, 
						strlen( seg->sl_label ) ) + 2;
	} else {
		tl = 0;
	}

	x1 = FRM2PIX (seg->sl_begin) - w->lyreDisp.virtual_x;
	wd = FRM2PIX (seg->sl_end - seg->sl_begin) + 1;

	if( wd < tl ) {
		wd = tl;
	}

	XClearArea( XtDisplay(w), XtWindow(w), x1, 0, 
					wd, w->core.height, expose);

}


/*
 * char *AddChar (str, c)
 *
 */

static char *AddChar( char *str, char c )
{
	char *tmp;
	int len;


	len = strlen( str ) + 1;

	tmp = XtRealloc( str, len );

	*(tmp+len-1) = c;
	*(tmp+len) = NULL;

	return( tmp );

}


/*
 * RedisplayCurEdit(w, sl, raise)
 *
 */

RedisplayCurEdit(LolaWidget w, register SegmentList *sl, int raise)
{
    register int        box_x;
    int                 label_y;
    int                 box_width;
    int                 virt_x = 0;

    virt_x = w->lyreDisp.virtual_x;

    label_y = w->core.height - 4 - w->lola.font_info->descent;

   /*
    * Draw the box
    */
    box_width = FRM2PIX(sl->sl_end - sl->sl_begin);
    box_x = FRM2PIX(sl->sl_begin) - virt_x;

    XSetLineAttributes( XtDisplay(w), w->lola.lolaGC, 1, LineOnOffDash,
			CapButt, JoinMiter );

    XDrawRectangle (XtDisplay (w), XtWindow (w), w->lola.lolaGC,
                        box_x, 2, box_width, w->core.height - 4);

    XSetLineAttributes( XtDisplay(w), w->lola.lolaGC, 1, LineSolid,
			CapButt, JoinMiter );

    if (sl->sl_label)
        XDrawString (XtDisplay (w), XtWindow (w), w->lola.lolaGC,
                     box_x + 2,
                     label_y - (raise * (w->lola.font_info->ascent + 
			w->lola.font_info->descent)),
                     sl->sl_label, strlen (sl->sl_label));
}


/*
 * NoAction()
 *
 */

static void NoAction()
{
	return;
}

