static char *rcsident = "$Header: /projects/cslu/speech/work/src/bin/auto_lyre/RCS/Lgraph.c,v 4.6 1993/05/28 21:27:40 johans Exp johans $";

/*
 * Lgraph.c - Line Graph Widget
 *
 * Graphics widget procedures
 *
 *------------------------------------------------------------*
 * Copyright 1988, Fil Alleva and Carnegie Mellon University
 *------------------------------------------------------------*
 * HISTORY
 * 21-Jun-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	Added play_data().
 *
 * 19-Jan-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	Added code to initialize lgraph.segments and lgraph.nsegments
 *	fields.
 *
 *  3-Jan-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	- Added code to handle fewer than one sample per pixel.
 *	- Modified COMPUTESEGMENTS to less floating pt. arithmetic
 *
 * 13-Apr-93 Johan Schalkwyk at Oregon Graduate Institute
 *       changed to ANSI style header declarations
 *
 */

/* Standard C library include file directives */
#include <stdio.h>
#include <c.h>

/* speech include file directives */
#include <ad.h>
#include <speech.h>
#include <fcntl.h>

/* X library include file directives */
#include <X11/Xos.h>
#include <X11/Xlib.h>
#include <X11/cursorfont.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>

/* Lyre include file directives */
#include <ScaleP.h>
#include <Scale.h>
#include <LgraphP.h>
#include <ToolUtil.h>

/* Private Definitions */


#define MyXtSetArg(args,i,arg,val)	{XtSetArg (args[i], arg, val); i++;}

/* Initialization of defaults */

#define OFFSET(field) XtOffset(LgraphWidget,lgraph.field)
#define GOFFSET(field) XtOffset(Widget,core.field)

static int Zero = 0;

static XtResource resources[] = {
    {XtNforeground, XtCForeground, XtRPixel, sizeof(Pixel),
        OFFSET(foreground_pixel), XtRString, "Black"},
    {XtNwidth, XtCWidth, XtRDimension, sizeof(Dimension),
	GOFFSET(width), XtRString, "10"},
    {XtNheight, XtCHeight, XtRDimension, sizeof(Dimension),
	GOFFSET(height), XtRString, "0"},
    {XtNreverseVideo, XtCReverseVideo, XtRBoolean, sizeof (Boolean),
	OFFSET (reverse_video), XtRString, "FALSE"},
    {XtNdata, XtNdata, XtRPointer, sizeof(caddr_t),
	OFFSET (input_data), XtRPointer, NULL},
    {XtNhscaleWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (hscale_widget), XtRPointer, NULL},
    {XtNvscaleWidget, XtCParameter, XtRPointer, sizeof(caddr_t),
	OFFSET (vscale_widget), XtRPointer, NULL},
};

#undef OFFSET
#undef GOFFSET


/* Procedure header declarations */
static void    Initialize (Widget request, Widget new, 
			   ArgList arglist, Cardinal *num_args);
static void    Realize (Widget w, XtValueMask *valueMask, 
			XSetWindowAttributes *attrs);
static void    Destroy (Widget w);
static void    Resize (Widget gw);
static void    Redisplay (Widget w, XEvent *event_in, Region region);
static Boolean SetValues (Widget gcurrent, Widget grequest, Widget gnew,
			  ArgList arglist, Cardinal *num_args);

static void    ComputeSegments (register LgraphWidget w);
static void    ComputeSegments1 (vdata_t *vdata, register XSegment *segments,
				 u_int nsegments, register int offset,
				 register float scale, double DatumPerPix);
static void    ComputeSegments2(vdata_t *vdata, register XSegment *segments,
				u_int nsegments, register int offset,
				register float scale, double DatumPerPix);
static void    ComputeSegments4(vdata_t *vdata, register XSegment *segments,
				u_int nsegments, register int offset,
				register float scale, double DatumPerPix);

static void data_amp (u_int cnt, caddr_t ptr, u_int size, 
		      int *max_amp, int *min_amp);
static void data_amp1 (register u_int cnt, register char *ptr, 
		       register u_int size, int *max_amp, int *min_amp);
static void data_amp2 (register u_int cnt, register short *ptr,
		       register u_int size, int *max_amp, int *min_amp);
static void data_amp4 (register u_int cnt, register int *ptr, 
		       register u_int size, int *max_amp, int *min_amp);

static void SetBracket (LgraphWidget w, XEvent *event, String *params, 
			Cardinal *num_params);
static void DrawBracket (LgraphWidget w, XEvent *event, String *params, 
			 Cardinal *num_params);
static void ReDrawBracket (LgraphWidget w, XEvent *event, String *params, 
			   Cardinal *num_params);
static void EraseBracket (LgraphWidget w, XEvent *event, String *params, 
			  Cardinal *num_params);
static void ToggleBracket (LgraphWidget w, XEvent *event, String *params, 
			   Cardinal *num_params);
static void Play  (LgraphWidget w, XEvent *event, String *params, 
		   Cardinal *num_params);

void play_data (u_int ssamp, u_int scnt, vdata_t *data);
int  dump_data (char *file, int ss, int cnt, vdata_t *vdata);


static char defaultTranslations[] = "\
<Key>P:        Play(a) \n\
<Key>p:        Play(a) \n\
<Enter>:       TrackMouse(e) \n\
<Leave>:       TrackMouse(l) \n\
<Btn1Down>:    TrackMouse(l) EraseBracket(l) DrawBracket(l) TrackMouse(e) \n\
<Btn1Motion>:  DrawBracket(l) ReDrawBracket(l) TrackMouse(b) \n\
<Btn1Up>:      TrackMouse(l) SetBracket(l) TrackMouse(e) \n\
<Btn2Down>:    Play(b) \n\
<Btn3Down>:    TrackMouse(l) EraseBracket(r) DrawBracket(r) TrackMouse(e) \n\
<Btn3Motion>:  DrawBracket(r) ReDrawBracket(r) TrackMouse(b) \n\
<Btn3Up>:      TrackMouse(l) SetBracket(r) TrackMouse(e) \n\
<Motion>:      TrackMouse(m) \n\
";


static XtActionsRec actions[] = {
	{"TrackMouse",		(XtActionProc) LyreDispTrackMouse},
	{"ToggleBracket",	(XtActionProc) ToggleBracket},
	{"Play",                (XtActionProc) Play},
	{"EraseBracket",	(XtActionProc) EraseBracket},
	{"DrawBracket", 	(XtActionProc) DrawBracket},
	{"ReDrawBracket",	(XtActionProc) ReDrawBracket},
	{"SetBracket",		(XtActionProc) SetBracket},
};


LgraphClassRec lgraphClassRec = {
    { /* core fields */
    /* superclass		*/	(WidgetClass) &lyreDispClassRec,
    /* class_name		*/	"Lgraph",
    /* widget_size		*/	sizeof(LgraphRec),
    /* class_initialize		*/	NULL,
    /* class_part_initialize	*/	NULL,
    /* class_inited		*/	FALSE,
    /* initialize		*/	Initialize,
    /* initialize_hook		*/	NULL,
    /* realize			*/	Realize,
    /* actions			*/	actions,
    /* num_actions		*/	XtNumber(actions),
    /* resources		*/	resources,
    /* resource_count		*/	XtNumber(resources),
    /* xrm_class		*/	NULL,
    /* compress_motion		*/	TRUE,
    /* compress_exposure	*/	(TRUE | XtExposeGraphicsExpose),
    /* compress_enterleave	*/	TRUE,
    /* visible_interest		*/	FALSE,
    /* destroy			*/	Destroy,
    /* resize			*/	Resize,
    /* expose			*/	Redisplay,
    /* set_values		*/	SetValues,
    /* set_values_hook		*/	NULL,
    /* set_values_almost	*/	XtInheritSetValuesAlmost,
    /* get_values_hook		*/	NULL,
    /* accept_focus		*/	NULL,
    /* version			*/	XtVersion,
    /* callback_private		*/	NULL,
    /* tm_table			*/	defaultTranslations,
    /* query_geometry           */	NULL,
    /* display_accelerator	*/	XtInheritDisplayAccelerator,	
    /* extension		*/	NULL,
    }
};

WidgetClass lgraphWidgetClass = (WidgetClass) &lgraphClassRec;


/*
 * Initialize (request, new)
 *
 * Initialize this widget class -> graphics display widget
 *
 */

static void Initialize (Widget request, Widget new, 
			ArgList arglist, Cardinal *num_args)
{
    LgraphWidget w = (LgraphWidget) new;
    XtGCMask  valuemask;
    XGCValues myXGCV;

    if (w->lgraph.reverse_video) {
	Pixel fg = w->lgraph.foreground_pixel;
	Pixel bg = w->core.background_pixel;

	if (w->core.border_pixel == fg)
	    w->core.border_pixel = bg;
	w->lgraph.foreground_pixel = bg;
	w->core.background_pixel = fg;
    }

    valuemask = GCForeground | GCBackground | GCLineWidth |
		GCGraphicsExposures;
    myXGCV.foreground = w->lgraph.foreground_pixel;
    myXGCV.background = w->core.background_pixel;
    myXGCV.line_width = 0;
    myXGCV.graphics_exposures = TRUE;	/* default */
    w->lgraph.myGC = XtGetGC ((Widget) w, valuemask, &myXGCV);

    valuemask = GCLineWidth | GCFunction;
    myXGCV.line_width = 0;
    myXGCV.function = GXinvert;
    w->lgraph.bracketGC = XtGetGC ((Widget) w, valuemask, &myXGCV);

    w->lgraph.data = w->lgraph.input_data;
    w->lgraph.input_data = 0;
    w->lgraph.brackets_on = FALSE;
    w->lgraph.nsegments = 0;
    w->lgraph.segments = NULL;
    SetScale ((LyreDispWidget)w->lgraph.hscale_widget, w->lyreDisp.secs_ppix);
}


/*
 * Realize (w, valueMask, attrs)
 *
 * Create graphics widget procedure
 *
 */

static void Realize (Widget w_in, XtValueMask *valueMask, 
		     XSetWindowAttributes *attrs)
{
  LgraphWidget w = (LgraphWidget) w_in;
  
  attrs->cursor = XCreateFontCursor( XtDisplay(w), XC_tcross );
  *valueMask |= CWCursor;

  XtCreateWindow ((Widget) w, InputOutput, (Visual *) CopyFromParent,
		  *valueMask, attrs);
  Resize ((Widget) w);
}


/*
 * Destroy (w)
 *
 * Destroy this graphics widget
 *
 */

static void Destroy (Widget w_in)
{
  LgraphWidget w = (LgraphWidget) w_in;

  XtDestroyGC (w->lgraph.myGC);
  XtFree ((char *) (w->lgraph.segments));
}


/*
 * Resize (gw)
 *
 * Resize of graphics widget requested 
 *
 */

static void Resize (Widget gw)
{
    LgraphWidget w = (LgraphWidget) gw;

    /*
     * don't do this computation if window hasn't been realized yet.
     */
    if (XtIsRealized ((Widget) w)) {
	ComputeSegments (w);
    }
}


/* 
 * Redisplay (w, event, region)
 *
 * Widget must be redisplayed after exposure
 *
 */

static void Redisplay (Widget w_in, XEvent *event_in, 
		       Region region)
{
  LgraphWidget  w = (LgraphWidget) w_in;
  XExposeEvent  *event = (XExposeEvent *) event_in;

    register XSegment  *xseg;
    register int        wx;
    register int        EndX;
    register int        StartX;
    register int        Width;
    register int        seg_index;
    register XSegment  *seg;

    if (w->lgraph.nsegments && w->lgraph.segments) {
	/*
	 * Draw the waveform 
	 */
	StartX = event->x;
	Width = event->width;
	seg_index = StartX + w->lyreDisp.virtual_x;

	if (seg_index < 0) {
	    Width -= (-seg_index);
	    StartX += (-seg_index);
	    seg_index = 0;
	}
	if ((seg_index + Width) > w->lgraph.nsegments)
	    Width = w->lgraph.nsegments - seg_index;
	if ((seg_index > w->lgraph.nsegments) || (Width <= 0))
	    return;

	seg = xseg = &w->lgraph.segments[seg_index];
	EndX = Width + StartX;
	for (wx = StartX; wx < EndX; wx++, xseg++) {
	    xseg->x1 = xseg->x2 = wx;
	}

	XDrawSegments (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
		       seg, Width);
	/*
	 * Draw the brackets if they're turned on 
	 */
	if (w->lgraph.brackets_on) {
	    int                 left_b_x =
	    w->lgraph.left_bracket_x - w->lyreDisp.virtual_x;
	    int                 right_b_x =
	    w->lgraph.right_bracket_x - w->lyreDisp.virtual_x;

	    if ((event->x <= left_b_x) && ((event->x + event->width) > 
					   left_b_x)) {
		XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
			   left_b_x, 4, left_b_x, w->core.height - 4);
		XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
			   left_b_x, 4, left_b_x + 4, 4);
		XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
			   left_b_x, w->core.height - 4,
			   left_b_x + 4, w->core.height - 4);
	    }
	    if ((event->x <= right_b_x) && ((event->x + event->width) > 
					    right_b_x)) {
		XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
			   right_b_x, 4, right_b_x, w->core.height - 4);
		XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
			   right_b_x, 4, right_b_x - 4, 4);
		XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
			   right_b_x, w->core.height - 4,
			   right_b_x - 4, w->core.height - 4);
	    }
	}
    }
}


/*
 * Boolean Setvalues (gcurrent, grequest, gnew)
 *
 */

static Boolean SetValues (Widget gcurrent, Widget grequest, Widget gnew,
			  ArgList arglist, Cardinal *num_args)
{
    LgraphWidget        current = (LgraphWidget) gcurrent;
    LgraphWidget        new = (LgraphWidget) gnew;
    LgraphWidget        req = (LgraphWidget) grequest;
    Boolean             redisplay = FALSE;
    Boolean             recompute = FALSE;
    XtGCMask            valuemask;
    XGCValues           myXGCV;

    if ((new->lgraph.foreground_pixel != current->lgraph.foreground_pixel)
	|| (new->core.background_pixel != current->core.background_pixel)) {
	valuemask = GCForeground | GCBackground | GCFont | GCLineWidth;
	myXGCV.foreground = new->lgraph.foreground_pixel;
	myXGCV.background = new->core.background_pixel;
	myXGCV.line_width = 0;
	XtDestroyGC (current->lgraph.myGC);
	new->lgraph.myGC = XtGetGC (gcurrent, valuemask, &myXGCV);
	redisplay = TRUE;
    }

    if (new->core.background_pixel != current->core.background_pixel) {
	valuemask = GCForeground | GCLineWidth;
	myXGCV.foreground = new->core.background_pixel;
	myXGCV.line_width = 0;
	redisplay = TRUE;
    }

    if (new->lgraph.input_data) {
	new->lgraph.data = new->lgraph.input_data;
	new->lgraph.input_data = 0;
	recompute = TRUE;
    }

    if (new->lyreDisp.secs_ppix != current->lyreDisp.secs_ppix) {
        SetScale ((LyreDispWidget)new->lgraph.hscale_widget, 
		  new->lyreDisp.secs_ppix);
	recompute = TRUE;
    }

    if (recompute) {
	ComputeSegments (new);
	new->lyreDisp.virtual_width = new->lgraph.nsegments;
	redisplay = TRUE;
    }

    if (current->lyreDisp.virtual_width != new->lyreDisp.virtual_width)
      XtCallCallbacks ((Widget) new, XtNwidthProc, 
		       (XtPointer) (new->lyreDisp.virtual_width));

    return (redisplay);
}


/*
 * ComputeSegments (w)
 *
 */

static void ComputeSegments (register LgraphWidget w)
{
    double      DatumPerPix;
    register vdata_t   *vdata = w->lgraph.data;
    int                 max, min;
    register float      scale;
    register int        offset;

    if (vdata == 0)
	return;

    XtFree ((char *) (w->lgraph.segments));
    w->lgraph.segments = 0;
    w->lgraph.nsegments = 0;

    DatumPerPix = 1000000000.0 * w->lyreDisp.secs_ppix / vdata->ns_per_datum;

    if ((vdata->ptr == 0) || (vdata->count == 0))
	return;

    {
	int                 range;
	Arg                 args[20];
	int                 arg_cnt;
	ScaleWidget         scaletemp;

	data_amp (vdata->count, vdata->ptr, vdata->bsize, &max, &min);
	range = max - min;
	if (range == 0)
	    range = 1;
	if (w->lgraph.vscale_widget) {
	    union {
		float f;
		unsigned int   i;
	    } unit_per_pix;

	    unit_per_pix.f = ((float) range) / ((float) w->core.height);

	    arg_cnt = 0;
	    MyXtSetArg (args, arg_cnt, XtNunitPerPix, unit_per_pix.i);
	    MyXtSetArg (args, arg_cnt, XtNzeroPixOffset,
			(int) (max / unit_per_pix.f));
	    XtSetValues ((Widget) (w->lgraph.vscale_widget), args, arg_cnt);
	    scaletemp =  w->lgraph.vscale_widget;
	    scaletemp->scale.unit_per_pix    = unit_per_pix.f;
	    scaletemp->scale.zero_pix_offset = (int) (max / unit_per_pix.f);
	}
	scale = (float) (w->core.height - w->lyreDisp.top_padding -
			 w->lyreDisp.bottom_padding) / range;
    }

    offset = (max * scale) + w->lyreDisp.top_padding;

    w->lgraph.nsegments = (vdata->count / DatumPerPix) - 1;
    w->lgraph.segments = (XSegment *) XtMalloc (w->lgraph.nsegments *
						sizeof (XSegment));

    switch (vdata->bsize) {
    case 1:
	ComputeSegments1 (vdata, w->lgraph.segments, w->lgraph.nsegments,
			  offset, scale, DatumPerPix);
	break;
    case 2:
	ComputeSegments2 (vdata, w->lgraph.segments, w->lgraph.nsegments,
			  offset, scale, DatumPerPix);
	break;
    case 4:
	ComputeSegments4 (vdata, w->lgraph.segments, w->lgraph.nsegments,
			  offset, scale, DatumPerPix);
	break;
    }
}

/*
 * The check to make sure that y2 is larger than y1 was put in for 
 *  the DECstation 5000/240 running Ultrix 4.2A and X11R5 -- if y1
 *  is equal to y2, nothing was drawn.  the fix makes a line of length
 *  2 pixels when one is wanted, but it is better than nothing
 */

#define COMPUTESEGMENTS(name,type,vdata,segments,nsegments,offset,scale,DatumPerPix) \
static void name (vdata_t *vdata, register XSegment *segments, \
		  u_int nsegments, register int offset, register float scale, \
		  double DatumPerPix) \
{ \
    double samp; \
    register type *eptr; \
    register type *ptr = (type *) vdata->ptr; \
    register type *tptr; \
    register int min, max; \
    register u_int xpix; \
 \
    for (samp = 0.0, xpix = 0; xpix < nsegments; samp += DatumPerPix, xpix++) { \
	tptr = ptr + (int) samp; \
	eptr = ptr + (int) (samp + DatumPerPix); \
 \
	min = max = *tptr; \
	for (; tptr <= eptr; tptr++) { \
	    if (*tptr > max) \
		max = *tptr; \
	    else if (*tptr < min) \
		min = *tptr; \
	} \
	segments->x1 = xpix; \
	segments->y1 = (int)(-max * scale) + offset; \
	segments->x2 = xpix; \
	segments->y2 = (int)(-min * scale) + offset; \
	if(segments->y1 >= segments->y2) segments->y2 = segments->y1 + 1;\
	segments++; \
    } \
} 

COMPUTESEGMENTS(ComputeSegments1, char, vdata, segments, nsegments,
		offset, scale, DatumPerPix)

COMPUTESEGMENTS(ComputeSegments2, short, vdata, segments, nsegments,
		offset, scale, DatumPerPix)

COMPUTESEGMENTS(ComputeSegments4, int, vdata, segments, nsegments,
		offset, scale, DatumPerPix)

#undef COMPUTESEGMENTS


/*
 * data_amp (cnt, ptr, size, max_amp, min_amp)
 *
 */

static void data_amp (u_int cnt, caddr_t ptr, u_int size, int *max_amp, 
		      int *min_amp)
{
    switch (size) {
    case 1:
	data_amp1(cnt, (char *) ptr, size, max_amp, min_amp);
	break;
    case 2:
	data_amp2(cnt, (short *) ptr, size, max_amp, min_amp);
	break;
    case 4:
	data_amp4(cnt, (int *) ptr, size, max_amp, min_amp);
	break;
    }
}


/*
 * data_amp1 (cnt, ptr, size, max_map, min_amp)
 *
 */

static void data_amp1 (register u_int cnt, register char *ptr, 
		       register u_int size, int *max_amp, int *min_amp)
{
    register int max, min;
    register char *eptr;

    for (min = max = *ptr, eptr = ptr + cnt; ptr < eptr; ptr++) {
	if (*ptr > max)
	    max = *ptr;
	else if (*ptr < min)
	    min = *ptr;
    }
    *min_amp = min;
    *max_amp = max;
}


/*
 * data_amp2 (cnt, ptr, size, max_amp, min_amp)
 *
 */

static void data_amp2 (register u_int cnt, register short *ptr, 
		       register u_int size, int *max_amp, int *min_amp)
{
    register int max, min;
    register short *eptr;

    for (min = max = *ptr, eptr = ptr + cnt; ptr < eptr; ptr++) {
	if (*ptr > max)
	    max = *ptr;
	else if (*ptr < min)
	    min = *ptr;
    }
    *min_amp = min;
    *max_amp = max;
}


/*
 * data_amp4 (cnt, ptr, size, max_map, min_amp)
 *
 */

static void data_amp4 (register u_int cnt, register int *ptr, 
		       register u_int size, int *max_amp, int *min_amp)
{
    register int max, min;
    register int *eptr;

    for (min = max = *ptr, eptr = ptr + cnt; ptr < eptr; ptr++) {
	if (*ptr > max)
	    max = *ptr;
	else if (*ptr < min)
	    min = *ptr;
    }
    *min_amp = min;
    *max_amp = max;
}


/*
 * SetBracket (w, event, params, num_params)
 *
 */

static void SetBracket (LgraphWidget w, XEvent *event, String *params, 
			Cardinal *num_params)
{
    w->lgraph.brackets_on = TRUE;

    if (params[0][0] == 'l') {
	int                 left_b_x =
	w->lgraph.left_bracket_x - w->lyreDisp.virtual_x;

	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
		   left_b_x, 4, left_b_x, w->core.height - 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
		   left_b_x, 4, left_b_x + 4, 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
		   left_b_x, w->core.height - 4,
		   left_b_x + 4, w->core.height - 4);
    }
    else {
	int                 right_b_x =
	w->lgraph.right_bracket_x - w->lyreDisp.virtual_x;

	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
		   right_b_x, 4, right_b_x, w->core.height - 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
		   right_b_x, 4, right_b_x - 4, 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.myGC,
		   right_b_x, w->core.height - 4,
		   right_b_x - 4, w->core.height - 4);
    }
}


/*
 * Draw_Bracket (w, event, params, num_params)
 * 
 */

static void DrawBracket (LgraphWidget w, XEvent *event, String *params, 
			 Cardinal *num_params)
{
    int                 left_b_x;
    int                 right_b_x;

    if (params[0][0] == 'l') {
	left_b_x = w->lgraph.left_bracket_x - w->lyreDisp.virtual_x;

	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   left_b_x, 4, left_b_x, w->core.height - 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   left_b_x, 4, left_b_x + 4, 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   left_b_x, w->core.height - 4,
		   left_b_x + 4, w->core.height - 4);
    }
    else {
	right_b_x = w->lgraph.right_bracket_x - w->lyreDisp.virtual_x;

	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   right_b_x, 4, right_b_x, w->core.height - 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   right_b_x, 4, right_b_x - 4, 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   right_b_x, w->core.height - 4,
		   right_b_x - 4, w->core.height - 4);
    }
}


/*
 * ReDrawBracket (w, event, params, num_params)
 *
 */

static void ReDrawBracket (LgraphWidget w, XEvent *event, String *params, 
			   Cardinal *num_params)
{
    int                 left_b_x;
    int                 right_b_x;
    int			vscale_width = 0;

    if (w->lgraph.vscale_widget)
	vscale_width = w->lgraph.vscale_widget->core.width;

    if (event->xbutton.x < vscale_width) {
	LScroll (w->lyreDisp.scroll_widget, -0.10);
	event->xbutton.x = vscale_width;
    } else
    if (event->xbutton.x > ((int) w->core.width)) {
	LScroll (w->lyreDisp.scroll_widget, 0.10);
	event->xbutton.x = w->core.width;
    }

    if (params[0][0] == 'l') {
    	w->lgraph.left_bracket_x = event->xbutton.x + w->lyreDisp.virtual_x;
	left_b_x = w->lgraph.left_bracket_x - w->lyreDisp.virtual_x;

	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   left_b_x, 4, left_b_x, w->core.height - 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   left_b_x, 4, left_b_x + 4, 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   left_b_x, w->core.height - 4,
		   left_b_x + 4, w->core.height - 4);
    }
    else {
    	w->lgraph.right_bracket_x = event->xbutton.x + w->lyreDisp.virtual_x;
	right_b_x = w->lgraph.right_bracket_x - w->lyreDisp.virtual_x;

	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   right_b_x, 4, right_b_x, w->core.height - 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   right_b_x, 4, right_b_x - 4, 4);
	XDrawLine (XtDisplay (w), XtWindow (w), w->lgraph.bracketGC,
		   right_b_x, w->core.height - 4,
		   right_b_x - 4, w->core.height - 4);
    }
}


/*
 * EraseBracket (w, event, params, num_params)
 *
 */

static void EraseBracket (LgraphWidget w, XEvent *event, String *params, 
			  Cardinal *num_params)
{
    w->lgraph.brackets_on = FALSE;

    if (params[0][0] == 'l') {
	XClearArea (XtDisplay (w), XtWindow (w),
		    w->lgraph.left_bracket_x - w->lyreDisp.virtual_x,
		    0, 5, w->core.height, TRUE);
    	w->lgraph.left_bracket_x = event->xbutton.x + w->lyreDisp.virtual_x;
    }
    else {
	XClearArea (XtDisplay (w), XtWindow (w),
		    w->lgraph.right_bracket_x - w->lyreDisp.virtual_x - 4,
		    0, 5, w->core.height, TRUE);
    	w->lgraph.right_bracket_x = event->xbutton.x + w->lyreDisp.virtual_x;
    }
}


/*
 * ToggleBracket (w, event, params, num_params)
 *
 */

static void ToggleBracket (LgraphWidget w, XEvent *event, String *params, 
			   Cardinal *num_params)
{
    w->lgraph.brackets_on = ! w->lgraph.brackets_on;
    XClearArea (XtDisplay (w), XtWindow (w),
	        w->lgraph.left_bracket_x - w->lyreDisp.virtual_x,
	        0, 5, w->core.height, TRUE);
    XClearArea (XtDisplay (w), XtWindow (w),
	        w->lgraph.right_bracket_x - w->lyreDisp.virtual_x - 4,
		0, 5, w->core.height, TRUE);
}


/*
 * Play (w, event, params, num_params)
 *
 */

static void Play  (LgraphWidget w, XEvent *event, String *params, 
		   Cardinal *num_params)
{
    float               start_time;
    float               end_time;
    float               tmp;
    int                 sample_rate;

    if (w->lgraph.data == 0)
      return;
    sample_rate = 1000000000 / w->lgraph.data->ns_per_datum;
    start_time = w->lgraph.left_bracket_x * w->lyreDisp.secs_ppix;
    end_time = w->lgraph.right_bracket_x * w->lyreDisp.secs_ppix;
    if (start_time > end_time) {
      tmp = start_time;
      start_time = end_time;
      end_time = tmp;
    }
    if (start_time < 0.0)
	start_time = 0.0;
    if (params[0][0] == 'a')
	play_data (0, 0, w->lgraph.data);
    else {
	play_data ((int)(start_time * sample_rate),
		   (int)(sample_rate * (end_time - start_time)),
	           w->lgraph.data);
    }
}


/*
 * play_data (ssamp, scnt, data)
 *
 * Play scnt samples data starting at sample ssamp. If scnt == 0
 * play all of the data.
 *
 */

void play_data (u_int ssamp, u_int scnt, vdata_t *data)
{
    char                tmp_file[128];
    char                command[1024];

    sprintf (tmp_file, "lyre%d.adc", getpid());
        
    if (scnt == 0)
	scnt = data->count;
    if (dump_data (tmp_file, ssamp, scnt, data))
	return;
    /* do not use explicit bin; must be on path */
    sprintf (command, "speechplay %s", tmp_file);
    system (command);
    unlink (tmp_file);
}


/*
 * dump_data (file, ss, cnt, vdata)
 *
 */

int dump_data (char *file, int ss, int cnt, vdata_t *vdata)
{
    struct ad_head adchead;
    int rate;

    if (cnt + ss > vdata->count)
      cnt = vdata->count - ss;

/*
    adchead.ad_channels = 1;
    adchead.ad_rate = vdata->ns_per_datum/250;
    adchead.ad_samples = cnt;
*/

    rate = (4000/(vdata->ns_per_datum/250)) * 1000;

    if( AdcWrite( file, (short *) (vdata->ptr+(ss * vdata->bsize)), 
						cnt, rate ) < 0 ) {
      return 1;
    }

    return (0);
}

