static char *rcsident = "$Header: /projects/cslu/speech/work/src/bin/auto_lyre/RCS/Adctool.c,v 4.8 1993/05/28 21:27:40 johans Exp johans $";

/*
 * Adctool.c - Adctool widget
 *------------------------------------------------------------*
 * Copyright 1988, Fil Alleva and Carnegie Mellon University
 *------------------------------------------------------------*
 * HISTORY
 * 09-Jan-91 Fil Alleva (faa@cs.cmu.edu)
 *	Converted controls to a pop-up menu.
 *
 * 23-Jun-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	Fixed the control of the mark widget so the it can be toggled on
 *	even if it starts out off.
 *
 * 25-Jan-89  Fil Alleva (faa) at Carnegie-Mellon University
 *	Changed the AdcRead to pick up the correct sampling rate from
 *	the adc file header.
 *
 * 25-March-1993 Johan Schalkwyk at Oregon Graduate Institute
 *       made stylistic changes, added lots and lots of comments
 *       and made the overall code more robust.
 *
 */

/* Standard C include file directives */
#include <malloc.h>
#include <stdio.h>
#include <stdlib.h>
#include <ctype.h>
#include <string.h>
#include <math.h>

/* Special Speech include file directives */
#include <speech.h>
#include <c.h>

/* X library include file directives */
#include <X11/Xlib.h>
#include <X11/Xos.h>
#include <X11/StringDefs.h>
#include <X11/IntrinsicP.h>
#include <X11/Shell.h>
#include <X11/Xmu/Misc.h>
#include <X11/Xaw/CommandP.h>
#include <X11/Xaw/DialogP.h>
#include <X11/Xaw/MenuButton.h>
#include <X11/Xaw/SimpleMenu.h>
#include <X11/Xaw/Sme.h>
#include <X11/Xaw/SmeBSB.h>


/* Lyre include file directives */
#include <LyreDisp.h>
#include <ToolUtil.h>
#include <ScaleP.h>
#include <Scale.h>
#include <LgraphP.h>
#include <Mark.h>
#include <Yoke.h>
#include <AdctoolP.h>

/* external procedure declarations */
char *salloc(char *str);

/* Forward declarations of procedures used in this module */
static void    Initialize (Widget request_in, Widget new_in,
			   ArgList arglist, Cardinal *num_args);
static Boolean SetValues (Widget current_in, Widget request_in, Widget new_in,
			  ArgList arglist, Cardinal *num_args);
static void LyreAdcRead (Widget w, WidgetList client_data);
static void PitchRead (Widget w, WidgetList client_data);
static void PitchWrite (Widget w, WidgetList client_data);
static void SetResolution (Widget w, WidgetList client_data);
static void SetChannel (Widget w, WidgetList client_data);
static void Destroy (Widget w, AdctoolWidget target_widget, caddr_t call_data);
static void ToggleMark (Widget cw, AdctoolWidget w, caddr_t call_data);
static void LolaSegWrite (Widget w, WidgetList client_data);
static void AdcSegWrite (Widget w, WidgetList client_data);

u_int mark_peaks (register u_int *pitch, register u_int pitch_count, 
		   register u_int *peak,  register u_int peak_count, 
		   register u_int *yes);
void   GetChannel (vdata_t *ac_data, vdata_t *sc_data, int channel, 
		   int channel_count);
int    AdcUnmap(AdctoolWidget tool);
int    AdcMap (AdctoolWidget tool);
int    AdcPollRead (AdctoolWidget tool, char *file);
int    MarkUnmap (AdctoolWidget tool);
int    MarkMap (AdctoolWidget tool);
int    MarkPollRead (AdctoolWidget tool, char *file);

static char defaultTranslations[] = "\
";

static XtActionsRec actions[] = {
	{NULL,NULL}
};

static Boolean defTRUE  = TRUE;
static Boolean defFALSE = FALSE;

#define offset(field) XtOffset(AdctoolWidget, field)

static XtResource resources[] = { 
   {XtNchild, XtNchild, XtRPointer, sizeof(caddr_t), 
      offset(adctool.child), XtRPointer, (caddr_t)NULL},
   {XtNmark, XtNmark, XtRPointer, sizeof(caddr_t), 
      offset(adctool.mark), XtRPointer, (caddr_t)NULL},
   {XtNchannel, XtNchannel, XtRInt, sizeof(int), 
      offset(adctool.channel), XtRString, "0"},
   {XtNvertScaleOn, XtNvertScaleOn, XtRBoolean, sizeof(Boolean), 
      offset(adctool.vert_scale_on), XtRBoolean, (caddr_t)&defTRUE},
   {XtNhorizScaleOn, XtNhorizScaleOn, XtRBoolean, sizeof(Boolean), 
      offset(adctool.horiz_scale_on), XtRBoolean, (caddr_t)&defTRUE},
   {XtNmarkOn, XtNmarkOn, XtRBoolean, sizeof(Boolean), 
      offset(adctool.mark_on), XtRBoolean, (caddr_t)&defFALSE},
   {XtNscrollWidget, XtNscrollWidget, XtRPointer, sizeof(caddr_t), 
      offset(adctool.scroll), XtRPointer, (caddr_t)NULL},
   {XtNtimeSeriesFn, XtNtimeSeriesFn, XtRPointer, sizeof(caddr_t), 
      offset(adctool.time_series_fn), XtRPointer, (caddr_t)NULL},
   {XtNdefTimeSeriesFn, XtCdefFn, XtRString, sizeof(String), 
      offset(adctool.def_time_series_fn), XtRString, ""},
   {XtNdefPitchFn, XtCdefFn, XtRString, sizeof(String), 
      offset(adctool.def_pitch_fn), XtRString, ""},
   {XtNsegOutDir, XtCParameter, XtRString, sizeof(String), 
      offset(adctool.segOutDir), XtRString, "."},
   {XtNlyreLabel, XtCParameter, XtRString, sizeof(String), 
      offset(adctool.lyre_label), XtRString, "Waveform Tool"},
};

#undef offset


AdctoolClassRec adctoolClassRec = {
  {
    (WidgetClass) &formClassRec,	/* superclass		  */	
    "Adctool",				/* class_name		  */
    sizeof(AdctoolRec),			/* size			  */
    NULL,				/* class_initialize	  */
    NULL,				/* class_part_initialize  */
    FALSE,				/* class_inited		  */
    Initialize,				/* initialize		  */
    NULL,				/* initialize_hook	  */
    XtInheritRealize,			/* realize		  */
    actions,				/* actions		  */
    XtNumber(actions),			/* num_actions		  */
    resources,				/* resources		  */
    XtNumber(resources),		/* resource_count	  */
    NULLQUARK,				/* xrm_class		  */
    FALSE,				/* compress_motion	  */
    TRUE,				/* compress_exposure	  */
    TRUE,				/* compress_enterleave    */
    FALSE,				/* visible_interest	  */
    NULL,				/* destroy		  */
    XtInheritResize,			/* resize		  */
    XtInheritExpose,			/* expose		  */
    SetValues,				/* set_values		  */
    NULL,				/* set_values_hook	  */
    XtInheritSetValuesAlmost,		/* set_values_almost	  */
    NULL,				/* get_values_hook	  */
    NULL,				/* accept_focus		  */
    XtVersion,				/* version		  */
    NULL,				/* callback_private	  */
    defaultTranslations,		/* tm_table		  */
    XtInheritQueryGeometry,		/* query_geometry	  */
    XtInheritDisplayAccelerator,	/* display_accelerator	  */
    NULL,				/* extension		  */
  },  /* CoreClass fields initialization */
  { /* composite_class fields */
    /* geometry_manager   */   XtInheritGeometryManager,
    /* change_managed     */   XtInheritChangeManaged,
    /* insert_child       */   XtInheritInsertChild,
    /* delete_child       */   XtInheritDeleteChild,
  },
  { /* constraint_class fields */
    /* subresourses       */   NULL,
    /* subresource_count  */   0,
    /* constraint_size    */   sizeof(AdctoolConstraintsRec),
    /* initialize         */   NULL,
    /* destroy            */   NULL,
    /* set_values         */   NULL
  },
  {
    0,
  }, /* FormClass fields */
  {
    0,                                     /* field not used    */
  },  /* AdctoolClass fields initialization */
};

WidgetClass adctoolWidgetClass = (WidgetClass) &adctoolClassRec;


#define MyXtSetArg(args,i,arg,val)	{XtSetArg (args[i], arg, val); i++;}


/*
 * Initialize (request, new)
 *
 * Initiliazes all resources and widgets descriptions concerning the
 * Adc tool widget. 
 *
 * Adctool -> command -> menu -> [Set Wave File   ] -> LyreAdcRead()
 *                            -> [Set Resolution  ] -> SetResolution()
 *                            -> [Set Pitch File  ] -> PitchRead()
 *                            -> [Write Pitch File] -> PitchWrite()
 *                            -> [Write Segment as] -> AdcSegWrite()
 *                            -> [Set Wave Channel] -> SetChannel()
 *                            -> [Toggle Pitch    ] -> ToggleMark()
 *                            -> [Delete Display  ] -> Destroy()
 *         -> hscale
 *         -> vscale
 *         -> mark
 *         -> child (graphics)
 *
 */

static void Initialize (Widget request_in, Widget new_in, ArgList arglist,
			Cardinal *num_args)
{
    AdctoolWidget request = (AdctoolWidget) request_in;
    AdctoolWidget new     = (AdctoolWidget) new_in;
    Widget  w, child, command, menu,
            entry, vscale = NULL, hscale = NULL, mark;
    Arg     args[20];
    int     arg_cnt;
    char    stmp[64];

    memset ((char *) &new->adctool.ac_data, 0, sizeof(vdata_t));
    memset ((char *) &new->adctool.sc_data, 0, sizeof(vdata_t));
    
    /*
     * Create command widget: [AdcTool] menu button
     *   Adctool -> command
     *
     */

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNright,  XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNbottom, XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNborderWidth,   2);
    MyXtSetArg (args, arg_cnt, XtNhorizDistance, 4);
    MyXtSetArg (args, arg_cnt, XtNvertDistance,  4);
    command = XtCreateManagedWidget(new->adctool.lyre_label, 
				    menuButtonWidgetClass, (Widget) new,
				    args, arg_cnt);
    
    /*
     * Create horizontal scale for graphics
     *  Adctool -> hscale
     *
     */

    if (new->adctool.horiz_scale_on) {
	arg_cnt = 0;
	MyXtSetArg (args, arg_cnt, XtNhorizDistance, 130); 
	MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
	MyXtSetArg (args, arg_cnt, XtNright,  XtChainRight);
	MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
	MyXtSetArg (args, arg_cnt, XtNbottom, XtChainTop);
	MyXtSetArg (args, arg_cnt, XtNresizable,  TRUE);
	MyXtSetArg (args, arg_cnt, XtNhorizontal, TRUE);
        MyXtSetArg (args, arg_cnt, XtNborderWidth, 1);
        MyXtSetArg (args, arg_cnt, XtNunitsLabel, "ms");
        MyXtSetArg (args, arg_cnt, XtNshowExtent, FALSE);
	hscale = XtCreateManagedWidget ("scale", scaleWidgetClass, 
					(Widget) new, args, arg_cnt);
        new->adctool.hscale = hscale;
    }

    /*
     * Create vertical scale for graphics
     *  Adctool -> vscale
     *
     */

    if (new->adctool.vert_scale_on) {
	arg_cnt = 0;
	MyXtSetArg (args, arg_cnt, XtNhorizDistance, 130);
	MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
	MyXtSetArg (args, arg_cnt, XtNright,  XtChainLeft);
	MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
	MyXtSetArg (args, arg_cnt, XtNbottom, XtChainBottom);
	if (new->adctool.horiz_scale_on)
	     MyXtSetArg (args, arg_cnt, XtNfromVert, hscale);
	MyXtSetArg (args, arg_cnt, XtNresizable,  TRUE);
	MyXtSetArg (args, arg_cnt, XtNhorizontal, FALSE);
        MyXtSetArg (args, arg_cnt, XtNborderWidth, 1);
	vscale = XtCreateManagedWidget ("scale", scaleWidgetClass, 
					(Widget) new, args, arg_cnt);
        new->adctool.vscale = vscale;
    }

    /*
     * Create widget for markings of speech signal
     *   Adctool -> mark
     *
     */

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNhorizDistance, 130);
    MyXtSetArg (args, arg_cnt, XtNvertDistance,  50);
    MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNright,  XtChainRight);
    MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNbottom, XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNresizable, TRUE);
    MyXtSetArg (args, arg_cnt, XtNborderWidth, 1);
    mark = XtCreateWidget ("mark", markWidgetClass, (Widget) 
			   new, args, arg_cnt);
    new->adctool.mark = mark;
    if (new->adctool.mark_on)
	XtManageChild (mark);

    /*
     * Create widget for menu dialog boxes
     *  Adctool -> child
     *
     */

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNhorizDistance, 130);
    MyXtSetArg (args, arg_cnt, XtNleft,   XtChainLeft);
    MyXtSetArg (args, arg_cnt, XtNright,  XtChainRight);
    MyXtSetArg (args, arg_cnt, XtNtop,    XtChainTop);
    MyXtSetArg (args, arg_cnt, XtNbottom, XtChainBottom);
    if (new->adctool.horiz_scale_on)
        MyXtSetArg (args, arg_cnt, XtNfromVert, hscale);
    MyXtSetArg (args, arg_cnt, XtNresizable,    TRUE);
    MyXtSetArg (args, arg_cnt, XtNvscaleWidget, vscale);
    MyXtSetArg (args, arg_cnt, XtNhscaleWidget, hscale);
    child = XtCreateManagedWidget ("graph", lgraphWidgetClass, (Widget) new,
				   args, arg_cnt);
    new->adctool.child = child;

    /*
     * Create popup menu for Adctool
     *   Adctool -> command (AdcTool button) -> menu
     *
     */

    menu = XtCreatePopupShell("menu", simpleMenuWidgetClass, command, 
			      NULL, 0);

    /*
     * Add first item to menu and its actions: [Set Wave File]
     *  Adctool -> command -> menu -> LyreAdcRead()
     *  See Also MakePopupL() --> ToolUtil.c
     *
     */

    entry = XtCreateManagedWidget("Set Wave File", smeBSBObjectClass, 
				  menu, NULL, 0);
    MakePopupL ((Widget)new, entry, (caddr_t)child,
		"Wave File Name:", new->adctool.def_time_series_fn,
		"Display", "Abort", (caddr_t)LyreAdcRead);

    /*
     * Create second meny entry: [Set Resolution]
     *  Adctool -> command -> menu -> SetResolution()
     *
     */

    entry = XtCreateManagedWidget("Set Resolution", smeBSBObjectClass, 
				  menu, NULL, 0);
    sprintf (stmp, "%g", ((LgraphWidget) child)->lyreDisp.secs_ppix);
    MakePopup ((Widget)new, entry, (caddr_t)child,
	       "Seconds per pixel:", stmp , "Set", "Abort", 
	       (caddr_t)SetResolution);

    /*
     * Third menu entry: [Set Pitch File]
     *  Adctool -> command -> menu -> PitchRead()
     *
     */

    entry = XtCreateManagedWidget("Set Pitch File", smeBSBObjectClass, 
				  menu, NULL, 0);
    MakePopup ((Widget)new, entry, (caddr_t)mark,
	       "Pitch file name:", "", "Display", "Abort", 
	       (caddr_t)PitchRead);

    /*
     * Fourth menu entry: [Write Pitch File]
     *  Adctool -> command -> menu -> PitchWrite()
     *
     */
     
    entry = XtCreateManagedWidget("Write Pitch File", smeBSBObjectClass, 
				  menu, NULL, 0);
    MakePopup ((Widget)new, entry, (caddr_t)mark, "Pitch file name:", 
	       "", "Write", "Abort", (caddr_t)PitchWrite);

    /*
     * Fifth menu entry: [Write Segment as Wave File]
     *  Adctool -> command -> menu -> AdcSegWrite()
     *
     */
     
    entry = XtCreateManagedWidget("Write Segment as Wave File", 
				  smeBSBObjectClass, menu, NULL, 0);
    MakePopup ((Widget)new, entry, (caddr_t)child,
	       "Name:", "t", "Write", "Abort", (caddr_t)AdcSegWrite);

    /*
     * Sixth menu entry: [Channel to Display]
     *  Adctool -> command -> menu -> SetChannel()
     *
     */
     
    entry = XtCreateManagedWidget("Set Wave Channel", smeBSBObjectClass, 
				  menu, NULL, 0);
    MakePopup ((Widget)new, entry, (caddr_t)child,
	       "Channel to Display", "0", "Set", "Abort", (caddr_t)SetChannel);

    /*
     * Seventh menu entry: [Toggle Pitch Marks]
     *  Adctool -> command -> menu -> ToggleMark
     *
     */
 
    entry = XtCreateManagedWidget("Toggle Pitch Marks", smeBSBObjectClass, 
				  menu, NULL, 0);
    XtAddCallback (entry, XtNcallback, (XtCallbackProc) ToggleMark, new);

    /*
     * Eigth menu entry: [Delete this display]
     *  Adctool -> command -> menu -> Destroy
     *
     */
     
    entry = XtCreateManagedWidget("Delete This Display", smeBSBObjectClass, 
				  menu, NULL, 0);
    XtAddCallback (entry, XtNcallback, (XtCallbackProc) Destroy, (Widget) new);
}



/*
 * Boolean SetValues (current, request, new)
 *
 * Set the values in the AdcTool widget
 *  More specifically old channel numbers
 * 
 */

static Boolean SetValues (Widget current_in, Widget request_in, 
			  Widget new_in, ArgList arglist, Cardinal *num_args)
{
    AdctoolWidget current = (AdctoolWidget) current_in;
    AdctoolWidget request = (AdctoolWidget) request_in;
    AdctoolWidget new     = (AdctoolWidget) new_in;

    Boolean redisplay = FALSE;
    Arg     args[10];
    int     arg_cnt = 0;

    if (new->adctool.channel != current->adctool.channel) {
	GetChannel (&new->adctool.ac_data, &new->adctool.sc_data,
		    new->adctool.channel, new->adctool.channel_count);

	arg_cnt = 0;
	MyXtSetArg (args, arg_cnt, XtNdata, &new->adctool.sc_data);
	XtSetValues (new->adctool.child, args, arg_cnt);
    }

    return (redisplay);
}


/*
 * LyreAdcRead(w, client_data)
 *
 * Menu #1: Read a waveform file from disc and display
 *          graphics.
 *
 * w          (in): parent widget for LyreAdcRead()
 * client_date(in): List of Widgets Currently created from parent
 *
 */

static void LyreAdcRead (Widget w, WidgetList client_data)
{
    LgraphWidget   child = (LgraphWidget)  client_data[0];
    DialogWidget   dw    = (DialogWidget)  client_data[1];
    AdctoolWidget  tool  = (AdctoolWidget) client_data[2];
    Arg            args[10];
    int            arg_cnt = 0;
    char           *file = XawDialogGetValueString ((Widget) dw);
    short	   *ptr  = 0;
    int len;
    float rate;

    /* 
     * Read waveform file and header  
     */

    if( ( len = SpeechRead( file, &ptr, &rate ) ) < 0 ) {
      XtAlert ("Could not open %s", file);
      return;
    }

    if (!(tool->adctool.time_series_fn = (char *) salloc(file))) {
      XtAlert("Malloc Failed on tool->adctool.time_series_fn");
      return;
    }

    if (tool->adctool.sc_data.ptr != tool->adctool.ac_data.ptr)
	XtFree (tool->adctool.sc_data.ptr);
    XtFree (tool->adctool.ac_data.ptr);
    tool->adctool.ac_data.ptr = (caddr_t) ptr;

    tool->adctool.ac_data.count = len;
    tool->adctool.ac_data.bsize = sizeof (short);
    if( rate != 0.0 )
	tool->adctool.ac_data.ns_per_datum = (int)(1.0e9 / rate);
    tool->adctool.channel_count = 1;

    /*
     * Extract channel number from header 
     */

    GetChannel (&tool->adctool.ac_data, &tool->adctool.sc_data,
		 tool->adctool.channel, tool->adctool.channel_count);

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNdata, &tool->adctool.sc_data);

    XtSetValues ((Widget) child, args, arg_cnt);

    {
      FL_UNION timeDuration;
      FL_UNION val;
      Widget   yoke;
      val.f = (((float)len) / rate);

      arg_cnt = 0;
      MyXtSetArg (args, arg_cnt, XtNyokeWidget, &yoke);
      XtGetValues ((Widget) child, args, arg_cnt);

      arg_cnt = 0;
      MyXtSetArg (args, arg_cnt, XtNtimeDuration, &timeDuration.l);
      XtGetValues (yoke, args, arg_cnt);

      arg_cnt = 0;
      MyXtSetArg (args, arg_cnt, XtNtimeDuration, val.l);
      XtSetValues (yoke, args, arg_cnt);
    }
}


/* 
 * AdcUnmap (tool)
 *
 * Make an Adc display disappear; space appears blank, not remapped 
 *
 */

int AdcUnmap(AdctoolWidget tool)
{
  XtUnmapWidget(tool->adctool.child);
}


/*
 * AdcMap (tool)
 *
 * Make an Adc display reappear 
 *
 */

int AdcMap (AdctoolWidget tool)
{
  XtMapWidget(tool->adctool.child);
}


/*
 * AdcPollRead (tool, file)
 *
 * Polling disc for Adc file (Used for auto-lyre)
 * 
 * tool (in): Adc tool data structure
 * file (in): name of Adc file to poll
 *
 */

int AdcPollRead (AdctoolWidget tool, char *file)
{
    LgraphWidget child = (LgraphWidget) tool->adctool.child;
    Arg          args[10];
    int          arg_cnt = 0;
    char         *fnp = tool->adctool.def_time_series_fn;
    short	 *ptr = 0;
    char         dir[128], basename[128], ext[32];
    int len;
    float rate;

    if(PollForFile (file, 1 /* Interval secs */, 1 /* Attempts */) == -1)
      return -1;

    if( (len = SpeechRead( file, &ptr, &rate ) ) < 0 ) {
	XtAlert ("Could not open %s", file);
	return( 1 );
    }

    if (!(tool->adctool.time_series_fn = (char *) salloc(file))) {
      XtAlert("Malloc Failed on tool->adctool.time_series_fn");
      return;
    }
    if (tool->adctool.sc_data.ptr != tool->adctool.ac_data.ptr)
	XtFree (tool->adctool.sc_data.ptr);
    XtFree (tool->adctool.ac_data.ptr);
    tool->adctool.ac_data.ptr = (caddr_t) ptr;

    tool->adctool.ac_data.count = len;
    tool->adctool.ac_data.bsize = sizeof (short);
    if (rate != 0.0)
	tool->adctool.ac_data.ns_per_datum = (int)(1.0e9/rate);
    tool->adctool.channel_count = 1;

    GetChannel (&tool->adctool.ac_data, &tool->adctool.sc_data,
		 tool->adctool.channel, tool->adctool.channel_count);

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNdata, &tool->adctool.sc_data);
    XtSetValues ((Widget) child, args, arg_cnt);

    {
      FL_UNION timeDuration;
      FL_UNION val;
      Widget yoke;
      val.f = ((float)len) / rate;

      arg_cnt = 0;
      MyXtSetArg (args, arg_cnt, XtNyokeWidget, &yoke);
      XtGetValues ((Widget) child, args, arg_cnt);

      arg_cnt = 0;
      MyXtSetArg (args, arg_cnt, XtNtimeDuration, &timeDuration.l);
      XtGetValues ((Widget) yoke, args, arg_cnt);

      arg_cnt = 0;
      MyXtSetArg (args, arg_cnt, XtNtimeDuration, val.l);
      XtSetValues ((Widget) yoke, args, arg_cnt);
    }
    return 0;
}


/* 
 * MarkUnMap(tool)
 *
 * Make a pitch mark display disappear; space appears blank, not remapped 
 *
 */

int MarkUnmap (AdctoolWidget tool)
{
  XtUnmapWidget(tool->adctool.mark);
}


/*
 * MarkMap (tool)
 * 
 * Make an pitch mark display reappear
 *
 */

int MarkMap (AdctoolWidget tool)
{
  XtMapWidget(tool->adctool.mark);
}


/*
 * int MarkPollRead (tool, file)
 *
 * Poll for then read a pitch file.  The tool argument is the
 * encompassing adctool.  The widget to display the marks is extracted
 * from that.  The bulk of the code was quickly adapted from
 * PitchRead, so some things may not be necessary.
 *
 * tool (in): Adc widget data structure
 * file (in): name of file to poll
 *
 */

int MarkPollRead (AdctoolWidget tool, char *file)
{
  Widget   mark = 0;
  Arg      args[10];
  int      arg_cnt = 0;
  char     peaks_file[128];
  vdata_t  *vdata = (vdata_t *) calloc (1, sizeof (vdata_t));
  vdata_t  *pdata = (vdata_t *) calloc (1, sizeof (vdata_t));
  vdata_t  *hdata = (vdata_t *) calloc (1, sizeof (vdata_t));
  int     no_peaks_file  = TRUE;

  if(PollForFile (file, 1 /* Interval secs */, 1 /* Attempts */) == -1)
    return (-1);

  /* found the pitch file, now look for peaks file once only */
  strcpy(peaks_file,file);
  sprintf(strrchr(peaks_file,'.'),".peaks");
  if (! ascii_ld_read ("", peaks_file, &vdata->ptr,
		       &vdata->bsize, &vdata->count))
    no_peaks_file = FALSE;

  arg_cnt = 0;
  MyXtSetArg (args, arg_cnt, XtNmark, &mark);
  XtGetValues ((Widget) tool, args, arg_cnt);

  if (ascii_ld_read ("", file, &pdata->ptr,
		       &pdata->bsize, &pdata->count))  {
      XtAlert ("Could not open pitch file %s", file);
      return( 1 );
    }

  /*  this is the "no_peaks" option from PitchRead */
  if(no_peaks_file)
    ascii_ld_read ("", file, &vdata->ptr,
		   &vdata->bsize, &vdata->count);
  
  memcpy ((char *) hdata, (char *) vdata, sizeof (vdata_t));
  hdata->ptr = (caddr_t) calloc (vdata->count, sizeof (int));
  
  mark_peaks ((u_int *) pdata->ptr, (u_int) pdata->count, 
	      (u_int *) vdata->ptr, (u_int) vdata->count,
	      (u_int *) hdata->ptr);
  free (pdata->ptr);
  free (pdata);
  
  if (tool->adctool.ac_data.ns_per_datum > 0)
    vdata->ns_per_datum = tool->adctool.ac_data.ns_per_datum;
  else
    vdata->ns_per_datum = 62500;
  
  arg_cnt = 0;
  MyXtSetArg (args, arg_cnt, XtNinputData, vdata);
  MyXtSetArg (args, arg_cnt, XtNinputHighlight, hdata);
  XtSetValues (mark, args, arg_cnt);
  return 0;
}


/*
 * PitchRead (w, client_data)
 *
 * Read a pitch file from disc.
 *
 * w           (in): parent widget where PitchRead was called from
 * client_data(out): data structure for passing data between widget calls
 *
 */

static void PitchRead (Widget w, WidgetList client_data)
{
    MarkWidget     mark = (MarkWidget) client_data[0];
    DialogWidget   dw   = (DialogWidget) client_data[1];
    AdctoolWidget  tool = (AdctoolWidget) client_data[2];
    Arg            args[10];
    int            arg_cnt = 0;
    char           *file = XawDialogGetValueString ((Widget) dw);
    char           peaks_file[128], pitch_file[128];
    vdata_t        *vdata = (vdata_t *) calloc (1, sizeof (vdata_t));
    vdata_t        *pdata = (vdata_t *) calloc (1, sizeof (vdata_t));
    vdata_t        *hdata = (vdata_t *) calloc (1, sizeof (vdata_t));
    char           *extension;
    int           no_peaks_file = TRUE;
    int           no_pitch_file = TRUE;

    *peaks_file = 0;
    *pitch_file = 0;

    /*
     * Check if extention is peaks or pitch or something else 
     */
    extension = strrchr (file, '.');
    if (extension) {
	if (strcmp (extension, ".peaks") == 0) {
	    strcpy (peaks_file, file);
	    strcpy (pitch_file, file);
	    sprintf (strrchr (pitch_file, '.'), ".pitch");
	}
	else if (strcmp (extension, ".pitch") == 0) {
	    strcpy (pitch_file, file);
	    strcpy (peaks_file, file);
	    sprintf (strrchr (peaks_file, '.'), ".peaks");
	}
	else {
	    strcpy (pitch_file, file);
	}
    }
    else {
	sprintf (peaks_file, "%s.peaks", file);
	sprintf (pitch_file, "%s.pitch", file);
    }

    if (*peaks_file)
	if (! ascii_ld_read ("", peaks_file, &vdata->ptr,
			    &vdata->bsize, &vdata->count))
	    no_peaks_file = FALSE;

    if (*pitch_file)
	if (! ascii_ld_read ("", pitch_file, &pdata->ptr,
			    &pdata->bsize, &pdata->count))
	    no_pitch_file = FALSE;

    if (no_pitch_file && no_peaks_file) {
	XtAlert ("Could not open [%s] or [%s]", peaks_file, pitch_file);
	return;
    }

    if (no_pitch_file)
	ascii_ld_read ("", peaks_file, &pdata->ptr,
			    &pdata->bsize, &pdata->count);
    if (no_peaks_file)
	ascii_ld_read ("", pitch_file, &vdata->ptr,
			    &vdata->bsize, &vdata->count);


    memcpy ((char *) hdata, (char *) vdata, sizeof (vdata_t));
    hdata->ptr = (caddr_t) calloc (vdata->count, sizeof (int));

    mark_peaks ((u_int *) pdata->ptr, (u_int) pdata->count, 
		(u_int *) vdata->ptr, (u_int) vdata->count,
		(u_int *) hdata->ptr);
    free (pdata->ptr);
    free (pdata);

    if (tool->adctool.ac_data.ns_per_datum > 0)
	vdata->ns_per_datum = tool->adctool.ac_data.ns_per_datum;
    else
	vdata->ns_per_datum = 62500;

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNinputData, vdata);
    MyXtSetArg (args, arg_cnt, XtNinputHighlight, hdata);
    XtSetValues ((Widget) mark, args, arg_cnt);
}


/*
 * PitchWrite (w, client_data)
 *
 * Write the pitch information to disc
 *
 * w           (in): parent widget where PitchRead was called from
 * client_data(out): data structure for passing data between widget calls
 *
 */

static void PitchWrite (Widget w, WidgetList client_data)
{
    MarkWidget    mark = (MarkWidget)    client_data[0];
    DialogWidget  dw   = (DialogWidget)  client_data[1];
    AdctoolWidget tool = (AdctoolWidget) client_data[2];
    Arg           args[10];
    int           arg_cnt = 0;
    char          *file = XawDialogGetValueString ((Widget) dw);
    char          pitch_file[128];
    vdata_t       *hdata;
    vdata_t       *pdata;
    int           i, *buf, pcnt;
    int		  *pptr, *hptr;

    if (! strrchr(file, '.'))
	sprintf (pitch_file, "%s.pitch", file);
    else
	strcpy (pitch_file, file);

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNstoreHighlight, &hdata);
    MyXtSetArg (args, arg_cnt, XtNstoreData, &pdata);
    XtGetValues ((Widget) mark, args, arg_cnt);

    if ((hdata == 0) && (pdata == 0)) {
	XtAlert ("No pitch data to write!");
	return;
    }

    buf = (int *) calloc (pdata->count, sizeof (int));
    hptr = (int *) hdata->ptr;
    pptr = (int *) pdata->ptr;
    for (pcnt = i = 0; i < pdata->count; i++) {
	if (hptr[i])
	    buf[pcnt++] = pptr[i];
    }

    ascii_ld_write (pitch_file, buf, sizeof (int), pcnt);
    free (buf);
}


/*
 * void SetResolution (w, client_data)
 *
 * Set Resolution of scales for graphics appearance.
 *
 * w           (in): parent widget where PitchRead was called from
 * client_data(out): data structure for passing data between widget calls
 *
 */

static void SetResolution (Widget w, WidgetList client_data)
{
    DialogWidget  dw = (DialogWidget) client_data[1];
    AdctoolWidget aw = (AdctoolWidget) client_data[2];
    Arg    args[10];
    int arg_cnt = 0;
    char *in = XawDialogGetValueString((Widget) dw);
    union {
      float f;
      int  l;
    } value;

    value.f = (float) atof(in);
    if (value.f==0.0) {
      XtAlert("Invalid Resolution");
      return;
    }

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNsecsPpix, value.l);
    XtSetValues (aw->adctool.child, args, arg_cnt);
    XtSetValues (aw->adctool.mark, args, arg_cnt);
}


/*
 * SetChannel(w, client_data)
 *
 * Set the channel for the Adc tool (part of header information)
 *
 * w           (in): parent widget where PitchRead was called from
 * client_data(out): data structure for passing data between widget calls
 *
 */

static void SetChannel (Widget w, WidgetList client_data)
{
    LgraphWidget  child = (LgraphWidget) client_data[0];
    DialogWidget  dw    = (DialogWidget) client_data[1];
    AdctoolWidget tool  = (AdctoolWidget) client_data[2];
    Arg  args[10];
    int  arg_cnt = 0;
    char *in     = XawDialogGetValueString((Widget) dw);
    int  channel = atoi (in);              

    arg_cnt = 0;
    MyXtSetArg (args, arg_cnt, XtNchannel, channel);
    XtSetValues ((Widget) tool, args, arg_cnt);
}


/*
 * Destroy (w, target_wdiget, call_data)
 *
 * Destroy the Adc widget and all its children.
 *
 */

static void Destroy (Widget w, AdctoolWidget target_widget, caddr_t call_data)
{
    if (target_widget->adctool.time_series_fn)
      free (target_widget->adctool.time_series_fn);
    XtDestroyWidget ((Widget) target_widget);
}


/*
 * ToggleMark(cw, w, call_data)
 * 
 * Toggle the markings on the Adc graphics.
 *
 * w           (in): parent widget where PitchRead was called from
 * client_data(out): data structure for passing data between widget calls
 *
 */

static void ToggleMark (Widget cw, AdctoolWidget w, caddr_t call_data)
{
    Widget mark = w->adctool.mark;

    if (mark != NULL) {
	if (w->adctool.mark_on) {
	    w->adctool.mark_on = FALSE;
	    XtUnmanageChild (mark);
	}
	else {
	    w->adctool.mark_on = TRUE;
	    /*
	     * The form widget is incredibly brain damaged.
	     * If a child widget wasn't mapped/managed when the
	     * widget is realized it won't be sizeed the correct
	     * way, I fix that here.
	     */
	    mark->core.width = w->core.width;
	    XtManageChild (mark);
	}
    }
}


/*
 * mark_peaks (pitch, pitch_count, peak, peak_count, yes)
 *
 * Mark all pitch peaks in the peaks vector
 *
 */

u_int mark_peaks (register u_int *pitch, register u_int pitch_count, 
		   register u_int *peak,  register u_int peak_count, 
		   register u_int *yes)
{
    register u_int  pitch_idx = 0, peak_idx = 0;
    register u_int  missing_pitch = 0;

    /*
     * Loop through the pitch peaks . 
     */
    while ((pitch_idx < pitch_count) && (peak_idx < peak_count)) {
	if (abs (*peak - *pitch) <= 2) {
	    *yes++ = TRUE;
	    peak++;
	    pitch++;
	    peak_idx++;
	    pitch_idx++;
	}
	else {
	    *yes++ = FALSE;
	    peak++;
	    peak_idx++;
	    if (*peak > *pitch) {
		pitch++;
		pitch_idx++;
		missing_pitch++;
	    }
	}
    }
    return (missing_pitch);
}


/*
 * GetChannel (ac_data, sc_data, channel, channel_count)
 *
 * From the speech header determine the channel.
 *
 */

void GetChannel (vdata_t *ac_data, vdata_t *sc_data, int channel, 
		 int channel_count)
{
    short              *dst;
    register short     *dptr, *sptr;
    register int        i;

    if (channel_count <= 1) {
	memcpy ((char *) sc_data, (char *) ac_data, sizeof (vdata_t));
    }
    else {
	XtFree (sc_data->ptr);
	memcpy ((char *) sc_data,(char *) ac_data, sizeof (vdata_t));
	dst = dptr = (short *) XtMalloc ((ac_data->count * ac_data->bsize) /
					 channel_count);
	sptr = (short *) ac_data->ptr;
	for (i = channel; i < ac_data->count; i += channel_count)
	    *dptr++ = sptr[i];
	sc_data->ptr = (caddr_t) dst;
	sc_data->count /= channel_count;
    }
}


/*
 * LolaSegWrite (w, client_data)
 *
 * Write out a lola file coorespond to the bracketed segment.
 * 
 * w               (in): Parent widget, where this procedure was called from
 * client_data (in/out): data pass pointer through widget calls
 *
 */

static void LolaSegWrite (Widget w, WidgetList client_data)
{
    LgraphWidget  lw   = (LgraphWidget) client_data[0];
    DialogWidget  dw   = (DialogWidget) client_data[1];
    AdctoolWidget tool = (AdctoolWidget) client_data[2];
    Arg                 args[10];
    int                 arg_cnt = 0;
    char               *file = XawDialogGetValueString ((Widget) dw);
    float               start_time;
    float               end_time;
    FILE               *fs;

    fs = fopen (file, "w");
    if (fs == NULL) {
      XtAlert ("Couldn't open %s for writing.\n", file);
      return;
    }

    start_time = lw->lgraph.left_bracket_x * lw->lyreDisp.secs_ppix;
    end_time = lw->lgraph.right_bracket_x * lw->lyreDisp.secs_ppix;

    fprintf (fs, "# Source adcfile name <%s>\n", tool->adctool.time_series_fn);
    if (start_time < end_time)
      fprintf (fs, "%.5f\t%.5f\tno_label\n", start_time, end_time);
    else
      fprintf (fs, "%.5f\t%.5f\tno_label\n", end_time, start_time);

    fclose (fs);
}


/*
 * AdcWrite(w, client_data)
 *
 * AdcSegWrite - write an adc segment
 *
 * DESCRIPTION
 *     Write out an adc file cooresponding to the bracketed segment.
 *
 */

static void AdcSegWrite (Widget w, WidgetList client_data)
{
    LgraphWidget   lw = (LgraphWidget) client_data[0];
    DialogWidget   dw = (DialogWidget) client_data[1];
    AdctoolWidget   tool = (AdctoolWidget) client_data[2];
    Arg   args[10];
    int   arg_cnt = 0;
    char  *name = XawDialogGetValueString ((Widget) dw);
    char  filename[256];
    char  tmp_str[256];
    float start_time;
    float end_time;
    float tmp;
    int   start_sample;
    int   end_sample;
    float samples_per_sec;
    char  *cptr;

    if (tool->adctool.sc_data.ptr == NULL)
      return;

    parse_path (tool->adctool.time_series_fn, 0, tmp_str, 0);
    sprintf (filename, "%s/%s-%s.wav", tool->adctool.segOutDir, tmp_str, name);
    
    start_time = lw->lgraph.left_bracket_x * lw->lyreDisp.secs_ppix;
    end_time = lw->lgraph.right_bracket_x * lw->lyreDisp.secs_ppix;

    if (start_time > end_time) {
      tmp = end_time;
      end_time = start_time;
      start_time = tmp;
    }
    samples_per_sec = 1e9 / tool->adctool.sc_data.ns_per_datum;
    start_sample = start_time * samples_per_sec;
    end_sample = end_time * samples_per_sec;

    start_sample = MAX(0, start_sample);
    start_sample = MIN(start_sample, tool->adctool.sc_data.count);

    end_sample = MAX(end_sample, start_sample);
    end_sample = MIN(end_sample, tool->adctool.sc_data.count);

    if( WavWrite( filename, (short *) (tool->adctool.sc_data.ptr +
		 (tool->adctool.sc_data.bsize * start_sample)), 
		 end_sample - start_sample, (int)samples_per_sec, 0) < 0 )
      XtAlert ("Couldn't write %s - %s", filename, ErrorString);
}
