static char SccsID[] = "@(#)lpcautor.c	1.1   12/15/89";
/*
  Copyright (C) 1994 Edmund Lai.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
*/
/*  === Speech Recognition Research Subroutine Library ===
 *
 *		==== LPCAUTOR.C ====
 *
 *  Description:
 *	Functions for Linear Predictive Analysis using the
 *	conventional autocorrelation method (Durbin recursion).
 *	The functions are essentially the same as in 'LPCAUTO.C'.
 *	However, they are designed for those programs that
 *	repeatedly call these functions.
 *
 *  Author:
 *	Edmund Lai
 *	Department of Electrical & Electronic Engineering
 *	The University of Western Australia
 *
 *============================================================*/

#include <math.h>
#include <malloc.h>

#define  pi	   M_PI


	/*==============================*/
	/*====== GLOBAL VARIABLES ======*/
	/*==============================*/
static int p = 0;		/* Analysis order */
static double *r,		/* Autocorrelation sequence */
	      *parcor,		/* Partial correlation coefficients */
	      *temp;		/* Temporary work space */


/*============= init_lpca ==============*/
/* Initialize the lpc analysis sequence */
/*======================================*/
void init_lpca (order)
int	order;
{
	p = order;		/* set the order of analysis */

	/* Allocate memory for temporary storage */
	r = (double *)malloc((order+1) * sizeof(double));
	parcor = (double *)malloc((order+1) * sizeof(double));
	temp = (double *)malloc((order+1) * sizeof(double));
}


/*========== close_lpca =========*/
/* End the lpc analysis sequence */
/*===============================*/
void close_lpca ()
{
	p = 0;
	free(r);
	free(parcor);
	free(temp);
}


/*==================== AUTOCORRELATION ======================*/
/*  Computes the LPC coefficients using the autocorrelation  */
/*  method with Durbin's recursive algorithm.                */
/*===========================================================*/
void  rlpc_autocorrelation (datalength,s,a,pcor)
int	datalength;
double	s[], a[], *pcor;
{
	int	i, j;
	double	EE;

	/* Initialize data arrays */
	for (i = 0; i <= p; i++) {
		a[i] = 0.0;
		*(parcor+i) = 0.0;
	}
	a[0] = 1.0;
		
	/* Compute the autocorrelation vector */
	for (i = 0; i <= p; i++) {
		*(r+i) = 0.0;
		for (j = 0; j < (datalength - i); j++)
			*(r+i) += (s[j] * s[j+i]);
	}

	/* Solve the Toeplitz set of equations */
	EE = *r;
	*(parcor+1) = *(r+1) / EE;
	a[1] = *(parcor+1);
	for (i = 2; i <= p; i++) {
		EE = (1.0 - (*(parcor+i-1) * *(parcor+i-1))) * EE;
		*(parcor+i) = *(r+i);
		for (j = 1; j < i; j++)
			*(parcor+i) = *(parcor+i) - *(r+i-j) * a[j];
		*(parcor+i) /= EE;
		a[i] = *(parcor+i);
		for (j = 1; j < i; j++)
			*(temp+j) = a[j] - *(parcor+i) * a[i-j];
		for (j = 1; j < i; j++)
			a[j] = *(temp+j);
	}

	for (i = 0; i <= p; i++)  *pcor++ = *(parcor+i);
}


/*================= rlpc_auto =================*/
/* Same as above except that the power of the */
/* data segment is returned instead of PARCOR */
/*============================================*/
void  rlpc_auto (datalength,s,a,power)
int	datalength;
double	s[], a[];
double  *power;
{
	int	i, j;
	double	EE;

	/* Initialize data arrays */
	for (i = 0; i <= p; i++) {
		a[i] = 0.0;
		*(parcor+i) = 0.0;
	}
	a[0] = 1.0;
		
	/* Compute the autocorrelation vector */
	for (i = 0; i <= p; i++) {
		*(r+i) = 0.0;
		for (j = 0; j < (datalength - i); j++)
			*(r+i) += (s[j] * s[j+i]);
	}
	*power = *r;

	/* Solve the Toeplitz set of equations */
	EE = *r;
	*(parcor+1) = *(r+1) / EE;
	a[1] = *(parcor+1);
	for (i = 2; i <= p; i++) {
		EE = (1.0 - (*(parcor+i-1) * *(parcor+i-1))) * EE;
		*(parcor+i) = *(r+i);
		for (j = 1; j < i; j++)
			*(parcor+i) = *(parcor+i) - *(r+i-j) * a[j];
		*(parcor+i) /= EE;
		a[i] = *(parcor+i);
		for (j = 1; j < i; j++)
			*(temp+j) = a[j] - *(parcor+i) * a[i-j];
		for (j = 1; j < i; j++)
			a[j] = *(temp+j);
	}
}


/*======================= LPC_AUTO2 =========================*/
/*  Computes the LPC coefficients using the autocorrelation  */
/*  method with Durbin's recursive algorithm.                */
/*  Same as rlpc_autocorrelation except that the auto-	     */
/*  correlation coefficients are supplied by calling program */
/*===========================================================*/
void  rlpc_auto2 (rr,a,pcor)
double	rr[], a[], pcor[];
{
	int	i, j;
	double	EE;

	/* Initialize data arrays */
	for (i = 0; i <= p; i++) {
		a[i] = 0.0;
		*(parcor+i) = 0.0;
	}
	a[0] = 1.0;
		
	/* Solve the Toeplitz set of equations */
	EE = rr[0];
	*(parcor+1) = rr[1] / EE;
	a[1] = *(parcor+1);
	for (i = 2; i <= p; i++) {
		EE = (1.0 - (*(parcor+i-1) * *(parcor+i-1))) * EE;
		*(parcor+i) = rr[i];
		for (j = 1; j < i; j++)
			*(parcor+i) = *(parcor+i) - rr[i-j] * a[j];
		*(parcor+i) /= EE;
		a[i] = *(parcor+i);
		for (j = 1; j < i; j++)
			*(temp+j) = a[j] - *(parcor+i) * a[i-j];
		for (j = 1; j < i; j++)
			a[j] = *(temp+j);
	}

	for (i = 0; i <= p; i++)  pcor[i] = *(parcor+i);
}

