/*
  title: ciips
  purpose: Format converting filter to interface CIIPS group .nrg files with
    the fview package.

  authors:  Gareth Lee.
  date:     23-10-93
  modified: 26-04-94

  Copyright (C) 1994 Gareth Lee.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  04-04-94: Support for extended FViewHeaderRecord (fview v1.2) added.
    This involves handling the color_preference and logged_status fields.
  26-04-94: Byte swapping option added using the -B swap|noswap flag to
    aid porting to platforms with different byte ordering.  Normally `noswap'
    should be used on Motorola/Sun platforms whilst `swap' should be enabled
    on Intel/DEC machines.
*/
#define VERSION "1.2"

#include <stdio.h>
#include <string.h>
#include <memory.h>

#include "sfheader.h"
#include "../fviewio.h"
#include "swappers.h"

char *filename = NULL;
char *magic = NULL;
FILE *fd = NULL;
AHeader afh;                                  /* Analysis_file_Header record */
FviewFeatureHeader ffh;
char text[10240];                                      /* Output text buffer */
int color_key = FVIEW_COLR_NOPREF;                /* no colour key specified */
int byte_swap = 0;                    /* perform byte swapping on input data */

extern char *optarg;

main(int argc, char *argv[])
{
  int i, j, nf, opt;
  long fpos1, fpos2;
  double *buf;

  /* read command line options */
  while ((opt = getopt(argc, argv, "B:C:f:m:")) != -1)
    switch (opt) {
    case 'B':
      if (*optarg == 's' || *optarg == 'S')
        byte_swap = 1;
      else
        byte_swap = 0;
      break;
    case 'C':
      color_key = atoi(optarg);
      if (color_key <= FVIEW_COLR_NOPREF)
      {
        printf("ciips: color preference should be >= 1\n");
        exit(-1);
      }
      break;
    case 'f':
      filename = optarg;
      break;
    case 'm':
      magic = optarg;
      break;
    case '?':
      fprintf(stderr, "usage: ciips -C colour_key -f file_name"
                      "-B [swap|noswap] -m magic number\n");
      fprintf(stderr, "ciips: option -%c not supported\n", opt);
      exit(-1);
    }
  
  if (filename != NULL)
  {
    fd = fopen(filename, "r");  
    if (fd == NULL)
    {
      fprintf(stderr, "Error: cannot open file %s\n", filename);
      fflush(stderr);
      exit(-1);
    }
  }
  else
  {
    fprintf(stderr, "ciips: no file name provided\n");
    fflush(stderr);
    exit(-1);    
  }

  /* read the header record */
  fread(&afh, sizeof(AHeader), 1, fd);
  if (byte_swap)
  {
    SWAP2(&afh.order);
    SWAP2(&afh.frame_length);
    SWAP2(&afh.adv);
  }

  if (magic != NULL)
  {
    if (strlen(magic) != 3)
      fprintf(stderr, "ciips: magic string must be 3 characters - ignored\n");
    else
    {
      if (strcmp(afh.type_of_file, magic) != 0)
      {
        fprintf(stderr, "ciips: file has wrong magic number (%s)\n",
                afh.type_of_file);
        fflush(stderr);
        exit(-1);
      }
    }
  }

  /* find the number of feature vectors in the file */
  fpos1 = ftell(fd);
  fseek(fd, 0, 2);
  fpos2 = ftell(fd);
  fseek(fd, fpos1, 0);
  nf = (fpos2 - fpos1) / (afh.order * sizeof(double));

  /* create and write an Fview header record */
  strcpy(magic, FVIEW_FEATURE_MAGIC);
  memcpy(ffh.magic, magic, 8);
  ffh.vector_dimension = afh.order;
  ffh.number_observations = nf;
  sprintf(text, "TYPE=%s\nSPEAKER_ID=%s\nSAMPLE_FREQ=%s\nORDER=%d\n"
          "FRAME_LENGTH=%d\nADVANCE=%d\nSPEECH=%s\nPRE_EMPHASIS=%c\n"
          "SAMPLE_BITS=%s\nSIGNED_DATA=%c\n",
          afh.type_of_file, afh.speaker_id, afh.sample_freq, afh.order,
          afh.frame_length, afh.adv, afh.speech, afh.preemphasis,
          afh.sample_bits, afh.signed_data);
  ffh.info_size = strlen(text) + 1;
  if (ffh.info_size > 10240)
  {
    fprintf(stderr, "ciips: textual information overflow\n");
    exit(-1);
  }
  ffh.logged = FVIEW_FEAT_UNKNOWN;        /* extracted from the command line */
  ffh.color_preference = color_key;       /* extracted from the command line */
  fwrite(&ffh, sizeof(FviewFeatureHeader), 1, stdout);
  fwrite(text, sizeof(char), ffh.info_size, stdout);

  /* copy the data across */
  buf = (double *) malloc(afh.order * sizeof(double));
  for (i = 0; i < nf; i++)
  {
    if (fread(buf, sizeof(double), afh.order, fd) != afh.order)
    {
      exit(-1);
      fprintf(stderr, "Error: data terminated prematurely\n");
      fflush(stdout);
    }
    if (byte_swap)
    {
      for (j = 0; j < afh.order; j++)
        SWAP8(&buf[j]);
    }
    if (fwrite(buf, sizeof(double), afh.order, stdout) != afh.order)
      exit(-1);
  }
  fclose(fd);
  fprintf(stderr, "ciips (%s): %s (%d obs)\n", VERSION, filename, nf);
}


