/*
  title: projectors.c
  purpose: Projection routines called from within fview.  Routines are
    provided for 2 and 3 dimensional projections and sweeps (slowed
    projections).  A spectrogram projector is also provided.
  
  authors:  Gareth Lee & Nicole Duball.
  date:     16-08-93
  modified: 22-04-94

  Copyright (C) 1994 Gareth Lee and Nicole Duball.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:  
  08-09-93: functions copied out of fview.c
  29-11-93: double buffering support added to display_3d_graphics() intended
    for use in the animation display mode.
  06-12-93: double buffer support completed.  DrawOrigin3() mod'd to support
    double buffered colors.
  08-12-93: sweep mode modified to operate correctly with double buffering.
  10-02-94: Code segments stripped out and placed in separate compilation
    units: timer.c, ctable.c, dbuffer.c.
  11-02-94: Code rationalized by the addition of some common functions such
    as FindRange().
  15-02-94: As continuation of previous change a new common function
    FindHighlightedVerts() implemented.
  01-04-94: 3D trace separation value made variable and linked to a slider
    within the 3D colours windows.
  02-04-94: minor bug fixed in draw_spectrogram() to ensure that highlighting
    lines are drawn acrossthe actual height of the spectrogram rather than the
    desired height.
  22-04-94: rounding function changed from nint() to ROUND() during port to
    Linux/i386.  The ROUND() macro can then be defined to a suitable
    library function on the architecture available.
*/

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <unistd.h>
#include <assert.h>

/* X lib & XView headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/cursor.h>
#include <xview/cms.h>

typedef REAL real;

/* Header files associated with other fview compilation units */
#include "fviewio.h"
#include "projectors.h"
#include "fview.h"
#include "interface.h"
#include "callbacks.h"
#include "entropy.h"
#include "timer.h"
#include "dbuffer.h"
#include "ctable.h"

#define DEG360 (23040)                   /* 360 degrees in 1/64 degree units */
#define ROUND(r) floor((double)(r))              /* standard library function */

int DepthScaling3D = DEFAULT_DEPTH_SCALING;  /* depth separation for 3D disp */
GC gc_3d, gc1_3d;                 /* Graphics contexts to use for 3D drawing */

/*****************************************************************************/

/*
  FindRange: find maximal and minimal values from within v[0..N-1] and return
  via pointers minp and maxp.
*/
void FindRange(real v[], int N, real *minp, real *maxp)
{
  int i;
  real min, max, proj;

  min = max = v[0];
  for (i = 1; i < N; i++)
  {
    proj = v[i];
    if (proj > max)
      max = proj;
    else
      if (proj < min)
        min = proj;
  }
  *minp = min;
  *maxp = max;
}

/*****************************************************************************/

/*
  FindHighlightedVerts: mark those vertices which occur within the `hp'
  entries of `order' so that they can be highlighted during subsequent
  drawing operations.
*/
void FindHighlightedVerts(int np, int order[], int hp, int mark[], int nverts)
{
  int i, j, k;

  /* reset all marks values and check that (nverts > 0) before continuing */
  for (i = 0; i < np; i++)
    mark[i] = 0;
  if (nverts == 0)
    return;

  /* set those marks entries cited in the order array */
  for (j = 0; j < hp; j++)
  {
    k = order[j];
    if (k >= 0 && k < np)
      mark[k] = 1;
    else
    {
      fprintf(stderr,"FindHighlightedVerts: index error within order array\n");
      exit(-1);
    }
  }
}
  
/*****************************************************************************/

/*
  OriginAdjust: ensure the min-max range calculated for auto_scale mode
    contains the origin.
*/
void OriginAdjust(real *min, real *max)
{
  if (*min > 0.0)
    *min = 0.0;
  else
    if (*max < 0.0)
      *max = 0.0;
}

/*****************************************************************************/

/*
  DrawOrigin2: Draw an origin symbol centered at (x,y) co-ordinates.
*/
void DrawOrigin2(int x, int y)
{
  const int radius = 8;                          /* radius of circle to draw */
  
  int x1 = x - radius;
  int x2 = x + radius;
  int y1 = y - radius;
  int y2 = y + radius;

  /* Draw black cross hairs and a surrounding circle */
  XSetForeground(display, gc, traj_color[BLACK].pixel);
  XDrawLine(display, win, gc, x1,  y, x2,  y);
  XDrawLine(display, win, gc,  x, y1,  x, y2);
  XDrawArc(display, win, gc, x1, y1, (2 * radius), (2 * radius), 0, DEG360);
}

/*****************************************************************************/
/*
  DrawOrigin3: Draw a three dimensional origin symbol centered at (x,y,z)
    co-ordinates.
*/
void DrawOrigin3(int x, int y, int z_offs,
                 unsigned long left_color, unsigned long right_color,
                 unsigned long text_color)
{
  const int radius = 8;                          /* radius of circle to draw */

  int x1, x2;
  int y1 = y - radius;
  int y2 = y + radius;

  if (z_offs < 5)
  {
    /* Draw cross hairs and a surrounding circle in a neutral colour */
    XSetForeground(display, gc, text_color);
    x1 = x - radius;
    x2 = x + radius;
    XDrawLine(display, win, gc, x1,  y, x2,  y);
    XDrawLine(display, win, gc,  (x - z_offs), y1,  (x - z_offs), y2);
    XDrawArc(display, win, gc, x1, y1, (2 * radius), (2 * radius), 0, DEG360);
  }
  else
  {
    /* Draw cross hairs and a surrounding circle in both colours */
    XSetForeground(display, gc, left_color);
    x1 = x - radius - z_offs;
    x2 = x + radius - z_offs;
    XDrawLine(display, win, gc, x1,  y, x2,  y);
    XDrawLine(display, win, gc,  (x - z_offs), y1,  (x - z_offs), y2);
    XDrawArc(display, win, gc, x1, y1, (2 * radius), (2 * radius), 0, DEG360);
    
    XSetForeground(display, gc, right_color);
    x1 = x - radius + z_offs;
    x2 = x + radius + z_offs;
    XDrawLine(display, win, gc, x1,  y, x2,  y);
    XDrawLine(display, win, gc,  (x + z_offs), y1,  (x + z_offs), y2);
    XDrawArc(display, win, gc, x1, y1, (2 * radius), (2 * radius), 0, DEG360);
  }
}

/*****************************************************************************/

/*
  DrawPointSet: Draw a set of points, each marked by an `x' and unconnected.
*/
void DrawPointSet(Display *dpy, Window win, GC gc, XPoint pts[], int npts)
{
  int i;

  for (i = 0; i < npts; i++)
    XDrawString(dpy, win, gc, pts[i].x - 3, pts[i].y + 4, "x", 1);
}

/*****************************************************************************/

/*
  ConstructFilenameString: construct and extended filename which indicates
    the Linear/Logged nature of the features being displayed.
*/
int ConstructFilenameString(SpeechRecord *sd, char *buf)
{
  int new_length;      /* string length after additional type chars appended */

  switch (sd->logged)
  {
  case FVIEW_FEAT_LINEAR:
    sprintf(buf, "%s (lin)", sd->file_name);
    new_length = sd->file_name_length + 6;
    break;

  case FVIEW_FEAT_LOGGED:
    sprintf(buf, "%s (log)", sd->file_name);
    new_length = sd->file_name_length + 6;
    break;

  case FVIEW_FEAT_UNKNOWN:
    if (sample_assume_logged)
      sprintf(buf, "%s (assumed log)", sd->file_name);
    else
      sprintf(buf, "%s (assumed lin)", sd->file_name);
    new_length = sd->file_name_length + 14;
    break;
  }

  /* Increase label box width to account for the additional characters */
  sd->label_box.width = (unsigned short) XTextWidth(FontInfo, buf, new_length);
  return new_length;
}

/*****************************************************************************/

/*
  draw_3d_graphics: Draw 3D projection of a single trajectory
*/
void draw_3d_graphics(unsigned int window_width, unsigned int window_height,
                      int label)
{
  unsigned long plane_mask;
  unsigned long text_color;                /* color index for drawing labels */
  unsigned long bg_color;              /* color index for drawing background */
  unsigned long left_color;                /* color index for left eye trace */
  unsigned long right_color;              /* color index for right eye trace */  
  int i, s, x, y;
  int width,height;
  int drawx1, drawx2, drawy1, drawy2;
  XPoint *left_points, *right_points;
  real xmin, xmax, ymin, ymax, zmin, zmax, proj;
  int index, zoffset, length;
  XGCValues xgcv;
  SpeechRecord *sd;
  char cbuf[1024];

  /* Swap the drawing buffer before starting to create the graphics */
  GetHiddenColors(&plane_mask, &text_color, &bg_color,
                   &left_color, &right_color);
  
  /* Reduce drawable area slightly to allow for offsets */
  drawx1 = X_MARGIN_3D;
  drawx2 = window_width - X_MARGIN_3D;
  drawy1 = Y_MARGIN_3D;
  drawy2 = window_height - Y_MARGIN_3D;
  
  /* Automatic scaling to ensure trajectory fills projection area */
  s = utterance_index;
  sd = spdata[s];
  left_points = (XPoint *) malloc(sd->npoints * sizeof(XPoint));
  right_points = (XPoint *) malloc(sd->npoints * sizeof(XPoint));

  FindRange(sd->project1, sd->npoints, &xmin, &xmax);
  FindRange(sd->project2, sd->npoints, &ymin, &ymax);
  FindRange(sd->project3, sd->npoints, &zmin, &zmax);
  if (display_origin_mode)
  {
    OriginAdjust(&xmin, &xmax);
    OriginAdjust(&ymin, &ymax);
    OriginAdjust(&zmin, &zmax);
  }

  /* set background color then draw left and right eye point sets */
  xgcv.plane_mask = plane_mask;
  xgcv.fill_style = FillSolid;
  XChangeGC(display, gc_3d, GCPlaneMask | GCFillStyle, &xgcv);

  XSetForeground(display, gc_3d, bg_color);
  XFillRectangle(display, win, gc_3d, 0, 0, window_width+1, window_height+1);
  
  for (i = 0; i < sd->npoints; i++)
  {
    /*
      draw trajectories using appropriate scaling to fill the rectangle
      specified by drawx1, drawx2, drawy1 and drawy2.
    */
    x = (short) ROUND((sd->project1[i] - xmin) *
                     (drawx2 - drawx1) / (xmax - xmin)) + drawx1;
    y = (short) ROUND((sd->project2[i] - ymin) *
                     (drawy2 - drawy1) / (ymax - ymin)) + drawy1;
    zoffset = (short) ROUND(DepthScaling3D *
                           ((sd->project3[i] - zmin) / (zmax - zmin)));

    /* left and right eye projections */
    left_points[i].x = x - zoffset;
    left_points[i].y = y;
    right_points[i].x = x + zoffset;
    right_points[i].y = y;
  }

  XSetForeground(display, gc_3d, left_color);
  if (sd->ptset)
    DrawPointSet(display, win, gc_3d, left_points, sd->npoints);
  else
    XDrawLines(display, win, gc_3d, left_points, sd->npoints, CoordModeOrigin);
  if (label & !sd->ptset)
  {
    XDrawString(display, win, gc_3d, left_points[0].x, left_points[0].y,
                "Start", 5);
    XDrawString(display, win, gc_3d, left_points[sd->npoints - 1].x,
                left_points[sd->npoints - 1].y, "End", 3);
  }
  XSetForeground(display, gc_3d, right_color);
  if (sd->ptset)
    DrawPointSet(display, win, gc_3d, right_points, sd->npoints);
  else
    XDrawLines(display, win, gc_3d, right_points, sd->npoints, CoordModeOrigin);
  if (label & !sd->ptset)
  {
    XDrawString(display, win, gc_3d, right_points[0].x, right_points[0].y,
                "Start", 5);
    XDrawString(display, win, gc_3d, right_points[sd->npoints - 1].x,
                right_points[sd->npoints - 1].y, "End", 3);
  }

  /* draw remaining labels */
  if (label && display_label_mode)
  {
    XSetForeground(display, gc_3d, text_color);
    sd->label_box.x = (short) 10;
    sd->label_box.y = (short) (window_height - 10);
    length = ConstructFilenameString(sd, cbuf);
    XDrawString(display, win, gc_3d, sd->label_box.x, sd->label_box.y,
                cbuf, length);
  }
  
  /* print highlighted vertices */
  if (highlighted && sd->nverts)
  {
    for (i = 0; i < highlighted; i++)
    {
      index = sd->order[i];
      XSetForeground(display, gc_3d, left_color);
      XDrawArc(display, win, gc_3d, left_points[index].x - 4,
               left_points[index].y - 4, 8, 8, 0, DEG360);
      XSetForeground(display, gc_3d, right_color);
      XDrawArc(display, win, gc_3d, right_points[index].x - 4,
               right_points[index].y - 4, 8, 8, 0, DEG360);
    }
  }

  if (display_origin_mode)
  {
    x = (short) ROUND(xmin * (drawx1 - drawx2) / (xmax - xmin)) + drawx1;
    y = (short) ROUND(ymin * (drawy1 - drawy2) / (ymax - ymin)) + drawy1;
    zoffset = (short) ROUND(DepthScaling3D * (zmin / (zmin - zmax)));
    DrawOrigin3(x, y, zoffset, left_color, right_color, text_color);
  }
  
  free(right_points);
  free(left_points);

  /* Display the previous drawing commands */
  SwapBuffers();
  SwapDisplayBuffer();
}

/*****************************************************************************/

/*
  draw_3d_sweep: Slowly draw 3D projection of a single trajectory.
*/
void draw_3d_sweep(unsigned int window_width, unsigned int window_height,
                   int label, unsigned int sleep_period)
{
  unsigned long plane_mask;
  unsigned long text_color;                /* color index for drawing labels */
  unsigned long bg_color;              /* color index for drawing background */
  unsigned long left_color;                /* color index for left eye trace */
  unsigned long right_color;              /* color index for right eye trace */  
  int i, j, s, x, y;
  int width,height;
  int drawx1, drawx2, drawy1, drawy2;
  int left_x1, left_y1, left_x2, left_y2;
  int right_x1, right_y1, right_x2, right_y2;
  real xmin, xmax, ymin, ymax, zmin, zmax, proj;
  int index, zoffset, length;
  XGCValues xgcv;
  int *vertex;
  SpeechRecord *sd;
  char cbuf[1024];

  /* Obtain colors for the exposed buffer into which drawing may occur */
  GetExposedColors(&plane_mask, &text_color, &bg_color,
                   &left_color, &right_color);
  
  /* Reduce drawable area slightly to allow for offsets */
  drawx1 = X_MARGIN_3D;
  drawx2 = window_width - X_MARGIN_3D;
  drawy1 = Y_MARGIN_3D;
  drawy2 = window_height - Y_MARGIN_3D;
  
  /* Automatic scaling to ensure trajectory fills projection area */
  s = utterance_index;
  sd = spdata[s];

  FindRange(sd->project1, sd->npoints, &xmin, &xmax);
  FindRange(sd->project2, sd->npoints, &ymin, &ymax);
  FindRange(sd->project3, sd->npoints, &zmin, &zmax);
  if (display_origin_mode)
  {
    OriginAdjust(&xmin, &xmax);
    OriginAdjust(&ymin, &ymax);
    OriginAdjust(&zmin, &zmax);
  }

  /* find which vertices need to be highlighted */
  vertex = (int *) malloc(sd->npoints * sizeof(int));
  FindHighlightedVerts(sd->npoints,sd->order,highlighted,vertex,sd->nverts);
  
  /* set background color then draw left and right eye point sets */
  xgcv.plane_mask = plane_mask;
  xgcv.fill_style = FillSolid;
  XChangeGC(display, gc_3d,  GCPlaneMask | GCFillStyle, &xgcv);
  XChangeGC(display, gc1_3d, GCPlaneMask | GCFillStyle, &xgcv);

  XSetForeground(display, gc_3d, bg_color);
  XFillRectangle(display, win, gc_3d, 0, 0, window_width+1, window_height+1);

  /* Draw the origin if required */
  if (display_origin_mode)
  {
    x = (short) ROUND(xmin * (drawx1 - drawx2) / (xmax - xmin)) + drawx1;
    y = (short) ROUND(ymin * (drawy1 - drawy2) / (ymax - ymin)) + drawy1;
    zoffset = (short) ROUND(DepthScaling3D * (zmin / (zmin - zmax)));
    DrawOrigin3(x, y, zoffset, left_color, right_color, text_color);
  }

  /* create duplicate GC to avoid spurious XSetForeground calls */
  XSetForeground(display, gc_3d, left_color);
  XSetForeground(display, gc1_3d, right_color);
  
  right_x1 = 0;
  right_y1 = 0;
  left_x1 = 0;
  left_y1 = 0;

  StartTimer(sleep_period);
  for (i = 0; i < sd->npoints; i++)
  {
    /*
      draw trajectories using appropriate scaling to fill the rectangle
      specified by drawx1, drawx2, drawy1 and drawy2.
    */
    x = (short) ROUND((sd->project1[i] - xmin) *
                     (drawx2 - drawx1) / (xmax - xmin)) + drawx1;
    y = (short) ROUND((sd->project2[i] - ymin) *
                     (drawy2 - drawy1) / (ymax - ymin)) + drawy1;
    zoffset = (short) ROUND(DepthScaling3D *
                           ((sd->project3[i] - zmin) / (zmax - zmin)));

    /* left and right eye projections */
    left_x1 = left_x2;
    left_x2 = x - zoffset;
    left_y1 = left_y2;
    left_y2 = y;
    right_x1 = right_x2;
    right_x2 = x + zoffset;
    right_y1 = right_y2;
    right_y2 = y;

    if (i == 0)
    {
      if (sd->ptset)
      {
        XDrawString(display, win, gc_3d,  left_x2 - 3,  left_y2 + 4,  "x", 1);
        XDrawString(display, win, gc1_3d, right_x2 - 3, right_y2 + 4, "x", 1);
      }
      else
        if (label)
        {
          /* Draw "Start" labels */
          XDrawString(display, win, gc_3d,  left_x2,  left_y2,  "Start", 5);
          XDrawString(display, win, gc1_3d, right_x2, right_y2, "Start", 5);
        }
    }
    else
    {
      if (sd->ptset)
      {
        XDrawString(display, win, gc_3d,  left_x2 - 3,  left_y2 + 4,  "x", 1);
        XDrawString(display, win, gc1_3d, right_x2 - 3, right_y2 + 4, "x", 1);
      }
      else
      {
        /* Draw left and right lines */
        XDrawLine(display,win, gc_3d,  left_x1,  left_y1,  left_x2,  left_y2);
        XDrawLine(display,win, gc1_3d, right_x1, right_y1, right_x2, right_y2);
      }

      /* highlight vertex */
      if (vertex[i])
      {
        XDrawArc(display, win, gc_3d, left_x2-4,  left_y2-4,  8, 8, 0, DEG360);
        XDrawArc(display, win, gc1_3d,right_x2-4, right_y2-4, 8, 8, 0, DEG360);
      }      

      XFlush(display);        /* Ensures updates are displayed on the server */
    }
#ifdef LINUX
    (void) pause();        /* wait for timer to ellapse generating interrupt */
#else
    (void) sigpause(0);                                 /* ditto under SunOS */
#endif
  }
  StopTimer();
  
  if (label)
  {
    if (!sd->ptset)
    {
      /* Draw "End" labels */
      XDrawString(display, win, gc_3d,  left_x2,  left_y2,  "End", 3);
      XDrawString(display, win, gc1_3d, right_x2, right_y2, "End", 3);
    }

    if (display_label_mode)
    {
      /* Draw trace label */
      XSetForeground(display, gc_3d, text_color);
      sd->label_box.x = (short) 10;
      sd->label_box.y = (short) (window_height - 10);
      length = ConstructFilenameString(sd, cbuf);
      XDrawString(display,win, gc_3d, sd->label_box.x, sd->label_box.y,
                  cbuf, length);
    }
  }
  free(vertex);
}

/*****************************************************************************/

/*
  draw_2d_graphics: Draw 2D projection of multiple trajectories.
*/
void draw_2d_graphics(unsigned int window_width, unsigned int window_height,
                   int label)
{
  int i,s;
  int width,height;
  XPoint *points;
  real scale;
  real min, max, xmin, xmax, ymin, ymax, proj;
  int index, x, y, length;
  int drawx1, drawx2, drawy1, drawy2;
  XSetWindowAttributes attributes;
  SpeechRecord *sd;
  char cbuf[1024];

  /* set screen background to white */
  attributes.background_pixel = WhitePixel(display, screen_num);
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);

  /* Automatic scaling to ensure trajectory fills projection area */
  sd = spdata[0];
  FindRange(sd->project1, sd->npoints, &xmin, &xmax);
  FindRange(sd->project2, sd->npoints, &ymin, &ymax);
  for (s = 1; s < samples; s++)
  {
    sd = spdata[s];
    FindRange(sd->project1, sd->npoints, &min, &max);
    if (min < xmin)
      xmin = min;
    if (max > xmax)
      xmax = max;
    FindRange(sd->project2, sd->npoints, &min, &max);
    if (min < ymin)
      ymin = min;
    if (max > ymax)
      ymax = max;
  }

  /* Adjust the auto_scale range so that the origin is visable */
  if (display_origin_mode)
  {
    OriginAdjust(&xmin, &xmax);
    OriginAdjust(&ymin, &ymax);
  }
  
  if (auto_scale)
  {
    drawx1 = 20;
    drawx2 = window_width - 20;
    drawy1 = 20;
    drawy2 = window_height - 20;    
  }
  else
  {
    width = window_width;
    height = window_height;
    scale = exp(((double)size) * 0.1);    /* must convert argument to double */
  }

  XClearWindow(display, win);
  for (s = 0; s < samples; s++) {
    sd = spdata[s];
    points = (XPoint *) malloc(sd->npoints * sizeof(XPoint));
    
    for (i = 0; i < sd->npoints; i++) {
      if (auto_scale)
      {
        /*
          draw trajectories using appropriate scaling to fill the rectangle
          specified by drawx1, drawx2, drawy1 and drawy2.
        */
        points[i].x = (short) ROUND((sd->project1[i] - xmin) *
                                   (drawx2 - drawx1) / (xmax - xmin)) + drawx1;
        points[i].y = (short) ROUND((sd->project2[i] - ymin) *
                                   (drawy2 - drawy1) / (ymax - ymin)) + drawy1;
      }
      else
      {
        /*
          Absolute scaling using enlargement and left-right up-down buttons
        */
        points[i].x = (short) ROUND((sd->project1[i]-xmin)*scale) + left_right;
        points[i].y = (short) ROUND((sd->project2[i]-ymin)*scale) + up_down;
      }
    }
    XSetForeground(display, gc, traj_color[sd->color].pixel);
    if (sd->ptset)
      DrawPointSet(display, win, gc, points, sd->npoints);
    else
      XDrawLines(display, win, gc, points, sd->npoints, CoordModeOrigin);
    if (label)
    {
      if (!sd->ptset)
      {
        XDrawString(display, win, gc, points[0].x, points[0].y, "Start", 5);
        XDrawString(display, win, gc, points[sd->npoints - 1].x,
                    points[sd->npoints - 1].y, "End", 3);
      }
      if (display_label_mode)
      {
        sd->label_box.x = (short) 10;
        sd->label_box.y  = (short)
          window_height - ((samples - s) * LABEL_BOX_HEIGHT);
        length = ConstructFilenameString(sd, cbuf);
        XDrawString(display, win, gc, sd->label_box.x, sd->label_box.y,
                    cbuf, length);
      }
    }

    /* print highlighted vertices */
    if (highlighted && sd->nverts)
    {
      for (i = 0; i < highlighted; i++)
      {
        index = sd->order[i];
        XDrawArc(display, win, gc, points[index].x - 4, points[index].y - 4,
                 8, 8, 0, DEG360);
      }
    }
    free(points);
  }

  /* Draw the origin if appropriate */
  if (display_origin_mode)
  {
    if (auto_scale)
    {
      x = (int) ROUND(xmin * (drawx1 - drawx2) / (xmax - xmin)) + drawx1;
      y = (int) ROUND(ymin * (drawy1 - drawy2) / (ymax - ymin)) + drawy1;
    }
    else
    {
      x = (int) ROUND((-xmin * scale) + left_right);
      y = (int) ROUND((-ymin * scale) + up_down);
    }
    DrawOrigin2(x, y);
  }
}

/*****************************************************************************/

/*
  draw_2d_sweep: Slowly draw 2D projection of multiple traces.
*/
void draw_2d_sweep(unsigned int window_width, unsigned int window_height,
                int label, unsigned int sleep_period)
{
  int i, j, s;
  int width, height, base_line, length;
  int x, y, x1, y1, x2, y2;
  real scale;
  real min, max, xmin, xmax, ymin, ymax, proj;
  int points[MAX_SAMPLES], max_points;
  int drawx1, drawx2, drawy1, drawy2;
  int *vertex[MAX_SAMPLES];
  XSetWindowAttributes attributes;
  SpeechRecord *sd;
  char cbuf[1024];

  /* set screen background to white */
  attributes.background_pixel = WhitePixel(display, screen_num);
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);

  /* Automatic scaling to ensure trajectory fills projection area */
  sd = spdata[0];
  FindRange(sd->project1, sd->npoints, &xmin, &xmax);
  FindRange(sd->project2, sd->npoints, &ymin, &ymax);
  for (s = 1; s < samples; s++)
  {
    sd = spdata[s];
    FindRange(sd->project1, sd->npoints, &min, &max);
    if (min < xmin)
      xmin = min;
    if (max > xmax)
      xmax = max;
    FindRange(sd->project2, sd->npoints, &min, &max);
    if (min < ymin)
      ymin = min;
    if (max > ymax)
      ymax = max;
  }

  /* Adjust the auto_scale range so that the origin is visable */
  if (display_origin_mode)
  {
    OriginAdjust(&xmin, &xmax);
    OriginAdjust(&ymin, &ymax);
  }

  /* pre-compute a list of the vertices that need to be highlighted */
  max_points = 0;
  for (s = 0; s < samples; s++)
  {
    sd = spdata[s];
    points[s] = sd->npoints;
    vertex[s] = (int *) malloc(points[s] * sizeof(int));
    FindHighlightedVerts(points[s],sd->order,highlighted,vertex[s],sd->nverts);

    if (points[s] > max_points)
      max_points = points[s];
  }
  
  if (auto_scale)
  {
    drawx1 = 20;
    drawx2 = window_width - 20;
    drawy1 = 20;
    drawy2 = window_height - 20;    
  }
  else
  {
    width = window_width;
    height = window_height;
    scale = exp(((double)size) * 0.1);    /* must convert argument to double */
  }

  XClearWindow(display,win);

  /* Firstly draw the origin if appropriate */
  if (display_origin_mode)
  {
    if (auto_scale)
    {
      x = (int) ROUND(xmin * (drawx1 - drawx2) / (xmax - xmin)) + drawx1;
      y = (int) ROUND(ymin * (drawy1 - drawy2) / (ymax - ymin)) + drawy1;
    }
    else
    {
      x = (int) ROUND((-xmin * scale) + left_right);
      y = (int) ROUND((-ymin * scale) + up_down);
    }
    DrawOrigin2(x, y);
  }

  StartTimer(sleep_period);
  for (i = 0; i < max_points; i++)
  {
    for (s = 0; s < samples; s++)
    {
      sd = spdata[s];
      
      /* only consider samples that have more than i points */
      if (i < sd->npoints)
      {
        if (auto_scale)
        {
          /*
            draw trajectories using appropriate scaling to fill the rectangle
            specified by drawx1, drawx2, drawy1 and drawy2.
          */
          if (i > 0)
          {
            x1 = (short) ROUND((sd->project1[i-1] - xmin) *
                              (drawx2 - drawx1) / (xmax - xmin)) + drawx1;
            y1 = (short) ROUND((sd->project2[i-1] - ymin) *
                              (drawy2 - drawy1) / (ymax - ymin)) + drawy1;
          }
          x2 = (short) ROUND((sd->project1[i] - xmin) *
                            (drawx2 - drawx1) / (xmax - xmin)) + drawx1;
          y2 = (short) ROUND((sd->project2[i] - ymin) *
                            (drawy2 - drawy1) / (ymax - ymin)) + drawy1;
        }
        else
        {
          /*
            Absolute scaling using enlargement and left-right up-down buttons
          */
          if (i > 0)
          {
            x1 = (short) ROUND((sd->project1[i-1] - xmin) * scale) +left_right;
            y1 = (short) ROUND((sd->project2[i-1] - ymin) * scale) + up_down;
          }
          x2 = (short) ROUND((sd->project1[i] - xmin) * scale) + left_right;
          y2 = (short) ROUND((sd->project2[i] - ymin) * scale) + up_down;
        }
        
        XSetForeground(display, gc, traj_color[sd->color].pixel);
        if (i == 0)
        {
          if (sd->ptset)
            XDrawString(display, win, gc, x2 - 3, y2 + 4, "x", 1);
          else
            if (label)
              XDrawString(display, win, gc, x2, y2 ,"Start",5);
        }
        else
        {
          if (sd->ptset)
            XDrawString(display, win, gc, x2 - 3, y2 + 4, "x", 1);
          else
            XDrawLine(display, win, gc, x1, y1, x2, y2);
          if (vertex[s][i])
            XDrawArc(display, win, gc, x2 - 4, y2 - 4, 8, 8, 0, DEG360);
          
          /* on final vertex */
          if (label && !sd->ptset && i == (points[s] - 1))
            XDrawString(display, win, gc, x2, y2, "End", 3);
        }
      }
    }
    XFlush(display);      /* ensure changes have been written to the display */
#ifdef LINUX
   (void) pause();         /* wait for timer to ellapse generating interrupt */
#else
   (void) sigpause(0);                                  /* ditto under SunOS */
#endif
  }
  StopTimer();

  /* label each of the traces */
  if (label && display_label_mode)
  {
    base_line = window_height - (samples * LABEL_BOX_HEIGHT);
    for (s = 0; s < samples; s++)
    {
      sd = spdata[s];
      XSetForeground(display, gc, traj_color[sd->color].pixel);
      sd->label_box.x = (short) 10;
      sd->label_box.y = (short) base_line;
      length = ConstructFilenameString(sd, cbuf);
      XDrawString(display, win, gc, sd->label_box.x, sd->label_box.y,
                  cbuf, length);
      base_line += LABEL_BOX_HEIGHT;
    }
  }

  for (s = 0; s < samples; s++)
      free(vertex[s]);

  fflush(stdout);
}

/*****************************************************************************/

/*
  draw_spectrogram: Display multiple traj. as a colour coded spectrogram.
*/
void draw_spectrogram(unsigned int window_width, unsigned int window_height,
                      int label)
{
  const int spectrogram_height = 150;    /* height of spectrogram box (pels) */
  const int spectrogram_spacing = 40; /* vertical space between boxes (pels) */
  
  int i, t, j, s, w, x, y, index;
  int start_index, ngrams;
  int pix_height, pix_width;
  int y_base, base_line;
  char buf[100];
  SpeechRecord *sd;
  XSetWindowAttributes attributes;
  real d, max, min, range;
  int *vertex[MAX_SAMPLES];
  int actual_height, length;
  real *energy, max_energy;
  XPoint *pts;
  char cbuf[1024];

  /* set background color then draw left and right eye point sets */
  attributes.background_pixel = spec_color[0].pixel;
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);
  XClearWindow(display, win);

  start_index = utterance_index;
  ngrams = window_height / (spectrogram_height + spectrogram_spacing);

  if (!spectrogram_stretch_mode)
  {
    /* find a common pix_width across all the utterances being displayed */
    for (i = 0; i < ngrams; i++)
    {
      s = start_index + i;
      if (s == samples)
        break;
      w = window_width / spdata[s]->npoints;
      if (i == 0)
        pix_width = w;
      else
        if (w < pix_width)
          pix_width = w;
    }
  }

  /* pre-compute a list of the vertices that need to be highlighted */
  for (i = 0; i < ngrams; i++)
  {
    s = start_index + i;
    if (s == samples)
      break;

    sd = spdata[s];
    vertex[i] = (int *) malloc(sd->npoints * sizeof(int));
    FindHighlightedVerts(sd->npoints, sd->order, highlighted, vertex[i],
                         sd->nverts);
  }

  /* loop to num of spectrograms that can be drawn within the window height */
  y_base = 0;
  for (i = 0; i < ngrams; i++)
  {
    s = start_index + i;
    if (s == samples)
      break;
    sd = spdata[s];

    if (spectrogram_stretch_mode)
      pix_width = window_width / sd->npoints;

    /* find range of values associated with the utterance */
    max = min = sd->point[0][0];
    for (t = 0; t < sd->npoints; t++)
      for (j = 0; j < dimension; j++)
      {
        d = sd->point[j][t];
        if (d > max)
          max = d;
        else
          if (d < min)
            min = d;
      }

    pix_height = spectrogram_height / dimension;
    actual_height = pix_height * dimension;
    
    /* draw spectrogram */
    x = 0;
    range = (real) SPECTRUM_SIZE * 0.999;
    for (t = 0; t < sd->npoints; t++)
    {
      y = y_base;
      for (j = 0; j < dimension; j++)
      {
        index = (int)
              floor(range * (sd->point[dimension-1-j][t] - min) / (max - min));
        XSetForeground(display, gc, spec_color[index].pixel);
        XFillRectangle(display, win, gc, x, y, pix_width, pix_height);
        y += pix_height;
      }

      if (vertex[i][t])
      {
        XSetForeground(display, gc, spec_color[15].pixel);
        XDrawLine(display, win, gc, x, y_base, x, y_base + actual_height);
      }
      x += pix_width;
    }

    /* draw energy profile */
    if (spectrogram_energy_mode)
    {
      energy = (real*) malloc(sd->npoints * sizeof(real));
      for (t = 0; t < sd->npoints; t++)
      {
        energy[t] = 0.0;
        for (j = 0; j < dimension; j++)
          energy[t] += sd->point[j][t];
        if (t == 0)
          max_energy = energy[t];
        else
          if (energy[t] > max_energy)
            max_energy = energy[t];
      }
      pts = (XPoint*) malloc(sd->npoints * sizeof(XPoint));
      for (t = 0; t < sd->npoints; t++)
      {
        pts[t].y = (short) y_base +
                       ROUND((1.0 - (energy[t] / max_energy)) * actual_height);
        pts[t].x = (short) (pix_width * t);
      }

      /* draw the entire line using a single call */
      XSetForeground(display, gc, spec_color[15].pixel);
      XDrawLines(display, win, gc, pts, sd->npoints, CoordModeOrigin);

      free(pts);
      free(energy);
    }

    /* Draw trace label */
    XSetForeground(display, gc, spec_color[15].pixel);
    sd->label_box.x = 10;
    sd->label_box.y = y_base + spectrogram_height + 20;
    length = ConstructFilenameString(sd, cbuf);
    XDrawString(display, win, gc, sd->label_box.x, sd->label_box.y,
                cbuf, length);

    y_base += (spectrogram_height + spectrogram_spacing);
  }  

  for (i = 0; i < ngrams; i++)
      free(vertex[i]);
}

/*****************************************************************************/

/* end of projectors.c */
