/*
  title: eigensys.c (previously gh.c)
  purpose:  perform matrix orthogonalization using the givens-householder
    method.

  authors: Chris deSilva 26 May 1993 (Converted to ANSI C by Gareth Lee,
    22 Dec 1993).
  modified: 24-04-94

  Copyright (C) 1994 Chris deSilva and Gareth Lee.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.
    
  changes (by Gareth Lee):
  21-01-94:  Memory leakage cured in gh_diag() by the addition of a free(tv)
    statement.
  30-03-94: Extra functions added to allow the existing routines to be
    interfaced to fview.
  24-04-94: EigenWorkProc() made more robust by trapping floating point
    exceptions thereby maintaining program control.
  24-04-94: raise(sig) function defined in terms of kill(getpid(),sig) when
    used with SunOS since it does not support raise() as a library function.
*/
#include <stdio.h>
#include <math.h>

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <unistd.h>
#include <assert.h>
#include <setjmp.h>
#include <signal.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>
#include <X11/Xatom.h>
#include <X11/keysym.h>

/* XView headers */
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/notice.h>
#include <xview/cursor.h>

/* default floating point type */
typedef REAL real;

/* Header files associated with other compilation units */
#include "projectors.h"
#include "fview.h"
#include "interface.h"
#include "callbacks.h"
#include "entropy.h"
#include "colours.h"
#include "ctable.h"
#include "eigensys.h"

#define        EPSILON        1.0e-9    /*accuracy of bisection*/

#ifdef SUNOS
/* SunOS does not provide raise() as a library function */
#define raise(s) kill(getpid(),s)
#endif

/*****************************************************************************/

int      EigenDone = 0;     /* indicates an eigensystem has been established */
int      EigenDim;             /* the dimension of the eigenvector datatypes */
double  *EigenRoots;              /* vector of root eigenvalues for the data */
double **EigenVectors;                             /* matrix of eigenvectors */
jmp_buf  EigenJump;                         /* for use in exception handling */

/*****************************************************************************/

/*
gh_diag
  diagonalizes a real symmetric matrix using the givens-householder method
  the input matrix is converted to tridiagonal form
  computes eigenvalues and matrix of eigenvectors, the eigenvectors are the
    rows of the matrix
  returns 0 if successful, 1 otherwise
*/
int gh_diag(double **mat, int sz, double **otm, double eval[], double **evecs)
/* mat = input matrix */
/* sz = size of matrix */
/* otm = orthogonal transformation matrix */
/* eval = eigenvalues */
/* evecs = matrix of eigenvectors */
{
  double *tv;                           /*temporary storage*/
  int  i,j,k;                         /*loop indices*/

  tv=(double *)malloc((unsigned)(sz*sizeof(double)));
  assert(tv!=NULL);
  if(tridiag(mat,sz,otm)!=0) return(1);
  if(cevtdm(mat,sz,eval,sz)!=0) return(1);
  for(i=0;i<sz;i++)
  {
    compevec(mat,sz,eval[i],tv);
    for(j=0;j<sz;j++)
    {
      evecs[i][j]=0.0;
      for(k=0;k<sz;k++) evecs[i][j]=evecs[i][j]+otm[j][k]*tv[k];
    }
  }
  free(tv);                /* added by gel (21-01-94) to cure memory leakage */
  return(0);
}

/*****************************************************************************/

/*
tridiag
  reduces a real symmetric matrix to tridiagonal form using householder
    reduction
  returns 0 if successful, 1 if malloc fails
*/
tridiag(double **mat, int sz, double **otm)
/* mat = input/output matrix */
/* sz = size of matrix */
/* otm = orthogonal transformation matrix */
{
  int  i,j,l,m;                         /*loop indices*/
  double sigma,h,k;                     /*intermediate values*/
  double *u,*p,*q;                      /*intermediate vectors*/

  /*allocate memory*/
  u=(double *)malloc((unsigned)(sz*sizeof(double)));
  p=(double *)malloc((unsigned)(sz*sizeof(double)));
  q=(double *)malloc((unsigned)(sz*sizeof(double)));
  assert(u != NULL && p != NULL && q != NULL);

  /*initialize transformation matrix*/
  for(i=0;i<sz;i++)
  {
    for(j=0;j<sz;j++) otm[i][j]=0.0;
    otm[i][i]=1.0;
  }
  /*apply householder reduction*/
  for(m=1;m<(sz-1);m++)
  {
    i=sz-m;
    sigma=mat[i][i-1]*mat[i][i-1];
    for(j=0;j<(i-1);j++)
    {
      sigma=sigma+mat[i][j]*mat[i][j];
      u[j]=mat[i][j];
    }
    if(mat[i][i-1]<0.0) u[i-1]=mat[i][i-1]-sqrt(sigma);
    else u[i-1]=mat[i][i-1]+sqrt(sigma);
    for(j=i;j<sz;j++) u[j]=0.0;
    h=0.0;
    for(j=0;j<sz;j++) h=h+u[j]*u[j];
    h=0.5*h;
    for(j=0;j<sz;j++)
    {
      p[j]=0.0;
      for(l=0;l<sz;l++) p[j]=p[j]+mat[j][l]*u[l];
      p[j]=p[j]/h;
    }
    k=0.0;
    for(j=0;j<sz;j++) k=k+u[j]*p[j];
    k=0.5*k/h;
    for(j=0;j<sz;j++) q[j]=p[j]-k*u[j];
    for(j=0;j<sz;j++)
    {
      for(l=0;l<sz;l++) mat[j][l]=mat[j][l]-q[j]*u[l]-u[j]*q[l];
    }
    /*compute new transformation matrix*/
    for(i=0;i<sz;i++)
    {
      p[i]=0.0;
      for(j=0;j<sz;j++) p[i]=p[i]+otm[i][j]*u[j];
      p[i]=p[i]/h;
    }
    for(i=0;i<sz;i++)
    {
      for(j=0;j<sz;j++) otm[i][j]=otm[i][j]-p[i]*u[j];
    }
  }
  /*release memory*/
  free ((char *)u);
  free ((char *)p);
  free ((char *)q);
  return(0);
}

/*****************************************************************************/

/*
cevtdm
  computes the eigenvalues of a real symmetric tridiagonal matrix,
    using givens' method
  returns 0 if successful, 1 if malloc fails
*/
cevtdm(double **tdm, int sz,double ev[], int ne)
/* tdm = tridiagonal matrix */
/* sz = size of matrix */
/* ev = array of eigenvalues */
/* ne = number of eigenvalues */
{
  int i;                                /*loop index*/
  double *a,*b;                         /*matrix coefficients*/
  double p,q;                           /*test points*/
  double sum;                           /*sum of matrix coefficients*/
  double dpq;                           /*distance between p and q*/
  double *ssv;                          /*values of sturm sequence*/
  int  scl,scr;                         /*sign changes left and right*/
  double m;                             /*midpoint*/
  int  scm;                             /*sign changes middle*/
  int  loops;                           /*termination count*/

  /*allocate memory*/
  a=(double *)malloc((unsigned)((sz+1)*sizeof(double)));
  b=(double *)malloc((unsigned)((sz+2)*sizeof(double)));
  ssv=(double *)malloc((unsigned)((sz+1)*sizeof(double)));
  assert(a != NULL && b != NULL && ssv != NULL);

  /*copy values of matrix coefficients*/
  a[0]=0.0;
  b[0]=0.0;
  b[1]=0.0;
  for(i=0;i<sz;i++) a[i+1]=tdm[i][i];
  for(i=1;i<sz;i++) b[i+1]=tdm[i][i-1];
  b[sz+1]=0.0;
  /*set starting point*/
  q=0.0;
  for(i=1;i<=sz;i++)
  {
    sum=fabs(a[i])+fabs(b[i])+fabs(b[i+1]);
    if(q<sum) q=sum;
  }
  for(i=0;i<ne;i++)
  {
    /*locate bounds for eigenvalue*/
    scr=scc(a,b,sz,q,ssv);
    if((scr+i)!=sz)
    {
      fprintf(stderr, "Error: eigensystem: Right endpoint is wrong !\n");
      raise(SIGUSR1);                     /* abort via the exception handler */
    }
    loops = 0;
    for(dpq=1.0;;)
    {
      p = q - dpq;
      scl = scc(a,b,sz,p,ssv);
      if(scl == scr)
      {
        q = p;
        dpq = 1.0;
      }
      else
        if(scl == (scr-1))
          break;
        else
          if(scl < (scr-1))
            dpq /= 2.0;
          else
          {
            fprintf(stderr,"Error: eigensystem: Bound determination failed\n");
            raise(SIGUSR1);                   /* abort via exception handler */
          }

      /* terminate with error if looping indefinitely */
      if (loops == 100000)
        raise(SIGUSR1);
      loops++;
    }
    /*do bisection*/
    chp(a,b,sz,p,ssv);
    if(ssv[sz]==0.0)
    {
      q=p;
    }
    else
    {
      for(;;)
      {
        m = 0.5 * (p+q);
        chp(a,b,sz,m,ssv);
        if(ssv[sz]==0.0)
        {
          q = m;
          break;
        }
        scm = scc(a,b,sz,m,ssv);
        if(scm == scr)
          q = m;
        else
        {
          if(scm == scl)
            p = m;
          else
          {
            fprintf(stderr, "Error: eigensystem: Bisection has failed.\n");
            raise(SIGUSR1);                   /* abort via exception handler */
          }
        }
        if((q - p) < EPSILON)
        {
          q = p;
          break;
        }
      }
    }
    ev[i]=q;
  }
  /*release memory*/
  free((char *)a);
  free((char *)b);
  free((char *)ssv);
  return(0);
}

/*****************************************************************************/

/*
chp
  computes the value of the characteristic polynomial of a real symmetric
    tridiagonal matrix
  returns 0
*/
chp(double *a, double *b, int sz,double p, double *v)
/* a,b = matrix coefficients */
/* sz = size of matrix */
/* p = point of evaluation */
/* v = values of sturm sequence functions */
{
  int  i;                               /*loop index*/

  v[0]=1.0;
  v[1]=a[1]-p;
  for(i=2;i<=sz;i++)
    v[i]=(a[i]-p)*v[i-1]-b[i]*b[i]*v[i-2];
  return(0);
}

/*****************************************************************************/

/*
scc
  counts the number of sign changes in the sturm sequence
  returns the number of sign changes 
*/
scc(double *a, double *b, int sz,double p, double *v)
/* a, b = matrix coefficients */
/* sz = size of matrix */
/* p = point of evaluation */
/* v = values of sturm sequence functions */
{
  int  i;                              /*loop index*/
  int  count;                          /*count of sign changes*/

  chp(a,b,sz,p,v);
  count=0;
  for(i=0;i<sz;i++)
  {
    if((v[i]*v[i+1])<0.0) count++;
  }
  return(count);
}

/*****************************************************************************/

/*
compevec
  computes the eigenvector of a real symmetric tridiagonal matrix corresponding
    to a given eigenvalue
  returns 0
*/
compevec(double **tdm, int sz, double ev, double vector[])
/* tdm = tridiagonal matrix */
/* sz = size of matrix */
/* ev = eigenvalue */
/* vector = eigenvector */
{
  int i;                                /*loop index*/
  double norm;                          /*norm of vector*/

  vector[0]=1.0;
  if(tdm[0][1]==0.0) vector[1]=1.0;
  else vector[1]=(ev-tdm[0][0])/tdm[0][1];
  norm=1+vector[1]*vector[1];
  for(i=2;i<sz;i++)
  {
    if(tdm[i-1][i]==0.0) vector[i]=1.0;
    else
    {
      vector[i]=(ev-tdm[i-1][i-1])*vector[i-1]-tdm[i-1][i-2]*vector[i-2];
      vector[i]=vector[i]/tdm[i-1][i];
    }
    norm=norm+vector[i]*vector[i];
  }
  norm=sqrt(norm);
  for(i=0;i<sz;i++) vector[i]=vector[i]/norm;
  return(0);
}

/*****************************************************************************/

/*
  compar: qsort comparison routine.
*/
int compar(const void *a, const void *b)
{
  double x, y;
  x = EigenRoots[*((int*) a)];
  y = EigenRoots[*((int*) b)];
  if (x < y)
    return 1;
  else
  {
    if (x == y)
      return 0;
    else
      if (x > y)
        return -1;
  }
}

/*****************************************************************************/

/*
  ResetEigenWidgets: reset to array of gauge widgets to their initial (zero)
    state.
*/
void ResetEigenWidgets(void)
{
  int i;

  /* scale all values relative the the maximum value */
  for (i = 0; i < dimension; i++)
    xv_set(SpecifyEgnGauge[i], PANEL_VALUE, 0, NULL);

  /* display error message */
  xv_set(SpecifyErrorMessage,
         XV_SHOW, TRUE,
         PANEL_LABEL_STRING, "Error: analysis failed.\n",
         NULL);
  XBell(display, 0);
}

/*****************************************************************************/

/*
  SetupEigenWidgets: set the array of guage widgets displayed within the
    Specify menu to reflect the relative root eigen values resulting from
    the analysis.
*/
void SetupEigenWidgets(void)
{
  int i, value;
  double max, ratio;

  /* find the maximum value */
  max = EigenRoots[0];
  for (i = 1; i < dimension; i++)
    if (EigenRoots[i] > max)
      max = EigenRoots[i];

  /* scale all values relative the the maximum value */
  for (i = 0; i < dimension; i++)
  {
    ratio = EigenRoots[i] / max;
    value = (int) (ratio * 99.0);
    xv_set(SpecifyEgnGauge[i], PANEL_VALUE, value, NULL);
  }

  /* display error message */
  xv_set(SpecifyErrorMessage,
         XV_SHOW, TRUE,
         PANEL_LABEL_STRING, "Analysis suceeded.\n",
         NULL);
}

/*****************************************************************************/

/*
  EigenHandler: exception handler for use within the eigensystem module.
*/
void EigenHandler(int x)
{
  longjmp(EigenJump, SIGFPE);   /* return indicating type of exception */
}

/*****************************************************************************/

/*
  EigenWorkProc: called via the SpecifyAnalyseButton to initiate eigenvector
    analysis on the dataset which is currently being displayed.
*/
void EigenWorkProc(void)
{
  const double eigen_value_threshold = -1.0e-8;

  int s, t, i, j, ret;
  SpeechRecord *sd;
  long count;
  int *index;
  double pti;
  double *sum, *eigenvals;
  double **sum2, **covar, **scratch, **eigenvectors;

  /* Ensure than FP/other exceptions can be recovered from */
  if (setjmp(EigenJump))
  {
    fprintf(stderr, "Floating point exception occurred: eigensystem failed\n");
    fflush(stderr);
    (void) signal(SIGFPE, SIG_DFL);           /* return to default behaviour */
    (void) signal(SIGUSR1, SIG_DFL);
    EigenDone = 0;
    ResetEigenWidgets();
    return;                        /* an FP exception has occurred so return */
  }
  else
  {
    (void) signal(SIGFPE, EigenHandler);      /* register exception handlers */
    (void) signal(SIGUSR1, EigenHandler);
  }
  xv_set(SpecifyErrorMessage, XV_SHOW, FALSE, NULL); /* hide previous errors */

  /* create matrices */
  sum = (double *) malloc(dimension * sizeof(double));
  sum2 = (double **) malloc(dimension * sizeof(double*));
  covar = (double **) malloc(dimension * sizeof(double*));
  for (i = 0; i < dimension; i++)
  {
    sum[i] = 0.0;
    sum2[i] = (double *) malloc(dimension * sizeof(double));
    for (j = 0; j < dimension; j++)
      sum2[i][j] = 0.0;
    covar[i] = (double *) malloc(dimension * sizeof(double));
  }
  
  /* find sum and sum2 matrices */
  count = 0;
  for (s = 0; s < samples; s++)
  {
    sd = spdata[s];
    for (t = 0; t < sd->npoints; t++)
      for (i = 0; i < dimension; i++)
      {
        /* note: point indexed in a perverse fashion */
        pti = (double) sd->point[i][t];
        sum[i] += pti;
        for (j = 0; j < dimension; j++)
          sum2[i][j] += pti * (double) sd->point[j][t];
      }
    count += sd->npoints;
    notify_dispatch();
  }

  /* derive covariance matrix */
  for (i = 0; i < dimension; i++)
  {
    sum[i] /= (double) count;
    for (j = 0; j < dimension; j++)
      sum2[i][j] /= (double) count;
  }
  for (i = 0; i < dimension; i++)
    for (j = 0; j < dimension; j++)
      covar[i][j] = sum2[i][j] - (sum[i] * sum[j]);

  /* solve for eigensystem */
  if (EigenRoots != NULL)
    free(EigenRoots);
  EigenRoots = (double *) malloc(dimension * sizeof(double));
  scratch = (double **) malloc(dimension * sizeof(double*));
  for (i = 0; i < dimension; i++)
    scratch[i] = (double *) malloc(dimension * sizeof(double));
  if (EigenVectors != NULL)
    for (i = 0; i < EigenDim; i++)
      free(EigenVectors[i]);
  EigenVectors = (double **) malloc(dimension * sizeof(double*));
  for (i = 0; i < dimension; i++)
    EigenVectors[i] = (double *) malloc(dimension * sizeof(double));
  EigenDim = dimension;
  ret = gh_diag(covar, dimension, scratch, EigenRoots, EigenVectors);
  if (ret == 1)
    raise(SIGUSR1);                       /* abort via the exception handler */
  else
    EigenDone = 1;
  printf("Eigenvalues:\n");
  for (i = 0; i < dimension; i++)
  {
    printf("%d: %e\n", i, EigenRoots[i]);   /* eigenval since not rooted yet */
    if (EigenRoots[i] < 0.0)
    {
      /*
        Assume very small -ve eigenvalues result from insufficient numerical
        precision and so round up to 0.0.  Large -ve eigenvals give rise to
        an exception.
      */
      if (EigenRoots[i] > eigen_value_threshold)
        EigenRoots[i] = 0.0;
      else
        raise(SIGUSR1);
    }
    EigenRoots[i] = sqrt(EigenRoots[i]);
  }
  fflush(stdout);

  /* sort eigenvectors into order of descending eigenroots */
  index = (int*) malloc(dimension * sizeof(int));
  for (i = 0; i < dimension; i++)
    index[i] = i;
  qsort(index, dimension, sizeof(int), compar);
  eigenvals = EigenRoots;
  EigenRoots = (double*) malloc(dimension * sizeof(double));
  eigenvectors = EigenVectors;
  EigenVectors = (double**) malloc(dimension * sizeof(double*));
  for (i = 0; i < dimension; i++)
  {
    EigenRoots[i] = eigenvals[index[i]];
    EigenVectors[i] = eigenvectors[index[i]];   /* index data using new ptrs */
  }
  free(index);

  SetupEigenWidgets();
  
  free(eigenvals);
  free(eigenvectors);                /* only free the pointers, not the data */
  for (i = 0; i < dimension; i++)
  {
    free(scratch[i]);
    free(covar[i]);
    free(sum2[i]);
  }
  free(sum);

  (void) signal(SIGFPE, SIG_DFL);             /* return to default behaviour */
  (void) signal(SIGUSR1, SIG_DFL);
}

/*****************************************************************************/

void SelectEigenVectors(int a, int b, int c)
{
  int i;

  assert(a>=0 && a<dimension && b>=0 && b<dimension && c>=0 || c<dimension);
  for (i = 0; i < dimension; i++)     /* copy eigenvectors across into bases */
  {
    w1[i] = (real) EigenVectors[a][i];
    w2[i] = (real) EigenVectors[b][i];
    w3[i] = (real) EigenVectors[c][i];
  }
}

/*****************************************************************************/

/* end of eigensys.c */
