/*
  title:    timit.c
  purpose:  Converters TIMIT format sample files into a format suitable for
    use with the fview package.

  author:   Gareth Lee
  date:     02-11-93
  modified: 04-04-93

  Copyright (C) 1994 Gareth Lee.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  02-11-93: Program (loosely) based on sfconverter.c
  04-04-94: Bug fix to ensure diagnostic messages are produced.
*/
#define VERSION "1.2"

#include <stdio.h>
#include <stdlib.h>
#include <assert.h>
#include <math.h>

#ifdef LINUX
#include <endian.h>
#endif
#ifdef SUNOS
#include "../endian.h"
#endif
#include "../fviewio.h"
#include "swappers.h"

/*****************************************************************************/

/* TIMIT standard header - stored in VAX record format - byte swapping */
/* therefore necessary for Sun machines */

typedef struct {
  short header_size;                           /* in VAX word (2-byte) units */
  short version;                                           /* version number */
  short channels;           /* number of channels on which sampling occurred */
  short rate;                                 /* sample rate (units unknown) */
  int   samples;                            /* number of samples in the file */
} TIMIThdr;

#define MAX_HDR_SIZE (10240)
#define VERSION_UNSET (-9999)

typedef short sample_t;          /* 16-bit data type to use for read samples */

/*****************************************************************************/

/* Globals */
char               filename[1024];

/* Header records */
TIMIThdr thdr;
FviewSampleHeader fsh;
char header[MAX_HDR_SIZE];
#if (BYTE_ORDER == LITTLE_ENDIAN)
  int byte_swap = 0;
#elif (BYTE_ORDER == BIG_ENDIAN)
  int byte_swap = 1;
#else
#error byte order undefined/unknown
#endif
 
/* for use with getopt */
extern char *optarg;
extern int optind, opterr;

/*****************************************************************************/

void main(int argc, char *argv[])
{
  int index, opt;
  FILE *fd;
  sample_t sample, *sample_buffer;
  int version = VERSION_UNSET;
  TIMIThdr swapped;

  assert(sizeof(sample_t) == 2); /* ensure types have the correct dimensions */
  assert(sizeof(int) == 4);
  
  /* Parse command line */
  while ((opt = getopt(argc, argv, "B:f:v:")) != -1)
    switch(opt) {

    case 'B':
      if (*optarg == 's' || *optarg == 'S')
        byte_swap = !byte_swap;
      break;

    case 'f':
      strcpy(filename, optarg);
      break;

    case 'v':
      version = atoi(optarg);
      break;
    }
  
  if ((fd = fopen(filename, "rb")) == NULL) {
    fprintf(stderr, "timit: error opening file %s for reading\n", filename);
    exit(-1);
  }

  /* read in header information and reverse the byte order */
  if (fread(&thdr, sizeof(TIMIThdr), 1, fd) != 1)
  {
    fprintf(stderr, "timit: error reading TIMIT header\n");
    exit(-1);
  }
  if (byte_swap)
  {
    SWAP2(&thdr.header_size);
    SWAP2(&thdr.version);
    SWAP2(&thdr.channels);
    SWAP2(&thdr.rate);
    SWAP4(&thdr.samples);
  }

  /* Check the TIMIT Header record (in 2-byte units) */
  if (thdr.header_size != 6)
  {
    fprintf(stderr, "timit: header size incorrect\n");
    exit(-1);
  }
  if (version != VERSION_UNSET && thdr.version != version)
  {
    fprintf(stderr, "timit: version field (%d) differs from that specified\n",
            thdr.version);
    exit(-1);
  }
  
  /* Create an Fview sample header record */
  memcpy(fsh.magic, FVIEW_SAMPLE_MAGIC, 8);
  fsh.number_samples = (unsigned) thdr.samples;
  fsh.sample_freq = thdr.rate;
  
  /* Create the textual header to go with the file */
  sprintf(header, "TIMIT\nversion=%d\nchannels=%d\nrate=%d\nsamples=%d\n",
          thdr.version, thdr.channels, thdr.rate, thdr.samples);

  /* Write out Fview headers to stdout */
  fsh.info_size = strlen(header) + 1;
  
  if (fwrite(&fsh, sizeof(FviewSampleHeader), 1, stdout) != 1)
  {
    fprintf(stderr, "timit: failed to write FviewSampleHeader\n");
    exit(-1);
  }
  fwrite(header, sizeof(char), fsh.info_size, stdout);
  
  /* copy 16 bit twos compliment samples across and byte-swap samples */
  sample_buffer = (sample_t*) malloc(thdr.samples * sizeof(sample_t));
  if (fread(sample_buffer, sizeof(sample_t), thdr.samples, fd) != thdr.samples)
  {
    fprintf(stderr, "timit: failed to read sample data\n");
    exit(-1);
  }
  if (byte_swap)
    for (index = 0; index < thdr.samples; index++)
      SWAP2(&sample_buffer[index]);

  if (fwrite(sample_buffer, sizeof(sample_t), thdr.samples, stdout) !=
      thdr.samples)
  {
    fprintf(stderr, "timit: failed to write sample data\n");
    exit(-1);
  }
  free(sample_buffer);
  
  fclose(fd);
  
  /* print out some usefile information */
  fprintf(stderr, "timit (%s):file %s, frames %d\n",
          VERSION, filename, thdr.samples);
  
  fflush(stderr);
}

/* end of timit.c */
