/*
  title:    interface.c
  purpose:  xview widget bindings for the fview program.

  author:   Gareth Lee and Hugh Sanderson.
  date:     15-07-93
  modified: 30-03-94

  Copyright (C) 1994 Gareth Lee and Hugh Sanderson.
     
  This program is free software; you can redistribute it and/or
  modify it under the terms of the GNU General Public License
  as published by the Free Software Foundation; either version 2
  of the License, or (at your option) any later version.
  
  This program is distributed in the hope that it will be useful,
  but WITHOUT ANY WARRANTY; without even the implied warranty of
  MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
  GNU General Public License for more details.
  
  You should have received a copy of the GNU General Public License
  along with this program; if not, write to the Free Software
  Foundation, Inc., 675 Mass Ave, Cambridge, MA 02139, USA.

  changes:
  11-08-93: CreateDisplayWindow() mod'd to ensure that the window background is
    always set to white.  This ensures that the trajectories drawn are always
    visible no matter what default screen background colour the user has set.
  10-02-94: include of dbuffer.h added.
  29-03-94: `Point set' button added to the main control menu for globally
    controlling point set display option.
  30-03-94: Eigensystem mode added to the Specify menu to allow a set of
    basis vectors be analysed from the data.  The associated root eigenvalues
    are also displayed.
*/

/* Standard headers */
#include <stdio.h>
#include <stdlib.h>
#include <math.h> 
#include <string.h>
#include <unistd.h>

/* X lib headers */
#include <X11/Xlib.h>
#include <X11/Xutil.h>
#include <X11/Xos.h>

/* XView headers */
#include <xview/xview.h>
#include <xview/panel.h>
#include <xview/cursor.h>
#include <xview/cms.h>
#include <xview/svrimage.h>

typedef REAL real;

/* Header files associated with other compilation units */
#include "projectors.h"
#include "callbacks.h"
#include "fview.h"
#include "interface.h"
#include "colours.h"
#include "ctable.h"
#include "dbuffer.h"

/* Icon Bitmaps */
#include "bitmaps.xbm"

/* `Parent Object' for orphans */
#define NO_PARENT ((Xv_opaque) 0)

/* Widget handles for objects */
Xv_opaque MainFrame; 
Xv_opaque MainPanel; 
Xv_opaque SampleFrame; 
Xv_opaque SamplePanel;
Xv_opaque SpecifyFrame; 
Xv_opaque SpecifyPanel; 
Xv_opaque RotationSelector1;
Xv_opaque RotationSelector2;
Xv_opaque DisplayFrame;
Xv_opaque DisplayPanel;
Xv_opaque DisplayCanvas;
Xv_opaque DisplaySlider;
Xv_opaque DisplayRandomizeButton;
Xv_opaque DisplayStoreButton;
Xv_opaque DisplayRecallButton;
Xv_opaque DisplayColorsButton;
Xv_opaque DisplayCheckBox;
Xv_opaque DisplayDecrButton;
Xv_opaque DisplayIncrButton;
Xv_opaque ControlAutoScaleButton;
Xv_opaque UpDownSlider;
Xv_opaque LeftRightSlider;
Xv_opaque EnlargeShrinkSlider;
Xv_opaque ControlAnimateButton;
Xv_opaque AngleSlider;
Xv_opaque ControlSweepButton;
Xv_opaque SweepRateSlider;
Xv_opaque IterationSlider;
Xv_opaque ControlSpecifyButton;
Xv_opaque SpecifyModeStack;
Xv_opaque SpecifyErrorMessage;
Xv_opaque SpecifySelector1;
Xv_opaque SpecifyRandEq1Choice;
Xv_opaque SpecifySelector2;
Xv_opaque SpecifyRandEq2Choice;
Xv_opaque SpecifySelector3;
Xv_opaque SpecifyRandEq3Choice;
Xv_opaque SpecifyEgnSelector;
Xv_opaque SpecifyEgnGauge[MAX_DIMENSION];
Xv_opaque SpecifyAnalyseButton;
Xv_opaque SpecifyProjectButton;
Xv_opaque ControlSamplesButton;
Xv_opaque SampleListnameText;
Xv_opaque SampleDimensionText;
Xv_opaque SampleNumSampsText;
Xv_opaque SampleFileList;
Xv_opaque SampleFormatList;
Xv_opaque SampleFrontendList;
Xv_opaque SampleUpdateButton;
Xv_opaque SampleParamsButton;
Xv_opaque SampleCheckBox;
Xv_opaque ParamsFrame;
Xv_opaque ParamsPanel;
Xv_opaque ParamsFormatNameText;
Xv_opaque ParamsFormatFilterText;
Xv_opaque ParamsFrontendNameText;
Xv_opaque ParamsFrontendFilterText;
Xv_opaque ParamsFormatList;
Xv_opaque ParamsFrontendList;
Xv_opaque ParamsFormatValueText;
Xv_opaque ParamsFrontendValueText;
Xv_opaque ControlHighlightButton;
Xv_opaque ControlPointSetButton;
Xv_opaque HighlightFrame;
Xv_opaque HighlightPanel;
Xv_opaque HighlightSlider;
Xv_opaque HighlightSelector;
Xv_opaque HighlightCheckBox;

Xv_Cursor BasicCursor;
Xv_Cursor BusyCursor;
Server_image ciips_image;
Server_image project_image;
Server_image SampleWin1Glyph;
Server_image SampleWin2Glyph;

Xv_opaque ciips_logo;       /* CIIPS image for use within the control window */
Icon project_icon;   /* Project icon for association with each of the frames */

/* Display and context variables */
int screen_num = 0;                                /* works on most machines */
unsigned int screen_width, screen_height;        /* size of display (pixels) */
Colormap DefColormap;                /* Default colour map (used externally) */
XFontStruct *FontInfo;                         /* Font information structure */
char *CName[40];         /* Must Leave this set to 40, cf. set_color_names() */
XColor traj_color[MAX_COLORS];       /* Colours for use drawing trajectories */
XColor spec_color[SPECTRUM_SIZE];    /* Colours for use drawing spectrograms */
XColor dim3_defs[4];              /* Colours for use drawing 3D trajectories */
XColor dim3_color[16];       /* Cols to use drawing double buffered 3D trajs */
int RecallAllowed = FALSE; /* Indicates the `Recall' button should be active */

/* Color set to use for drawing trajectories */
const char* TrajectoryColors[MAX_COLORS] =
{
  "black", "blue", "red", "green", "maroon",
  "dark slate grey", "khaki", "magenta", "NavyBlue", "goldenrod",
  "pink", "purple", "thistle", "sienna", "aquamarine",
  "orchid", "violet", "tomato", "OliveDrab", "orange"
};

/*****************************************************************************/

/*
  SetCursors: Sets the Cursor type as specified in all frames and canvases.
*/
void SetCursors(Xv_Cursor cursor)
{
  Xv_Window win;

  win = (Xv_Window) xv_get(MainFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(SampleFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(DisplayFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  xv_set(canvas_paint_window(DisplayCanvas), WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(ParamsFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(HighlightFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);
  win = (Xv_Window) xv_get(SpecifyFrame, FRAME_NTH_SUBWINDOW, 1, NULL);
  xv_set(win, WIN_CURSOR, cursor, NULL);

  XFlush(display);
}

/*****************************************************************************/

/* Colors to use for drawing spectra */
const char* SpectrumColors[SPECTRUM_SIZE] =
{
  "black", "midnight blue", "dark slate blue", "blue violet",
  "medium slate blue", "LightSlateBlue", "turquoise3", "turquoise",
  "aquamarine", "green yellow", "yellow2",
  "gold",
  "orange",
  "orange red",
  "red",
  "white"
};

/*****************************************************************************/

void getGC(Window win, GC *gc, XFontStruct *font_info)
{
  static char dash_list[] = {12, 24};

  unsigned long valuemask = 0; /* ignore XGCvalues and use defaults */
  XGCValues values;
  unsigned int line_width = 1;
  int line_style = LineSolid;
  int cap_style = CapRound;
  int join_style = JoinRound;
  int dash_offset = 0;
  int list_length = 2;
 
  /* Create default Graphics Context */
  *gc = XCreateGC(display, win, valuemask, &values);
 
  /* specify font */
  XSetFont(display, *gc, font_info->fid);

  /* specify black foreground since default window background is
   * white and default foreground is undefined. */
  XSetForeground(display, *gc, BlackPixel(display, screen_num));

  /* set line attributes */
  XSetLineAttributes(display, *gc, line_width, line_style,
                     cap_style, join_style);

  /* set dashes */
  XSetDashes(display, *gc, dash_offset, dash_list, list_length);
} /* end of getGC */
 
/*****************************************************************************/

void load_font(XFontStruct **font_info)
{
  char *fontname = DEFAULT_FONT_NAME;

  /* Load font and get font information structure. */
  if ((*font_info = XLoadQueryFont(display, fontname)) == NULL)
  {
    printf("Error: Cannot open the font '%s'\n", fontname);
    Quit(-1);
  }
} /* end of load_font */

/*****************************************************************************/

/*
  get_colors: Allocate the colours that will be needed from the (shared) colour
    map.
*/
int get_colors(char **name)
{
  int i, j;
  int default_depth;
  XVisualInfo visual_info;
  XColor exact;

  default_depth = DefaultDepth(display, screen_num);
  DefColormap = DefaultColormap(display, screen_num);
  if (default_depth == 1) {
    /* must be StaticGray, use black and white */
    return(0);
  }
 
  if (!XMatchVisualInfo(display, screen_num, default_depth, PseudoColor,
                                                               &visual_info)) {
    if (!XMatchVisualInfo(display, screen_num, default_depth, DirectColor,
                                                               &visual_info)) {
      /* No PseudoColor visual available at default_depth.
       * Some applications might try for a GrayScale visual
       * here if they can use gray to advantage, before
       * giving up and using black and white.
       */
      return(0);
    }  
  }   

  /* Setup trajectory colours */
  for (i = 0; i < MAX_COLORS; i++)
    if (!XAllocNamedColor(display, DefColormap, TrajectoryColors[i],
                          &traj_color[i], &exact))
    {
      printf("Error: could not allocate Trajectory color %s (i = %d)\n",
             TrajectoryColors[i], i);
      Quit(-1);
    }

  /* Setup spectrogram colours */
  for (i = 0; i < SPECTRUM_SIZE; i++)
    if (!XAllocNamedColor(display, DefColormap, SpectrumColors[i],
                          &spec_color[i], &exact))
    {
      printf("Error: could not allocate Spectrum color %s (i = %d)\n",
             TrajectoryColors[i], i);
      Quit(-1);
    }

  SetupColorEntries();                         /* Set color allocation table */
  
  return 1;
}  /* End of get_colors */

/*****************************************************************************/

/*
  ChangeBandWidgets: customise any widgets that depend upon the number of
    dimensions set.
*/
void ChangeBandWidgets(void)
{
  int i, width, gauges_exposed;
  char buf[1024];

  /* reset the two RotationSelector widgets */
  xv_set(RotationSelector1,
         XV_SHOW, FALSE,
         PANEL_CHOICE_STRINGS,
           "Band 1",
           NULL,
         NULL);
  xv_set(RotationSelector2,
         XV_SHOW, FALSE,
         PANEL_CHOICE_STRINGS,
           "Band 1",
           NULL,
         NULL);
  for (i = 1; i < dimension; i++)
  {
    sprintf(buf, "Band %d", i + 1);

    xv_set(RotationSelector1, PANEL_CHOICE_STRING, i, buf, NULL);
    xv_set(RotationSelector2, PANEL_CHOICE_STRING, i, buf, NULL);
  }
  xv_set(RotationSelector1, XV_SHOW, TRUE, NULL);
  xv_set(RotationSelector2, XV_SHOW, TRUE, NULL);

  rotation_dim1 = ROTATION_DIM1;
  xv_set(RotationSelector1, PANEL_VALUE, ROTATION_DIM1, NULL);
  rotation_dim2 = ROTATION_DIM2;
  xv_set(RotationSelector2, PANEL_VALUE, ROTATION_DIM2, NULL);

  /*
    Resize the specify window and reposition/reset the widgets contained
      therein.
  */
  width = 140 + (dimension * 38);
  xv_set(SpecifyPanel, XV_SHOW, FALSE, NULL);
  xv_set(SpecifyFrame, XV_WIDTH, width, NULL);
  xv_set(SpecifyProjectButton, XV_X, width - 70, NULL);  

  i = (width / 2) - 120;
  xv_set(SpecifyRandEq1Choice, XV_X, i, NULL);
  xv_set(SpecifyRandEq2Choice, XV_X, i, NULL);
  xv_set(SpecifyRandEq3Choice, XV_X, i, NULL);

  /* Relabel all the selectors */
  xv_set(SpecifySelector1, PANEL_CHOICE_STRINGS, "1", NULL, NULL);
  xv_set(SpecifySelector2, PANEL_CHOICE_STRINGS, "1", NULL, NULL);
  xv_set(SpecifySelector3, PANEL_CHOICE_STRINGS, "1", NULL, NULL);
  xv_set(SpecifyEgnSelector, PANEL_CHOICE_STRINGS, "1", NULL, NULL);
  for (i = 1; i < dimension; i++)
  {
    sprintf(buf, "%d", i + 1);

    xv_set(SpecifySelector1, PANEL_CHOICE_STRING, i, buf, NULL);
    xv_set(SpecifySelector2, PANEL_CHOICE_STRING, i, buf, NULL);
    xv_set(SpecifySelector3, PANEL_CHOICE_STRING, i, buf, NULL);
    xv_set(SpecifyEgnSelector, PANEL_CHOICE_STRING, i, buf, NULL);
  }

  /*
    Setup guages for displaying root eigenvalues.  If gauges are currently
    exposed then set XV_SHOW on the first dimension gauges else leave them
    hidden.
    */
  gauges_exposed = (int) xv_get(SpecifyModeStack, PANEL_VALUE);
  if (gauges_exposed)
  {
    for (i = 0; i < dimension; i++)
      xv_set(SpecifyEgnGauge[i],
             XV_X, (i == 0) ? 5 : 117 + (35 * i),
             PANEL_VALUE, 0,
             XV_SHOW, TRUE,
             NULL);
  }
  else
  {
    for (i = 0; i < dimension; i++)
      xv_set(SpecifyEgnGauge[i],
             XV_X, (i == 0) ? 5 :117 + (35 * i),
             PANEL_VALUE, 0,
             XV_SHOW, FALSE,
             NULL);
  }

  for (i = dimension; i < MAX_DIMENSION; i++)
    xv_set(SpecifyEgnGauge[i],
           XV_SHOW, FALSE,
           NULL);

    

  /* Reset all selector and choice widgets back to their default values */
  xv_set(SpecifySelector1, PANEL_VALUE, 0x0001, NULL);
  xv_set(SpecifyRandEq1Choice, PANEL_VALUE, 0, NULL);
  xv_set(SpecifySelector2, PANEL_VALUE, 0x0002, NULL);
  xv_set(SpecifyRandEq2Choice, PANEL_VALUE, 0, NULL);
  xv_set(SpecifySelector3, PANEL_VALUE, 0x0004, NULL);
  xv_set(SpecifyRandEq3Choice, PANEL_VALUE, 0, NULL);
  xv_set(SpecifyEgnSelector, PANEL_VALUE, 0x0007, NULL);

  /* only expose the panel once all changes have been made */
  xv_set(SpecifyPanel, XV_SHOW, TRUE, NULL);
}

/*****************************************************************************/

/*
  PrepareConverterLists: examine records in the formats and front_ends linked
    lists and prepare their lists accordingly.
*/
void PrepareConverterLists(void)
{
  int index;
  FilterRecord *fptr;
  char buf[1024];
  
  /* Create Formats (conversion filters) list */
  index = 0;
  for (fptr = &Formats; fptr != NULL; fptr = fptr->next)
  {
    sprintf(buf, "%s [->%c]", fptr->name, fptr->format);
    xv_set(SampleFormatList,
           PANEL_LIST_INSERT, index,
           PANEL_LIST_STRING, index, buf,
           PANEL_LIST_CLIENT_DATA, index, fptr,
           NULL);
    
    fptr->index = index;
    index++;
  }
  xv_set(SampleFormatList,
         PANEL_LIST_SELECT, 0, TRUE,
         XV_SHOW, TRUE,
         NULL);
  
  /* Create Front-End filters list */ 
  index = 0;
  for (fptr = &Frontends; fptr != NULL; fptr = fptr->next)
  {
    sprintf(buf, "[%c->] %s", fptr->format, fptr->name);
    xv_set(SampleFrontendList,
           PANEL_LIST_INSERT, index,
           PANEL_LIST_STRING, index, buf,
           PANEL_LIST_CLIENT_DATA, index, fptr,
           NULL);
    
    fptr->index = index;
    index++;
  }
  xv_set(SampleFrontendList,
         PANEL_LIST_SELECT, 0, TRUE,
         XV_SHOW, TRUE,
         NULL);
}

/*****************************************************************************/

/*
  CreateSampleWindow: A pop-up window providing a scrollable list of the
    samples (utterances) within the list file.  Samples can then be
    selected and deselected from the projection display at run time.
    NB: Window geometry is modified between Contracted and Expanded states
    by SampleOptionsCallback().
*/
void CreateSampleWindow(Xv_opaque owner)
{
  unsigned int sample_width = 260;  /* contracted window width, expanded=500 */
  unsigned int sample_height = 310;                         /* window height */

  unsigned flags;

  SampleFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, (screen_width - sample_width - 10),
              XV_Y, 10,
              XV_WIDTH,  sample_width,
              XV_HEIGHT, sample_height,
              XV_LABEL, "List File",
              XV_SHOW, FALSE,              
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              FRAME_DONE_PROC, SampleDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  SamplePanel = xv_get(SampleFrame, FRAME_CMD_PANEL);

  SampleListnameText =
    xv_create(SamplePanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 10,
              PANEL_VALUE_DISPLAY_WIDTH, 150,
              PANEL_LABEL_STRING, "File Name:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_NOTIFY_PROC, SampleListnameCallback,
              NULL);
  SampleDimensionText =
    xv_create(SamplePanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 30,
              PANEL_VALUE_DISPLAY_WIDTH, 84,
              PANEL_LABEL_STRING, "Feature Dimensions:",
              PANEL_VALUE, "0",
              PANEL_READ_ONLY, TRUE,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              NULL);
  SampleNumSampsText =
    xv_create(SamplePanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 50,
              PANEL_VALUE_DISPLAY_WIDTH, 117,
              PANEL_LABEL_STRING, "Samples in List:",
              PANEL_VALUE, "0",
              PANEL_READ_ONLY, TRUE,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              NULL);
  SampleFileList =
    xv_create(SamplePanel, PANEL_LIST,
              XV_X, 10,
              XV_Y, 75,
              PANEL_LIST_WIDTH, 225,
              PANEL_LIST_DISPLAY_ROWS, 8,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_NOTIFY_PROC, SampleListCallback,
              PANEL_READ_ONLY, FALSE,
              PANEL_CHOOSE_ONE, FALSE,
              NULL);

  /* setup widget value from default variables */
  flags = 0;
  if (sample_display_info)
    flags |= 0x0001;
  if (sample_assume_logged)
    flags |= 0x0004;
  SampleCheckBox =
    xv_create(SamplePanel, PANEL_CHECK_BOX,
              XV_X, 10,
              XV_Y, 247,
              PANEL_VALUE, flags,
              PANEL_CHOICE_NROWS, 3,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Options:",
              PANEL_CHOICE_STRINGS,
                "Information",
                "Assume Logged",
                NULL,
              PANEL_NOTIFY_PROC, SampleOptionsCallback,
              NULL);

  SampleWin1Glyph = (Server_image)
    xv_create(NO_PARENT, SERVER_IMAGE,
              XV_WIDTH, SampleIcon1_width,
              XV_HEIGHT, SampleIcon1_height,
              SERVER_IMAGE_X_BITS, SampleIcon1_bits,
              NULL);

  SampleWin2Glyph = (Server_image)
    xv_create(NO_PARENT, SERVER_IMAGE,
              XV_WIDTH, SampleIcon2_width,
              XV_HEIGHT, SampleIcon2_height,
              SERVER_IMAGE_X_BITS, SampleIcon2_bits,
              NULL);

  /* Window State button */
  xv_create(SamplePanel, PANEL_BUTTON,
            XV_X, 210,
            XV_Y, 255,
            PANEL_LABEL_IMAGE, SampleWin1Glyph,
            PANEL_NOTIFY_PROC, SampleWinStateCallback,
            NULL);

  SampleFormatList =
    xv_create(SamplePanel, PANEL_LIST,
              XV_X, 270,
              XV_Y, 10,
              PANEL_LIST_TITLE, "File Formats:",
              PANEL_LIST_WIDTH, 205,
              PANEL_LIST_DISPLAY_ROWS, 4,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_ONE, TRUE,
              PANEL_NOTIFY_PROC, SampleFormatCallback,
              XV_SHOW, FALSE,
              NULL);

  SampleFrontendList =
    xv_create(SamplePanel, PANEL_LIST,
              XV_X, 270,
              XV_Y, 140,
              PANEL_LIST_TITLE, "Front Ends:",
              PANEL_LIST_WIDTH, 205,
              PANEL_LIST_DISPLAY_ROWS, 4,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_ONE, TRUE,
              PANEL_NOTIFY_PROC, SampleFrontendCallback,
              XV_SHOW, FALSE,
              NULL);

  SampleUpdateButton =
    xv_create(SamplePanel, PANEL_BUTTON,
              XV_X, 290,
              XV_Y, 270,
              PANEL_LABEL_STRING, "Update",
              PANEL_NOTIFY_PROC, SampleUpdateCallback,
              XV_SHOW, FALSE,
              NULL);

  SampleParamsButton =
    xv_create(SamplePanel, PANEL_BUTTON,
              XV_X, 400,
              XV_Y, 270,
              PANEL_LABEL_STRING, "Parameters",
              PANEL_NOTIFY_PROC, SampleParamsCallback,
              XV_SHOW, FALSE,
              NULL);
}

/*****************************************************************************/

/*
  SampleFillScrollList: Fills up a scrolling list widget by traversing down
    through a linked list of OptionRecords.
*/ 
void SampleFillScrollList(Xv_opaque list, OptionRecord *option)
{
  int index;
  OptionRecord *optr;
  char buf[1024];

  index = 0;
  for (optr = option; optr != NULL; optr = optr->next)
  {
    sprintf(buf, "%s = %s", optr->name, optr->value);
    xv_set(list,
           PANEL_LIST_INSERT, index,
           PANEL_LIST_STRING, index, buf,
           PANEL_LIST_CLIENT_DATA, index, optr,
           NULL);
    optr->index = index;
    index++;        
  }
}

/*****************************************************************************/

/*
  SampleEmptyScrollList: Empties a scrolling list widget previously filled 
    by SampleFillScrollList().
*/ 
void SampleEmptyScrollList(Xv_opaque list)
{
  xv_set(list, PANEL_LIST_DELETE_ROWS, 0, 99, NULL);
}

/*****************************************************************************/

void CreateParamsWindow(Xv_opaque owner)
{
  const int params_width = 500;
  const int params_height = 190;
  
  ParamsFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, 100,
              XV_Y, 100,
              XV_WIDTH,  params_width,
              XV_HEIGHT, params_height,
              XV_LABEL, "Filter Parameters",
              XV_SHOW, FALSE,              
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              FRAME_DONE_PROC, ParamsDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  ParamsPanel = /* xv_get(ParamsFrame, FRAME_CMD_PANEL); */
    xv_create(ParamsFrame, PANEL,
              XV_X, 0,
              XV_Y, 0,
              XV_WIDTH , WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, WIN_EXTEND_TO_EDGE,
              WIN_BORDER, FALSE,
              NULL);

  ParamsFormatNameText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 10,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Format:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_READ_ONLY, TRUE,
              PANEL_VALUE, DEF_FORMAT_FILTER,
              NULL);

  ParamsFormatFilterText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 35,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Filter:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_READ_ONLY, TRUE,
              NULL);

  ParamsFormatList =
    xv_create(ParamsPanel, PANEL_LIST,
              XV_X, 10,
              XV_Y, 60,
              PANEL_LIST_WIDTH, 205,
              PANEL_LIST_DISPLAY_ROWS, 4,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_ONE, TRUE,
              PANEL_NOTIFY_PROC, ParamsFormatCallback,
              NULL);

    ParamsFormatValueText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 10,
              XV_Y, 160,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Value:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_NOTIFY_PROC, ParamsFormatTextCallback,
              PANEL_READ_ONLY, TRUE,
              PANEL_CLIENT_DATA, NULL,
              NULL);

  ParamsFrontendNameText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 260,
              XV_Y, 10,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Front End:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_READ_ONLY, TRUE,
              PANEL_VALUE, DEF_FRONTEND_FILTER,
              NULL);

  ParamsFrontendFilterText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 260,
              XV_Y, 35,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Filter:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_READ_ONLY, TRUE,
              NULL);

  ParamsFrontendList =
    xv_create(ParamsPanel, PANEL_LIST,
              XV_X, 260,
              XV_Y, 60,
              PANEL_LIST_WIDTH, 205,
              PANEL_LIST_DISPLAY_ROWS, 4,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_ONE, TRUE,
              PANEL_NOTIFY_PROC, ParamsFrontendCallback,
              NULL);

  ParamsFrontendValueText =
    xv_create(ParamsPanel, PANEL_TEXT,
              XV_X, 260,
              XV_Y, 160,
              PANEL_VALUE_DISPLAY_WIDTH, 160,
              PANEL_LABEL_STRING, "Value:",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_NOTIFY_PROC, ParamsFrontendTextCallback,
              PANEL_READ_ONLY, TRUE,
              PANEL_CLIENT_DATA, NULL,
              NULL);
}

/*****************************************************************************/

/*
  CreateSpecifyWindow: pop-up window with controls for selecting new basis
    functions for projection.  Note: Panel sizes and widget positions are
    customized by CreateBandWidgets() depending on the current value of
    'dimension'.
*/
void CreateSpecifyWindow(Xv_opaque owner)
{
  int i;
  
  SpecifyFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, 100,
              XV_Y, 100,
              XV_HEIGHT, 240,
              XV_LABEL, "Projection Basis Vectors",
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              XV_SHOW, FALSE,
              FRAME_DONE_PROC, SpecifyDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  SpecifyPanel = xv_get(SpecifyFrame, FRAME_CMD_PANEL);

  SpecifyModeStack = 
    xv_create(SpecifyPanel, PANEL_CHOICE_STACK,
              XV_X, 5,
              XV_Y, 5,
              PANEL_LABEL_STRING, "Basis Selection:",
              PANEL_CHOICE_STRINGS,
              "Manual", "Eigensystem",
              NULL,
              PANEL_NOTIFY_PROC, SpecifyModeCallback,
              NULL);

  /* Create Rotation axis selector radio-buttons */
  SpecifySelector1 =
    xv_create(SpecifyPanel, PANEL_TOGGLE,
              XV_X, 5,
              XV_Y, 45,
              PANEL_VALUE, 0x0001,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Subspace 1:",
              PANEL_NOTIFY_PROC, SpecifyToggleCallback,
              PANEL_CLIENT_DATA, 1,
              NULL);

  SpecifyRandEq1Choice =
    xv_create(SpecifyPanel, PANEL_CHOICE,
              XV_Y, 75,
              PANEL_LABEL_STRING, "Basis 1:",
              PANEL_CHOICE_STRINGS,
                "Diagonal", "Random", NULL,
              NULL);

  SpecifySelector2 =
    xv_create(SpecifyPanel, PANEL_TOGGLE,
              XV_X, 5,
              XV_Y, 110,
              PANEL_VALUE, 0x0002,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Subspace 2:",
              PANEL_NOTIFY_PROC, SpecifyToggleCallback,
              PANEL_CLIENT_DATA, 2,
              NULL);

  SpecifyRandEq2Choice =
    xv_create(SpecifyPanel, PANEL_CHOICE,
              XV_Y, 140,
              PANEL_LABEL_STRING, "Basis 2:",
              PANEL_CHOICE_STRINGS,
                "Diagonal", "Random", NULL,
              NULL);

  SpecifySelector3 =
    xv_create(SpecifyPanel, PANEL_TOGGLE,
              XV_X, 5,
              XV_Y, 175,
              PANEL_VALUE, 0x0004,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Subspace 3:",
              PANEL_NOTIFY_PROC, SpecifyToggleCallback,
              PANEL_CLIENT_DATA, 3,
              NULL);

  SpecifyRandEq3Choice =
    xv_create(SpecifyPanel, PANEL_CHOICE,
              XV_Y, 205,
              PANEL_LABEL_STRING, "Basis 3:",
              PANEL_CHOICE_STRINGS,
                "Diagonal", "Random", NULL,
              NULL);

  SpecifyEgnGauge[0] =
    xv_create(SpecifyPanel, PANEL_GAUGE,
              XV_Y, 40,
              PANEL_LABEL_STRING, "Root Eigenvals:",
              PANEL_SLIDER_WIDTH, 100,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 99,
              PANEL_TICKS, 11,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_VALUE, 0,
              PANEL_DIRECTION, PANEL_VERTICAL,
              XV_SHOW, FALSE,
              NULL);
  for (i = 1; i < MAX_DIMENSION; i++)
    SpecifyEgnGauge[i] =
      xv_create(SpecifyPanel, PANEL_GAUGE,
                XV_Y, 40,
                PANEL_SLIDER_WIDTH, 100,
                PANEL_MIN_VALUE, 0,
                PANEL_MAX_VALUE, 99,
                PANEL_TICKS, 11,
                PANEL_SHOW_RANGE, FALSE,
                PANEL_VALUE, 0,
                PANEL_DIRECTION, PANEL_VERTICAL,
                XV_SHOW, FALSE,
                NULL);
  
  SpecifyEgnSelector =
    xv_create(SpecifyPanel, PANEL_TOGGLE,
              XV_X, 5,
              XV_Y, 170,
              PANEL_VALUE, 0x0007,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "EigenVectors:",
              PANEL_NOTIFY_PROC, SpecifyEigenCallback,
              XV_SHOW, FALSE,
              NULL);

  SpecifyAnalyseButton =
    xv_create(SpecifyPanel, PANEL_BUTTON,
              XV_X, 20,
              XV_Y, 215,
              PANEL_LABEL_STRING, "Analyse",
              PANEL_NOTIFY_PROC, SpecifyAnalyseCallback,
              XV_SHOW, FALSE,
              NULL);

  SpecifyErrorMessage =
    xv_create(SpecifyPanel, PANEL_MESSAGE,
              XV_X, 105,
              XV_Y, 217,
              XV_SHOW, FALSE,
              PANEL_LABEL_STRING, "",
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              NULL);

  SpecifyProjectButton =
    xv_create(SpecifyPanel, PANEL_BUTTON,
              XV_Y, 215,
              PANEL_LABEL_STRING, "Project",
              PANEL_NOTIFY_PROC, SpecifyProjectCallback,
              NULL);
}

/*****************************************************************************/

/*
  HighlightWindow: Creates a window to allow control of vertex Prediction
    and display of highlighted vertices.
*/
void CreateHighlightWindow(Xv_opaque owner)
{
  const int highlight_width = 300;
  const int highlight_height = 140;
  
  HighlightFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, 100,
              XV_Y, 140,
              XV_WIDTH, highlight_width,
              XV_HEIGHT, highlight_height,
              XV_LABEL, "Highlight Vertices",
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              XV_SHOW, FALSE,
              FRAME_DONE_PROC, HighlightDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  HighlightPanel = xv_get(HighlightFrame, FRAME_CMD_PANEL);

  /* Create Rotation axis selector radio-buttons */
  HighlightSelector =
    xv_create(HighlightPanel, PANEL_CHOICE,
              XV_X, 10,
              XV_Y, 10,
              PANEL_VALUE, 0,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_NONE, FALSE,
              PANEL_LABEL_STRING, "Predictor Order:",
              PANEL_CHOICE_STRINGS, "Zeroth", "First", "Second", NULL,
              PANEL_NOTIFY_PROC,HighlightSelectorCallback,
              NULL);

  HighlightCheckBox =
    xv_create(HighlightPanel, PANEL_CHECK_BOX,
              XV_X, 10,
              XV_Y, 40,
              PANEL_VALUE, 0,
              PANEL_CHOICE_NROWS, 2,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Options:",
              PANEL_CHOICE_STRINGS, "Direction", "Velocity", "Verbose", NULL,
              NULL);
  
  HighlightSlider =
    xv_create(HighlightPanel, PANEL_SLIDER,
              XV_X, 10,
              XV_Y, 105,
              PANEL_LABEL_STRING, "Vertices:",
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, TRUE,
              PANEL_SHOW_VALUE, TRUE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 10,
              PANEL_VALUE, 0,
              PANEL_INACTIVE, TRUE,
              PANEL_NOTIFY_PROC, HighlightSliderCallback,
              NULL);
}

/*****************************************************************************/

CreateDisplayWindow(Xv_opaque owner)
{
  unsigned flags;
  XSetWindowAttributes attributes;
  Server_image UpArrowGlyph, DownArrowGlyph;
  
  DisplayFrame =
    xv_create(owner, FRAME_CMD,
              XV_X, 0,
              XV_Y, 0,
              XV_WIDTH, 750,
              XV_HEIGHT, 700,
              XV_LABEL, "Trajectory Display",
              XV_SHOW, FALSE,
              FRAME_SHOW_RESIZE_CORNER, TRUE,
              XV_VISUAL_CLASS, PseudoColor,
              FRAME_DONE_PROC, DisplayDismissCallback,
              FRAME_CMD_DEFAULT_PIN_STATE, FRAME_CMD_PIN_IN,
              NULL);

  DisplayPanel = xv_get(DisplayFrame, FRAME_CMD_PANEL);
  xv_set(DisplayPanel,
         XV_X, 0,
         XV_Y, 0,
         XV_WIDTH, WIN_EXTEND_TO_EDGE,
         XV_HEIGHT, 60,
         WIN_BORDER, TRUE,
         NULL);
  DisplayCanvas =
    xv_create(DisplayFrame, CANVAS,
              XV_X, 0,
              XV_Y, 60,
              XV_WIDTH, WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, WIN_EXTEND_TO_EDGE,
              CANVAS_X_PAINT_WINDOW, FALSE,
              CANVAS_RETAINED, TRUE,
              CANVAS_FIXED_IMAGE, TRUE,
              CANVAS_REPAINT_PROC, RefreshGraphics,
              CANVAS_RESIZE_PROC, RefreshGraphics,
              NULL);
  /* Set event handler for canvas --- only handle down-button mouse events */
  xv_set(canvas_paint_window(DisplayCanvas),
         WIN_EVENT_PROC, DisplayEventHandler,
         WIN_CONSUME_EVENTS, WIN_MOUSE_BUTTONS,
         NULL);
  xv_create(DisplayPanel, PANEL_CHOICE_STACK,
              XV_X, 10,
              XV_Y, 6,
              PANEL_LABEL_STRING, "Format:",
              PANEL_CHOICE_STRINGS,
                "2 Dimensions", "3 Dimensions", "Spectrogram",
                NULL,
              PANEL_NOTIFY_PROC, DisplayFormatCallback,
              NULL);

  DisplayRandomizeButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 190,
              XV_Y, 8,
              PANEL_LABEL_STRING, "Randomize",
              PANEL_NOTIFY_PROC, DisplayRandomizeCallback,
              NULL);

  /*
    NB: These two buttons share a common callback routine. The rotuine tells
      between them using the PANEL_CLIENT_DATA value.  The "Recall" button is
      made active on the first press of the "Store" button.
  */
  DisplayStoreButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 290,
              XV_Y, 8,
              PANEL_LABEL_STRING, "Store",
              PANEL_NOTIFY_PROC, DisplayStoreRecallCallback,
              PANEL_CLIENT_DATA, 0,
              NULL);
  DisplayRecallButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 350,
              XV_Y, 8,
              PANEL_LABEL_STRING, "Recall",
              PANEL_NOTIFY_PROC, DisplayStoreRecallCallback,
              PANEL_CLIENT_DATA, 1,
              NULL);
  
  DisplayColorsButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 415,
              XV_Y, 8,
              PANEL_LABEL_STRING, "Colours",
              PANEL_NOTIFY_PROC, DisplayColorsCallback,
              PANEL_CLIENT_DATA, 1,
              PANEL_INACTIVE, TRUE,
              NULL);

  flags = 0x0000;
  if (compress_raw_mode)
    flags |= 0x0001;
  if (compress_scale_mode)
    flags |= 0x0002;
  if (display_label_mode)
    flags |= 0x0004;
  if (display_origin_mode)
    flags |= 0x0008;
  DisplayCheckBox =
    xv_create(DisplayPanel, PANEL_CHECK_BOX,
              XV_X, 10,
              XV_Y, 30,
              PANEL_VALUE, flags,
              PANEL_CHOICE_NROWS, 1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_LABEL_STRING, "Options:",
              PANEL_CHOICE_STRINGS,
                "Raw", "Scaled", "Labels", "Origin",
                NULL,
              PANEL_NOTIFY_PROC, DisplayCheckBoxCallback,
              NULL);
  DisplaySlider =
    xv_create(DisplayPanel, PANEL_SLIDER,
              XV_X, 600,
              XV_Y, 4,
              PANEL_VALUE, (int) floor(99.9 * (1.0 - compress_alpha)),
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 99,
              PANEL_MIN_TICK_STRING, "Linear",
              PANEL_MAX_TICK_STRING, "Compressed",
              PANEL_NOTIFY_PROC, DisplaySliderCallback,
              NULL);

  /* Create icon for display within DisplayDecrButton */
  UpArrowGlyph = (Server_image)
    xv_create(NO_PARENT, SERVER_IMAGE,
              XV_WIDTH, up_arrow_width,
              XV_HEIGHT, up_arrow_height,
              SERVER_IMAGE_X_BITS, up_arrow_bits,
              NULL);

  DisplayDecrButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 490,
              XV_Y, 8,
              PANEL_LABEL_IMAGE, UpArrowGlyph,
              PANEL_NOTIFY_PROC, DisplayArrowCallback,
              PANEL_INACTIVE, TRUE,
              PANEL_CLIENT_DATA, 0,
              NULL);

  /* Create icon for display within DisplayIncrButton */
  DownArrowGlyph = (Server_image)
    xv_create(NO_PARENT, SERVER_IMAGE,
              XV_WIDTH, down_arrow_width,
              XV_HEIGHT, down_arrow_height,
              SERVER_IMAGE_X_BITS, down_arrow_bits,
              NULL);

  DisplayIncrButton =
    xv_create(DisplayPanel, PANEL_BUTTON,
              XV_X, 545,
              XV_Y, 8,
              PANEL_LABEL_IMAGE, DownArrowGlyph,
              PANEL_NOTIFY_PROC, DisplayArrowCallback,
              PANEL_INACTIVE, TRUE,
              PANEL_CLIENT_DATA, 1,
              NULL);

  /* Register Event Handler for Keyboard Events within the window */
/*
  xv_set(canvas_paint_window(DisplayCanvas),
         WIN_EVENT_PROC, DisplayEventHandler,
         WIN_CONSUME_X_EVENT_MASK, KeyPressMask,
         WIN_IGNORE_X_EVENT_MASK, KeyReleaseMask,
         NULL);
*/
  
  /* Obtain handles into window */
  display = (Display *) xv_get(DisplayCanvas,XV_DISPLAY);
  win = (Window) xv_get(canvas_paint_window(DisplayCanvas),XV_XID);

  /* set background color then draw left and right eye point sets */
  attributes.background_pixel = WhitePixel(display, screen_num);
  XChangeWindowAttributes(display, win, CWBackPixel, &attributes);
  XClearWindow(display, win);

  /* Prepare Canvas for drawing */
  (void) get_colors(CName);                           /* Setup color palette */
  load_font(&FontInfo);                                /* Load Display Font */
  getGC(win, &gc, FontInfo);              /* create GC for text and drawing */

  /* Establish an initial three_dimensional mode colour map */
  SetupDoubleBuffer();
  SwapDisplayBuffer();
}

/*****************************************************************************/

/*
  SetWindowManager: Commands controlling window manager and specifying the
    icon to use etc. .
*/
void SetWindowManager(int argc, char* argv[])
{
  Server_image hour_glass_image, hour_glass_mask;
  Xv_singlecolor black, white;
  
  project_image = (Server_image)
    xv_create(NO_PARENT, SERVER_IMAGE,
              XV_WIDTH, project_width,
              XV_HEIGHT, project_height,
              SERVER_IMAGE_X_BITS, project_bits,
              NULL);
  project_icon = (Icon)
    xv_create(NO_PARENT, ICON,
              XV_WIDTH, project_width,
              XV_HEIGHT, project_height,
              ICON_IMAGE, project_image,
              ICON_TRANSPARENT, TRUE,
              NULL);

  /* Create the set of cursors */
  BasicCursor = (Xv_Cursor)
    xv_create(XV_NULL, CURSOR,
              CURSOR_SRC_CHAR,  OLC_BASIC_PTR,
              CURSOR_MASK_CHAR, OLC_BASIC_MASK_PTR,
              NULL);
  BusyCursor = (Xv_Cursor)
    xv_create(XV_NULL, CURSOR,
              CURSOR_SRC_CHAR,  OLC_BUSY_PTR,
              CURSOR_MASK_CHAR, OLC_BUSY_MASK_PTR,
              NULL);
}

/*****************************************************************************/

void CreateControlWindow(int argc, char* argv[])
{
  unsigned int control_width  = 460;                /* window width (pixels) */
  unsigned int control_height = 330;               /* window height (pixels) */
  char control_title[1024];

  /* setup xv graphics system */
  xv_init(XV_INIT_ARGS, argc, argv, NULL);

  /* Create Control window frame and panel */
  sprintf(control_title, "fview, v%s", FVIEW_VERSION);
  MainFrame =
    xv_create(NO_PARENT, FRAME,
              XV_WIDTH,  control_width,
              XV_HEIGHT, control_height,
              XV_LABEL, control_title,
              FRAME_SHOW_RESIZE_CORNER, FALSE,
              XV_SHOW, FALSE,
              NULL);

  /* Creates window containing Canvas for graphics output */
  CreateDisplayWindow(MainFrame);

  /* Create Icon associated with the windows */
  SetWindowManager(argc, argv);
  
  /*
    Get total screen and adjust position of MainFrame accordingly.
    Also set frame icon.
  */
  screen_width  = DisplayWidth(display,  screen_num);
  screen_height = DisplayHeight(display, screen_num);
  xv_set(MainFrame,
         XV_X, (screen_width - control_width - 30),
         XV_Y, (screen_height - control_height - 30),
         FRAME_ICON, project_icon,
         NULL);

  MainPanel =
    xv_create(MainFrame, PANEL,
              XV_X, 0,
              XV_Y, 0,
              XV_WIDTH , WIN_EXTEND_TO_EDGE,
              XV_HEIGHT, WIN_EXTEND_TO_EDGE,
              WIN_BORDER, FALSE,
              NULL);

  /* Create CIIPS icon for display in auto_scale mode */
  ciips_image = (Server_image)
    xv_create(0, SERVER_IMAGE,
              XV_WIDTH, ciips_width,
              XV_HEIGHT, ciips_height,
              SERVER_IMAGE_X_BITS, ciips_bits,
              NULL);

  ciips_logo =
    xv_create(MainPanel, PANEL_MESSAGE,
              XV_X, 240,
              XV_Y, 70,
              PANEL_LABEL_IMAGE, ciips_image,
              XV_SHOW, FALSE,
              NULL);

  /* Create Rotation axis selector radio-buttons */
  RotationSelector1 =
    xv_create(MainPanel, PANEL_CHOICE_STACK,
              XV_X, 10,
              XV_Y, 10,
              PANEL_VALUE, rotation_dim1,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_NONE, FALSE,
              PANEL_LABEL_STRING, "Rotation Axis 1:",
              PANEL_NOTIFY_PROC, Rotation1Callback,
              NULL);

  RotationSelector2 =
    xv_create(MainPanel, PANEL_CHOICE_STACK,
              XV_X, 240,
              XV_Y, 10,
              PANEL_VALUE, rotation_dim2,
              PANEL_LAYOUT, PANEL_HORIZONTAL,
              PANEL_CHOOSE_NONE, FALSE,
              PANEL_LABEL_STRING, "Rotation Axis 2:",
              PANEL_NOTIFY_PROC, Rotation2Callback,
              NULL);

  /* Create Sliders for Position and Scale of Projection */
  UpDownSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 280,
              XV_Y, 50,
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_VERTICAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, -1000,
              PANEL_MAX_VALUE, 1000,
              PANEL_MIN_TICK_STRING, "Down",
              PANEL_MAX_TICK_STRING, "Up",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, ViewSliderCallback,
              NULL);
  
  LeftRightSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 300,
              XV_Y, 190,
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, -1000,
              PANEL_MAX_VALUE, 1000,
              PANEL_MIN_TICK_STRING, "Left",
              PANEL_MAX_TICK_STRING, "Right",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, ViewSliderCallback,
              NULL);

  EnlargeShrinkSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 360,
              XV_Y, 50,
              PANEL_SLIDER_WIDTH, 100,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_VERTICAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, -100,
              PANEL_MAX_VALUE, 100,
              PANEL_MIN_TICK_STRING, "Reduce",
              PANEL_MAX_TICK_STRING, "Enlarge",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, ViewSliderCallback,
              NULL);

  ControlAutoScaleButton =
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 320,
              XV_Y, 250,
              PANEL_LABEL_STRING, "AutoScale",
              PANEL_NOTIFY_PROC, AutoCallback,
              NULL);

  /* Hide buttons from outset if initially in autoscale mode */
  if (auto_scale)
  {
    xv_set(UpDownSlider, XV_SHOW, FALSE, NULL);
    xv_set(LeftRightSlider, XV_SHOW, FALSE, NULL);
    xv_set(EnlargeShrinkSlider, XV_SHOW, FALSE, NULL);
    xv_set(ciips_logo, XV_SHOW, TRUE, NULL);
  }

  /* Controls for Animation System */
  AngleSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 10,
              XV_Y, 50,
              PANEL_LABEL_STRING, "Angle:",
              PANEL_SLIDER_WIDTH, 140,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 1000,
              PANEL_MIN_TICK_STRING, "Small",
              PANEL_MAX_TICK_STRING, "Large",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, AngleSliderCallback,
              NULL);

  IterationSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 10,
              XV_Y, 110,
              PANEL_LABEL_STRING, "Range:",
              PANEL_SLIDER_WIDTH, 140,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, iterations,
              PANEL_MAX_VALUE, 1000,
              PANEL_MIN_TICK_STRING, "Small",
              PANEL_MAX_TICK_STRING, "Large",
              PANEL_VALUE, 0,
              PANEL_NOTIFY_PROC, IterationSliderCallback,
              NULL);

  ControlAnimateButton = 
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 100,
              XV_Y, 155,
              PANEL_LABEL_STRING, "Animate",
              PANEL_NOTIFY_PROC, AnimateCallback,
              NULL);

  /* Display options */
  SweepRateSlider =
    xv_create(MainPanel, PANEL_SLIDER,
              XV_X, 10,
              XV_Y, 210,
              PANEL_LABEL_STRING, "Sweep:",
              PANEL_SLIDER_WIDTH, 140,
              PANEL_TICKS, 10,
              PANEL_DIRECTION, PANEL_HORIZONTAL,
              PANEL_SLIDER_END_BOXES, FALSE,
              PANEL_SHOW_RANGE, FALSE,
              PANEL_SHOW_VALUE, FALSE,
              PANEL_MIN_VALUE, 0,
              PANEL_MAX_VALUE, 200,
              PANEL_MIN_TICK_STRING, "Fast",
              PANEL_MAX_TICK_STRING, "Slow",
              PANEL_VALUE, 0,
              NULL);

  ControlSweepButton =
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 100,
              XV_Y, 255,
              PANEL_LABEL_STRING, "Sweep",
              PANEL_NOTIFY_PROC, SweepGraphicsCallback,
              NULL);

  /* Create Remaining buttons along base of panel */
  ControlSamplesButton = 
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 30,
              XV_Y, 300,
              PANEL_LABEL_STRING, "Samples",
              PANEL_NOTIFY_PROC, SamplesCallback,
              NULL);

  ControlSpecifyButton = 
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 110,
              XV_Y, 300,
              PANEL_LABEL_STRING, "Projection",
              PANEL_NOTIFY_PROC, SpecifyCallback,
              NULL);

  ControlHighlightButton =
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 200,
              XV_Y, 300,
              PANEL_LABEL_STRING, "Highlight",
              PANEL_NOTIFY_PROC, HighlightCallback,
              NULL);

  ControlPointSetButton =
    xv_create(MainPanel, PANEL_BUTTON,
              XV_X, 290,
              XV_Y, 300,
              PANEL_LABEL_STRING, "Point set",
              PANEL_NOTIFY_PROC, PointSetCallback,
              NULL);

    xv_create(MainPanel, PANEL_BUTTON,
            XV_X, 380,
            XV_Y, 300,
            PANEL_LABEL_STRING, "Quit",
            PANEL_NOTIFY_PROC, QuitCallback,
            NULL);

  /*
    Create (hidden) pop-up windows for selecting utterances to view, for
    selecting both format and front-end filters and for controlling the
    parameters of the filters selected.
  */
  CreateSampleWindow(MainFrame);
  CreateParamsWindow(MainFrame);
  
  /* Creates (hidden) pop-up window for setting projection axes */
  CreateSpecifyWindow(MainFrame);

  /* Creates (hidden) pop-up window for controllong prediction/highlighting */
  CreateHighlightWindow(MainFrame);

  /* Creates subwindow allowing control of Pallette for 3D modes */
  CreateColorWindow(MainFrame);
}

/*****************************************************************************/

/* end of interface.c */
