/**************************************************************************
*
* ROUTINE
*               cli
*
* FUNCTION
*               UNIX Command line interface.
*		Read and check the UNIX command line.
*		Report and log job characteristics and performance.
* 
* SYNOPSIS
*	subroutine cli (ifile, ofile, l, ll, lp, np, scale, descale, ber, 
*    +                  mask, stype, eccbits, sbits)
*
*   formal 
*
*			data	I/O
*	name		type	type	function
*       -------------------------------------------------------------------
*	ifile		char	i/o	input file name
*	ofile		char	i/o	output file name
*	l		int	i/o	Code word length
*	ll		int	i/o	LPC analysis frame size
*	lp		int	i/o	Pitch analysis frame size
*	np		int	i/o	Pitch order
*	scale		float	i/o	Input speech scaling factor
*	descale		i	i/o	Output speech scaling factor
*	ber		float	i/o	% bit error rate
*	mask		int	o	Error mask
*	stype		char	i/o	Spectrum quantizer type
*	sbits		int	i/o	Spectrum bit allocation
*	eccbits		int	i/o	Error control bits/frame
*	ssum		int	i/o	Sum of spectrum bit allocation
*
*   global common 
*                       data    I/O
*       name            type    type    function
*       -------------------------------------------------------------------
*	cbgbits		int	i/o
*	frame		int	i
*	ncsize		int	i/o
*	no		int	i/o
*	pbits[]		int	i/o
*	gamma2		float	i/o
*	ptype[]		int	i/o
*	cbgtype[]	int	i/o
*
***************************************************************************
*			CELP COMMAND 
*NAME:
*     celp - execute the CELP coder
*
*     The celp command generates a codebook-excited-linear-prediction
*     processed output file from an input file.
*
*SYNOPSIS:
*     celp [-i ifile] [-o ofile] [-p pfile] [-q qfile] [-m mfile] [-l lfile]
*     celp [-c chan]  [-o ofile]
*
*ARGUMENTS:
*     -i      Input file i*2 direct access format (unformatted, consecutive
*             16-bit signed samples).
*             For defaults, see celp.c declaration statements.
* 
*     -o      Speech output file (.spd) i*2 direct access format 
*	      (unformatted, consecutive 16-bit signed samples) and
*	      bit stream channel file (.chan) hexadecimal format.
*             For defaults, see celp.c declaration statements.
*
*     -p      Parameter file specifying celp characteristics.
*             For defaults, see celp.c declaration statements.
*             Parameters are stored in a ascii text file in the order below
*             with one number per line:
*
*             512	[ncsize = Code book size]
*             60	[l      = Code word length]
*             240	[ll     = LPC analysis frame size]
*             10	[no     = LPC filter order]
*             60	[lp     = Pitch analysis frame size]
*             1         [np     = Pitch order]
*             0.8	[gamma  = Noise weighting factor]
*             1.0	[scale  = Input speech scaling factor]
*	      1.0       [descale = Output speech scaling factor]
*	      0.0	[ber	= % bit error rate]
*	      1		[mxsw    = modified excitation logical switch]
*	      0.0       [prewt   = prefilter logical switch]
*	      hier      [pstype  = type of fractional pitch search]
*
*     -q      Quantization characteristics file.
*             For defaults, see celp.c declaration statements.
*             The file has 3 sections: cbgain, pitch, and spectrum.
*             Each section type is followed by a quantization type,
*             which can be one of the following:  "max", "uniform", "vq",
*             "log", "opt", or "none".  If the quantization type is not 
*	      "none",  { the next line is the bit allocation; i.e.:
*
*             cbgain
*             none		[unquantized cbgain]
*             pitch
*             max		[Max quantization]
*             8 6 5		[8 bit delay, 6 bit delta delay, 5 bit gain]
*             spectrum
*             kang		[Kang's quantization]
*             3 4 4 4 4 3 3 3 3 3 	[lsp1=3,...,lsp10=3]
*
*     -m      Mask file specifying protected and unprotectected bits
*	      in the bit stream when introducing bit errors to the 
*	      unpermuted bit stream.  (Note, this protection is separate
*	      from the Hamming FEC).  Each line of the mfile corresponds
*	      to a bit where each line is a 1 (protected) or 0 (not
*	      protected).
*
*     -l      Log file output containing run time information.
*             Defaults to appending to a file called "celp.log".
*             If the file name "none" is specified, no log file
*             is generated.
*
*
*     -c      Input channel file (.chan) hexadecimal format.  Channel
*	      files generated from previous analysis runs are used as
*	      imputs to a "synthesis only" run.  During this mode, the
*	      -i switch is invalid.
*EXAMPLES:
*
*     celp
*             This causes celp to process ifile.spd into ofile.spd (and a
*	      nonpostfiltered output in ofilenpf.spd) using the
*             defaults specified by the C declaration statements in
*             celp.c, writes the bit stream file ofile.chan and generates 
*	      a log appended to the file "celp.log".  (If SUNGRAPH is 
*	      enabled, a set of files, with .sg_data extension, as defined
*	      in sungraph_open.com is generated.)
*
*     celp -i speech/dam27 -o dam27.48 -p celp48.p -q celp48.q -l log
*
*             celp processes speech/dam27.spd into the normal, highpassed
*	      and nonpostfiltered output, dam27.48.spd, dam27_48hpf.spd
*	      and dam27_48npf.spd, respectively, writes the bit stream
*	      file dam27_48.chan and appends log information to a file
*	      called log.  The celp parameters specified by celp48.p and  
*	      quantization characteristics of celp48.q are used.  (If  
*	      SUNGRAPH is enabled, a set of files, with .sg_data extension, 
*             as defined in sungraph_open.com is generated.)
*	      
*
*     celp -c speech/512_dam27.chan -o 512_dam27b
*
*	      celp synthesizes speech/512_dam27.chan channel file into 
*	      the normal, highpassed and nonpostfiltered output, 
*	      512_dam27b.spd, 512_dam27bhpf.spd and 512_dam27bnpf.spd,
*	      respectively.
*
***************************************************************************
*
* CALLED BY
*
*       celp
*
* CALLS
*
*       
*
**************************************************************************/
#define TRUE            1
#define FALSE           0
#define MAXARG		14
#define MAXTYPE		3
#define MAXQTYPE	5
#include <stdio.h>
#include <sys/time.h>
#include <sys/types.h>
#include <sys/stat.h>
#include <math.h>
#include "ccsub.h"
extern int cbgbits, frame, ncsize, no, pbits[], mxsw;
extern float gamma2, prewt;
extern char ptype[], cbgtype[], pstype[];
static char lfile[82] = "celp.log";
FILE *fp;

cli (ifile,ofile,l,ll,lp,np,scale,descale,ber,mask,stype,sbits,eccbits,ssum,argc,argv)
int *l, *ll, *lp, *np, mask[], sbits[], *eccbits, *ssum, argc;
float *scale, *ber, *descale;
char ifile[], ofile[], stype[], *argv[];

{
	/*note:  ncsize, no, and gamma are external variables	*/

  int psum, rate, bits, rate2, i, j, flag, qflag, iflag, cflag;
  long tim;
  char prog[82], dir[82], host[20], temp[12];
  static char qtype[][10] = {"kang", "log", "max2", "arcsin", "none"},
              pfile[82] = "", qfile[82] = "", mfile[82] = "";
  struct stat statbuf;


  /*	*** parse the command line:
	celp [-i ifile] [-c chan] [-o ofile] [-p pfile] 
	     [-q qfile] [-m mfile] [-l lfile] 				*/

  if (argc > MAXARG + 1)
  {
    fprintf(stderr,"cli: Too many arguments %d\n",argc);
    fprintf(stderr,"%s%s\n","celp [-i ifile] [-o ofile] [-p pfile] ",
                            "[-q qfile] [-m mfile] [-l lfile]");
    fprintf(stderr,"%s\n","celp [-c chan] [-o ofile]");
  }
    
  if ((argc%2) == 0)
  {
    fprintf(stderr,"cli: Odd number of arguments %d\n",argc);
    fprintf(stderr,"%s%s\n","celp [-i ifile] [-o ofile] [-p pfile] ",
                            "[-q qfile] [-m mfile] [-l lfile]");
    fprintf(stderr,"%s\n","celp [-c chan] [-o ofile]");
    exit(1);
  }
  strcpy(prog, *argv);
  flag = iflag = cflag = FALSE;
  while (--argc > 0)
  {
    if (strncmp("-",*++argv,1) == 0)
    {
      switch (argv[0][1])
      {
        case 'i':
        case 'I':
          if (strncmp("-",*++argv,1) == 0)
          {
            fprintf(stderr,"cli: Missing input file\n");
            flag = TRUE;
            --argv;
          }
          else
          {
            strcpy(ifile,*argv);
            --argc;
	    iflag = TRUE;
          }
          break;
        case 'o':
        case 'O':
          if (strncmp("-",*++argv,1) == 0)
          {
            fprintf(stderr,"cli: Missing output file\n");
            flag = TRUE;
            --argv;
          }
          else
          {
            strcpy(ofile,*argv);
            --argc;
          }
          break;
        case 'p':
        case 'P':
          if (strncmp("-",*++argv,1) == 0)
          {
            fprintf(stderr,"cli: Missing parameter file\n");
            flag = TRUE;
            --argv;
          }
          else
          {
            strcpy(pfile,*argv);
            --argc;
          }
          break;
        case 'q':
        case 'Q':
          if (strncmp("-",*++argv,1) == 0)
          {
            fprintf(stderr,"cli: Missing quantization file\n");
            flag = TRUE;
            --argv;
          }
          else
          {
            strcpy(qfile,*argv);
            --argc;
          }
          break;
        case 'm':
        case 'M':
          if (strncmp("-",*++argv,1) == 0)
          {
            fprintf(stderr,"cli: Missing mask file\n");
            flag = TRUE;
            --argv;
          }
          else
          {
            strcpy(mfile,*argv);
            --argc;
          }
          break;
        case 'l':
        case 'L':
          if (strncmp("-",*++argv,1) == 0)
          {
            fprintf(stderr,"cli: Missing log file\n");
            flag = TRUE;
            --argv;
          }
          else
          {
            strcpy(lfile,*argv);
            --argc;
          }
          break;
	case 'c':
        case 'C':
          if (strncmp("-",*++argv,1) == 0)
          {
            fprintf(stderr,"cli: Missing channel file\n");
            flag = TRUE;
            --argv;
          }
          else
          {
            strcpy(ifile,*argv);
            --argc;
	    cflag = TRUE;
          }
          break;
        default:
          printf("cli: Bad switch %s\n",*argv);
          flag = TRUE;
      }
    }
    else
    {
      printf("cli: Bad switch %s\n",*argv);
      flag = TRUE;
    }
  }
  if (iflag && cflag)
  {
    fprintf(stderr, "%s\n", "cli: the -i switch is invalid with the -c switch");
    flag = TRUE;
  }
  if (flag)
  {
    fprintf(stderr,"%s%s\n","celp [-i ifile] [-o ofile] [-p pfile] ",
                            "[-q qfile] [-m mfile] [-l lfile]");
    fprintf(stderr,"%s\n","celp [-c chan] [-o ofile]");
    exit(1);
  }

  /*	*** read input parameter file if requested
		(all parameters are required)				*/

  if (*pfile)
  {
    fp = fopen(pfile,"r");
    if (fp == NULL)
    {
      perror("cli: Error opening the parameter file");
      exit(0);
    }
    fscanf(fp,"%d %d %d %d %d %d %f %f %f %f %d %f %s", &ncsize, l, ll, &no,
           lp, np, &gamma2, scale, descale, ber, &mxsw, &prewt, pstype);
    fclose(fp);
  }

  /*	*** read quantization characteristics file if requested
		(not all parameters are required)			*/

  if (*qfile)
  {
    fp = fopen(qfile,"r");
    if (fp == NULL)
    {
      perror("cli: Error opening the quantization characteristics file");
      exit(0);
    }
    while (fscanf(fp,"%s",temp) != EOF)
    {
      if (strcmp(temp,"cbgain") == 0)
      {
        fscanf(fp, "%s", cbgtype);
        if (strcmp(cbgtype, "none"))
          fscanf(fp, "%d", &cbgbits);
      }
      else if (strcmp(temp,"pitch") == 0)
      {
        fscanf(fp, "%s", ptype);
        if (strcmp(ptype, "none"))
          for (i = 0; i < *np + 2; i++)
            fscanf(fp, "%d", &pbits[i]);
      }
      else if (strcmp(temp,"spectrum") == 0)
      {
        fscanf(fp, "%s", stype);
        if (strcmp(ptype, "none") && strcmp(ptype, "opt"))
          for (i = 0; i < no; i++)
            fscanf(fp, "%d", &sbits[i]);
      }
    }
    fclose(fp);
  }

  /*	*** calculate data rate if fully quantized			*/

  /*	    (assuming spectrum update rate (ll) = spectrum analysis rate)
	    (assuming pitch update rate (lp) = pitch analysis rate)
	    (assuming 8khz sampling rate)				*/

  qflag = FALSE;
  if (strcmp(cbgtype, "none") && strcmp(ptype, "none") &&
      strcmp(stype, "none"))
  {
    qflag = TRUE;
    psum = 2*(pbits[0]+pbits[1]) + 4*(pbits[2]);
    for (*ssum = 0, i = 0; i < no; i++)
      *ssum += sbits[i];
    bits = nint(*ll / (*l) * log10((float)ncsize) / log10(2.) +
           *ll / (*l) * cbgbits + psum + *ssum + *eccbits);
    rate = nint(bits * 8000.0 / (*ll));

    /*	   1 bit for sync:						*/

    rate2 = rate + nint(1.0 * 8000.0 / (*ll));
  }

  /*	*** read bit error characteristics file if requested		*/

  if (*mfile)
  {
    fp = fopen(mfile, "r");
    if (fp == NULL)
    {
      perror("cli: Error opening the bit error characteristics file");
      exit(0);
    }
    for (i = 0; i < bits; i++)
      fscanf(fp, "%1d", mask[i]);
    fclose(fp);
  }	   

  /*	*** echo out parameters						*/

  gethostname(host,20);
  stat(prog, &statbuf); 
  getwd(dir);
  time(&tim);

  printf("             User: %s@%s\n", getlogin(), host);
  printf("             Time: %s", ctime(&tim));
  printf("          Program: %s\n", prog);
  printf("         Modified: %s", ctime(&statbuf.st_mtime));
  printf("        Directory: %s\n", dir);
  if (cflag)
  {
    printf("     Channel file: %s\n", ifile);
    printf("      Output file: %s\n", ofile);
  }
  else
  {
    printf("       Input file: %s\n", ifile);
    printf("      Output file: %s\n", ofile);
    printf("   Parameter file: %s\n", pfile);
    printf("Quantization file: %s\n", qfile);
    printf("BER and mask file: %s\n", mfile);
    printf("         Log file: %s\n", lfile);
    printf("           Code book size (ncsize)=%12d\n",ncsize);
    printf("              Code word length (l)=%12d\n",*l);
    printf("      LPC analysis frame size (ll)=%12d\n",*ll);
    printf("             LPC filter order (no)=%12d\n",no);
    printf("    Pitch analysis frame size (lp)=%12d\n",*lp);
    printf("         Pitch analysis order (np)=%12d\n",*np);
    printf(" Modified Excitation switch (mxsw)=%12d\n",mxsw);
    printf("Fractional pitch analysis (pstype)=%12s\n",pstype);
    printf("Prefilter weighting factor (prewt)=%12f\n",prewt);
    printf("    Noise weighting factor (gamma)=%12f\n",gamma2);
    printf("      Input scaling factor (scale)=%12f\n",*scale);
    printf("   Output scaling factor (descale)=%12f\n",*descale);
    printf("            %% bit error rate (BER)=%12f\n",*ber);
    if (strcmp(cbgtype, "none")) 
      printf(" CB Gain: %8s quantizer, bit allocation: %2d\n", cbgtype, cbgbits);
    if (strcmp(ptype, "none")) 
    {
      printf("   Pitch: %8s quantizer, bit allocation: ", ptype);
      for (i = 0; i < *np+2; i++)
        printf("%2d",pbits[i]);
      printf("\n");
    }
    if (strcmp(stype, "none")) 
    {
      printf("Spectrum: %8s quantizer, bit allocation: ", stype);
      for (i = 0; i < no; i++)
        printf("%2d",sbits[i]);
      printf("\n");
    }
    if (qflag) printf("Data rate = %4d bps %3d bpf (%4d bps with 1 bit sync)\n",
                    rate, bits, rate2);

    /*	*** verify parameters						*/

    if (ncsize > MAXNCSIZE) 
    {
      fprintf(stderr, "cli:  Error - ncsize too big\n");
      exit(1);
    }
    if (*l > MAXL)           
    {
      fprintf(stderr, "cli:  Error - l too big\n");
      exit(1);
    }
    if (*ll > MAXLL)         
    {
      fprintf(stderr, "cli:  Error - ll too big\n");
      exit(1);
    }
    if (no > MAXNO)         
    {
      fprintf(stderr, "cli:  Error - no too big\n");
      exit(1);
    }
    if (*lp > MAXLP)         
    {
      fprintf(stderr, "cli:  Error - lp too big\n");
      exit(1);
    }
    if (*np > MAXNP)         
    {
      fprintf(stderr, "cli:  Error - np too big\n");
      exit(1);
    }
    if (gamma2 > 1.0)        
    {
      fprintf(stderr, "cli:  Error - gamma > 1\n");
      exit(1);
    }
    if (*scale < 0.0)        
    {
      fprintf(stderr, "cli:  Error - scale < 0\n");
      exit(1);
    }
    if (*ber < 0.0)     
    {
      fprintf(stderr, "cli:  Error - ber < 0\n");
      exit(1);
    }
    if (*ll%*l) 
    {
      fprintf(stderr, "cli:  *** Error - ll & l are inconsistent\n");
      exit(1);
    }
    if (*ll%*lp) 
    {
      fprintf(stderr, "cli:  *** Error - ll & lp are inconsistent\n");
      exit(1);
    }
    if (cbgbits < 5 || cbgbits > 7) 
    {
      fprintf(stderr, "cli:  Error - bad cbgbits (not implemented)\n");
      exit(1);
    }
    if (pbits[0] > 8) 
    {
      fprintf(stderr, "cli:  *** Error - bad pbits[0] (not implemented)\n");
      exit(1);
    }
    if (pbits[1] > 7 || pbits[1] < 3) 
    {
      fprintf(stderr, "cli:  *** Error - bad pbits[1] (not implemented)\n");
      exit(1);
    }
    for (i = 2; i < *np+2; i++)
    {
      if (pbits[i]  <  3 || pbits[i] > 5)    
      {
        fprintf(stderr, "cli:  *** Error - bad pbits (not implemented)\n");
        exit(1);
      }
    }
    flag = TRUE;
    for (j = 0; j < MAXQTYPE; j++)
    {
      if (strcmp(cbgtype, qtype[j]) == 0) flag = FALSE;
    }
    if (flag) 
    {
      fprintf(stderr, "cli:  *** Error - bad cbgain quantizer type\n");
      exit(1);
    }
    flag = TRUE;
    for (j = 0; j < MAXQTYPE; j++)
    {
      if (strcmp(ptype, qtype[j]) == 0) flag = FALSE;
    }
    if (flag) 
    {
      fprintf(stderr, "cli:  *** Error - bad pitch quantizer type\n");
      exit(1);
    }
    flag = TRUE;
    for (j = 0; j < MAXQTYPE; j++)
    {
      if (strcmp(stype, qtype[j]) == 0) flag = FALSE;
    }
    if (flag) 
    {
      fprintf(stderr, "cli:  *** Error - bad spectrum quantizer type\n");
      exit(1);
    }

    /*	*** generate log file if requested				*/

    if (strcmp(lfile, "none"))
    {  
      if (strcmp(lfile, "celp.log") == 0)
      {
        fp = fopen(lfile, "a");
        if (fp == NULL)
        {
          perror("cli: Error opening the log file");
          exit(0);
        }
      }
      else
      {
        fp = fopen(lfile, "w");
        if (fp == NULL)
        {
          perror("cli: Error opening the log file");
          exit(0);
        }
      }
      fprintf(fp, "             User: %s@%s\n", getlogin(), host);
      fprintf(fp, "             Time: %s", ctime(&tim));
      fprintf(fp, "          Program: %s\n", prog);
      fprintf(fp, "         Modified: %s", ctime(&statbuf.st_mtime));
      fprintf(fp, "        Directory: %s\n", dir);
      fprintf(fp, "       Input file: %s\n", ifile);
      fprintf(fp, "      Output file: %s\n", ofile);
      fprintf(fp, "   Parameter file: %s\n", pfile);
      fprintf(fp, "Quantization file: %s\n", qfile);
      fprintf(fp, "BER and mask file: %s\n", mfile);
      fprintf(fp, "         Log file: %s\n", lfile);
      fprintf(fp, "           Code book size (ncsize)=%12d\n",ncsize);
      fprintf(fp, "              Code word length (l)=%12d\n",*l);
      fprintf(fp, "      LPC analysis frame size (ll)=%12d\n",*ll);
      fprintf(fp, "             LPC filter order (no)=%12d\n",no);
      fprintf(fp, "    Pitch analysis frame size (lp)=%12d\n",*lp);
      fprintf(fp, "         Pitch analysis order (np)=%12d\n",*np);
      fprintf(fp, " Modified Excitation switch (mxsw)=%12d\n",mxsw);
      fprintf(fp, "Fractional pitch analysis (pstype)=%12s\n",pstype);
      fprintf(fp, "Prefilter weighting factor (prewt)=%12f\n",prewt);      
      fprintf(fp, "    Noise weighting factor (gamma)=%12f\n",gamma2);
      fprintf(fp, "      Input scaling factor (scale)=%12f\n",*scale);
      fprintf(fp, "   Output scaling factor (descale)=%12f\n",*descale);
      fprintf(fp, "            %% bit error rate (BER)=%12f\n",*ber);
      if (strcmp(cbgtype, "none")) 
        fprintf(fp, " CB Gain: %8s quantizer, bit allocation: %2d\n",
                   cbgtype, cbgbits);
      if (strcmp(ptype, "none")) 
      {
        fprintf(fp, "   Pitch: %8s quantizer, bit allocation: ", ptype);
        for (i = 0; i < *np+2; i++)
          fprintf(fp, "%2d",pbits[i]);
        fprintf(fp, "\n");
      }
      if (strcmp(stype, "none")) 
      {
        fprintf(fp, "Spectrum: %8s quantizer, bit allocation: ", stype);
        for (i = 0; i < no; i++)
          fprintf(fp, "%2d",sbits[i]);
        fprintf(fp, "\n");
      }
      if (qflag)
        fprintf(fp, "Data rate = %4d bps %3d bpf (%4d bps with 1 bit sync)\n",
                  rate, bits, rate2);
    }
  }
}
/**************************************************************************
*
* ROUTINE
*               cliend
*
* FUNCTION
*               End of command line interface entry.
*		Report and log job characteristics and performance.
* 
* SYNOPSIS
*
*	entry cliend(sumsnr, framesnr, realerror,
*     &	            sumdm, framedm, sumdm2, framedm2)
*
*   formal 
*
*			data	I/O
*	name		type	type	function
*       ------------------------------------------------------------------
*	sumsnr		float	i	sum of segmental signal-to-noise
*	framesnr	int	i	count of frames used for snr sum
*	realerror	float	i	bit error rate
*	sumdm		float	i	sum of speech distortion measures
*						(see dist.c)
*	framedm		int	i	count of frames used for sumdm
*	sumdm2		float	i	sum of spectral distortion measures
*						(see dist.c)
*	framedm2	int	i	count of frames used for sumdm2
*
**************************************************************************/
#include <sys/times.h>
cliend(sumsnr, framesnr, realerror, sumdm, framedm, sumdm2, framedm2)
int framesnr, framedm, framedm2;
float sumsnr, realerror, sumdm[], sumdm2[];
{
  long tim;
  struct tms buffer;
  /*			*calculate run time, print time & date		*/
  
  time(&tim);
  times(&buffer);
  printf("Finished processing %d frames of data on %s", frame, ctime(&tim));

  /*			*actual bit error rate				*/
  
  printf("Overall bit error rate = %8.4f%%\n", realerror);

  /*			*average segmental signal-to-noise		*/
  
  if (framesnr <= 0) framesnr = 1;
  if (sumsnr / framesnr > 0.0)
    printf("Segmental SNR = %8.4f dB (%d subframes averaged for SNR)\n",
           10 * log10(sumsnr/framesnr),framesnr);
  else 
    printf("Segmental SNR = %8.4f    (%d subframes averaged for SNR)\n",
           sumsnr/framesnr,framesnr);

  /*			*speech distortion				*/
  
  if (framedm != 0.0)
  {
    printf("Speech distortion:\n");
    printf("likelihood measures   %7.4f     %7.4f\n",
           sumdm[0]/framedm,sumdm[1]/framedm);
    printf("cosh measures         %7.4f dB, %7.4f dB, %7.4f dB, %7.4f dB\n",
           sumdm[2]/framedm, sumdm[3]/framedm,
           sumdm[4]/framedm, sumdm[5]/framedm);
    printf("cepstral distance     %7.4f dB, %7.4f dB, %7.4f dB\n",
           sumdm[6]/framedm,sumdm[7]/framedm,sumdm[8]/framedm);
    printf("average distance      %7.4f dB\n", sumdm[9]/framedm);
  }
  printf("%d subframes averaged for distance measures\n", framedm);

  /*			*spectral distortion				*/
  
  if (framedm2 != 0.0)
  {
    printf("Spectral distortion:\n");
    printf("likelihood measures   %7.4f     %7.4f\n",
           sumdm2[0]/framedm2,sumdm2[1]/framedm2);
    printf("cosh measures         %7.4f dB, %7.4f dB, %7.4f dB, %7.4f dB\n",
           sumdm2[2]/framedm2, sumdm2[3]/framedm2,
           sumdm2[4]/framedm2, sumdm2[5]/framedm2);
    printf("cepstral distance     %7.4f dB, %7.4f dB, %7.4f dB\n",
           sumdm2[6]/framedm2,sumdm2[7]/framedm2,sumdm2[8]/framedm2);
    printf("average distance      %7.4f dB\n", sumdm2[9]/framedm2);
  }
  printf("%d subframes averaged for distance measures\n", framedm2);
  
  /*			*print runtime, etc. (& ring bell)		*/
  
  printf("Elapsed times (s):  %.1f user, %.1f system, %.1f total\7\n",
          buffer.tms_utime/60., buffer.tms_stime/60.,
          (buffer.tms_utime + buffer.tms_stime)/60.);

  /*			*write above to log file			*/
  
  if (strcmp(lfile, "none"))
  {
    fprintf(fp, "Finished processing %d frames of data on %s",
                 frame, ctime(&tim));

    fprintf(fp, "Overall bit error rate = %8.4f%%\n", realerror);

    if (sumsnr/framesnr > 0.0)
      fprintf(fp, "Segmental SNR = %8.4f dB (%d subframes averaged for SNR)\n",
                   10 * log10(sumsnr/framesnr), framesnr);
    else 
      fprintf(fp, "Segmental SNR = %8.4f    (%d subframes averaged for SNR)\n",
                   sumsnr/framesnr, framesnr);

    if (framedm != 0.0)
    {
      fprintf(fp, "Speech distortion:\n");
      fprintf(fp, "likelihood measures %7.4f     %7.4f\n",
                 sumdm[0]/framedm,sumdm[1]/framedm);
      fprintf(fp, "cosh measures       %7.4f dB, %7.4f dB, %7.4f dB, %7.4f dB\n",
                 sumdm[2]/framedm, sumdm[3]/framedm,
                 sumdm[4]/framedm, sumdm[5]/framedm);
      fprintf(fp, "cepstral distance   %7.4f dB, %7.4f dB, %7.4f dB\n",
                 sumdm[6]/framedm,sumdm[7]/framedm,sumdm[8]/framedm);
      fprintf(fp, "average distance    %7.4f dB\n", sumdm[9]/framedm);
    }

    fprintf(fp, "%d subframes averaged for distance measures\n", framedm);

    if (framedm2 != 0.0)
    {
      fprintf(fp, "Spectral distortion:\n");
      fprintf(fp, "likelihood measures   %7.4f     %7.4f\n",
                 sumdm2[0]/framedm2,sumdm2[1]/framedm2);
      fprintf(fp, "cosh measures       %7.4f dB, %7.4f dB, %7.4f dB, %7.4f dB\n",
                 sumdm2[2]/framedm2, sumdm2[3]/framedm2,
                 sumdm2[4]/framedm2, sumdm2[5]/framedm2);
      fprintf(fp, "cepstral distance   %7.4f dB, %7.4f dB, %7.4f dB\n",
                 sumdm2[6]/framedm2,sumdm2[7]/framedm2,sumdm2[8]/framedm2);
      fprintf(fp, "average distance    %7.4f dB\n", sumdm2[9]/framedm2);
    }

    fprintf(fp, "%d subframes averaged for distance measures\n", framedm2);

    fprintf(fp, "Elapsed times (s):  %f user, %f system, %f total\6\n",
                 buffer.tms_utime/60., buffer.tms_stime/60.,
                 (buffer.tms_utime + buffer.tms_stime)/60.);
    fprintf(fp, "%s%s\n","...................................",
                         "...................................");
    fclose(fp);
  }

}
