/**************************************************************************
*
* ROUTINE
*		autohf
*
* FUNCTION
*		LPC autocorrelation analysis with high frequency compensation
*
*	        NOTE:  If high frequency correction is not used (i.e., lambda=0),
*	               faster procedures may be used (e.g., Durbin's recursion
*                      or LeRoux & Guegen fixed point method.
*
* SYNOPSIS
*		subroutine autohf(si, w, n, p, lambda, omega, a, rc)
*
*   formal 
*                       data    I/O
*       name            type    type    function
*       -------------------------------------------------------------------
*	si(n)		float	i	signal input
*	w(n)		float	i	window (i.e., Hamming)
*	n		int	i	length of input sequence
*	p		int	i	order of LPC polynomial
*	lambda		float	i	hF correction scale factor
*	omega		float	i	bandwidth expansion factor
*	a		float	o	LPC coefficients (1 to m+1)
*	rc		float	o	reflection coefficients (1 to m)
*					(voiced-> +rc1)
*
*   external 
*                       data    I/O
*       name            type    type    function
*       -------------------------------------------------------------------
*	frame		int	i	
*
***************************************************************************
*	
* DESCRIPTION
*
*  	Subroutine to perform HF corrected autocorrelation LPC analysis.
*	First, autocorrelation coefficients are calculated and high
*	frequency corrected to partially compensate for the analog
*	antialiasing filter*.  (Traditionally, this technique has only been
*	applied to covariance analysis, but it applies to autocorrelation
*	analysis as well).  Next, the autocorrelation function is converted
*	to reflection coefficients by the Schur recursion (aka LeRoux &
*	Guegen).  Then, the reflection coefficients are converted to LPC
*	predictor coefficients.  Finally, the predictors are bandwidth
*	expanded by omega.
*
*	CELP's LPC predictor coefficient convention is:
*              p+1         -(i-1)
*       A(z) = SUM   a   z          where a  = +1.0
*              i=1    i                    1
*
*	The sign convention used defines the first reflection coefficient
*	as the normalized first autocorrelation coefficient, which results
*	in positive values of rc(1) for voiced speech.
*
***************************************************************************
*
* CALLED BY
*
*	celp
*
* CALLS
*
*	actorc	bwexp	cor	pctorc	rctopc
*
***************************************************************************
*	
* REFERENCES
*
*	*Atal & Schroeder, Predictive Coding of Speech Signals
*	 and Subjective Error Criteria, IEEE TASSP, June 1979.
*
**************************************************************************/
#define	TRUE	1
#define	FALSE	0
#include <math.h>
#include <stdio.h>
#include "ccsub.h"
extern int frame;
autohf(si, w, n, p, lambda, omega, a, rc)
int n, p;
float si[], w[], lambda, omega, a[], rc[];
{
  int i, unstable;
  float c0, c[MAXNO], alpha, lemin, atemp[MAXNO+1], s[MAXLL];
  
  setr(MAXNO+1, 0.0, atemp);
  for (i = 0;  i < n ; i++)
    s[i] = si[i] * w[i];		/* apply window			*/
  unstable = FALSE;	
  cor(s, n, p, &c0, c);			/* calculate autocorrelations	*/
  if (c0 < 0.0) 
    unstable = TRUE;
  actorc(c0, c, rc, p, &alpha);		/* convert autocorrelations	*/
  					/* to rc's			*/
  lemin = lambda * c0 * alpha;		/* unnormalized prediction 	*/
  					/* error scaled by lambda	*/
  c0 = c0 + 0.375 * lemin;
  c[0] = c[0] - 0.25 * lemin;		/* high frequency correction	*/
  c[1] = c[1] + 0.0625 * lemin;		/* (eq. 16 Atal & Schroeder)	*/
  actorc(c0, c, rc, p, &alpha);		/* convert corrected		*/
  					/* autocorrelations to rc's	*/
  rctopc(rc, atemp, p);			/*convert corrected rc's to pc's*/
  bwexp(omega, atemp, a, p);		/* expand corrected pc's	*/
  pctorc(a, rc, p);			/* match rc's to expanded pc's	*/
  					/* and test for stability	*/
  for (i = 0; i < p ; i++)
  {
    if (fabs(rc[i]) > 1.0)
      unstable = TRUE;
  }
  if (unstable)
  {
    printf("autohf:  unstable lpc analysis at frame %d\n",frame);
    for (i = 0; i < p; i++)
    {
      a[i+1] = 0.0;
      rc[i] = 0.0;
    }
  }
}
