c==========================================================================
c
c ROUTINE
c               decodeham
c
c FUNCTION
c               This subroutine decodes the bitstream generated by 
c	encodeham.  It will correct a single error, and detect 2 
c	errors.
c               
c
c SYNOPSIS
c               subroutine decodeham(codelength1,codelength2,hmatrix,
c		     syndrometable,paritybit,codeword,twoerror,synflag)
c
c   formal 
c
c                       data    I/O
c       name            type    type    function
c       -------------------------------------------------------------------
c       codelength1	int	i	number of data bits
c	codelength2	int	i	number of information bits
c	hmatrix		int	i	vector to encode an decode by
c	syndrometable	int	i	errormasks used to correct single
c					errors
c	paritybit	int	i	overall parity bit
c	codeword	int	i/o	encoded/decoded stream 
c	twoerror	log	o	flag for 2 error detect
c	synflag		int	o	value 0 or 1, 1 if syndrome .ne. 0
c	
c
c==========================================================================
c
c DESCRIPTION
c
c		This subroutine is part of a set of subroutines which perform
c	a Generalized Hamming Code.  As you know, Hamming codes are perfect
c	codes and can only detect and correct one error.  We added an overall
c 	parity checkbit, which allows us to detect 2 errors.  When 2 errors 
c	are detected, (in subroutine dcodham.f) no correction attempt is
c	made.  This would most likely result in more errors.  Instead, a flag
c	is sent to the calling program notifying it of multiple errors so
c	that smoothing may be attempted.  The Hamming codes presently supported
c	by the routines are (63,57), (31,26), (15,11), and shortened variations
c	thereof.  It could be made even more general by making minor modifications
c	to the dectobin.f subroutine.  This routine at present will calculate 
c	a maximum of 6 bits.
c
c	Hamming routines consist of the following files:
c
c		matrixgen - generates the hmatrix and sydrometable.
c		dectobin - does a simple decimal to binary conversion.
c		encodeham - generates the code word and overall paritybit.
c		decodeham - recovers infobits, checks for errors, corrects 1
c					error, and sends out flag for smoothing.
c
c 
c		This subroutine, decodeham, is responsible for checking for errors,
c	correcting the error if there is only one, and sending a smoothing flag
c	to the calling routine if there is more than one.
c
c==========================================================================
c	
c REFERENCES
c
c	Lin and Costello : Error Control Coding
c	Berlekamp : Algebraic Coding Theory
c
c**************************************************************************
c
	subroutine decodeham(codelength1,codelength2,hmatrix,
     1  	syndrometable,paritybit,codeword,twoerror,synflag)
	implicit undefined(a-z)
	integer codelength1,codelength2,hmatrix(codelength1)
	integer syndrometable(codelength1),codeword(codelength1)
	integer paritybit, parityflag, errorflag, synflag, i, j, temp3
	logical twoerror
c
	twoerror=.false.
	errorflag=0
	parityflag=1
c			
c	This part of the routine checks the overall parity of the code word
c	and compares it with the overall paritybit sent.  If they are not the
c	same that means there is at least one error.  If, later on in the routine,
c	the syndrome check indicates that there is an error and the parity is
c	correct in this part of the routine, that indicates there are two errors.
c	One of the weaknesses of this method is that there is no way of knowing
c	if we have 3,5,7,... errors.  We always smooth if there are 2,4,6,...
c	errors.
c
	if (parityflag.eq.1) then
	   synflag=0
	   do 10 i=1,codelength1
	      synflag= synflag .xor. codeword(i)
10	   continue
	   if (paritybit.ne.synflag)errorflag=errorflag+1
	end if
c
c	This part of the routine generates the syndrome.  The syndrome will
c	equal zero if there are no errors.  synflag accumulates the syndrome
c	and is used as the offset in the syndrome table, which tells the 
c	routine which bit is in error.
c
	synflag=0
	temp3=0
	do 30 i=1,codelength1
	   if(codeword(i).ne.0)synflag = synflag .xor. hmatrix(i)
	   if(codeword(i).ne.0)temp3 = temp3 + 1
30	continue
c
c	*** Check to see if the parityflag is set and if it is then check
c	to see if the parity bit was in error.
c	If the parityflag was set and there was an error in the syndrome,
c	the errorflag should equal 1.
c	If it doesn't, then there are more errors than can be corrected
c	and the infobits are passed on unchanged.
c
	if (synflag.ne.0)then
	   if(errorflag.ne.1 .and. parityflag .eq. 1)then
	      twoerror = .true.
	      goto 20
	   end if
	   j=syndrometable(synflag)
	   codeword(j)=codeword(j) .xor. 1
	end if
c
c	*** If the syndrome is equal to zero and the errorflag is set
c	(not likely, but must be checked) then more than one error has
c	occured, but it cannot be corrected, so I pass on the infobits
c	the same as if there were no errors.
c
20	return
	end
