C==========================================================================
C
C ROUTINE
C		codebook (main)
C
C FUNCTION
C		Generate code book vector for CELP coder
C
C==========================================================================
C
C DESCRIPTION
C
C	Generate a sparse ternary code book vector x by center clipping
C	and limiting a zero-mean unit-variance Gaussian sequence.  For
C	a desired sparsity, the center clipping threshold "THRESH" is
C	determined from F (the cumulative distribution function of a
C	standardized normal random variable):
C
C		sparsity = 1 - 2[1 - F(THRESH)]
C
C		Sparsity	THRESH
C		--------	------
C		75%		1.15
C		77%		1.2
C		86.6%		1.5
C		90%		1.645
C
C==========================================================================
C	
C REFERENCES
C
C	CRC Standard Mathematical Tables,
C	The Normal Probability Function and Related Functions.
C
C==========================================================================
C 
C FILES
C
C	The data is output in file "codebook.h"
C
C**************************************************************************
C
	program codebook
	implicit undefined(a-z)
	integer M, L, MAXCODE, WIDTH, i, j, k
	parameter (M = 512, L = 60)
	parameter (MAXCODE = 2*(M-1)+L)
	parameter (WIDTH = 20)
	real x(0:MAXCODE+1), THRESH
	parameter (THRESH = 1.2)
c
c			*open file for code book data
	open(unit=10, file='codebook.h', status='new')
	k = 0
	do 50 i = 0, MAXCODE/2
c			*get a pair of G(0,1) samples
	   call noise(x(k), x(k+1))
c			*center clip ternary samples
	   do 20 j = 0, 1
	      if (abs(x(k+j)) .lt. THRESH) then
	         x(k+j) = 0.0
	      else
	         x(k+j) = sign(1.0, x(k+j))
	      end if
20	   continue
	   k = k + 2
50	continue
c			*write samples to file (skip first & last sample)
	k = 1
	do 80 i = 1, MAXCODE/WIDTH
c	   write (10, 90) (int(x(j)), j=k,k-1+WIDTH)
	   write (10, 90) (x(j), j=k,k-1+WIDTH)
	   k = k + WIDTH
80	continue
c	write (10, 91) (int(x(j)), j=k,k-1+mod(MAXCODE,WIDTH))
	write (10, 91) (x(j), j=k,k-1+mod(MAXCODE,WIDTH))
	stop 'codebook.h generated'
c90	format(1x, 20(i3,','))
90	format(1x, 20(f4.0,','))
c91	format(1x, i3, ',', i3)
91	format(1x, f4.0, ',', f4.0)
	end
C==========================================================================
C
C ROUTINE
C               noise
C
C FUNCTION
C
C               Generates gaussian noise using the polar method.
C SYNOPSIS
C               noise(x1, x2)
C   formal 
C
C                       data    I/O
C       name            type    type    function
C       -------------------------------------------------------------------
C       x1              r       o       a sample of noise source
C       x2              r       o       another sample of noise source
C
C==========================================================================
C       
C USAGE
C
C	noise generates two samples of a Gaussian noise
C	source for each call using the polar method.
C
C==========================================================================
C
C REFERENCES
C
C       Knuth, The Art of Programming, Volume 2
C
C**************************************************************************
C*-
	subroutine noise (x1, x2)
	implicit undefined(a-z)
	real x1, x2
	integer random, i, j
	real f(2), f1, f2, s
c
c	f(i) are samples from a uniform distribution
c	in the range of 0.0 inclusive to 1.0 inclusive. 
c
10	do 30 i = 1, 2
	   do 20 j = 1, 4
	      f(i) = (float(random()+32768))/65535.
20	   continue
30	continue
	f1 = 2.*f(1) - 1.
	f2 = 2.*f(2) - 1.
	s  = f1*f1 + f2*f2
	if (s .ge. 1.) goto 10
	s  = sqrt(-2.*alog(s)/s)
	x1 = f1*s
	x2 = f2*s
	return
	end
C==========================================================================
C
C ROUTINE
C		random
C
C FUNCTION
C
C		Pseudo random number generator.
C
C SYNOPSIS
C		function random()
C
C   formal 
C                       data    I/O
C       name            type    type    function
C       -------------------------------------------------------------------
C	random		i	fun	uniformly distributed
C					over -32768 to 32767
C==========================================================================
C	
C DESCRIPTION
C
C	See reference.
C
C==========================================================================
C	
C REFERENCE
C
C	Knuth, The Art of Programming, Volume 2, p. 27.
C
C==========================================================================
C*-
	function random ()
	implicit undefined(a-z)
	integer random
	integer MIDTAP, MAXTAP
	parameter (MIDTAP=2, MAXTAP=5)
	integer y(MAXTAP), j, k, temp
	save y, j, k
	data y /-21161, -8478, 30892, -10216, 16950/
	data j/MIDTAP/, k/MAXTAP/
c
c	simulate 2's complement 16-bit addition
c
	temp = and (y(k) + y(j), 65535)
cAlli	temp = iand (y(k) + y(j), 65535)
	if (temp .gt. 32767) temp = temp - 65536
	y(k) = temp
	random = temp
	k = k - 1
	if (k .le. 0) k = MAXTAP
	j = j - 1
	if (j .le. 0) j = MAXTAP
	return
	end

