;; =======================================================================
;;                       Window FrameGraphics Library for X11
;; 
;;  CopyRight(c) Robert L. Joseph
;; =======================================================================


(in-package "PG" :use '("LISP" "XLIB" "USER"))

(export '(pg-refresh-window    pg-create-window       pg-kill-window
	  pg-clear-window      pg-show-window         pg-hide-window
	  pg-window-height     pg-window-width        pg-draw-line 
	  pg-write-text        pg-text-width          pg-text-height
	  pg-with-window       pg-frame-rect          pg-invert-rect
	  pg-erase-rect        pg-window-struct       pg-init-graphics
          pg-move              pg-move-to             pg-line
	  pg-window-name       pg-set-window-name     pg-string-height
          pg-line-to           pg-draw-string         pg-string-width 
          default-event-handler  current-host	      win-p
          win-gc               pg-X-rect-draw         pg-font-height
	  *DISPLAY*	       *BG-PIX*		      *FG-PIX*
	  *ROOT*	       *SCREEN*		      *FG-COLOR*
	  *BG-COLOR*		build-point	      point-v
	  point-h	       point-p               copy-point
          *FONT*	       pg-paint-rect
))

(defvar *GRAPHICS-INIT-FLAG* nil "Flag that indicates graphics have already been inited.")
; ========================================================================
;                          Window  Data  Type
; ========================================================================

(defvar *event-debug* nil)
(defparameter *exposure-event-mask*
  (xlib:make-event-mask :exposure :structure-notify :Button-Press
      :Button-Release :Button-Motion :Enter-window
;     :Button-Motion
  ))

(defun toggle-backingstore ()
   (declare (special user::*tree-window*))
    "This function will help us test the usefulness of a backing store."
   (if (boundp 'user::*tree-window*)
      (if (eq (xlib:window-backing-store (win-struct user::*tree-window*))
				:always)
          (setf (xlib:window-backing-store (win-struct user::*tree-window*))
				:not-useful)
	  (setf (xlib:window-backing-store (win-struct user::*tree-window*))
                                :always)
      )
   )
       )

(defstruct (win (:print-function (lambda (structure stream z)
			(declare (ignore z))
                        (format stream "#<WIN: ~S>"
                                (getf (xlib:window-plist (win-struct
                                                        structure)) 'name)
                        )
				 )))
  struct
  gc
  pen-x
  pen-y
  char-hght
  char-wdth)

(defmacro pg-window-struct (w)
  `(win-struct ,w))

(defmacro handler-arg (w)
   (declare (type xlib:window w))
`   (getf (xlib:window-plist ,w) 'handler-arg)
)

; ========================================================================
;                      Prodigy  Graphics  Functions...
; ========================================================================
;;; The current implementation permits only one font.

(defvar *FG-DISPLAY* ())
(defvar *FG-SCREEN*  ())
(defvar *FG-ROOT*    ())
(defvar *FG-BG-PIX*  ())
(defvar *FG-FG-PIX*  ())
(defvar *FG-COLOR-MAP* NIL "Global FrameGraphics colormap")
(defvar *FG-FG-COLOR* nil "Global FrameGraphics foreground color.")
(defvar *FG-BG-COLOR* nil "Global FrameGraphics background color.")
(defvar *FG-FONT* nil "Global FrameGraphics font.")




(defun pg-init-graphics (host font-name)
    (declare (special *DISPLAY* *SCREEN* *ROOT* *BG-PIX* *FG-PIX* *FONT*
		      *FONT-WIDTH* *FONT-HEIGHT* *GRAPHICS-INIT-FLAG*
		      *FG-COLOR* *BG-COLOR* *COLOR-MAP*)
	     (type display *DISPLAY*)
	     (type screen *SCREEN*)
	     (type window *ROOT*)
	     (type font *FONT*)
	     (type card32 *BG-PIX* *FG-PIX*)
	     (type colormap *COLOR-MAP*)
	     (type color *FG-COLOR* *BG-COLOR*)
	     (string font-name))
    "This function opens the server, the screen and a sets a few parameters as
     special variables."

#+:franz-inc (eval-when (compile eval load) (require :clx))

(unless *GRAPHICS-INIT-FLAG*

        (setf *DISPLAY* (xlib:open-display host))
        (setf *SCREEN* (xlib:display-default-screen *DISPLAY*))
        (setf *ROOT* (xlib:screen-root *SCREEN*))
        (setf *BG-PIX* (xlib:screen-white-pixel *SCREEN*))
        (setf *FG-PIX* (xlib:screen-black-pixel *SCREEN*))
	(setf *FONT* (xlib:open-font *DISPLAY* font-name))
        (setf *FONT-WIDTH* (pg-string-width *FONT* "A"))
	(setf *FONT-HEIGHT* (pg-string-height *FONT* "I"))
	(setf *COLOR-MAP* (xlib:screen-default-colormap *SCREEN*))
	(setf *FG-COLOR* (car (xlib:query-colors *COLOR-MAP* 
							(list *FG-PIX*))))
	(setf *BG-COLOR* (car (xlib:query-colors *COLOR-MAP* 
							(list *BG-PIX*))))
	(setf *GRAPHICS-INIT-FLAG* t) ; never init again.

)
   
)


(defun pg-internal-no-op (&rest x)
   (declare (ignore x))
   "Do nothing for the pg system."
nil)
;;; this function creates an x11 window and returns a win structure.
;;; the first four key word args are event handling functions.  See 
;;; the code just before event-case to understand the args that they
;;; take.  The override allows direct placement of the window under
;;; twm in x11R3, and uspp is for R4.  The system automatically figures
;;; out what you want to do, so don't worry about which system you 
;;; are using.  Handler-arg is (one of) the args passed to the handler
;;; functions.  If it is unspecified (or specified as nil) then the 
;;; window in which the event occurs is passed.  Name is the name on
;;; on the window.


(defun pg-create-window (x y w h &key (exposure-function 'pg-internal-no-op)
				      (button-press 'pg-internal-no-op)
				      (config-function 'pg-internal-no-op)
				      (enter-function 'pg-internal-no-op)
				      (override :off override-default-p)
				      (uspp nil)
				      (handler-arg nil)
				      (name "NoName"))



	(declare (special *DISPLAY* *SCREEN* *ROOT* *BG-PIX* *FG-PIX* *FONT*)
	         (ignore override-default-p))

  ;;; release 3 should use the override keyword :on to place directly 
  ;;; windows, while release 4 should use uspp t and set things on the
  ;;; property list.  The semantics of these two system seem to have been
  ;;; switch between release 3 and release 4 under twm.

  (when (= 3 (xlib:display-release-number *display*))
     (when uspp
	(setf override :on)
	(setf uspp nil))
  )

  (let* ((chght (xlib:text-width *FONT* "A"))
	 (cwdth (xlib:text-extents *FONT* "I"))
	 (wi (xlib:create-window :parent *ROOT*
				 :x x :y y
				 :width w :height h
				 :border *FG-PIX*
				 :background *BG-PIX*
				 :border-width 1
		                 :event-mask *exposure-event-mask*
				 :override-redirect override
	     ))
	 (gc (xlib:create-gcontext :drawable wi
				   :foreground *FG-PIX*
				   :background *BG-PIX*
				   :line-width 0 
				   :line-style :solid
				   :font *FONT*
				   :cap-style :round
				   :fill-style :solid
				   :arc-mode :chord
				   :exposures :on))
	 (win (make-win :struct wi 
			:gc gc
			:char-hght chght
			:char-wdth cwdth
			:pen-x 0
			:pen-y 0)))
#+:cmu	(ext::enable-clx-event-handling *DISPLAY*
   				       'default-event-handler)

  (xlib:set-standard-properties wi
     :name (format nil "~A" name)
     :icon-name (format nil "~A" name)
  )

  ;;; this also is to make twm work correctly when running under
  ;;; x11 r3 or x11 r4 (ie, don't use under r3)

  (when (/= 3 (xlib:display-release-number *display*))
   (xlib:set-standard-properties wi 
       :user-specified-position-p uspp
       :x x :y y)
  )

   (when *event-debug* (format t "PG-CREATE-WINDOW:Window name~A~%" name)
	)
   (if (not handler-arg) (setf handler-arg wi))
   (setf (getf (xlib:window-plist wi) 'handler-arg) handler-arg)
   (setf (getf (xlib:window-plist wi) 'name) name)
   (setf (getf (xlib:window-plist wi) 'Exposure) exposure-function)
   (setf (getf (xlib:window-plist wi) 'ButtonPress) button-press)
   (setf (getf (xlib:window-plist wi) 'Config) config-function)
   (setf (getf (xlib:window-plist wi) 'Enter) enter-function)
    win))

(defun win-disp (win)
    (xlib:window-display (win-struct win)))

(defun win-font (win)
    (xlib:gcontext-font (win-gc win)))

(defun win-bg (win)
    (xlib:gcontext-background (win-gc win)))

(defun win-fg (win)
    (xlib:gcontext-foreground (win-gc win)))

(defun win-wdth (win)
	(xlib:drawable-width (win-struct win)))

(defun win-hght (win)
	(xlib:drawable-height (win-struct win)))


(defun Map-Notify (event-debug event-window)
  (declare (ignore event-window))
  (when event-debug (format t "Map-notify~%"))
  t)

(defun Unmap-Notify (event-debug event-window)
  (declare (ignore event-window))
  (when event-debug (format t "Unmap-notify~%"))
  t)

(defun Enter-Notify (event-debug event-window)
   (when event-debug (format t "Enter-notify~%"))
     (call-specific-event-handler event-window
					 'Enter (handler-arg event-window))
;   (set-current-window (getf (xlib:window-plist event-window) 'frame))
  t)

(defun Circulate-Notify (event-debug)
  (when event-debug (format t "Circulate-notify~%"))
  t)

(defun Gravity-Notify (event-debug)
  (when event-debug (format t "gravity-notify~%"))
  t)

(defun Reparent-Notify (event-debug x y)
    (when event-debug (format t "reparent-notify ~s ~s~%" x y))
  t)

;;;       snarfed in it's entirety from hemlock code
;;; removes any events still drifting about on the queue that are meant for
;;; the window that is about to be destroyed
(defun deleting-window-drop-events (display win)
  (xlib:display-finish-output display)
  
  (let ((result nil))
    (xlib:process-event
     display :timeout 0
     :handler #'(lambda (&key event-window window &allow-other-keys)
		  (if (or (eq event-window win) (eq window win))
		      (setf result t)
		      nil)))
    result)
  )

(defun Destroy-Notify (event-debug event-window)
  ;; do nothing, actually, probably destroy aggregate ?
  (when event-debug (format t " destroy-notify ~s~%" (xlib:window-id
						      event-window)))
  (deleting-window-drop-events (xlib:window-display event-window)
			       event-window)

  t)

(defun Configure-Notify (event-debug event-window x y width height)
    (when event-debug (format t "Configure-notify ~s ~s ~s ~s~%" 
					x y width height))
    (if (getf (xlib:window-plist event-window) 'Config)
       (funcall (getf (xlib:window-plist event-window) 'Config)
		    (handler-arg event-window) x y width height))
  t)

(defun Exposure (event-debug event-window count display)
   (declare (ignore count))
    (when event-debug (format t "Exposure-notify~%")
    )
      ;;  throw away extra exposure events on this window
      ;;  drop out of this loop, once any other events show up
      (loop
        (unless (xlib:event-case (display :discard-p nil :timeout 0)
                  (:EXPOSURE ((:event-window foo-window))
                             (when (eq foo-window event-window)
                               t))
                  (t () nil)) 	; ;  any other event, return nil (causes
          ;;  event-case to terminate), which causes
          ;;  loop to terminate
          (return)))

;;  done throwing away interim exposure
; events

     (call-specific-event-handler event-window 
				'Exposure (handler-arg event-window))
     t)

;;; This was moved here from some framegraphics stuff so that it 
;;; could be used in ButtonPress.

(defstruct (point  (:constructor build-point))
  (h 0 :type integer)
  (v 0 :type integer))

(defun ButtonPress (event-debug event-window x y button state)
    (declare (type window event-window)
	(type xlib:int16 x y)
	(type xlib:card8 button)
	(type xlib:card16 state)); unsigned, so plusp can be used.
    (when event-debug 
	  (format t "Button ~A pressed at location ~A:~A~%" button x y))
    (call-specific-event-handler event-window
	'ButtonPress
	(handler-arg event-window) 
	(build-point :h x :v y)
	(= 3 button)
	(plusp (logand (xlib:make-state-mask :shift) state))
	t ; this does nothing right now.   
    )
    t)

(defun ButtonRelease (event-debug x y)
    (when event-debug (format t "ButtonReleasex - ~A y - ~A~%" x y))
  t)

(defun ButtonMotion (event-debug)
    (when event-debug (format t "Button-motion~%"))
  t)

(defun Motion-Notify (event-debug)
      (when event-debug (format t "Motion-Notify~%"))
  t)
(defun default-event-handler (display)
  "Event handler for the pg-windows"
  ;; taken from /afs/cs/project/garnet/opal/code/windows.lisp
  ;; yes indeed, every clause should return T, not NIL!
  (when *event-debug* (format t "In default-event-handler~%"))
  (xlib:event-case (display :discard-p t :timeout 0)
    (:MAP-NOTIFY (event-window) (Map-notify *event-debug* event-window))
    (:UNMAP-NOTIFY (event-window) (Unmap-notify *event-debug* event-window))
    (:CIRCULATE-NOTIFY () (Circulate-notify *event-debug*))
    (:REPARENT-NOTIFY (x y) (Reparent-Notify *event-debug* x y)) 
    (:GRAVITY-NOTIFY () (Gravity-notify *event-debug*))
    (:DESTROY-NOTIFY (event-window) 
	(Destroy-notify *event-debug* event-window))
    (:CONFIGURE-NOTIFY (event-window x y width height)
		       (Configure-Notify *event-debug* event-window x y 
							width height))
    (:EXPOSURE (event-window count)
	       (Exposure *event-debug* event-window count display))
	(:ENTER-NOTIFY  (event-window)
	  (Enter-Notify *event-debug* event-window))
    (:NO-EXPOSURE () t)
    (:LEAVE-NOTIFY () t) ; not masked for but necessary for
			 ; edit-a-string
    (:BUTTON-PRESS (event-window x y code state)
         (ButtonPress *event-debug* event-window x y code state))
    (:BUTTON-RELEASE () t) ; This is handled in a different loop for 
			 ; framegraphics and not at all for prodigy.
			 ; The case is needed to prevent an illegal event.
			 ; dkahn.
    (:Motion-notify ()
	(when *event-debug* 
	      (format t "Motion Notify in default-event-handler~%")))
    (OTHERWISE (event-key) (format t "illegal event-- ~S~%." event-key) t)))


; Call-specific-event-handler pulls the name of a function off of 
; the window's p-list and makes the call.
(defun call-specific-event-handler (event-window indicator &rest l)
     (if (atom l)
	 (setf l (list l)))
     (if (getf (xlib:window-plist event-window) indicator)
	 (apply (getf (xlib:window-plist event-window) indicator) l)
     )
)


(defun pg-kill-window (win)
   (declare (special *display*)
	    (type display *display*)
	    (type win win))
;  (xlib:unmap-window (win-struct win))
  (xlib:destroy-window (win-struct win))
  (xlib:free-gcontext (win-gc win))
  (xlib:display-finish-output *display*)
;  (xlib:close-display (win-disp win))
;  (setf (win-disp win) nil)
;  (setf (win-struct win) nil)
;  (setf (win-gc win) nil)
;  (setf (win-font win) nil)
;  (setf (win-fg win) nil)
;  (setf (win-bg win) nil)
;  (setf (win-pen-x win) nil)
;  (setf (win-pen-y win) nil)
;  (setf (win-hght win) nil)
;  (setf (win-wdth win) nil)
)



(defun pg-refresh-window (w)
  (declare (type win w))
  "PG-REFRESH-WINDOW causes output to be flushed to the window in w."
  (xlib:display-finish-output (win-disp w)))

;;; pg-clear-window now works by setting a gcontext such that the
;;; the foreground is the background and then draws a rectangle
;;; with fill-p non-nil.  This system recognizes cliping regions
;;; while the old xlib:clear-area method did not.

(defun pg-clear-window (w)
  (declare (type win w))
  "PG-CLEAR-WINDOW causes the window in w to be cleared."
;  (xlib:clear-area (win-struct w))
   (xlib:with-gcontext ((win-gc w) :foreground (win-bg w))
	(xlib:draw-rectangle (win-struct w) (win-gc w)
			    0 0 (win-wdth w) (win-hght w) t))
;  (xlib:display-finish-output (win-disp w))
)


(defun pg-show-window (w)
  (declare (type win w))
  "PG-SHOW-WINDOW will cause the window in w to be mapped on the 
   screen if it is iconified and the output is flushed to the window."
    (xlib:map-window (win-struct w))
  (xlib:display-finish-output (win-disp w))
)


(defun pg-hide-window (w)
  (declare (type win w))
  "PG-HIDE-WINDOW will unmap (iconify) the window in w and flush output."
    (xlib:unmap-window (win-struct w))
;  (xlib:display-finish-output (win-disp w))
)

(defun pg-draw-line (w x1 y1 x2 y2)
  (declare (type win w))
  "PG-DRAW-LINE draws a line in the window in w from (x1 y1) to (x2 y2)
   and then flushes output."
  (declare (type w))
  (xlib:draw-line (win-struct w) (win-gc w) x1 y1 x2 y2)
;  (xlib:display-force-output (win-disp w))
)

(defun pg-write-text (w x y s)
  (declare (type win w))
  "PG-WRITE-TEXT writes the string s at the location (x y) in the window
   in w.  It then flushes output."
    (xlib:draw-image-glyphs (win-struct w)  (win-gc w) x y s)
;  (xlib:display-force-output (win-disp w))
)

(defun pg-draw-string (w s)
  (declare (type win w))
  "PG-DRAW-STRING writes the string s at the location (x y) in the window
   in w.  It then flushes output."
    (xlib:draw-image-glyphs (win-struct w)  (win-gc w) 
              (win-pen-x w) (win-pen-y w) s)
;  (xlib:display-force-output (win-disp w))
)

;;; the functions fg-text-width and fg-text-height both ignore their window
;;; arguments.  They should really be passed a font so that the system can 
;;; allow multiple fonts, but thats another story...
(defun pg-text-width (w s)
  (declare (special *FONT*) (ignore w))
  (xlib:text-width *FONT* s)
)
#|
(defun pg-text-width (w s)
  (declare (type win w))
  "PG-TEXT-WIDTH returns the width of the string in pixels for the default
   font in w."
    (xlib:text-width (win-font w) s))
|#
(defun pg-string-width (f s)
     (declare (type font f) (string s))
     "Returns the width of the string in the given font."
     (xlib:text-width f s))

(defun pg-string-height (f s)
    (declare (type font f) (string s))
    "Returns the height of the string in the given font."
     (let (d1 d2 d3 d4 d5 ascent descent d8)
    (declare (ignore d1 d2 d3 d4 d5 d8))
    (multiple-value-setq (d1 d2 d3 d4 d5 ascent descent d8)
      (xlib:text-extents f s))
    (+ ascent descent)))


(defun pg-text-height (w s)
(declare (special *FONT*) 
	 (type xlib:font *FONT*)
	 (ignore w))
  (let (d1 d2 d3 d4 d5 ascent descent d8)
    (declare (ignore d1 d2 d3 d4 d5 d8))
    (multiple-value-setq (d1 d2 d3 d4 d5 ascent descent d8)
      (xlib:text-extents *FONT* s))
    (+ ascent descent))
)

(defun pg-font-height (font)
    (declare (type xlib:font font))
    "Returns the maximun height of a character in the font."
    (+	(xlib:max-char-descent font)
	(xlib:max-char-ascent font)))


#|
(defun pg-text-height (w s)
  (declare (type win w))
  "PG-TEXT-HEIGHT returns the height of the text in the string s for the 
   font in w."
    (let (d1 d2 d3 d4 d5 ascent descent d8)
    (declare (ignore d1 d2 d3 d4 d5 d8))
    (multiple-value-setq (d1 d2 d3 d4 d5 ascent descent d8)
      (xlib:text-extents (win-font w)  s))
    (+ ascent descent)))
|#

(defun pg-window-height (w)
  (declare (type win w))
  "PG-WINDOW-HEIGHT returns the height of the window in w in pixels"
    (win-hght w))


(defun pg-window-width (w)
  (declare (type win w))
  "PG-WINDOW-WIDTH returns the width of the window in w in pixels."
    (win-wdth w))


(defmacro pg-with-window (w &rest forms)
  (declare (type win w))
   "PG-WITH-WINDOW binds w to the  global-variable *PG-WINDOW* so 
    that the forms in the second argument can use *PG-WINDOW*."
  `(progn (setq *pg-window* ,w)
	  ,@forms 
;	  (xlib:display-finish-output (win-disp *pg-window*))
))
           

;;; NOTE:  This is a new syntax.  A win must be passed in.  
;;; Originally it needed a pg-with-window.  No longer.
(defun pg-erase-rect (win lft top rgt bot)
  (xlib:clear-area (win-struct win) :x lft :y top
	 :width (abs (- rgt lft)) :height (abs (- bot top)))
  (xlib:display-force-output (win-disp win)))

#|
(defun pg-erase-rect (lft top rgt bot)
  (declare (special *PG-WINDOW*))
  "PG-ERASE-RECT erases a rectange between lft and rgt and between
   top and bot.  It must be in a PG-WITH-WINDOW construct."
  (xlib:clear-area (win-struct *pg-window*) :x lft :y top
	 :width (abs (- rgt lft)) :height (abs (- bot top)))
  (xlib:display-force-output (win-disp *pg-window*))
)
|#
;;; Put the actual call to xlib:draw-rectangle in one place

(defmacro pg-X-rect-draw (win gc x y w h fill-p)
  ;; Note: fill-p is assumed determined at compile time.
  ;;   if you don't like this, you'll have to change it
  (unless (or (eq fill-p 't) (eq fill-p 'nil))
    (break "fg-X-rect-draw expected a constant for fill-p.. check me out."))
  `(xlib:draw-rectangle ,win ,gc ,x ,y
			,(if fill-p `(1+ ,w) w)
			,(if fill-p `(1+ ,h) h)
			,fill-p))
;;; This function has been modified so as not to be compatible with the 
;;; original pg-frame-rect.  It no longer needs a with-window macro.
(defun pg-frame-rect (window lft top rgt bot)
  (declare (type win window)
           (type xlib:int16 lft top))
	   
  "PG-FRAME-RECT draws a rectange between lft and rgt and between
   top and bottom."
  (xlib:draw-rectangle (win-struct window) (win-gc window)
	lft top (abs (- rgt lft 1)) (abs (- bot top 1)) nil)
)


;;; NOTE:  This is a new syntax.  A win must be passed in.  We never used
;;; this function in a prodigy domain so we have no trouble changing it.
;;; Originally it needed a pg-with-window.  No longer.
(defun pg-invert-rect (win lft top rgt bot)
  "PG-INVERT-RECT will invert the pixel coloers in the region between
   lft and rgt and between top and bot."
  (let ((invert-gc (win-gc win)))
     (xlib:with-gcontext (invert-gc :function boole-c2)
        (xlib:draw-rectangle (win-struct win) invert-gc
	   lft top (abs (- rgt lft 1)) (abs (- bot top 1)) t)))
  (xlib:display-force-output (win-disp win)))
#|
(defun pg-invert-rect (lft top rgt bot)
  (declare (special *PG-WINDOW*))
  (let ((invert-gc (win-gc *pg-window*)))
     (xlib:with-gcontext (invert-gc :function boole-c2)
        (draw-rectangle (win-struct *pg-window*) invert-gc
	   lft top (abs (- rgt lft 1)) (abs (- bot top 1)) t)))
;  (xlib:display-force-output (win-disp *pg-window*))
)
|#
(defun pg-window-name (win)
   (declare (type win win))
   (xlib:get-property (win-struct win)
	:wm_name :transform #'code-char :result-type 'string))

(defun pg-set-window-name (win name)
   (declare (type win win)
	    (string name))
   (xlib:change-property (win-struct win) :wm_name name :string 8))

;The following code will return the correct host on which to create
; a graphics window.  It uses the display variable minus the last four
; characters to determine which X server to use.  At CMU this will be the 
; host name of the machine on which the X server is running unless it is 
; running on the same machine as the lisp process in which case the 
; string will be "unix".
; get proper host isn't used any more.
(defun get-proper-host ()
    (declare (special USER::*DISPLAY-ENV*))

    (if (null USER::*DISPLAY-ENV*) nil ; no X windows
        (subseq USER::*DISPLAY-ENV* 0 (position #\: USER::*DISPLAY-ENV*))
    )

)

(defmacro current-host ()
   "The reason we use this macro is to make setf check for spelling
    in the var name.  This must be used before init-graphics is called."
   'user::*display-server-name*)


;; ========================================================================


(pushnew :pg-system lisp::*features*)
(provide 'pg-system)
(provide 'pg-window) ;;; for framegraphics


;; The code after this point either is not used by PRODIGY because 
;; it was written after PRODIGY or works differently then the 
;; original PRODIGY code.  This code does not need pg-with-window
;; which (unfortunately) some PRODIGY code does.
#|
;(defmacro pg-with-window (w &rest forms)
;  `(progn (let ((*current-window* ,w))
;	  ,@forms 
;	  (xlib:display-finish-output (win-disp *current-window*)))))
           


(defun pg-frame-rect (win lft top rgt bot)
  (xlib:draw-rectangle (win-struct win) (win-gc win)
	lft top (abs (- rgt lft 1)) (abs (- bot top 1)) nil)
  (xlib:display-force-output (win-disp win)))


(defun pg-paint-rect (win r)
    (let ((gc (win-gc win))
	  (lft (get-rect-left r))
	  (top (get-rect-top r))
	  (rgt (get-rect-right r))
	  (bot (get-rect-bottom r)))	     
	 (xlib:with-gcontext (gc :function boole-1)
	     (xlib:draw-rectangle (win-struct win) gc
		 lft top (abs (- rgt lft 1)) (abs (- bot top 1)) t)))
    (xlib:display-force-output (win-disp win)))

;; ========================================================================
|#
(defun pg-move-to (w x y)
  (setf (win-pen-x w) x)
  (setf (win-pen-y w) y))

(defun pg-move (w x y)
  (setf (win-pen-x w) (+ (win-pen-x w) x))
  (setf (win-pen-y w) (+ (win-pen-y w) y)))

(defun pg-line (w x y)
    (let* ((x1 (win-pen-x w))
	   (y1 (win-pen-y w)) 
	   (x2 (+ x1 x))
	   (y2 (+ y1 y)))  
	  (xlib:draw-line (win-struct w) (win-gc w) x1 y1 x2 y2)
	  (setf (win-pen-x w) x2)
	  (setf (win-pen-y w) y2)
	  (xlib:display-force-output (win-disp w))))

(defun pg-line-to (w x2 y2)
    (let* ((x1 (win-pen-x w))
	   (y1 (win-pen-y w)))
	  (xlib:draw-line (win-struct w) (win-gc w) x1 y1 x2 y2)
	  (setf (win-pen-x w) x2)
	  (setf (win-pen-y w) y2)
	  (xlib:display-force-output (win-disp w))))

#+:xlib
(defun domain-window-data (&optional (stream *standard-output*))
  (declare (special *INIT-DOMAIN-X* 
		    *INIT-DOMAIN-Y* 
		    *INIT-DOMAIN-WIDTH*
		    *INIT-DOMAIN-HEIGHT*))
  "Returns lisp code that can be used to set the size of the domain
   window in a demo script before the window is created.  The code can
   write to a previously opened stream or defaults to stdout. This code
   will never work perfectly because the window manager usually repositions
   the windows a little."
   (if (member :xlib *features*)
	(let ((window (pg-window-struct *domain-window*)))
   		(format stream "(setf *INIT-DOMAIN-X* ~D)~%" 
				(xlib:drawable-x window))
		(format stream "(setf *INIT-DOMAIN-Y* (- ~D 19))" 
				(xlib:drawable-y window))
		(format stream ";  The -19 is because twm relocates the
			   ;  window and adds 19 to its y value.~%")
		(format stream "(setf *INIT-DOMAIN-WIDTH* ~D)~%" 
				(xlib:drawable-width window))
		(format stream "(setf *INIT-DOMAIN-HEIGHT* ~D)~%" 
				(xlib:drawable-height window))
   )
))

(provide 'window)

