 
(proclaim '(special *STATIC* *CONTEXT-ARRAY* *AXIOMS* 
		    *AT-LEAST-ONE-GENERATORS* *VARIABLE-EXPANSION*))

(eval-when (compile) 
  (load-path *PLANNER-PATH* "data-types"))

;;;
;;;==============================================================================
;;;
;;;
;;;
;;; Top level function that builds the context array.  The context array
;;; represents all of the potential immediate subgoals when a literal is
;;; created as a subgoal of a particular operator.  The literals are placed
;;; along the rows of the array and the operators (in which they occur in
;;; the context of) are placed along the columns of the array.  The elements
;;; of the array contain the operators that could be used to achieve the
;;; literal and the corresponding subgoals.  For example, an element would
;;; look like: 
;;;     (((push-thru-dr
;;;        ((next-to object door) (next-to robot object)
;;;         (inroom object room) (inroom robot room)))
;;;       (carry-thru-dr
;;;        ((inroom object room) (inroom robot room)
;;;         (next-to robot door))))
;;; which represents the things (inroom object room) could subgoal on
;;; in the context of UNLOCK.
;;;
(defun create-context-array (ops types)
  (let ((context-array (make-array (list (+ 2 (length ops))
					 (1+ (length types))
					 2))))
    (setq *CONTEXT-ARRAY* context-array)
    (place-ops-in-array (cons '*finish* ops) context-array 1)
    (place-types-in-array types context-array 1)
    (fill-in-context-array types ops ops context-array)))
;;;
;;;
;;;
;;; Places the list of operators along the top of the x-axis.
;;;
(defun place-ops-in-array (ops array index)
  (cond ((null ops) nil)
	(t (setf (aref array index 0 0)(car ops))
	   (place-ops-in-array (cdr ops) array (1+ index)))))
;;;
;;;
;;;
;;; Places the list of types along the y-axis.
;;;
(defun place-types-in-array (types array index)
  (cond ((null types) nil)
	(t (setf (aref array 0 index 0)(car types))
	   (place-types-in-array (cdr types) array (1+ index)))))
;;;
;;;
;;;
;;; Finds all the operators that contain the given formula in the preconditions.
;;;
(defun find-ops-with-precondition (ops formula)
  (cond ((null ops) nil)
	((formula-in-preconditions formula (get (car ops) 'typed-preconds))
	 (cons (car ops)(find-ops-with-precondition (cdr ops) formula)))
	(t (find-ops-with-precondition (cdr ops) formula))))
;;;
(defun formula-in-preconditions (formula preconditions)
  (cond ((null preconditions) nil)
	((has-vars (formula-instance (car preconditions)))
	 (cond ((equal (extract formula) 
		       (strip-negation (formula-type (car preconditions)))))
	       (t (formula-in-preconditions formula (cdr preconditions)))))
	(t 
	 (cond ((equal 
		 (extract-instance formula) 
		 (extract-instance (strip-negation 
				    (formula-instance (car preconditions))))))
	       (t (formula-in-preconditions formula (cdr preconditions)))))))
;;;
;;;
;;;
;;; Fills in each element of the context array.
;;;
(defun fill-in-context-array (types context-ops all-ops context-array)
  (cond ((null types))
	(t (fill-in-row (car types) 
			(find-ops-with-precondition context-ops (car types)) 
			all-ops
			context-array)
	   (fill-in-context-array (cdr types) context-ops all-ops context-array))))
;;;
(defun fill-in-row (type precondition-ops ops context-array)
  (cond ((null precondition-ops))
	(t (fill-in-row-col type (car precondition-ops) ops context-array)
	   (fill-in-row type (cdr precondition-ops) ops context-array))))
;;;
(defun fill-in-row-col (type op ops context-array)
  (setf (aref context-array (determine-column context-array op)
	                    (determine-row context-array type) 
			    0)
	(determine-potential-subgoals type op   
				      (find-ops-with-primary-add ops type))))
;;;
;;;
;;;
;;; Find the row that the literal occurs in.
;;;
(defun determine-row (array type)
  (do ((row (1- (array-dimension array 1))(1- row)))
      ((equal type (aref array 0 row 0)) row)
    (if (= 1 row)(error "Literal: ~a not found in context array" type))))
;;;
;;;
;;;
;;; Find the column that the literal occurs in.
;;;
(defun determine-column (array op)
  (do ((col (1- (array-dimension array 0))(1- col)))
      ((eq op (aref array col 0 0)) col)
    (if (= 1 col)(error "Operator: ~a not found in context array" op))))

;;;
;;;
;;;
;;; Finds all the operators that have the given literal as a primary add.
;;;
(defun find-ops-with-primary-add (ops primary-add)
  (cond ((null ops) nil)
	((primary-add-in-effects primary-add (get (car ops) 'typed-effects))
	 (cons (car ops)(find-ops-with-primary-add (cdr ops) primary-add)))
	(t (find-ops-with-primary-add (cdr ops) primary-add))))
;;;
;;;
;;;
;;; Determines whether the effects contain the given primary add.
;;;
(defun primary-add-in-effects (primary-add effects)
  (cond ((null effects) nil)
	((and (eq 'primary (formula-status (car effects)))
	      (equal (extract primary-add)(strip-negation (formula-type (car effects))))))
	(t (primary-add-in-effects primary-add (cdr effects)))))
;;;
;;;
;;;
;;; Finds the subgoals for each of the add-ops and enters them in the array.
;;;
(defun determine-potential-subgoals (type context-op add-ops)
  (if add-ops
      (let* ((typed-subgoals (remove-static-failures 	
			      (remove-duplicates 
			       (potential-subgoals (extract type) 
						   context-op (car add-ops))
			       :test #'equal)))
	     (inst-subgoals 
	      (and *VARIABLE-EXPANSION*
		   (find-all-potential-inst-subgoals type add-ops)))
	     (subgoals (if *VARIABLE-EXPANSION*
			   (potential-inst inst-subgoals typed-subgoals)
			 typed-subgoals)))
	(cond ((null subgoals)
	       (determine-potential-subgoals type context-op (cdr add-ops)))
	      (t (cons (list (car add-ops) subgoals)
		       (determine-potential-subgoals type context-op 
						     (cdr add-ops))))))))

(defun find-all-potential-inst-subgoals (type add-ops)
  (remove-duplicates 
   (mapcar #'(lambda (x)(extract-instance x))
	   (mapcan #'(lambda (x)
		       (potential-inst-subgoals 
			x
			(find-matching-effects (extract type)
					       (get (car add-ops) 
						    'unrenamed-typed-effects))
			(car add-ops)))
		   (generate-descendents type t)))
   :test #'equal))


(defun potential-inst-subgoals (type add-insts add-op)
  (cond ((null add-insts) nil)
	(t (let ((mapping (match (list (car add-insts))(list type)'(((nil))))))
	     (cond (mapping
		    (append (exp-match-failures 
			     (get add-op 'abs-preconds) 
			     (remove-dotted-pairs (car mapping)))
			    (potential-inst-subgoals type (cdr add-insts)
						     add-op)))
		   (t (potential-inst-subgoals type (cdr add-insts) add-op)))))
	))

(defun remove-dotted-pairs (bindings)
  (cond ((null bindings) nil)
	((null (caar bindings))
	 (remove-dotted-pairs (cdr bindings)))
	(t (cons (list (caar bindings)(cdar bindings))
		 (remove-dotted-pairs (cdr bindings))))))
;;; 
;;;
;;;
;;; Given the set of potential instantiated subgoals and the set of
;;; potential typed subgoals, intersect the two to get the real set of
;;; potential instantiated subgoals.
;;;
(defun potential-inst (inst-subgoals typed-subgoals)
  (cond ((null inst-subgoals) nil)
	((not (member (extract (car inst-subgoals)) typed-subgoals 
		      :test #'equal))
	 (potential-inst (cdr inst-subgoals) typed-subgoals))
	(t (cons (car inst-subgoals)(potential-inst (cdr inst-subgoals)
						    typed-subgoals)))))
	 
	 
;;;
;;;
;;;
;;; This function finds all of the potential subgoals of a literal based
;;; on the context in which the literal occurs and the operator that 
;;; achieves the literal.
;;;
;;; Note that if add-op and pre-op are equal then it is an operator that 
;;; can subgoal on itself.  We treat this separately since the variables
;;; need to be renamed.  This is done by storing two versions of the 
;;; operators, one with the original variables and one with the renamed 
;;; variables.  For all other cases we use the renamed variables.  In this
;;; case we match the original variables against the renamed ones.
;;;
(defun potential-subgoals (lit-type context-op add-op)
   (cond ((eq add-op context-op) ; This occurs when an operator recurses on itself
	 (check-each-matching-pre
	  lit-type
	  (get context-op 'renamed-vars)
	  (get add-op 'unrenamed-typed-preconds)
	  (get context-op 'context-preconds)
	  (find-matching-effects lit-type (get add-op 'unrenamed-typed-effects))
	  (find-matching-preconds lit-type (get context-op 'typed-preconds))))
	(t 
	 (check-each-matching-pre
	  lit-type
	  (get context-op 'renamed-vars)
	  (get add-op 'typed-preconds)
	  (get context-op 'context-preconds)
	  (find-matching-effects lit-type (get add-op 'typed-effects))
	  (find-matching-preconds lit-type (get context-op 'typed-preconds))))
 ))
;;;
;;;
;;;
;;; Attempts to match add-op-pre against context-op-pre in order to determine 
;;; which subgoals could be generated when subgoaling on a given predicate.
;;; Recurses through each of the precondition instances in context-op-pre.
;;;
(defun check-each-matching-pre (lit-type context-op-vars add-op-pre 
					 context-op-pre add-insts context-insts)
  (cond ((or (null context-insts)(null add-op-pre)) nil) 
	(t (append
	    (check-each-matching-add context-op-vars add-op-pre context-op-pre
				     add-insts (car context-insts))
	    (check-each-matching-pre lit-type context-op-vars add-op-pre 
				     context-op-pre add-insts 
				     (cdr context-insts))))))
;;;
;;;
;;;
;;; Continuation of check-each-matching-pre.  This function recurses through
;;; each of the adds corresponding to add-op.  Note that this function calls
;;; the matcher and first matches the effects of add-op to a precondition of
;;; context-op.  If they don't match then there is no chance of subgoaling 
;;; and we can go on to the next add-inst.  If they do match, then it
;;; determines if add-op-pre holds in the context of context-op-pre.  The
;;; context-op-vars are passed to the matcher to indicate that those variables
;;; should be treated as constants.  This prevents the system from adding 
;;; restrictions to context-op-pre which would violate the proof that add-op-pre
;;; will always hold.
;;;
(defun check-each-matching-add (context-op-vars add-op-pre context-op-pre 
					    add-insts pre-inst)
  (cond ((null add-insts) nil)
	(t (let ((mapping (match (list (car add-insts))(list pre-inst)
				 '(((nil))) context-op-vars)))
	     (cond (mapping
		    (append 
		     (check-each-context context-op-vars add-op-pre add-op-pre 
					 context-op-pre mapping 
					 (augment-goal (negate-goal 
							(car add-insts))
						       *AXIOMS* nil)
					 (vars-in-predicate (car add-insts)))
		     (check-each-matching-add context-op-vars add-op-pre
					      context-op-pre (cdr add-insts) 
					      pre-inst)))
		   (t (check-each-matching-add context-op-vars add-op-pre
					       context-op-pre (cdr add-insts) 
					       pre-inst)))))))

;;; 
;;; The use of "or"s in a precondition can create multiple contexts
;;; that the set of preconditions must be tested against.  The
;;; potential subgoals are the union of matching the preconditons
;;; against each of these context.
;;;
(defun check-each-context (context-op-vars preconds add-op-pre 
			   context-op-pre blsts true-conds produced)
  (cond ((null context-op-pre) nil)
	(t (append (check-matching-add context-op-vars preconds add-op-pre 
				       (car context-op-pre) blsts true-conds
				       produced nil)
		   (check-each-context context-op-vars preconds add-op-pre
				       (cdr context-op-pre) blsts 
				       true-conds produced)))))

;;;
;;;
;;; Check each of the preconditions against the context.  The context
;;; is the set of conditions that must hold based on where the
;;; particular subgoal is generated from.  This code also checks 
;;; that all of the variables are generated or else the generators for
;;; those variables are also tested.
;;; produced = those variables that have a successful generator
;;; failed = those variables that have a generator that may get
;;;          subgoaled on.
;;;
(defun check-matching-add (context-op-vars preconds add-op-pre 
			   context-op-pre blsts true-conds produced failed)
  (cond ((null preconds) nil)
	      ;; needs statics to map variables -- removed later
	;; Check if it is a nonstatic precondition that can't be
	;; subgoaled on.
	((and (not (eq 'static (formula-status (car preconds))))
	      ;; keep soft constraints on inference rules to order levels
	      (operator-p (formula-operator (car preconds)))
	      ;; check that if the condition can be ignored
	      (not-subgoalable (formula-instance (car preconds))))
	 (check-matching-add context-op-vars (cdr preconds) add-op-pre 
			     context-op-pre blsts true-conds 
			     (append produced (generated-vars (car preconds)
							      add-op-pre
							      produced))
			     failed))
	;;Check if it is static condition that does not generate any vars.
	((and (eq 'static (formula-status (car preconds)))
	      (not (generated-vars (car preconds) add-op-pre produced)))
	 (check-matching-add context-op-vars (cdr preconds) add-op-pre 
			     context-op-pre blsts true-conds produced failed))
	;;Check if it depends on a variable of another condition that
	;;failed to match.
	((intersect failed (vars-in-predicate (formula-instance (car preconds))))
	 (cons (strip-negation (formula-type (car preconds)))
	       (check-matching-add context-op-vars (cdr preconds) add-op-pre 
				   context-op-pre blsts true-conds produced 
				   (append failed 
					   (generated-vars (car preconds) 
							   add-op-pre produced)))))
	;; Check to see if the precondition matches the true-conds.
	((match (list (formula-instance (car preconds))) true-conds blsts)
	 (check-matching-add context-op-vars (cdr preconds) add-op-pre
			     context-op-pre blsts true-conds 
			     (append produced (generated-vars (car preconds)
							      add-op-pre
							      produced))
			     failed))
	;; Check if it is an at-least-one-generator.
	((and (boundp '*AT-LEAST-ONE-GENERATORS*)
	      (test-at-least-one-generators (formula-instance (car preconds))
					    produced))
	 (check-matching-add context-op-vars (cdr preconds) add-op-pre
			     context-op-pre blsts true-conds
			     (append produced (generated-vars (car preconds)
							      add-op-pre
							      produced))
			     failed))
	;; Check if it holds in the context.
	(t (let ((new-blsts (match (list (formula-instance (car preconds)))
				   context-op-pre blsts context-op-vars)))
	     ; If the length is 1, it matches with a unique set of bindings
	     (cond ((= 1 (length new-blsts))
		    (check-matching-add context-op-vars (cdr preconds) add-op-pre
					context-op-pre new-blsts true-conds
					(append produced (generated-vars 
							  (car preconds) 
							  add-op-pre
							  produced))
					failed))
		   (t (cons (strip-negation (formula-type (car preconds)))
			    (check-matching-add context-op-vars (cdr preconds) 
						add-op-pre context-op-pre blsts 
						true-conds produced
						(append failed (generated-vars 
								(car preconds) 
								add-op-pre
								produced)))))
		   )))))


(defun test-at-least-one-generators (literal produced)
  (let ((generators (car (member literal *AT-LEAST-ONE-GENERATORS*
			     :test #'(lambda (x y)(eq (car x)(car y)))))))
    (cond ((null generators) nil)
	  ((at-least-one (cdr literal) (cdr generators) produced)))))

(defun at-least-one (args generators produced)
  (cond ((null args))
	((car generators)
	 (cond ((or (not (is-variable (car args)))
		    (member (car args) produced))
		(at-least-one (cdr args)(cdr generators) produced))))
	;; (null (car generators))
	((not (member (car args) produced))
	 (at-least-one (cdr args) generators (cons (car args) produced)))))


;;;
;;;
;;;
;;; This functions returns the match failures, but first removes the static
;;; match failures.
;;;
(defun remove-static-failures (failures)
  (cond ((null failures) nil)
	((member (car failures) *STATIC* :test #'equal)
	 (remove-static-failures (cdr failures)))
	(t (cons (car failures)(remove-static-failures (cdr failures))))))
;;;
;;;
;;;
;;; Simply extracts the instances from a list of formulas.
;;;
(defun extract-instances (formulas)
  (cond ((null formulas) nil)
	(t (cons (formula-instance (car formulas))
		 (extract-instances (cdr formulas))))))

;;;
;;;
;;;
;;; A predicate is a generator if it contains a variable that does not
;;; occur in the primary add and it occurs in another precondition.
;;;
(defun generated-vars (lit-type preconds produced)
  (generates-vars (vars-in-predicate (formula-instance lit-type))
		  lit-type preconds produced))
;;;
(defun generates-vars (vars lit-type preconds produced)
  (cond ((null vars) nil)
	((member (car vars) produced)
	 (generates-vars (cdr vars) lit-type preconds produced))
	((occurs-in-preconds (car vars) preconds)
	 (cons (car vars)(generates-vars (cdr vars) lit-type preconds produced)))
	(t (generates-vars (cdr vars) lit-type preconds produced))))
;;;
;;;
;;;
;;; Tests whether a variable occurs in another precondition.
;;;
(defun occurs-in-preconds (var preconds)
  (cond ((null preconds) nil)
	((eq 'static (formula-status (car preconds)))
	 (occurs-in-preconds var (cdr preconds)))
	((occurs-in var (formula-instance (car preconds))))
	(t (occurs-in-preconds var (cdr preconds)))))
;;;
;;;
;;;
;;; Tests whether a variable occurs in the given literal.
;;;
(defun occurs-in (var lit)
  (cond ((null lit) nil)
	((eq var (car lit)))
	(t (occurs-in var (cdr lit)))))
;;;
;;;
;;;
;;; Returns the variables in a lit-type.
;;;
(defun vars-in-predicate (static)
  (cond ((null static) nil)
	((is-variable (car static))
	 (cons (car static)(vars-in-predicate (cdr static))))
	(t (vars-in-predicate (cdr static)))))
;;;
;;;
;;;
;;; Finds all the preconditions that match predicate.
;;;
(defun find-matching-preconds (lit-type list)
  (cond ((null list) nil)
	((equal lit-type (strip-negation (formula-type (car list))))
	 (cons (formula-instance (car list))
	       (find-matching-preconds lit-type (cdr list))))
	(t (find-matching-preconds lit-type (cdr list)))))
;;;
;;;
;;;
;;; Finds all the effects that match predicate.
;;;
(defun find-matching-effects (lit-type list)
  (cond ((null list) nil)
	((and (eq 'primary (formula-status (car list)))
	      (equal lit-type (strip-negation (formula-type (car list)))))
	 (cons (formula-instance (car list))
	       (find-matching-effects lit-type (cdr list))))
	(t (find-matching-effects lit-type (cdr list)))))


