;****************************************************************
; Functions for generating, formatting, and feeding rules.  
;;
;; Author: Oren Etzioni
;; Modified: Julie Roomy
;; Modified: Rob Spiger
;;
;; Many of these functions are not used in "julie-static" but will
;; be usefull if goal clobber and prerequisit violation are added 
;; in.
;****************************************************************


; Notes:

; Troubleshooting note.  Sometimes calling simplify returns T instead
; of some more interesting expression.  When this happens sometimes
; the expression is used to make the meta-fn (known <node> T) in a
; rule.  This use of known causes an error.  To remove it see the
; modification by Rob made on 4/17/92 down in make-op-rej-lhs.



; Justification for choice of meta-fns:
; node-rejetion rules: IS-TOP-LEVEL-GOAL is used because if the goal
; is true in the initial state then I don't want to reject the node.
; op-goal clobbering rules: IN-GOAL-EXP is used because even if the goal 
; is true I want to avoid clobbering it!
; op-pv rules: IS-TOP-LEVEL-GOAL is used because I only
; need to worry about prerequisites if the goal has not been achieved yet!

;A rule stored in *frules* has the following form:
;goal, state-preds, op-name, sole-op, goal-context, lhs-list (for
;subsumption check).


; If the predicate that generates the bindings is negated,
; then the bindings generation (on the lhs of the bindings rule) will
; fail.  This occurs for either a select or a reject rule.
; The solution is to make the rules be reject rules and add
; a candidate-bindings clause in the beginning of the rule, so that
; clause will be the generator.  This doesn't work for bindings 
; selection rules, because these are executed before the candidate
; bindings are generated. (probably accounts for some of my speedup).


; The rules in *frules* should be simplified and ready to
; roll. All feed-failure-rules should do is format.


;****************************************************************
; Rules from Failures:

; Interface function to rej.lisp Format-rules takes care of
; prettifying this.  Creates all the rules and stores them in *frules*
; without formatting.  DF checks that no rule is subsumed.  You want
; to do the check only after all the rules have been formed!

; ops is a non-empty list, usually consisting of one element.  When
; ops has more than one element the lhses need to be conjoined (with
; the duplicates removed!) 

; Two cases: reject rule for one op, or reject rule
; for an op that matches the goal in multiple ways.
(defun create-reject-rule (goal-lit ops sole-op)
  (let* (
	 (goal-context (Literal-goal-stack goal-lit))
	 (dec-goal (Literal-name goal-lit))
	 (op-name (Operator-name (car ops)))
	 (rules (get op-name 'rules)))
    (if (> (length ops) 1)
	(create-multiple-reject-rules
	 goal-context dec-goal op-name rules ops sole-op)
      (let* ((op (car ops))
	     (op-lhs (Operator-failure-condition op))
	     (context-and-lhs
	      (make-op-rej-lhs dec-goal op-lhs goal-context op-name))
	     (rule-context (car context-and-lhs))
	     (rule-lhs (cadr context-and-lhs)) 
	     (lhs-list (third context-and-lhs))
	     )
	(unless (mem-equal lhs-list rules)
		(setf (get op-name 'rules) (cons lhs-list rules))
		;; modified 3/9/92 by Julie Roomy to remove frules that
                ;; have a sole-op 
		(if sole-op
		    (update-sole-op-frules dec-goal op-name rule-lhs 
					   rule-context lhs-list)
		  (update-frules dec-goal op-name rule-lhs rule-context 
				 sole-op lhs-list))

		(when (form-bindings-rule? op-name dec-goal op-lhs
					   (Operator-bindings op))
		      (update-brules dec-goal op-name rule-lhs rule-context
				     (Operator-bindings op))))))))



; Creates a seperate bindings rule for each op, and
; then creates an op-rej rule by conjoining the lhses of the indiv
; ops.
(defun create-multiple-reject-rules
  (goal-context dec-goal op-name rules ops sole-op)
  (let ((combined-op-lhs
    (loop for op in ops
     collect (operator-failure-condition op) into comb
     finally (return (rob-simplify (cons 'AND comb) 'no-sub)))))

;;     (rob-simplify (cons 'AND 
;;                         (mapcar #'(lambda (op) 
;;                                  (operator-failure-condition op))
;;                          ops)) 'no-subs)))

;;  (let ((combined-op-lhs
;;	 (iter:iterate
;;	  (iter:for op iter:in ops)
;;	  (iter:collect (Operator-failure-condition op) iter:into comb)
;;	  (iter:finally (return (rob-simplify (cons 'AND comb) 'no-sub))))))
;; )

    (unless (mem-equal combined-op-lhs (get 'op-name 'rules))
	    (let*
		(
		 (context-and-lhs
		  (make-op-rej-lhs dec-goal combined-op-lhs
				   goal-context op-name))
		 (lhs-list (third context-and-lhs))
		 (rule-lhs (cadr context-and-lhs))
		 (rule-context (car context-and-lhs))
		 )
	      (setf (get op-name 'rules) (cons lhs-list rules))
	      (update-frules dec-goal op-name rule-lhs rule-context 
			     sole-op lhs-list)      
	      )
	    (iter:iterate
	     (iter:for op iter:in ops)
	     (when
	      (form-bindings-rule? op-name dec-goal
				   (Operator-failure-condition op)
				   (Operator-bindings op))
	      (let*
		  ((context-and-lhs
		    (make-op-rej-lhs dec-goal (Operator-failure-condition op)
				     goal-context op-name))
		   (rule-lhs (cadr context-and-lhs))
		   (rule-context (car context-and-lhs)))
		(update-brules dec-goal op-name rule-lhs rule-context
			       (Operator-bindings op)))))
	    
	    )
    )
  )


;****************************************************************	
; helper functions for create-reject-rule.


; The lhs-list is used by rule-subsumed? which is called by DF.
(defun update-frules (dec-goal op-name rule-lhs rule-context 
			       sole-op lhs-list)
  (setq *frules*
	(append
	 (list
	  (list dec-goal rule-lhs op-name sole-op
		rule-context lhs-list)) *frules*)))

(defun update-sole-op-frules (dec-goal op-name rule-lhs rule-context lhs-list)
  (setq *sole-op-frules* 
	(append
	 (list
	  (list dec-goal rule-lhs op-name rule-context lhs-list)) 
	 *sole-op-frules*)))

(defun update-brules (dec-goal op-name rule-lhs rule-context
			     op-bindings)
  (setq *brules*
	(append
	 (list
	  (list dec-goal rule-lhs op-name op-bindings
		rule-context)) *brules*)))

(defun rej-rule-debug (dec-goal op-name rule-lhs rule-context)
  (format
   t 
   "~% RULE CREATED goal: ~s lhs: ~% ~s goal context: ~s ~s ~%"
   dec-goal rule-lhs rule-context op-name))


; Sub binds into vg and store rule.
(defun update-pvrules (lbl vg g &optional gc-cond)
    (iter:iterate
     (iter:for bl iter:in lbl)
     (update-pvrules-helper bl vg g gc-cond)))

; caar to get rid of binds and list around the dec-goal.
; The dec-goal is variablized, and the bl substititutes bindings
(defun update-pvrules-helper (bl vg g gc-cond)
  (setq g (caar (bind-clauses (list g) bl nil)))
  (let ((r (list vg g gc-cond)))
    (unless (or (mem-equal r *gc-rules*)
		(mem-equal r *pv-rules*))
	    (setq *pv-rules* (cons r *pv-rules*)))))


; Sub binds into pg and store rule.
(defun update-gcrules (lbl pg g &optional gc-cond)
  (iter:iterate
   (iter:for bl iter:in lbl)
   (update-gcrules-helper bl pg g gc-cond)))

; caar to get rid of binds and list around the pg.
(defun update-gcrules-helper (bl pg dec-goal gc-cond)
  (setq pg (caar (bind-clauses (list pg) bl nil)))
  (unless (or (mem-equal dec-goal *false-list*)
	      (mem-equal pg *false-list*))
	  (let ((r (list dec-goal pg gc-cond)))
	    (setq *gc-rules* (cons r *gc-rules*)))))



(defun update-op-gcrules (op lbl pg dec-goal)
  (iter:iterate
   (iter:for bl iter:in lbl)
   (setq pg (caar (bind-clauses (list pg) bl nil)))
   (setq *op-gc-rules*
	 (cons (list dec-goal pg op) *op-gc-rules*))))


(defun update-op-pvrules (lbl vg vop dec-goal op)
  (setq dec-goal (varify-literal dec-goal))
  (iter:iterate
   (iter:for bl iter:in lbl)
   (setq dec-goal (caar (bind-clauses (list dec-goal) bl nil)))
   (setq *op-pv-rules*
	 (cons (list dec-goal vg op) *op-pv-rules*))))


;****************************************************************


; Display function for failure rules.
(defun format-rules (&optional (rules *frules*))
  (let ((rule-num 0))
    (iter:iterate
     (iter:for r iter:in rules)
     (unless (fourth r)			;sole-op
	     (setq rule-num (1+ rule-num))
	     (format t "~%~%IF ")
	     (format t "~s ~%" (second r)) ;lhs.
	     (format t "    ~s ~%" (fifth r)) ;context
	     (format t "THEN (reject operator ~s)~%"
		     (third r)))
     )
    (format t "Formed rules: ~s *frules* ~s ~%" rule-num
	    (length rules))
    ))


; Interface function to Prodigy.
; Suboptimal order for context predicates on lhs.  I can fix this.
; hack. 
(defun feed-failure-rules ()
  (iter:iterate (iter:for r iter:in *frules*)
		(let* (
		       (op (third r))
		       (rhs (list 'reject 'operator op))
		       (lhs (second r))
		       (context (fifth r))
		       (unique-sig (make-unique-sig 'op-fails))
		       (rule-type 'op-reject)
		       (nm (generate-random-rule-name)))
		  (unless (fourth r)	;sole-op
			  (feed-generated-rule (append lhs context) rhs
					       unique-sig nm
					       *static-est-savings*
					       rule-type
					       )))))

(defun feed-bindings-rules ()
  (iter:iterate
   (iter:for r iter:in *brules*)
   (feed-bindings-rule r)))


(defun feed-bindings-rule (r)
   (let* ((context (fifth r))
	  (op-lhs (second r))
	  (op-name (third r))
	  (striped-lhs (convert-lhs op-lhs))
	  (lhs (make-bindings-lhs context striped-lhs (first r) op-name
				  ))
	  (good-bindings (generate-bindings-for-rule
			  op-name
			  (fourth r) ;bindings
			  ))
	  (rhs (list 'select 'bindings (varify-list good-bindings)))
	  (unique-sig (make-unique-sig 'bindings-fail))
	  (nm (generate-random-rule-name)))
     (feed-generated-rule lhs rhs unique-sig nm
			  *static-est-savings* 'bindings-select)))


  

; Converts an op-lhs to bindings-lhs with same node name so that
; on-goal-stack preds will refer to the right node.
(defun make-bindings-lhs (context lhs goal op)
  (let* ((node (if context
		  (second (first context)) ;the node name
		(generate-random-var)))
	 )
    (if (not lhs)
	(error "Ill formed lhs for bindings rules ~%")
      `(and (current-node ,node)
	    ,@(when goal `((current-goal ,node ,(varify-literal goal))))
	    ,@context
	    (current-op ,node ,op)
	    (known ,node
		   ,(negate-formula lhs)
		   )))))


;Strips the operator-lhs down to the "known node" part, and places
; a not before the rule.
(defun convert-lhs (lhs)
 (third (fifth lhs)))
		


; This list of pairs is ordered according to the params and placed
; in the rule.  
;; The new static adds in the bindings generated from the preconditions
;; so it is not necessary to access the precondition field in this function.
(defun generate-bindings-for-rule (name bindings)
  (let* ((params (get name 'params)))
    (order-binding-list bindings params)
    ))

; Order the binding list according to the params.
(defun order-binding-list (bindings params)
  (if (null params) nil
    (let ((b (cadr (assoc (car params) bindings))))
      (if b
	  (append (list b) (order-binding-list bindings (cdr params)))
	(append (list (generate-random-var))
		(order-binding-list bindings (cdr params)))))))
	


; The comma-atsign gets rid of NIL if there's no goal or lhs, two
; levels of parentheses are needed around current-goal and known to
; compensate for the atsign.
; varify-literal for the goal should not be done earlier, because
; simplify-lhs assumes that dec-goal is constified.
(defun make-op-rej-lhs (goal lhs goal-context op)

  (let* ((node (generate-random-var))
	 (context-and-lhs (do-lhs goal lhs goal-context node))
	 (lhs (cadr context-and-lhs))
	 (context (car context-and-lhs))
	 (context-lhs-list
	  (mapcar #'(lambda (ogs-expr)
		      (negate-literal (third ogs-expr))) context))
		  )
    `(,context				;goal context list.
      (and (current-node ,node)		;lhs.
	   ,@(when goal
		   `((current-goal ,node
				   ,(varify-literal goal))))
	   (candidate-op ,node ,op)
	   ,@(when lhs (when (not (equal lhs t)) `((known ,node ,lhs))))
;; on the line above, the second when clause was added by Rob on
;; 4/17/92 to remove (KNOWN <node> T) because it was silly to have in
;; the rules and it was causing an error when it was fed to prodigy

	   )
;; before mod (next line)  by Rob on 4/17/92
;;      ,(remove-nils (adjoin lhs context-lhs-list :test #'equal))  ;lhs-list
;; mod below
;; was modified because remove-nils fails if the list contains an
;; atom.  The atom t was coming up and causing an error.
;;
      ,(remove-if #'(lambda (x) (null x)) (adjoin lhs context-lhs-list
												  :test #'equal)) ;lhs-list



;; mod above
      )
    ))



;  Called by all make-X-lhs routines.
; removed call to Steve's simplify routine. 
(defun do-lhs (g no-context-lhs &optional goal-context node)
  (let* (
     (context (mapcar #'(lambda (goal)
                         `(on-goal-stack
                           ,node
                           ,(varify-literal goal)))
               goal-context))

;;	 (context (iter:iterate 
;;		   (iter:for goal iter:in goal-context)
;;		   (iter:collect `(on-goal-stack ,node
;;					    ,(varify-literal goal))
;;			    iter:into context)
;;		   (iter:finally (return context))))
	 )
    (list context (varify-lhs no-context-lhs))))






(defun literal? (lhs)
  (and (listp lhs)
       (atom (car lhs))
       (not (lexpr-p (car lhs)))))
       




; Function to decide whether bindings rules are necessary.
; If all the variable's in an operator's preconditions are bound
; in the goal, then no bindings rules are necessary.
; So I compare the length of the goal with the length of the params.
; The lhs here has to be raw, because when
; create-multiple-reject-rules calls this function, the full lhs has
; not been composed yet.
(defun form-bindings-rule? (op-name goal lhs binds)
  (setq lhs (cadr (do-lhs goal lhs))) ; cadr=no-context-lhs.
  (and lhs ;known stuff not added yet.
       (unbound-param? op-name lhs binds)))


; This should return T only when there's a param to the op that is not
; bound in the goal, but is constrained by the lhs.  The lhs here is a
; boolean expression.  Need the op-bindings argument since the goal
; can match the operator in a number of ways, and the bindings tell us
; how. Algorithm: find all the unbound params, and check for
; membership of their randomly-generated-var bindings in the lhs.

(defun unbound-param? (op-name lhs binds)
  (let* ((params (get op-name 'params))
	 (unbound
	  (iter:iterate
	   (iter:for p iter:in params)
	   (let ((bind (cadr (assoc p binds))))
	     (cond
	      ((null bind) (error "param ~s for operator ~s has no binding ~%"
				   p op-name))
	      ((not (rob-is-var? bind)) nil) ;prevent my-var? from crash
	      ((my-var? bind) (iter:collect bind))
	      )))))			;otherwise do nothing!
     (iter:iterate
      (iter:for u iter:in unbound)
      (iter:thereis (var-appears? u lhs)))))

; REturns T if u appears in lhs.
(defun var-appears? (u lhs)
  (cond
   ((null lhs) nil)
   ((listp lhs) (or (var-appears? u (car lhs))
		    (var-appears? u (cdr lhs))))
   (t (equal lhs u))))
				  
    
  

;****************************************************************
; Rules from Goal interactions:

(defun format-op-g-rules (&optional (rules *op-gc-rules*) pv)
  (iter:iterate (iter:for r iter:in rules)
		(format t "~%")
		(format t "IF current-goal=~s" (first r))
		(format t " AND~%")
		(format t "top-level-goal=~s" (second r))
		(format t "~%")
		(if pv
		(format t "THEN (prefer operator ~s) ~%" (third r))
		(format t "THEN (prefer OTHER operator to ~s) ~%"
			(third r))))
  (length rules)
  )


(defun format-g-rules (&optional (rules *gc-rules*))
  (iter:iterate (iter:for r iter:in rules)
		(format t "~%")
		(format t "IF current-goal=~s" (first r))
		(format t " AND~%")
		(format t "top-level-goal=~s" (second r))
		(format t "~%")
		(format-stack (third r))
		(format t "THEN (prefer goal ~s to goal ~s)~%"
			(first r) (second r)))
  (length rules)
  )


; hack to distinguish gc and pv rules using the est-savings.
; the est-savings for gc rules are twice as big.
(defun feed-g-rules (gc)
  (let ((rules (if gc *gc-rules* *pv-rules*)))
    (iter:iterate (iter:for r iter:in rules)
		  (let* ((vg1 (varify-literal (first r)))
			 (vg2 (varify-literal (second r)))
			 (rhs (list 'prefer 'goal vg1 vg2))
			 (lhs (make-goal-lhs vg1 vg2 r))
			 (unique-sig (make-unique-sig 'goal-interacts))
			 (nm (generate-random-rule-name)))
		    (feed-generated-rule lhs rhs unique-sig nm
					 (if gc 
					     (* 2 *static-est-savings*)
					   *static-est-savings*)
					 'goal-pref))
		  )))


; prefer-known-op not currently used.
(defun feed-op-g-rules (rules gc &optional prefer-known-op)
  (iter:iterate (iter:for r iter:in rules)
		(let* ((cand-op (generate-random-var))
		       (rhs (append
			     (list 'prefer 'operator)
			     (if prefer-known-op
				 (list (third r) cand-op)
			       (list cand-op  (third r)))))
		       (lhs (make-goal-op-lhs r cand-op gc))
		       (unique-sig (make-unique-sig 'op-interacts))
		       (nm (generate-random-rule-name)))
		  (feed-generated-rule lhs rhs unique-sig nm
				       *static-est-savings* 'op-pref))
		))



; (third r) are the conditions on the goal preference.  
; Sometimes (eg in BW) there are no such conditions.
; cadr of do-lhs because there's no context!
(defun make-goal-lhs (vg1 vg2 r)
  (let* ((node (generate-random-var))
	 (lhs (third r))
	 )
    `(and (current-node ,node)
	  (candidate-goal ,node ,vg1)
	  (candidate-goal ,node ,vg2)
	  ,@(when lhs
		  `((known ,node
			   ,(cadr (do-lhs (car r) lhs))
			   ))))))

; super hack.  No room for lhs yet.
(defun make-goal-op-lhs (r cand-op gc)
  (let* ((node (generate-random-var))
	 (op (third r))
	 (vg1 (varify-literal (car r)))
	 (vg2 (varify-literal (second r)))
	 )
    (if gc
	`(and (current-node ,node)
	      (current-goal ,node ,vg1) 
	      (in-goal-exp ,node ,vg2) ;protected goal.
	      (candidate-op ,node ,cand-op)
	      (candidate-op ,node ,op)
	      (not-equal ,cand-op ,op))
	`(and (current-node ,node)
	      (current-goal ,node ,vg1) 
	      (is-top-level-goal ,node ,vg2) ;goal to be achieved.
	      (candidate-op ,node ,cand-op)
	      (candidate-op ,node ,op)
	      (not-equal ,cand-op ,op)))))



;****************************************************************
; Common functions:

; Function to turn statically generated rule into a learned rule
; so that I can feed it to Prodigy and use the UA.
; Need to invent:
;; unique-sig (tc-name v1 v2 v3) ;ie all the vars in the rule.
;; name. 
;; est-savings.
;;load-new-scrs? make-new-scrs?
; The learning program argument indicates who generated this rule.
(defun feed-generated-rule (lhs rhs unique-sig nm est-savings
				rule-type &optional (learning-program 'static))
  (setf (get nm 'rule-type) rule-type)
  (setf (get nm 'lhs) lhs)
  (setf (get nm 'rhs) rhs)
  (when (eq (car rhs) 'prefer)
	(setf (get nm 'priority) 0))
  (setf (get nm 'was-learned) learning-program)
  (setf (get nm 'static) t) ; to fool ebl's utility evaln.
  (normalize-scr-for-ebs nm) 
  (setf (get nm 'unique-sig) unique-sig)
  (dynamically-add-scr nm)
  (when (member nm *LEARNED-RULES* :test #'equal)
      (error "rule ~s already exists ~%" nm))
  (push nm *LEARNED-RULES*)
  (push nm *learned-rules-in-sys*)
  (push nm *NEW-LEARNED-RULES*)
  (cond ((boundp '*PROB-NM*)
	 (setf (get nm 'problem) *PROB-NM*))
	(t (setf (get nm 'problem) 'Unnamed-problem)))
  (setf (get nm 'est-savings) est-savings)
  (setf (get nm 'cum-savings) est-savings)
  (format t "~%New Learned Rule: ~a" nm )	 
  nm
  )


; Functions to format *frules* into Prodigy readable rules.
(defun generate-random-rule-name ()
  (setq *rule-name-counter* (1+ *rule-name-counter*))
  (intern (concatenate 'string "R" (princ-to-string *rule-name-counter*)
		       )))
		       
(defun make-unique-sig (type)
  (list type (generate-random-var)))


; Doesn't feed default sc-rules, since these will be loaded
; at runtime.
(defun feed-rules (unsolvable-probs)
  (feed-goal-pref-rules)
  (feed-goal-rejection-rules)
  (feed-operator-rejection-rules)
  (feed-failure-rules)
  (feed-bindings-rules)
  (feed-success-rules)
  (feed-node-rules)
  (feed-g-rules t) ;gc
  (when unsolvable-probs
     (feed-node-rejection-rules))
  (feed-g-rules nil) ;pv
  (feed-op-g-rules *op-gc-rules* t) 
  (feed-op-g-rules *op-pv-rules* nil)
  )


; Could be extended to handle quantification.
(defun negate-formula (f)
  (when f
	(let ((term (car f)))
	  (cond  
	   ((eq term 'or) (cons 'and (negate-formula (cdr f))))
	   ((eq term 'and) (cons 'or (negate-formula (cdr f))))
	   ((eq term '~) (cadr f))
	   ((rules-atomic-formula? term)
	    (append (list `(~ ,term)) (negate-formula (cdr f))))
	   (t (cons (negate-formula term)
		    (negate-formula (cdr f))))))))

(defun rules-atomic-formula? (f)
  (not (or (eq (car f) 'and)
	   (eq (car f) 'or)
	   (eq (car f) '~))))


(defun negate-literal (l)
  (if (eq (car l) '~)
      (cadr l)
    `(~ ,l)))

(defun format-stack (stack)
  (iter:iterate (iter:for c iter:in stack)
		(format t "   NOT ~s~%" c)))


;****************************************************************
; Node rejection rules.

(defun create-node-reject-rule (lhs lit)
  (when *debug* (format t "~% NODE rule created ~s ~s ~%"
			(lit-dec-goal lit) lhs))
  (setq *node-rules* (append
		      (list (list (lit-dec-goal lit) lhs))
		      *node-rules*))
  )

(defun feed-goal-pref-rules ()
 (iter:iterate
 (iter:for rule iter:in *preference-rules*)
  (let* (     
    (lhs (second (second rule)))
    (rhs (second (third  rule)))
    (nm (first rule))
    (unique-sig (make-unique-sig 'goal-pref)))
   (feed-generated-rule lhs rhs unique-sig nm *static-est-savings*
           'goal-pref))))
 
(defun feed-goal-rejection-rules ()
 (iter:iterate
 (iter:for rule iter:in *goal-rejection-rules*)
  (let* (     
    (lhs (second (second rule)))
    (rhs (second (third  rule)))
    (nm (first rule))
    (unique-sig (make-unique-sig 'goal-reject)))
   (feed-generated-rule lhs rhs unique-sig nm *static-est-savings*
           'goal-reject))))
 
(defun feed-operator-rejection-rules ()
 (iter:iterate
 (iter:for rule iter:in *operator-rejection-rules*)
  (let* (     
    (lhs (second (second rule)))
    (rhs (second (third  rule)))
    (nm (first rule))
    (unique-sig (make-unique-sig 'op-reject)))
   (feed-generated-rule lhs rhs unique-sig nm *static-est-savings*
           'op-reject))))
 
(defun feed-node-rules ()
  (iter:iterate
   (iter:for r iter:in *node-rules*)
   (let* ((goal (car r))
	  (node (generate-random-var))
	  (lhs (make-node-lhs (second r) node goal))
	  (rhs (list 'reject 'node node))
	  (unique-sig (make-unique-sig 'node-fails))
	  (nm (generate-random-rule-name)))	  
     (feed-generated-rule lhs rhs unique-sig nm *static-est-savings*
			  'node-reject))))

; 'cadr' because the first of do-lhs is NIL (because the goal context
; is nil. hack.
(defun make-node-lhs (lhs node goal)
  (setq lhs (cadr (do-lhs goal lhs)))
  `(and (candidate-node ,node)
	(is-top-level-goal ,node ,(varify-literal goal))
	(known ,node ,lhs)))



;****************************************************************
; SUCCESS

;;; What about bindings rules?
;;; Check for pref rule subsumption?
;;; Checks that no equivalent/more general rejection rule has been learned.

(defun create-pref-rule (dec-goal op sole-op lhs)
  (let* (
	 (op-name (Operator-name op))
	 (rules (get op-name 'rules))
	 (lhs-and-bad-op (make-op-pref-lhs dec-goal lhs op-name))
	 (bad-op (cadr lhs-and-bad-op))
	 )
    (setq lhs (car lhs-and-bad-op))
    (unless
     (or
      (more-general-rule-already-learned? (negate-formula lhs) rules)
      sole-op)
     (update-srules dec-goal op-name lhs sole-op bad-op))))

		      
(defun update-srules (dec-goal op-name rule-lhs sole-op bad-op)
  (setq *srules*
	(append
	 (list
	  (list dec-goal rule-lhs op-name sole-op bad-op
		)) *srules*)))


(defun feed-success-rules ()
  (iter:iterate (iter:for r iter:in *srules*)
		(let* (
		       (op (third r))
		       (bad-op (fifth r))
		       (rhs (list 'prefer 'operator op bad-op))
		       (lhs (second r))
		       (unique-sig (make-unique-sig 'op-succeeds))
		       (rule-type 'op-pref)
		       (nm (generate-random-rule-name)))
		  (unless (fourth r)	;sole-op
			  (feed-generated-rule lhs rhs
					       unique-sig nm
					       *static-est-savings*
					       rule-type
					       ))
		  )))


; The comma-atsign gets rid of NIL if there's no goal or lhs, two
; levels of parentheses are needed around current-goal and known to
; compensate for the atsign.  varify-literal for the goal should not
; be done earlier, because simplify-lhs assumes that dec-goal is
; constified.

(defun make-op-pref-lhs (goal lhs op)
  (let* ((node (generate-random-var))
	 (bad-op (generate-random-var))
	 )
    (setq lhs (varify-lhs lhs))
    (when (= (length lhs) 1) (setq lhs (car lhs)))
    `(
      (and (current-node ,node)		;lhs.
	   ,@(when goal
		   `((current-goal ,node
				   ,(varify-literal goal))))
	   (candidate-op ,node ,op)
	   (candidate-op ,node ,bad-op)
	   ,@(when lhs `((known ,node ,lhs)))
	   )
      ,bad-op
      )))



;;------------------------
;; utility routines moved from uts.lisp in original static
;; varify-literal
(defun varify-literal (c)
  (cond
   ((eq (car c) '~)
    (list (car c) (cons (car (cadr c)) (varify-list (cdr (cadr c))))))
   (t
    (cons (car c)
          (varify-list (cdr c))))))

;; varify-list
(defun varify-list (vars)
  (iter:iterate
   (iter:for x iter:in vars)
   (iter:collect (varify-atom x))))

;;varify-atom
; The case stuff is done so that the var will print properly.
; Don't variabilize actual constants!
(defun varify-atom (x)
  (if (random-const? x)
      (intern (concatenate 'string "<"
                           (write-to-string x :case :upcase)
                           ">"))
    x))


;; varify-lhs
(defun varify-lhs (l)
  (cond ((null l) nil)
        ((listp l)
         (cons (varify-lhs (car l)) (varify-lhs (cdr l))))
        (t (varify-atom l))))


;;mem-equal
(defun mem-equal (e l)
  (member e l :test #'equal))


;; normalize-scr-for-ebs
; from domain-check.lisp. Added check for 'static.
(defun normalize-scr-for-ebs (rule-nm)
  (cond ((and (get rule-nm 'was-learned)
              (not (get rule-nm 'static)))
         (or (get rule-nm 'lhs-for-ebs)
             (error "normalize-scr: learned rule incomplete"))
         (setf (get rule-nm 'sig-for-ebs) (get rule-nm 'unique-sig)))
        ((member (get rule-nm 'rule-type) '(node-reject node-select))
         (normalize-node-filter-rule rule-nm))
        ((member (get rule-nm 'rule-type) '(goal-reject goal-select))
         (normalize-goal-filter-rule rule-nm))
        ((member (get rule-nm 'rule-type) '(op-reject op-select))
         (normalize-op-filter-rule rule-nm))
	((member (get rule-nm 'rule-type) '(bindings-reject bindings-select))
	 (normalize-bindings-filter-rule rule-nm))
	((member (get rule-nm 'rule-type)
                 '(node-pref goal-pref op-pref bindings-pref))
         nil)
        (t (break))))

(defun feed-node-rejection-rules ()
 (let* (
   (lhs nil)
   (rhs nil)
   (nm nil)
   (unique-sig nil))
  (loop for rule in *top-level-node-rejection-rules*
   do 
    (progn
     (setq lhs (second (second rule)))
     (setq rhs (second (third  rule)))
     (setq nm  (first rule))
     (setq unique-sig (make-unique-sig 'node-reject))
     (feed-generated-rule lhs rhs unique-sig nm *static-est-savings*
           'node-reject)))))
