;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;
;; rules.lisp forms search control rules based upon node failure or
;; node success
;;
;; Author: Oren Etzioni
;; Modified: Julie Roomy 
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;

;;------------------------------
;; form-operator-rejection-rules
; If |ops|<2 then don't form an op-rej rule, only a bindings rule.
; Encodes highest failure heuristic. Could change that by calling
; form-op-rej recursively even when create-reject-rule is called.  If
; an op matches g under multiple bindings, then the op name will
; appear more than once in ops.  Then we'd like to create an op rej
; rule whose lhs is the conjunction of the conditions in the indiv
; rules (with duplicates removed!).  The corresponding bindings
; selection rule is the appropriate one as well.
(defun form-operator-rejection-rules (goal)
  (let* ((ops (literal-operators goal))
	 (sole-op (< (length ops) 2)))
    (operator-rejection-rules ops goal sole-op)))



;;-------------------------
;; operator-rejection-rules
;; Fully recursive search for op-rej rules.  If operator fails, create
;; a rejection rule for the operator and don't search below this failure
;; (but continue searching in sibling operators).  Otherwise look for
;; failed literal preconditions of the operator (and continue searching 
;; in sibling operators).
(defun operator-rejection-rules (ops goal sole-op)
  (unless
   (null ops)
   (let ((op (first ops)))
     (if (failed-label? (Operator-label op))
	 (let ((instances (instances-of-op op ops)))
	   (create-reject-rule goal instances sole-op)
	   (if (= (length instances) 1)
	       (operator-rejection-rules (rest ops) goal sole-op)
	     (operator-rejection-rules (set-difference ops instances
						       :test #'op-name-equal)
				       goal sole-op)))

       (progn
	 ;; recursively do rest of operators
	 ;; at lower levels
	 (check-lower-levels (Operator-preconditions op))
	 (operator-rejection-rules (rest ops) goal sole-op))))))


;;-------------------
;; check-lower-levels
;;  If the literal precondition is a holds, stop. 
;;  If the precondition is labeled success, continue since a 
;;   failed-operator is possible at a lower level.
;;  If the literal is a failure node, continue (see README for why).
;;  If the literal is recursive but not failure, stop.
;;  Otherwise continue.
;; Notes:  It could continue after recursion, but I think it shouldn't.
;;   This makes slight difference in BW.
(defun check-lower-levels (node)
  (cond 
   ((InternalNode-p node)
    (iter:iterate
     (iter:for operand iter:in (InternalNode-operands node))
     (check-lower-levels operand)))

   ((Literal-p node)
    (let ((label (Literal-label node)))
      (unless (or (success-label? label)
		  (and (rec-label? label)
		       (not (failed-label? label))))
	      (form-operator-rejection-rules node))))))

    


;;--------------
;; op-name-equal
(defun op-name-equal (o1 o2)
  (equal (Operator-name o1) (Operator-name o2)))


;;----------------
;; instances-of-op
;; Returns all instances of op in ops
(defun instances-of-op (op ops)
  (iter:iterate
   (iter:for x iter:in ops)
   (when (op-name-equal op x)
         (iter:collect x))))


