;;-------------------------------------------------------------------
;;-  Author        :  Rob Spiger 
;;- Sponsoring Prof:  Oren Etzioni
;;-    Date        :  2/9/93
;;-------------------------------------------------------------------

;;Modifications:
;;
;; Modification on 10/2/92
;; 
;; Added the negation of the operands of disjunctions to the
;; true-list.
;;
;; So, when evaulating (OR A B C) when A is evaluated ~B and ~C are on
;;  the true-list.
;;
;;---------------------------------------------------------------
;;GLOBAL VARIABLES:
;;
;;*rob-simplify-not-vars* 
;;
;;     Gets set to those variables which should not be
;;      bound in the expression.  They may be variables
;;      that exist in another expression that's related to
;;      this one.
;;     But other variables not in this list might get
;;      set to these variables.  For example, if 
;;      *rob-simplify-not-vars* = (<x>)
;;      and you had the exp (AND (is-equal <x> <y>) (holding <y>)
;;                               (holding <x>))
;;      the variable <x> could not be bound, and the expression
;;      would become (AND (holding <x>) (holding <x>))
;;
;;*rob-simplify-or-passed*
;;
;;     Set to true if no variable substitutions should be made
;;     at this level in the simplification. 
;;
;;     Gets set to true when an OR expression is passed
;;      because you shouldn't instantiate things in 
;;      (is-equal <x> <y>) if there was an OR branch up
;;      above in the expression.  [This is because
;;      you might have something like
;;      (AND (holding <x>)
;;         (OR (is-equal <x> a) (is-equal <x> b)))
;;      and you wouldn't know which instantiation
;;      to pass up for <x>.]
;;
;;     Also gets set to true if the second input to rob-simplify 
;;      specifies that no substitutions should be made.
;;
;;
;;  *rob-simplify-w-negation-axioms*
;;
;;  Will be T if the simplifier should use the negations axioms to
;;  simplify.
;;  Otherwise, it will be nil.
;;
;;
;;PRECEDENCE:
;;
;;If the value of *rob-simplify-or-passed* is true then
;; the value of *rob-simplify-not-variables* is irrelevant.
;;--------------------------------------------------------------
;;
;;
;;
;;The only exported function is the function rob-simplify.
;;But the user may rewrite rob-simplify-predicate to interface
;; the simplifier for other usages.
;;Don't forget to call initialize-rob-simplify to set up the
;; negation axioms for the simplifier.


(defun initialize-rob-simplify ()
 (setq *negation-list-both-ways*
  (remove-duplicates
   (append *negation-list* 
           (mapcar #'(lambda (x) `(,(second x) ,(first x)))
            *negation-list*))
   :test #'equal))
 (setq *negation-list-predicates*
  (remove-duplicates 
   (mapcar #'(lambda (ax) (predicate-name (first ax)))
    *negation-list-both-ways*))))

;;Notes on general theory of simplification.
;;
;;When you have (AND x y) both x and y must be true at the
;;  same time for the expression to be true.  In other words,
;;  if either x being true would negate y, or
;;            y being true would negate x, then the expression must
;;  be false.
;;
;;Likewise, if you have (OR x y) in order for the expression to
;; be reduced to false, both x and y must be false at the same
;; time.  So, if one can show that ~y negates ~x or
;; ~x negates ~y then one has shown the expression to be true.
;;
;;
;;Futhermore, if x is true and x negates y then 
;; the simplifier infers that x implies ~y.
;; As an example:
;;
;; You have the negation axiom (dr-open <X>) negates (locked <x>)
;;
;; Then the expression 
;;      (AND (dr-open <V1>) (OR (~ (locked <V1>)) (holding b)))
;;
;; would reduce to (dr-open <V1>) because the disjunction would
;;     evaluate to true since (~ (locked <V1>)) would be infered to
;;     be true.

;;----------------------------------------------------------------
;;Beginning of some gory details.
;;----------------------------------------------------------------  
;;
;;The simplifier does:
;;
;;(OR A ~A) = T
;;(AND A ~A) = F
;;
;;(OR T A) = T
;;(OR F A) = (OR A)
;;(OR A) = A
;;(AND A) = A
;;(AND F A) = F
;;(AND T A) = (AND A)
;;
;;Where T = True
;;      F = False
;;
;;
;;
;;Each function defines a different simplifying transformation, the
;;functions are applied repeatedly until no transformation makes any
;;additional progress.  At this point, the simplifier quits.
;;
;;
;;But to analyze the first two laws above, A has to be an atomic
;;expression or reducable by the simplifier to an atomic expression.
;;(It can't be a complicated expression, meaning in Boolean logic it 
;;would have to be a single variable, or reduce to a single variable.)  
;;
;;The simplifier would NOT be able to reduce:
;;
;;(OR (AND B A) (AND B ~A)) at all.
;;
;;It would be able to reduce (OR (AND B (OR A ~A)))
;;in generally the following internal steps:
;;
;;(OR (AND (B (OR A ~A)))
;;(OR (AND (B T)))
;;(OR (AND B))
;;(OR B)
;;B
;;
;;----------------------------------------------------
;;The way the simplifier works is it has a "true list" which is a list 
;;of what is known to be true at each level of the expression.
;;Basically, for a conjunctive expression to end up being true,
;;all of is operands must end up being true, they are put into
;;the true list, but, the true list only holds atomic expressions.
;;
;;For example, if you have the expression:
;;
;;(AND A B C (OR ~C B) (AND D (OR ~A (OR D C) E)))
;;
;;At each level of the unsimplified expression these would be the
;;true lists:
;;
;;Depth  Expression level                    True list
;;----------------------------------------------------------------
;;1      (AND A B C sub-exp1  sub-exp2)      a,b,c
;;2      (OR ~C B)                           A,B,C   <-sub-exp1
;;2      (AND D sub-exp3)                    A,B,C,d <-sub-exp2
;;3      (OR ~A sub-exp4 E)                  A,B,C,D <-sub-exp3
;;4      (OR D C)                            A,B,C,D <-sub-exp4
;;
;;Also important is if a variable was just added to the true list at the
;;that level.  If it was just added, it's not used to simplify itself at that
;;level.  The variables above in the true list are in lowercase if they 
;;have just been added.
;;
;;In the table above, the simplifier sees if knowing what's in the 
;;true-list is true can help simplfiy the expression at each level.
;;For example, by appling the knowledge of the true list, the
;;above table could be simplfied to the following:
;;
;;Depth  Expression simplified by applying the true-list
;;--------------------------------------------------------------
;;1      (AND A B C sub-exp1 sub-exp2)
;;2      (OR F T)
;;2      (AND D sub-exp3)
;;3      (OR F sub-exp4 E)
;;4      (OR T T)
;;
;;This would eventually simplify the entire expression to:
;;
;;(AND A B C D)
;;
;;But the major flaw is if the conjunctive operands aren't atomic
;;expressions, they aren't added to the true-list.  
;;
;;Now you should see why the simplifier wouldn't reduce
;;(OR (AND B A) (AND B ~A)) at all.
;;
;;----------------------------------------------------------------
;;End of gory details
;;----------------------------------------------------------------  
;;



;;-------------------------------------------------------------
;;ONLY OUTSIDE CALLABLE FUNCTION
;;inputs: 1.  an expression to be simplified
;;        2.  optional list of those variables which
;;            will not be bound in the expression.
;;            (see *rob-simplify-not-vars*)
;;            If this is set to 'No-sub then no
;;            substituitions will occur.
;;        3.  A list of things know to be true.
;;            For example, when anayzling the conditional
;;            of a control rule, one knows that the negation
;;            of the goals are true.  
;;        4.  Set to true if you wish to access then negation axioms.
;;            Default is nil, which doesn't use them.
;;
;;Outputs: 
;;  If the second input param is set to 'No-sub then output
;;    is merely the simplified expression. 
;;
;;  If the second param isn't set to 'No-sub then output =
;;         (exp, blist) 
;;         exp is the simplified expression.
;;         blist is the bindings which were instantiated.
;;
;;
;;
(defun rob-simplify (exp &optional (simplify-not-vars nil) 
                    (true-list nil) (use-axioms nil)
					(use-invariant-list nil))
 (if use-axioms
  (setq *rob-simplify-w-negation-axioms* t)
  (setq *rob-simplify-w-negation-axioms* nil))
 (if use-invariant-list 
  (setq *rob-simplify-w-invariant-list* t)
  (setq *rob-simplify-w-invariant-list* nil))
 (if (eq simplify-not-vars 'No-sub)
  (progn  ;;don't instantiate anything
   (setq *rob-simplify-not-vars* nil)
   (setq *rob-simplify-or-passed* T))
  (progn  ;;do instantiate
   (setq *rob-simplify-not-vars* simplify-not-vars)
   (setq *rob-simplify-or-passed* nil)))
 (let (
   (result (sub-rs exp true-list)))
  (if (eq simplify-not-vars 'No-sub)
   (first result)
   result)))

;;-------------------------------------------------------------
;; USER SETABLE-FUNCTIONS
;;-------------------------------------------------------------
;; Function: rob-simplify-predicate
;;
;; Input:  1.  A possibly-negated-predicate to be simplifed.
;;         2.  A list of those things known to be true.
;;
;; Output:
;;Must return two things in a list.
;;  1.  What gets put back into the expression instead of the
;;      predicate.
;;  2.  The bindings that resulted from simplifying the predicate
;;
;; Function must conform to the following or correctness of simplified
;;   expression is NOT guaranteed:
;;
;;1.      *rob-simplify-not-vars* holds those variables which
;;        should not be instantiated as something else.
;;
;;2.      no bindings should be made if *rob-simplify-or-passed* is T
;; 
;;3.      all bindings made MUST be passed back.
;;
;;4.      Should not use the negation axioms unless *rob-simplify-w-
;;        axioms* is set to T.  Also, won't use the invariant list
;;        unless this is true.
;;

(defun rob-simplify-pred (pred true-list)
 (simplify-predicate pred true-list))

;;( (rob-simplify-pred-default pred))
;;------------------------------------------------------------------
(defun rob-simplify-pred-default (pred true-list)
 (if (AND (eq (predicate-name pred) 'is-equal)
          (able-to-simplify-is-equal pred))
  `(T ,(simplifier-is-equal-to-blist pred))
  (list (rob-simplify-evaluate pred true-list))))  
;;------------------------------------------------------------------

;;returns pred or nil
(defun rob-simplify-evaluate (pred true-list)
 pred)   ;;need to changed to look at false lists
         ;;need to see if (is-equal a b) is possible
;;------------------------------------------------------------------

;;returns true of false
;;returns false if the is-equals is false and not possible
;;returns false if the is-equals might be true but
;; can't be instantiated because both of it's variables are
;; on the don't instantiate list
;;------------------------------------------------------------------

(defun able-to-simplify-is-equal (pred)
 (unless *rob-simplify-or-passed*
  (OR (treat-as-variable (second pred))
      (treat-as-variable (second pred)))))
;;------------------------------------------------------------------

(defun treat-as-variable (var)
 (AND (rob-is-var? var)
      (not (member var *rob-simplify-not-vars*))))
;;------------------------------------------------------------------

;;converts the is-equal to a blist
;; stores the instantiation
;; handles tricky stuff of what type of variable to
;; have go to what type.
;;
(defun simplifier-is-equal-to-blist (pred)
 (let* (
   (var1 (second pred))
   (var2 (third pred))
   (bpair
    (cond
     ((treat-as-variable var1)
      `(,var1 ,var2))
     ((treat-as-variable var2)
      `(,var2 ,var1))
     (t (format t "~%two not-variables passed to function")
        (format t "~% simplifier-is-equal-to-blist")))))
  (list bpair)))
;;------------------------------------------------------------------
;; END USER SETABLE FUNCTIONS
;;------------------------------------------------------------------

;;GLOBAL VARIABLES

(defvar *rob-simplify-or-passed* nil)
  ;;true if no variable substitutions should be made
  ;; at this level in the simplification. 
  ;; Gets set to true when an OR expression is passed
  ;;  because you shouldn't instantiate things in 
  ;;  (is-equal <x> <y>) if there was an OR branch up
  ;;  above in the expression.  [This is because
  ;;  you might have something like
  ;;  (AND (holding <x>)
  ;;     (OR (is-equal <x> a) (is-equal <x> b)))
  ;;  and you wouldn't know which instantiation
  ;;  to pass up for <x>.]

(defvar *rob-simplify-not-vars* nil)
  ;; Gets set to those variables which should not be
  ;; instantiated in the expression.  They may variables
  ;; that exists another expression that's related to
  ;; this one that you don't want to bind.
  ;; But other variables not in this list might get
  ;;  set to these variables.  For example, if 
  ;; *rob-simplify-not-vars* = (<x>)
  ;; and you had the exp (AND (is-equal <x> <y>) (holding <y>)
  ;;                          (holding <x>))
  ;; the variable <x> could not be changed, and the expression
  ;; would become (AND (holding <x>) (holding <x>))
  ;;
(defvar *rob-simplify-w-negation-axioms* nil)
  ;;Set to true only if it is specified the negation axioms
  ;; should be used.
(defvar *rob-simplify-w-invariant-list* nil)
  ;;Set to true only if the invariant list is to be used to 
  ;; simplify.  Care should be taken because using the invariant
  ;; list will simplify a lot, but does result in a loss of 
  ;; information.  Consider the following two control rules,
  ;; which the first of which was simplified with the invariant
  ;; list and the second without.  The first one holds much less
  ;; information:

;;((|sr48|
;;  (LHS
;;   (AND (CURRENT-NODE <NODE>) (CANDIDATE-GOAL <NODE> (ARM-EMPTY))
;;        (CANDIDATE-GOAL <NODE> (INROOM ROBOT <G-1>))
;;        (KNOWN <NODE>
;;               (AND (IS-KEY <V40> <V46>) (DR-TO-RM <V40> <V45>)
;;                    (~ (DR-OPEN <V40>)) (~ (UNLOCKED <V40>))
;;                    (~ (HOLDING <V46>))))))
;;  (RHS (PREFER GOAL (INROOM ROBOT <G-1>) (ARM-EMPTY))) (CUM-SAVINGS 30)
;;  (EST-SAVINGS 30) (PROBLEM NIL) (MATCH-TIME 0) (UNIQUE-SIG (GOAL-PREF <V196>))
;;  (STATIC T) (WAS-LEARNED STATIC) (PRIORITY 0) (RULE-TYPE GOAL-PREF))
;;
;;((|sr49|
;;  (LHS
;;   (AND (CURRENT-NODE <NODE>) (CANDIDATE-GOAL <NODE> (ARM-EMPTY))
;;        (CANDIDATE-GOAL <NODE> (INROOM ROBOT <G-1>))
;;        (KNOWN <NODE>
;;               (AND (IS-KEY <V40> <V46>) (DR-TO-RM <V40> <V45>)
;;                    (DR-TO-RM <V40> <G-1>) (IS-DOOR <V40>)
;;                    (CONNECTS <V40> <G-1> <V39>) (~ (DR-OPEN <V40>))
;;                    (~ (UNLOCKED <V40>)) (~ (HOLDING <V46>))))))
;;  (RHS (PREFER GOAL (INROOM ROBOT <G-1>) (ARM-EMPTY))) (CUM-SAVINGS 30)
;;  (EST-SAVINGS 30) (PROBLEM NIL) (MATCH-TIME 0) (UNIQUE-SIG (GOAL-PREF <V196>))
;;  (STATIC T) (WAS-LEARNED STATIC) (PRIORITY 0) (RULE-TYPE GOAL-PREF))
;;
;; But this kind of simplification is good if you are only trying to
;; get the expression to simplify to T or nil and don't mind losing
;; information in the process.  For example, when you are trying to 
;; compute prerequisite violations and you want to use every bit of 
;; simplification you can get to simplify the failure conditionals to
;; T.
;;
;;




;;-------------------------------------------------------------
;;returns a the simplified (list exp)
;;and the list of bindings which occured all
;; across that expression.
;;-------------------------------------------------------------
;;Function: sub-rs
;;
;;Does all the work.  Recursively simplifies an expression.
;;
(defun sub-rs (exp true-list)
 (setq exp (drop-and-or-if-one-operand exp))
 (cond
  ((atom exp)
   `(,exp nil))
  ((keyword-or exp)
   (rob-simplify-or exp true-list))
  ((keyword-and exp)
   (let* (
     (result   (rob-simplify-and exp true-list))
     (blist    (second result))
     (result   (drop-and-or-if-one-operand
                (scan-t-nil-from-and
                 (raise-top-level-ands (first result))))))
    `(,result ,blist)))

  ((atomic-formula? exp)
   (rob-simplify-pred-using-true-list exp true-list))
  (T 
   `(,exp nil))))  ;;do nothing for forall or exists

;;------------------------------------------------------------------
(defun rob-simplify-pred-using-true-list (exp true-list)
 (cond
  ((member exp true-list :test #'cannot-mutually-exist)
   `(nil nil))
  ((member exp true-list :test #'equal)
   `(T nil))
  ((member exp true-list :test #'exactly-support-to-be-t)
   `(T nil))
  ((AND *rob-simplify-w-invariant-list* (rob-holds? exp true-list))
   `(T nil))
  (t
   (rob-simplify-pred exp true-list))))

;;------------------------------------------------------------------
;;Function cannot-mutally-exist
;;returns T iff both predicates cannot be true at the same time.

(defun cannot-mutually-exist (pred1 pred2)
 (rob-simplify-negate-exactly pred1 pred2))

;;------------------------------------------------------------------
;;Function exactly-support-to-be-t
;;Returns T iff pred1 implies that pred2 is true.
;;  (Will be true if pred1 negates ~pred2 exactly.
;;
(defun exactly-support-to-be-t (pred2 pred1)
 (rob-simplify-negate-exactly pred1 (negate-exp pred2)))


;;------------------------------------------------------------------
;;Simplifies a disjunctive expression.
;; When an OR statement is passed, no substitutions should be made
;;  in it's operands, so *rob-simplify-or-passed* is set.
;;
;;This function tries to simplify it's operands.  If some
;;simplification takes place, it calls itself again to try to simplify
;; more.  This happens until no further simplification takes place.

;; 

(defun rob-simplify-or (exp true-list)
 (let* (
   (do-reset (not *rob-simplify-or-passed*)))
      ;;true if this call is the one changing the global variable
      ;; and so if at the end of the call it should be reset to FALSE
  (setq *rob-simplify-or-passed* t)
  (let* (
    (result   `(OR ,@(sub-rob-simplify-or (rest exp) true-list)))
      ;;blist is nil because nothing was bound below the OR expression
    (blist    nil)
    (new-exp  (drop-and-or-if-one-operand
               (scan-t-nil-from-or
                (raise-top-level-ors result)))))
   (when do-reset
    (setq *rob-simplify-or-passed* nil))
   (if (equal exp new-exp)  ;;did expression simplify more on this call?
    `(,new-exp ,blist)      ;;no, return expression
    (sub-rs new-exp true-list))))) ;;yes, so try to simplify more.
  

;;------------------------------------------------------------------
;;When simplify an expression like (OR A B C) when you try
;; to simplify A you can assume both B and C are false.
;; Likewise, when you simplify B you can assume both A and C are 
;;  false.  (The same is true when simplifying C.)
;;
;;So what this function does when simplifying A it puts ~B and ~C
;; into the true-list.
;;It does likewise when simplifying B or C.
;;



(defun sub-rob-simplify-or (operands true-list)
 (let* (
   (operands (remove-duplicates operands :test #'equal))
   (false-list (remove-if-not #'atomic-formula? operands)))
  (remove-duplicates
   (mapcan 
     #'(lambda (operand)
        (let* (
          (false-list (remove-if #'(lambda (x) (equal x operand)) false-list))
          (true-list  (true-list-append true-list 
                              (mapcar #'negate-exp false-list)))
          (result     (first (sub-rs operand true-list))))
         (if (keyword-or result)
          (rest result)
          (list result))))
     operands) :test #'equal)))


;;------------------------------------------------------------------
;;returns T if the operands simplify to true


(defun can-any-two-operands-not-be-false-at-same-time (operands)
 (let* (
   (atomic-operands (remove-if-not #'atomic-formula? operands))
   (negated-atomic-operands (mapcar #'negate-exp atomic-operands)))
  (when (> (length atomic-operands) 1)
   (not (not
    (intersection negated-atomic-operands 
                  negated-atomic-operands
                  :test #'cannot-mutually-exist))))))
    
     
 
   
   
;;------------------------------------------------------------------
;;Simplifies a conjuctive expression.
;;exp = what to simplify
;;true-list = what is known to be true.  On the first call in the
;;recursion of this algorithm, org-true-list is set to the true-list.
;; This is because org-true-list is what was known to be true from
;; the level above this conjunctive expression.
;;
;;Both last-blist and org-true-list should never be set when not
;;calling this function from itself.
;;
;;This function tries to simplify the conjunctions operands.  If
;;completely no simplification takes place, the value is returned.
;; If some simplification takes place, it calls itself again to try to
;; simplify more.  This happens until no further simplification takes
;; place.

(defun rob-simplify-and (exp true-list &optional (last-blist nil)
                                                 (org-true-list 'please-set))
 (let* (
   (org-true-list (if (eq org-true-list 'please-set)
                   true-list
                   org-true-list))
   (this-lvl-atoms (remove-duplicates (rest exp) :test #'equal))
   (this-lvl-atoms (remove-if-not #'atomic-formula? this-lvl-atoms))
   (true-list      (true-list-append true-list this-lvl-atoms))


   (result     (sub-rob-simplify-and (rest exp) true-list last-blist org-true-list))
    ;;returns a list with each element as (simplified operand, blist)
    ;;the last operand had the largest blist.  Also, added later and
    ;;tacked onto the end of all this is the new-true-list

   (new-true-list (remove-duplicates (first (last result)) :test #'equal))
   (result     (butlast result))

   (operands   (save-nils-mapcan #'first result))
   (operands   (remove-duplicates operands :test #'equal))
   (blist      (second (first (last result))))
    ;;the most current blist is the last one used to instantiate with
   (new-exp    `(AND ,@operands))
    ;;put the AND back in from of the operaands
   (new-result `(,new-exp ,blist)))
;;(format t "~%exp: ~s" (np exp))
;;(format t "~%blist: ~s" blist)
;;(format t "~%new-exp: ~s" (np new-exp))
;;(format t "~%is-equal: ~s" (equal exp new-exp))
;;(format t "~%true-list: ~s" true-list)
;;(format t "~%org-true-list: ~s" org-true-list)
;;(format t "~%new-true-list: ~s" new-true-list)
;;(format t "~%union : ~s" 
;;              (union org-true-list true-list :test #'equal))
  (if (and 
       (set-equality new-true-list 
              (union org-true-list true-list :test #'equal))
       (equal new-exp exp)) ;;did we simplify some on this call?
   new-result              ;;no, so that's as much as we can simplify
						   ;;return the result
   (rob-simplify-and new-exp new-true-list blist org-true-list))))
                           ;;yes, so lets try to simplify again.
       ;;Also, something subtle here is that
       ;;the blist may have gotten larger in processing 
       ;;each operand.  So, another call is needed to make sure
       ;;all the operands got instantiated with the complete blist.
;;------------------------------------------------------------------
;;Instaniate first operand with blist
;;Simplify first operand
;; look at new binding list returned
;; if new-blist conflicts with old blist return a FALSE
;;Recursively call function with bigger blist.
;;
;; When it's all done, the function rob-simplify-and checks to
;;  make sure the expression hasn't changed.  (This will happen
;;  if the blist grows or something inside any operand gets changed.)
;;  If the expression has changed, it tries to simplify again, with
;;  the bigger blist.
;;
;; What the above is saying is if the blist grows after the first
;; operand has already been simplified.  You have a problem because
;; the first operand only got instantiated with part of the blist,
;; and not the whole blist.  This problem is solved by re-calling
;; this function with all the operands again with the cumulative
;; blist from the previous call.  This brings the complete blist
;; back to the first operand so it can be instantiated with it.
;;
;;Output: Returns each operand paired with current blist.
;;         Where the current blist is the blist found from
;;          combining the blist of all the previous operands.
;;
;;Org-true-list is those things known to be true from a level above
;; this conjunctive expression.
;;
;;True-list is the union of what is known to be true in this 
;; conjunctive expression and Org-true-list.
;;
;;This function is also recalled if the true-list changes just like
;; it is recalled if the blist changes.
;;
(defun sub-rob-simplify-and (operands true-list blist org-true-list)
 (if (null operands)
  (list true-list);;return new true list at very end of return value

  (let* (
    (operand        (instantiate (first operands) blist))
    (true-list      (instantiate true-list        blist))
    (true-list      (remove-duplicates true-list :test #'equal))
;;instantiatation causes duplicates to appear in true-list
;; so they are removed.
    (other-operands (rest operands))

    (true-list      (remove-if #'(lambda (x) 
                                  (equal x operand)) 
                     true-list))  ;;remove this atom from true-list
;;Have to prevent a predicate from being put on the true list to
;; simplify itself, otherwise everything would simplify itself to T.

    (true-list      (union true-list org-true-list :test #'equal))

;;But if the predicate was on the true list from higher up, then
;; it is okay if the predicate simplies itself at a deeper level
;; in the whole expression.  This is allowed by adding the orginal
;; true list which comes from higher up in the expression after
;; removing the predicate from the true list generated at this level.
;;
;; For example, (AND A (AND (OR A B) C)) should simplify to (AND A C)
;; Since the A from the conjunctive expression should be used in the
;;  true list to simplify the A in the conjunctive expression.


    (result         (rob-simplify-and-operand true-list operand))
    (operand-exp    (first result))
;;    (true-list      (if (and (atomic-formula? (first operand-exp))
	 ;;                             (not (null (first operand-exp))))
	 ;;))   old code
    (true-list      (if (atomic-formula? (first operand-exp))
                      (union (list (first operand-exp)) true-list)
                      true-list))
    (true-list      (true-list-append nil true-list))
 ;;expand true list to include this AND expressions previous
 ;; operands
    (new-blist      (second result))

    (combo-blist-cond (combine-or-fail-blists blist new-blist))
    (combined-blist   (first  combo-blist-cond))
    (new-cond         (second combo-blist-cond)))
   (unless (equal new-cond '(AND))
    (setq other-operands   (cons new-cond other-operands)))

   (cons `(,(first result) ,combined-blist)
    (sub-rob-simplify-and other-operands true-list combined-blist org-true-list)))))

;;------------------------------------------------------------------

;;Tries to simplify an operand of the and AND expression.
;; Pretty much just calls the simplifer on the operand
;;  and returns the result.  But if the simplified operand's
;;  keyword is AND then the AND is removed because it is
;;  redunant because we already know this is a conjunctive 
;;  expression.

(defun rob-simplify-and-operand (true-list operand)
 (let* (
 
  (result (sub-rs operand true-list))
   (exp    (first result))
   (blist  (second result)))
  (if (keyword-and exp)
   `(,(rest exp) ,blist)
   `(,(list exp) ,blist))))

;;------------------------------------------------------------------


;;returns a (blist, conditional) when the conditional is true, the
;;   bindings lists have succeeded in being matched together.
;;
;;need to check each bpair in blist1 against all the bpairs
;;  in blist2 and look for conflicts.
;;
;;May have case were two variables are assigned to same one, this
;; is okay.
;; For example, (<y> <x>) and (<z> <x>) don't conflict it simply
;;  says <y>=<x>=<z>.  
;;
;;  But (<x> <y>) and (<x> <z>) means the same thing too.  Only
;;   it also requires that (is-equal <y> (z>), information that
;;   was already there in the above case, so we turn the conditional
;;   (is-equal <y> <z>).

(defun combine-or-fail-blists (blist1 blist2)
 (let (
   (conditionals nil))
  (iter:iterate
  (iter:for bpair1 iter:in blist1)
   (iter:iterate
   (iter:for bpair2 iter:in blist2)
    (let (
      (var11 (first  bpair1))
      (var12 (second bpair1))
      (var21 (first  bpair2))
      (var22 (second bpair2)))
     (when (eq var11 var21)
      (unless (eq var12 var22)
       (setq conditionals (cons `(is-equal ,var12 ,var22)
                                conditionals)))))))
  (let (
    (new-blist (union blist1 blist2 :test #'equal)))
   `(,new-blist (AND ,@conditionals)))))


;;------------------------------------------------------------------
;;------------------------------------------------------------------
;;Scans an AND expression or sub-expression and loks for
;;sub-expressions which are ANDed and pulls them up to top level.
;;
;;  (AND (holding <x>) (AND (holding <y>) (asdf))) becomes -->
;;  (AND (hodling <x>) (holding <y>) (asdf))
;;

(defun raise-top-level-ands (exp)
 (if (keyword-and exp)
  `(AND ,@(sub-raise-top-level-ands exp))
  exp))


(defun sub-raise-top-level-ands (exp)
 (if (atom exp)
  (list exp)
  (if (keyword-and exp)
   (save-nils-mapcan #'sub-raise-top-level-ands (rest exp))
   (list exp))))
;;------------------------------------------------------------------
;;Scans an OR expression or subexpression which are ORs and brings
;; them up to the top level.
;;
;; For example, (OR (holding <x>) (OR (asdf) (fda))) becomes
;;              (OR (holding <x>) (asdf) (fda))
;;
(defun raise-top-level-ors (exp)
 (if (keyword-or exp)
  `(OR ,@(sub-raise-top-level-ors exp))
  exp))

(defun sub-raise-top-level-ors (exp)
 (if (atom exp)
  (list exp)
  (if (keyword-or exp)
   (mapcan #'sub-raise-top-level-ors (rest exp))
   (list exp))))
;;------------------------------------------------------------------
;;Scans the operands of an AND expression for T's or nil's
;;  It simply removes T's
;;  If it finds a nil, it returns the whole expression as false.

(defun scan-t-nil-from-and (exp)
 (unless (equal exp '(AND))
  (if (keyword-and exp)
   (unless (remove-if-not #'equal-nil exp)
    (let (
      (exp (remove-if #'equal-t exp)))
     (if (equal exp '(AND))
      T
      exp)))
   exp)))
;;------------------------------------------------------------------
;;Scans the operands of an OR expression for T's or nil's.  
;;  It simply removes nils.
;;  If it find a T, it returns the whole expression as T
;;
(defun scan-t-nil-from-or (exp)
 (unless (equal exp '(OR))
  (if (keyword-or exp)
   (if (find-if #'equal-t exp)
    T
    (let (
      (exp (remove-if #'equal-nil exp)))
     (unless (equal exp '(OR))
      exp)))
   exp)))
;;------------------------------------------------------------------
;;
;; Recursively removes layers of sinlge operand conjuctions or 
;;   disjunctions.  For example, 
;;   (AND (OR (AND (Holding <x>)))) --> (Holding <x>)
;;
(defun drop-and-or-if-one-operand (exp)
 (let (
   (result (sub-drop-and-or-if-one-operand exp)))
  (if (equal result exp)
   result
   (drop-and-or-if-one-operand result))))

;;------------------------------------------------------------------
;; Handles two cases:
;;   (OR x)  becomes x.
;;   (AND x) becomes x.
;; Otherwise does nothing.
;;
(defun sub-drop-and-or-if-one-operand (exp)
 (if (atom exp)
  exp
  (if (AND (= 2 (length exp))
           (OR (keyword-and exp)
               (keyword-or  exp)))
   (second exp)
   exp)))
;;------------------------------------------------------------------
;;Returns T if the expression is T
(defun equal-t (x)
 (eq x t))
;;------------------------------------------------------------------
;;Returns T if the expression is NIL
(defun equal-nil (x)
 (null x))
;;------------------------------------------------------------------
;;All of the following functions return T or nil if the keyword
;; of exp is the one in the function name.
;;------------------------------------------------------------------
(defun keyword-and (exp)
 (unless (atom exp)
  (eq 'and (first exp))))
;;------------------------------------------------------------------
(defun keyword-or (exp)
 (unless (atom exp)
  (eq 'or  (first exp))))
;;------------------------------------------------------------------
(defun keyword-exists (exp)
 (unless (atom exp)
  (eq 'exists (first exp))))
;;------------------------------------------------------------------
(defun keyword-forall (exp)
 (unless (atom exp)
  (eq 'forall (first exp))))
;;------------------------------------------------------------------
(defun keyword-not-exists (exp)
 (unless (atom exp)
  (when (eq '~ (first exp)))
   (keyword-exists (second exp))))
;;------------------------------------------------------------------




;;-----------------------------------------------------------------
;; Need to keep nil values when combining operands for AND
;; Mapcan doesn't keep these in when it applies append, but this
;;  function saves them.
;;
;;Examples:
;;
;;<cl> (mapcan #'(lambda (x) nil) '(1 2 3))  RETURNS NIL
;;
;;<cl> (save-nils-mapcan #'(lambda (x) nil) '(1 2 3))
;;     RETURNS (NIL NIL NIL) 
;;
;;
(defun save-nils-mapcan (func list)
 (let* (
   (result-with-nils   (mapcar func list))
   (result-with-falses (mapcar #'(lambda (x) (if (null x) '(FALSE) x))
                           result-with-nils))
   (result-with-falses (apply #'append result-with-falses))
   (result-with-nils   (mapcar #'(lambda (x) (unless (eq x 'FALSE) x))
                           result-with-falses)))
   result-with-nils))


;;depending on whether or not the negation axioms should be used,
;; returns T if pred1 negates pred2.

(defun rob-simplify-negate-exactly (pred1 pred2)
 (if *rob-simplify-w-negation-axioms*
  (rob-negate-exactly-w-axioms pred1 pred2)
  (rob-negate-exactly          pred1 pred2)))


;;returns true if by using the invariant list one can
;; figure out that predicate is true based on what
;; in known to be true in the true list.

;;Notes:  I (Rob) am trying to apply the holds? function to be more
;;  generally so it can be used by the simplifier.
;;  
;;  I assume that the goal stack is equivalent to a list of things
;;  which are known to be false.  I do not understand the importance
;;  of the order of the goal stack, or reversing that order.

; A lot of issues come up when using the invariant list to simplify expressions.
; One has to be sure that one doesn't conclude more information that the invariant list
; contains.  The invariant list was orginally designed to simplify what needed to be subgoaled
; upon when creating the PSG.  This led to certain implicit assumptions.  These include 
; assumptions like static predicates would never be on the goal stack.  Also, any variable was 
; unbound since the relevant goals had constants in the parts that mattered.  (The psgs
; are created by subgoaling on things like (on g-1 g-2) where g-1 and g-2 are constants.  Any other
; subgoals which arise while creating the PSG might have variables in them, but what those 
; variables bind to eventually are unimportant.)  When 
; using the invariant list through the holds? function to simplify expressions one does have
; a keen interest in what the other variables are bound to.
; For example, when creating the PSG it is okay to conclude something like
;   (on <V1> g-1) when it is known that (~ (holding g-1)) and (~ (clear g-1)) because what
;   <V1> comes out to be doesn't matter.  But when simplifying an expression <V1> is bound
;   to something specific already so the conclusion of (on <V1> g-1) would not be correct.
;
; The above problems can be solved by a little preparation before calling the holds? function
; when using the information the holds? function returns to simplify an expression.  
; 1.  Replace all variables by constants with the same name.
; 2.  Remove all static predicates from the stack.
; 3.  Call holds? are return the result.


(defun rob-holds? (predicate true-list)
 (let* (
   (false-list (mapcar #'negate-exp true-list))
   (false-list (remove-if  #'is-static-pred false-list))
   (pred-false-list `(,predicate ,false-list))
   (variables  (find-variables pred-false-list))
   (blist      (mapcar #'(lambda (var) `(,var ,(generate-random-const))) variables))
   (pred-false-list (instantiate pred-false-list blist))
   (pred       (first pred-false-list))
   (false-list (second pred-false-list)))

  (holds? pred false-list)))


;;appends the two true-lists together, but removes
;; any unacceptable elements for true lists from listb.
;;
;; Unacceptable elements are ones with poor predicates
;; for being of any use in simplifying the rest of the
;; expression.

(defun true-list-append (lista listb)
 (let* (
   (listb  (remove-if #'bad-true-list-predicate listb)))
  (union lista listb :test #'equal)))

;;returns T if an atomic-formula should not be put into
;; the true list.
(defun bad-true-list-predicate (exp)
 (or (null exp)
     (member (predicate-name exp)
          '(known
            candidate-goal
            current-node))))


          
         
