;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;
;;  Static Analyzer for Full Prodigy language
;;
;;  label-PSG.lisp contains the lisp code to label the already
;;  created PSG the top level function is called label-PSGs
;;
;;  Notes: All the labeling functions follow the same format.
;;   Label-* returns a list containing (label failure-condition).
;;   For information on labeling logic see label.doc.
;;
;;
;;  Author: Julie Roomy
;;  Sponsoring Professor: Oren Etzioni
;;  Mods: Rob Spiger
;;
;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;;


;;----------
;; label PSGs
;; Labels each level of the PSG proprogating failure conditions.
(defun label-PSGs (roots)
  (iter:iterate
   (iter:for root iter:in roots)
     (label-literal root)
     (special-simplify-root-failure-cond root)

))

;;Sees if the failure condition being true and the negation of the goal is a valid
;; state.  If not, then the failure condition can never be true when the goal isn't true.
;; So, the failure condition can be easily simplified to false.  This occurs in two PSG
;; roots in the blocksworld.  Should not be removed.
;;

(defun special-simplify-root-failure-cond (root)
 (let* (
   (lit (literal-name root))
   (lit (make-all-goal-consts-into-vars lit))
   (fc  (literal-failure-condition root))
   (fc  (make-all-goal-consts-into-vars fc))
   (true-lits (if (keyword-and fc) (remove-if-not #'atomic-formula? fc)
                                   (when (atomic-formula? fc)
                                    (list fc)))))  ;;known to be true if fc is going to be t
   (when fc 
    (when (eq t (first (rob-simplify lit nil true-lits t t)))
     (progn
      (setf (literal-label root) 'UNKNOWN)
      (setf (literal-failure-condition root) nil))))))





;;---------------
;; Label Operator
;; Label the operator based upon the following rules.
;; Operator gets same label and failure condition as the node pointed
;;    to by its preconditions field.
(defun Label-Operator (operator)

  ;; if the operator is already labeled just return the label.
  (if (Operator-label operator)
      (list (Operator-label operator) (Operator-failure-condition operator))

    ;; otherwise determine its label from its preconditions
    (let* ((precond (Operator-preconditions operator))
	   (label-plus-failure
	    (cond ((InternalNode-p precond)
		   (Label-Internal-Node precond))
		  ((Literal-p precond)
		   (Label-Literal precond)))))
      (setf (Operator-label operator) (first label-plus-failure))
      (setf (Operator-failure-condition operator) (second label-plus-failure))
      label-plus-failure)))


;;--------------------
;; Label Internal Node
;; Based upon the type of the Internal Node call the appropriate function.
;; each of the label-KEYWORD functions is based upon a chart in label.doc
(defun Label-Internal-Node (Internal)

  ;; if the internal-node is already labeled just return the label.
  (if (InternalNode-label Internal)
      (list (InternalNode-label internal)
	    (InternalNode-failure-condition internal))

    ;; otherwise determine its label from its preconditions
    (let* ((type (InternalNode-name Internal))
	   (childrens-labels (Label-Internal-children Internal))

	   (node-label
	    (cond ((or (eq type 'AND)
		       (eq type 'EXISTS))
		   (label-AND internal childrens-labels))
		  ((eq type 'OR)
		   (label-OR Internal childrens-labels))
		  ((eq type 'NOT)
		   (label-NOT Internal childrens-labels))
		  ((eq type 'FORALL)
		   (label-FORALL Internal childrens-labels))
		  (T "error"))))

      (setf (InternalNode-label Internal) (first node-label))
      (setf (InternalNode-failure-condition Internal) (second node-label))
      node-label)))



;;------------------------ 
;; label-Internal-children
;; collect all the labels and failure conditions of the children.
(defun label-Internal-children (node)
  (iter:iterate
   (iter:for operand iter:in (InternalNode-operands node))
   (if (InternalNode-p operand)
       (iter:collect (Label-Internal-Node operand)))
   (if (Literal-p operand)
       (iter:collect (Label-Literal operand)))))
  

;;-----------------------------------------
;;-----------------------------------------
;; labeling logic (see label.doc)

;;----------
;; label-AND
;; Use label chart to determine node's label and failure conditions
;;   based upon it's children.
(defun label-AND (and-node childrens-labels)
  (let*

      ;; if any child fails, OR together failure condition
      ((failure-condition
	(join-failure-conditions childrens-labels 'OR))

       ;; if any child labeled unknown or recursive failure mark recursion
       (recursion
	(any-child-recursive childrens-labels))

       ;; now determine the label
       (label-and-cond
	(if (null failure-condition)
	    (if (null recursion)

		;; no failed children and no recursion then success
		(list 'SUCCESS nil)
	  
	      ;; no failed children but at least one recursion then unknown
	      (list 'UNKNOWN nil))

	  ;; at least one failed child but no recursion then pure failure
	  (if (null recursion)
	      (list 'FAILURE failure-condition)
	
	   ;; at least one failed child and one recusion then recursive-failure
	   (list 'REC-FAILURE failure-condition)))))

    (setf (InternalNode-label and-node) (first label-and-cond))
    (setf (InternalNode-failure-condition and-node) (second label-and-cond))
    label-and-cond))



;;---------
;; label-or
;; Use label chart to determine node's label and failure conditions
;;   based upon it's children.
(defun label-or (or-node childrens-labels)
  (cond 
   ;; if already labeled, done
   ((InternalNode-label or-node) nil)

   ;; if any child success, label node 'SUCCESS
   ((any-child childrens-labels 'SUCCESS)
    (setf (InternalNode-label or-node) 'SUCCESS))

   ;; otherwise if any child labeled unknown, label node 'UNKNOWN
   ((any-child childrens-labels 'UNKNOWN)
    (setf (InternalNode-label or-node) 'UNKNOWN))

   ;; otherwise if any child recursive-failure, label node recursive-failure
   ;; and set failure condition to AND of childrens conditions
   ((any-child childrens-labels 'REC-FAILURE)
    (setf (InternalNode-label or-node) 'REC-FAILURE)
    (setf (InternalNode-failure-condition or-node)
	  (join-failure-conditions childrens-labels 'AND)))

   ;; otherwise if any child failure, label node FAILURE
   ((any-child childrens-labels 'FAILURE)
    (setf (InternalNode-label or-node) 'FAILURE)
    (setf (InternalNode-failure-condition or-node)
	  (join-failure-conditions childrens-labels 'AND)))

   (T "error"))

  (list (InternalNode-label or-node)
	(InternalNode-failure-condition or-node)))


;;----------
;; label-not
;; Use label chart to determine node's label and failure conditions
;;   based upon it's children.
(defun label-not (not-node childrens-labels)
  (cond
   ;; if already labeled, done
   ((InternalNode-label not-node) nil)

   ;; there should be only one child
   ((eq (length childrens-labels) 1)
    (let
     ((child (first childrens-labels)))
     (cond
      
      ;; if child success => failure (t)
      ((eq (first child) 'SUCCESS)
       (setf (InternalNode-label not-node) 'FAILURE)
       (setf (InternalNode-failure-condition not-node) t))


      ;; if child unknown => unknown
      ((eq (first child) 'UNKNOWN)
       (setf (InternalNode-label not-node) 'UNKNOWN))


      ;; if child failure (k1) => failure (not k1)
      ((eq (first child) 'FAILURE)
       ;; determine new failure condition: (not k1)
       (setf (InternalNode-failure-condition not-node)
	     (simplify-not (second child)))
       ;; if new failure condition == nil, then node is success [failure (nil)]
       (if (null (InternalNode-failure-condition not-node))
	   (setf (InternalNode-label not-node) 'SUCCESS)
	 ;; otherwise it is an ordinary failure node [failure (not k1)]
	 (setf (InternalNode-label not-node) 'FAILURE)))


      ;; if child recursive-failure (k1) => recursive-failure (not k1)
      ((eq (first child) 'REC-FAILURE)
       ;; determine new failure condition: (not k1)
       (setf (InternalNode-failure-condition not-node)
	     (simplify-not (second child)))

       ;; if new failure condition == nil, then node is unknown 
       (if (null (InternalNode-failure-condition not-node))
	   (setf (InternalNode-label not-node) 'UNKNOWN)
	 ;; otherwise it is an ordinary REC-FAILURE node
	 (setf (InternalNode-label not-node) 'REC-FAILURE)))

      
      (T "error"))))
       

   (T "error"))

  (list (InternalNode-label not-node)
	(InternalNode-failure-condition not-node)))
		   

;;-------------
;; label-FORALL
;; Use label chart to determine node's label and failure conditions
;;   based upon it's children.
(defun label-FORALL (forall-node childrens-labels)

  (cond
   ;; if already labeled, done
   ((InternalNode-label forall-node) nil)

   ;; there should be only two children
   ((eq (length childrens-labels) 2)
    
    (let* ((generator (first childrens-labels))
	  (F-expr (second childrens-labels))
	  (vars (find-variables (make-all-goal-consts-into-vars `(,generator ,F-expr)))))
      
      (cond

       ;; if all of the variables in the forall statement aren't in the params list
       ;; for the FORALL statement, then some variables which are existentially quantified
       ;; are present in the universally quantified expression.  This results in a complex
       ;; failure conditional which requires the generators for the existentially quantified
       ;; variables which are present.  This is not computed in this version of STATIC.
       
       ((not (is-subset vars (internalNode-params forall-node)))
	 (setf (InternalNode-label forall-node) 'UNKNOWN))
	
;;  otherwise....	 
       
;;        if F-expr always succeeds, so does FORALL expression
;;        if ~G always succeeds, so does FORALL expression
       ((any-child childrens-labels 'SUCCESS)
         (setf (InternalNode-label forall-node) 'SUCCESS))

;;        otherwise if either expression is UNKNOWN, FORALL is UNKNOWN 
;;        NOTE: This is done AFTER checking to see if either is success!
       ((any-child childrens-labels 'UNKNOWN)
	(setf (InternalNode-label forall-node) 'UNKNOWN))

;;        both ~G and f-exp must be failure.
;;        So forall expression will fail only when both of them fail.

       (T 
        (setf (InternalNode-label forall-node) 'FAILURE)
        (setf (InternalNode-failure-condition forall-node)
              (rob-simplify `(AND ,(second generator) ,(second F-expr)) 'no-sub))))))

;; other than two children 
   (T "error"))

  ;; return value
  (list (InternalNode-label forall-node)
	(InternalNode-failure-condition forall-node)))

   


;;-------------
;; simplify-not
;; Since the ebl simplifier does not remove double negation.  When
;; negating an expression first check to see if the expression is 
;; already negated, if so strip the first negation to effectively 
;; double negate.
(defun simplify-not (exp)
  (if (eq (first exp) '~)      ;; expression already negated, 
      (rob-simplify (second exp) 'no-sub)  ;; then remove the negation.
    (rob-simplify (negate-sentence exp) 'no-sub))) ;; otherwise negate the expression.
;;    (rob-simplify (list '~ exp) 'no-sub))) ;; otherwise negate the expression.


	   
;;----------------------
;; join-failure-conditions
;; collect the non-nil failure conditions and if there is more than
;; one, join them together with the keyword join.  If there is only
;; one, return it, otherwise return nil.
;; Note: children is a list of (failure-label failure-condition).
(defun join-failure-conditions (children join)
  (let ((conditions 
	 (iter:iterate
	  (iter:for child iter:in children)
	  (if (eq (first child) 'FAILURE)            ;; pure failure
	      (iter:collect (second child)))
	  (if (eq (first child) 'REC-FAILURE)  ;; recursive failure
	      (iter:collect (second child))))))
    (cond
     ((eq (length conditions) 1)
      (first conditions))

     ((> (length conditions) 1)
      (first (rob-simplify (cons join conditions)))))))   ; ex: (AND cond)



;;--------------------
;; any-child-recursive
;; Returns true after the first recursively labeled child (nil otherwise).
(defun any-child-recursive (children)
  (if (null children)
      nil
    (if (or (eq (first (first children)) 'REC-FAILURE)
	     (eq (first (first children)) 'UNKNOWN))
	T
      (any-child-recursive (rest children)))))


;;------------------
;; any-child
(defun any-child (children keyword)
  (cond
   ((null children)
    nil)
   ((eq (first (first children)) keyword)
    T)
   (T (any-child (rest children) keyword))))


;;--------------
;; Label-Literal
;; If the literal is not already labeled, For each operator which 
;; is a child of the literal, call Label_Operator. If there exists
;; an op which is lableled success label lit success.  Else if 
;; there exists an operator which is unknown lable lit unknown.
;; Else label lit failure and set lit's failure condition to conjunction
;; of operators failure conditions, AND the negation of itself
;; ex:  (and (not lit) (op1 failure) (op2 failure))
;;
;;MODIFICATION (1/8/93) by spiger@wolf.cs
;; The literal label also possibly contains some binding constraints for the literal.
;; These arise from creating the PSG using constants roots of the PSG.  This results
;;  in a tree to achieve the root of the PSG considering the PSG root literal to be
;;  an atomic expression consisting of constants.  Later these constants are replaced by
;;  variables of the same name.  But the thing which arises it that if the PSG was actually
;;  constructed using variables in the root literals one could have the variables in the 
;;  root literals be bound to constants.  Since static doesn't allow this,
;;  it keeps track of what bindings it don't allow as a result of this.  These are 
;;  put in the bad-bindings field of each literal node.  These bindings need to propagated 
;;  up the PSG in the failure conditional since they represent a limitation of what the
;;  root literal variables can be bound to.  At a literal node the bad-bindings list is
;;  just like the failure condition of the operators which could have worked if any of the
;;  bindings had been allowed.  So the failure condition of a literal is ANDed with a
;;  conjunction of (Not-equal bindingx) for each binding pair in bad-bindings.
;;
(defun label-literal (lit)

  ;; if the literal is already labeled return label and failure-condition
  (if (Literal-label lit)
      (convert-literal-label lit)

    ;; otherwise, look at operators
    (let*
	((operator-labels
	  (iter:iterate
	   (iter:for op iter:in (Literal-operators lit))
	   (iter:collect (label-operator op))))
         (operator-labels (if (literal-bad-bindings lit) 
                              (cons `(FAILURE (AND ,@(blist-to-noteqlist
                                                     (literal-bad-bindings lit)))) 
                                     operator-labels)
                              operator-labels))
	 (label-and-cond
	  (cond

	   ;; if any child is labeled successs, label literal success
	   ((any-child operator-labels 'SUCCESS)
	    (list 'SUCCESS))

	   ;; otherwise if any child labeled recursively
	   ((any-child operator-labels 'UNKNOWN)
	    (list 'UNKNOWN))

	   ;; otherwise if any child labeled recursive failure, 
	   ;; label the node recursive failure and AND together all failure
	   ;; conditions
	   ((any-child operator-labels 'REC-FAILURE)
	    (list 
	     'REC-FAILURE 
          (join-failure-conditions operator-labels 'AND)))


	   ;; otherwise literal is labeled failure and condition is 
	   ;; AND of operators failure conditions.
	   (T
	    (list 
	     'FAILURE 
          (join-failure-conditions operator-labels 'AND))))))

      ;; set new values and return new values

;; Notes: 
;;  It is desired not to propagate the failure conditions of shared
;;  literals up the tree.   This is probably because the rules are
;;  trying to make a choice between operators.  In other words, if more
;;  than one operator has the same expression in its failure
;;  condition, the expression isn't very useful for choosing between
;;  the operators.
;;    
;;  So in labeling and proprogating failure conditions up the tree,
;;    if a literal is completely shared, its failure condition is set
;;    to nil and is not propagated up the tree.
;; 

      (setf (Literal-label lit) (first label-and-cond))

      (if (Literal-completely-shared lit)                   ;;if true
          (setf (Literal-failure-condition lit) 'nil)
          (setf (Literal-failure-condition lit) (second label-and-cond)))
      (convert-literal-label lit))))


;;------------
;; add-literal
;; Adds the negation of the literal as a failure condition.  The negation
;; is added to the list of labels.
(defun add-literal (labels literal)
  (cons 
   (list 'FAILURE (simplify-not literal))
   labels))

  
;;----------------------
;; convert-literal-label
;; based upon the literal label return the logic label.  Also set the 
;; the failure-condition field (if it is not nil) and return the 
;; failure-condition field.
(defun convert-literal-label (lit)
  (let 
      ((label (Literal-label lit)))

    (cond
     ;; success labels
     ((or
       (eq label 'S-HOLDS)       ;; literal holds in current state.
       (eq label 'P-HOLDS))      ;; predicate always holds
      (setq label 'SUCCESS))

     ;;failure labels
     ((eq label 'UNACHIEVABLE)     ;; no operators achieve literal
      (setf (Literal-failure-condition lit) 
	    (simplify-not (Literal-name lit)))
      (setq label 'FAILURE))

     ((eq label 'GS-CYCLE)         ;; goal-stack cycle
      (setf (Literal-failure-condition lit) T)
      (setq label 'REC-FAILURE))

     ;; unknown labels
     ;; nil labels correspond to tree not being fully expanded
     ;;   (*limit* < depth).
     ((null label)
      (setq label 'UNKNOWN)))

    ;; everything else remains itself 
    ;; (ie, UNKNOWN => UNKNOWN, REC-FAILURE => REC-FAILURE ...)
                

    (list label
          
;;;failure-condition is not simply (literal-failure-condition lit) anymore
;;; because the negation of the literal needs to be ANDed to the failure
;;  condition when it is passed up.  This will need to be done in the
;;  cases where the label is FAILURE or REC-FAILURE
;;  But the literal should not be ANDEd to the failure conditional
;;  in the special cases of (1) The literal's label is REC-FAILURE because
;;  it is a GS-CYCLE.   In this case only T should be returned.
;;  (2)  The literal's failure conditional is nil because it is a 
;;;  completely shared node.  This this case, only nil should be returned.


          (if
           (AND (not (is-open-world-pred (literal-name lit)))
                (not (literal-completely-shared lit))
                (OR (eq label 'FAILURE)
                    (AND (eq label 'REC-FAILURE)
                         (not (eq (literal-label lit) 'GS-CYCLE)))))
           (rob-simplify `(AND ,(negate-exp (literal-name lit))
                 ,(literal-failure-condition lit)) 'no-sub)


           (literal-failure-condition lit)))))

        

